<?php
/**
 * Holds abstract functionality to extend to different stores.
 *
 * @package BuddyBossApp\InAppPurchases
 */

namespace BuddyBossApp\InAppPurchases;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\Tools\Logger;
use WP_Error as WP_Error;

/**
 * Abstract class for stores.
 */
abstract class StoreAbstract {

	/**
	 * Integration type.
	 *
	 * @var bool $type.
	 */
	protected $type = false;

	/**
	 * Integration label.
	 *
	 * @var bool $label
	 */
	protected $label = false;

	/**
	 * Store product types.
	 *
	 * @var array $store_product_types
	 */
	public $store_product_types = array();

	/**
	 * StoreAbstract constructor.
	 */
	private function __construct() {
		// ... leave empty, see Singleton below
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return void
	 */
	public static function instance() {
	}

	/**
	 * Integration label
	 *
	 * @return string
	 */
	public function get_label() {
		return $this->label;
	}

	/**
	 * Integration type
	 *
	 * @return string
	 */
	public function get_type() {
		return $this->type;
	}

	/**
	 * Function to be overridden in sub-class
	 *
	 * @param string $integration_type    Integration type.
	 * @param string $integration_label   Integration label.
	 * @param array  $store_product_types Store product types.
	 */
	public function set_up( $integration_type, $integration_label, $store_product_types ) {

		$this->type                = $integration_type;
		$this->label               = $integration_label;
		$this->store_product_types = $store_product_types;
		$this->hooks();
	}

	/**
	 * Function to add actions/filters
	 */
	private function hooks() {
		add_filter( "bbapp_iap_get_registered_iap", array( $this, 'register_iap' ) );
	}

	/**
	 * Register current instance iap type for easy access using the filter.
	 *
	 * @param array $registered_iap Registered IAP product.
	 *
	 * @return mixed
	 */
	public function register_iap( $registered_iap ) {
		$registered_iap[ $this->type ] = $this->label;

		return $registered_iap;
	}

	/**
	 * Return IAP Product Type.
	 *
	 * @param string $product_type Product type.
	 *
	 * @return bool|mixed
	 */
	public function get_product_type( $product_type ) {
		if ( isset( $this->store_product_types[ $product_type ] ) ) {
			return $this->store_product_types[ $product_type ];
		}

		return false;
	}

	/**
	 * Renders the product settings.
	 *
	 * @param string $integration Integration name.
	 * @param array  $item        Item data.
	 */
	abstract public function render_product_settings( $integration, $item );

	/**
	 * Save the product settings.
	 *
	 * @param string $integration Integration name.
	 * @param array  $item        Item data.
	 */
	abstract public function save_product_settings( $integration, $item );

	/**
	 * Quick access function for bbapp_iap_get_products
	 *
	 * @param array $args Array of argumetns.
	 *
	 * @return array
	 */
	public function get_products( $args ) {
		return bbapp_iap_get_products( $args );
	}

	/**
	 * Quick access function for bbapp_iap_get_product.
	 *
	 * @param int $product_id Product id.
	 *
	 * @return array|bool|\direct
	 */
	public function get_product( $product_id ) {
		return bbapp_iap_get_product( $product_id );
	}

	/**
	 * Helper function default appending iap_type
	 *
	 * @param array $product Product data.
	 *
	 * @return WP_Error|array
	 * @uses bbapp_iap_create_product
	 */
	public function create_product( $product ) {
		$default = array(
			'iap_type' => $this->type,
		);
		$product = array_merge( $default, $product );

		return bbapp_iap_create_product( $product );
	}

	/**
	 * Helper function to update product.
	 *
	 * @param int   $order_id Order id.
	 * @param array $data     Purchase data.
	 *
	 * @return bool
	 */
	public function update_product( $order_id, $data ) {
		return bbapp_iap_update_product( $order_id, $data );
	}

	/**
	 * Should be over-ride by iap type.
	 *
	 * @param array $data Purchase data.
	 *
	 * @return array
	 * @uses StoreAbstract::_process_payment()
	 */
	abstract public function process_payment( $data );

	/**
	 * Process Payment Token.
	 *
	 * @param array $data Purchase data.
	 *
	 * @return array
	 */
	public function _process_payment( $data ) { // phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore
		if ( IAP_LOG ) {
			Logger::instance()->add( 'iap_log', 'BuddyBossApp\InAppPurchases\Iap->_process_payment()' );
		}

		/**
		 * Data Variable Structure.
		 * [order_id] << Unique identifier for order.
		 * [bbapp_product_id] << Unique identifier for BuddyBossAppProduct stored in db.
		 * [iap_receipt_token] << InAppPurchase token generated on front-end(mobile)
		 * [store_product_id] << Store product id for ios or android
		 * [store_product_type] << Type of Store Product eg. consumable, non_consumable.
		 * [bbapp_product_type] << Type of Product type eg. free,paid.
		 * [integration_type] << Integration type eg. learndash-course, memberpress, woo-membership,
		 */
		$default = array(
			'test_mode'          => false,
			'iap_receipt_token'  => null,
			'store_product_id'   => null,
			'store_product_type' => null,
			'bbapp_product_type' => null,
			'integration_types'  => null,
		);
		$data    = array_merge( $default, $data );

		/**
		 * Test mode will help to test order flow without actual validation process.
		 * This is useful to test integration without doing actual payment.
		 */
		if ( ! empty( $data['test_mode'] ) ) {
			$transaction_data = array(
				'transaction_date'      => gmdate( 'Y-m-d H:i:s' ),
				'transaction_date_ms'   => strtotime( gmdate( 'Y-m-d H:i:s' ) ) * 1000,
				'parent_transaction_id' => sha1( time() ),
				'transaction_id'        => sha1( time() ),
				'data'                  => $data,
				'expire_at'             => false, // should be given if product is recurring.
			);

			// NOTE : Very.
			if ( in_array( $data['store_product_type'], array( 'auto_renewable' ), true ) ) {
				$gmtime                        = strtotime( gmdate( 'Y-m-d H:i:s' ) );
				$transaction_data['expire_at'] = gmdate( 'Y-m-d H:i:s', strtotime( '+5 minutes', $gmtime ) );
			}

			$transaction_data['data']['sandbox']             = true;
			$transaction_data['data']['transaction_history'] = __( 'Generated using test transaction mode.', 'buddyboss-app' );
			$transaction_data['data']['test_mode']           = 1;

			return $transaction_data;
		} else {
			// Trim the values to make sure no whitespace are there.
			if ( isset( $data['store_product_id'] ) ) {
				$data['store_product_id'] = trim( $data['store_product_id'] );
			}

			if ( isset( $data['store_product_type'] ) ) {
				$data['store_product_type'] = trim( $data['store_product_type'] );
			}

			if ( isset( $data['bbapp_product_type'] ) ) {
				$data['bbapp_product_type'] = trim( $data['bbapp_product_type'] );
			}

			if ( isset( $data['integration_types'] ) ) {
				$data['integration_types'] = trim( $data['integration_types'] );
			}

			$data['is_store_kit_2'] = ! empty( $data['is_store_kit_2'] );
			$data['purchase_id']    = ! empty( $data['purchase_id'] ) ? $data['purchase_id'] : '';
			$data['order_id']       = ! empty( $data['order_id'] ) ? $data['order_id'] : '';

			if ( isset( $data['bbapp_product_type'] ) && 'free' === $data['bbapp_product_type'] ) {
				$transaction_data                                = array(
					'transaction_date'      => gmdate( 'Y-m-d H:i:s' ),
					'transaction_date_ms'   => strtotime( gmdate( 'Y-m-d H:i:s' ) ) * 1000,
					'parent_transaction_id' => sha1( time() ),
					'transaction_id'        => sha1( time() ),
					'data'                  => $data,
					'expire_at'             => false, // should be given if product is recurring.
				);
				$transaction_data['data']['transaction_history'] = __( 'Generated transaction for free product.', 'buddyboss-app' );

				return $transaction_data;
			} else {
				$data = $this->process_payment( $data );
			}
		}

		return $data;
	}

	/**
	 * Checks if transaction id is already present in db. Return orders..
	 *
	 * @param string $origin_transaction_id Transaction id.
	 *
	 * @return array|bool|null|object|void
	 */
	public function do_transaction_exists( $origin_transaction_id ) {
		global $wpdb;

		$table                      = bbapp_iap()->get_global_dbprefix() . 'bbapp_iap_ordermeta';
		$store_product_type_require = $wpdb->prepare( "(SELECT count(*) FROM `{$table}` WHERE meta_key = '_device_platform' AND meta_value=%s AND order_id=meta.order_id)", $this->type ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$results                    = $wpdb->get_results( $wpdb->prepare( "SELECT order_id FROM `{$table}` as meta WHERE meta_key='_parent_transaction_id' AND meta_value=%s AND {$store_product_type_require}=1", $origin_transaction_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( empty( $results ) ) {
			return false;
		}

		$orders = array();

		foreach ( $results as $r => $v ) {
			$orders[] = Orders::instance()->get_by_id( $v->order_id );
		}

		return $orders;
	}

	/**
	 * Validates the order id.
	 *
	 * @param int $order_id Order id.
	 *
	 * @return bool
	 */
	public function _validate( $order_id ) { // phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore
		$order = Orders::instance()->get_by_id( $order_id );

		// if order doesn't exist.
		if ( empty( $order ) ) {
			return false;
		}

		// if no 'subscribed' order don't need to be validated as far as now.
		if ( ! in_array( $order->order_status, array( 'subscribed' ), true ) ) {
			return false;
		}

		// if expire_at is greater than now UTC than don't do anything.
		if ( strtotime( $order->expire_at ) > strtotime( gmdate( 'Y-m-d H:i:s' ) ) ) {
			Orders::instance()->update_order_next_validation_retry( $order->id, '+6 hours' );

			return false;
		}

		$iap_receipt_token     = Orders::instance()->get_meta( $order_id, '_iap_receipt_token' );
		$bbapp_product_id      = $order->bbapp_product_id;
		$store_product_type    = Orders::instance()->get_meta( $order_id, '_store_product_type' );
		$store_product_id      = Orders::instance()->get_meta( $order_id, '_store_product_id' );
		$test_mode             = Orders::instance()->get_meta( $order_id, '_transaction_data_test_mode' );
		$parent_transaction_id = Orders::instance()->get_meta( $order_id, '_parent_transaction_id' );
		$is_production         = '1' !== (string) Orders::instance()->get_meta( $order->id, '_transaction_data_sandbox' );
		$integration_types     = maybe_unserialize( $order->integration_types );

		foreach ( $integration_types as $integration_type ) {
			$integration = bbapp_iap()->integration[ $integration_type ];

			if ( empty( $iap_receipt_token ) || empty( $bbapp_product_id ) || empty( $store_product_type ) || empty( $store_product_id ) ) {
				Orders::instance()->update_order_next_validation_retry( $order->id, '+24 hours' );
				Orders::instance()->add_history( $order->id, 'iap-error', __( 'Error renewing order data found invalid.', 'buddyboss-app' ) );

				return false;
			}

			if ( ! $integration ) {
				Orders::instance()->update_order_next_validation_retry( $order->id, '+24 hours' );
				Orders::instance()->add_history( $order->id, 'iap-error', __( 'Error renewing order integration not present.', 'buddyboss-app' ) );

				return false;
			}

			$args = array(
				'order_id'              => $order_id,
				'iap_receipt_token'     => $iap_receipt_token,
				'bbapp_product_id'      => $bbapp_product_id,
				'store_product_type'    => trim( $store_product_type ),
				'store_product_id'      => trim( $store_product_id ),
				'integration_type'      => trim( $integration_type ),
				'parent_transaction_id' => $parent_transaction_id,
				'is_production'         => $is_production,
			);

			// if it's a test mode than fake it & bail it now.
			if ( '1' === (string) $test_mode ) {
				// does fake validates & return fake transaction data.
				$validate = $this->fakeValidate( $args, $order );
			} else {
				// else do the real validation.

				/**
				 * Should Return
				 * [status] // will determine if order is expired or not.
				 * [transaction_date]
				 * [parent_transaction_id]
				 * [transaction_id]
				 * [data] // will be stored in order meta.
				 * [expire_at]
				 */

				$validate = $this->validate( $args );
			}

			$order_status = 'subscribed';
			$expire_at    = $order->expire_at;

			// CASE : When there is error or something.
			if ( is_wp_error( $validate ) || empty( $validate ) ) {
				Orders::instance()->update_order_next_validation_retry( $order->id, '+12 hours' );
				$error = $validate->get_error_message();
				/* translators: %s: Validate error. */
				Orders::instance()->add_history( $order->id, 'iap-error', sprintf( __( 'Error Completing Order : %s', 'buddyboss-app' ), $error ) );

				// NOTE : This case is when we get error from backend(something went wrong).
				Orders::instance()->add_history( $order->id, 'warning', __( 'Order has expired or something went wrong.', 'buddyboss-app' ) );
				// trigger the item integration action.
				$integration->_on_order_expired( $order );

				return false;
			}

			// CASE : When subscription is expired but store is still retrying to renew.
			if ( 'retrying' === $validate['status'] ) {
				Orders::instance()->add_history( $order->id, 'warning', __( 'Store is retrying for payment.', 'buddyboss-app' ) );
				Orders::instance()->update_order_next_validation_retry( $order->id, '+12 hours' );

				return false;
			}

			// CASE : When subscription is expired 100%.
			if ( 'expired' === $validate['status'] ) {
				$order_status = 'expired';
				$date_format  = get_option( 'date_format' ) . ' @ ' . get_option( 'time_format' );
				$expired_at   = gmdate( $date_format, strtotime( $order->expire_at ) ) . ' GMT';

				if ( isset( $validate['history'] ) ) {
					foreach ( $validate['history'] as $history ) {
						Orders::instance()->add_history( $order->id, 'warning', $history );
					}
				}

				/* translators: %s: Expired at time. */
				Orders::instance()->add_history( $order->id, 'warning', sprintf( __( 'Order expired at %s', 'buddyboss-app' ), $expired_at ) );

				// trigger the item integration action.
				$integration->_on_order_expired( $order );
			}

			// CASE : When subscription is valid/active.
			if ( 'subscribed' === $validate['status'] ) {
				$order_status = 'subscribed';

				// Update meta information.
				if ( isset( $validate['transaction_id'] ) ) {
					Orders::instance()->update_meta( $order->id, '_transaction_id', $validate['transaction_id'] );
				}

				if ( isset( $validate['parent_transaction_id'] ) ) {
					Orders::instance()->update_meta( $order->id, '_parent_transaction_id', $validate['parent_transaction_id'] );
				}

				if ( isset( $validate['transaction_date'] ) ) {
					Orders::instance()->update_meta( $order->id, '_transaction_date', $validate['transaction_date'] );
				}

				// Store extras data from processPayment.
				foreach ( $validate['data'] as $k => $v ) {
					Orders::instance()->update_meta( $order->id, "_transaction_data_{$k}", $v );
				}

				Orders::instance()->add_history( $order->id, 'info', __( 'Order has been renewed successfully.', 'buddyboss-app' ) );

				// Set next check time.
				$expire_at = $validate['expire_at'];

				if ( '1' === (string) Orders::instance()->get_meta( $order->id, '_transaction_data_sandbox' ) ) {
					// next check +1 minute from expire_at only for sandbox.
					Orders::instance()->update_order_next_validation_retry(
						$order->id,
						'+1 minute',
						date( 'Y-m-d H:i:s', strtotime( $expire_at ) ) // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
					);
				} else {
					Orders::instance()->update_order_next_validation_retry( $order->id, '+1 day', gmdate( 'Y-m-d H:i:s', strtotime( $expire_at ) ) ); // next check +1 day from expire_at.
				}

				// Make sure item is activated.
				$integration->_on_order_activate( $order );
			}

			// Update expire at anyway.
			Orders::instance()->update_order(
				$order->id,
				array(
					'expire_at'    => $expire_at,
					'order_status' => $order_status,
				)
			);
		}

		return true;
	}

	/**
	 * Validates order payment status
	 *
	 * @param array $data Purchase data.
	 *
	 * @return mixed
	 */
	public function validate( $data ) {
		return new WP_Error( 'no_validation_implemented', __( 'No correct validation method has been implemented in IAP type.', 'buddyboss-app' ) );
	}

	/**
	 * Fake - Validates order payment status
	 *
	 * @param array  $data  Purchase data.
	 * @param object $order Order data.
	 *
	 * @return mixed
	 */
	final public function fakeValidate( $data, $order ) { //phpcs:ignore WordPress.NamingConventions.ValidFunctionName.MethodNameInvalid
		$validate        = array(
			'status'                => 'subscribed',
			'transaction_date'      => gmdate( 'Y-m-d H:i:s' ),
			'parent_transaction_id' => sha1( time() ),
			'transaction_id'        => sha1( time() ),
			'data'                  => array(),
			'expire_at'             => '',
			'history'               => array(),
		);
		$renew_count_key = '_transaction_data_test_mode_renew_count';
		$renew_count     = (int) Orders::instance()->get_meta( $order->id, $renew_count_key );
		$renew_count ++;

		if ( $renew_count > 5 ) {
			$validate['status'] = 'expired';
		} else {
			Orders::instance()->update_meta( $order->id, $renew_count_key, $renew_count );
		}

		if ( in_array( $data['store_product_type'], array( 'auto_renewable' ), true ) ) {
			// Prepare next fake expire at time.
			$_time = strtotime( $validate['transaction_date'] );

			$validate['expire_at'] = gmdate( 'Y-m-d H:i:s', strtotime( '+5 minutes', $_time ) ); // add more five minutes to last expire.

		}

		return $validate;
	}
}
