<?php
/**
 * Holds functionality related IAP orders.
 *
 * @package BuddyBossApp\InAppPurchases
 */

namespace BuddyBossApp\InAppPurchases;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\Admin\InAppPurchases\Helpers;
use BuddyBossApp\Admin\InAppPurchases\ProductHelper;
use BuddyBossApp\RestErrors;
use BuddyBossApp\Tools\Logger;
use WP_Error as WP_Error;
use WP_User;

/**
 * IAO orders class.
 */
final class Orders {

	/**
	 * Class instance.
	 *
	 * @var null $instance
	 */
	private static $instance = null;

	/**
	 * Order table name.
	 *
	 * @var string $table_name
	 */
	public $table_name = 'bbapp_iap_orders';

	/**
	 * Order meta table name.
	 *
	 * @var string $table_meta_name
	 */
	public $table_meta_name = 'bbapp_iap_ordermeta';

	/**
	 * Orders constructor.
	 */
	private function __construct() {
		// ... leave empty, see Singleton below.
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return Orders
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->set_table_name();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Set database table name.
	 *
	 * @return void
	 */
	public function set_table_name() {
		global $wpdb;

		$prefix = $wpdb->prefix;

		if ( is_multisite() ) {
			switch_to_blog( 1 );
			$prefix = $wpdb->prefix;
			restore_current_blog();
		}

		$this->table_name      = $prefix . $this->table_name;
		$this->table_meta_name = $prefix . $this->table_meta_name;
	}

	/**
	 * All hooks here
	 *
	 * @return void
	 */
	public function hooks() {
		add_action( 'init', array( $this, 'handle_notification' ) );
		add_action( 'delete_user', array( $this, 'revoke_order_for_delete_user' ), 10, 1 );
		add_action( 'wp_login', array( $this, 'web_login_check_pending_orders_make_as_complete' ), 10, 2 );
		add_action( 'bp_rest_signup_create_item', array( $this, 'bp_rest_signup_create_item_with_order_id' ), 10, 3 );
		add_action( 'bbapp_auth_rest_user_registration_after', array( $this, 'bbapp_auth_rest_user_registration_after_with_order_id' ), 10, 2 );
	}

	/**
	 * This function checks user pending orders as soon as possible when they login & assign it.
	 * This is useful when user has purchased the product when user is not logged in phase.
	 * We find user orders by matching their email to iap unassigned orders and then we
	 * change product status to complete.
	 *
	 * Reference - check order IAP without user authentication.
	 *
	 * @param string  $user_login User login.
	 * @param WP_User $user       User object.
	 *
	 * @note after 2.1.70 action changed from 'init' to 'wp_login'.
	 *
	 * @return void
	 */
	public function web_login_check_pending_orders_make_as_complete( $user_login, $user ) {
		if ( is_user_logged_in() ) {
			// Get order is which has custom order status.
			$order_ids = self::instance()->get_orders_ids_by_meta( '_custom_order_status', 'pending_complete' );

			if ( ! empty( $order_ids ) ) {
				$orders_obj = $this->get_orders(
					array(
						'id'           => $order_ids,
						'order_status' => 'pending',
						'user_email'   => $user->user_email,
					)
				);

				if ( ! empty( $orders_obj ) ) {
					foreach ( $orders_obj as $order ) {
						$_iap_receipt_token = $this->get_meta( $order->id, '_iap_receipt_token' );
						$order_args         = array(
							'user_id'           => get_current_user_id(),
							'order_id'          => $order->id,
							'iap_receipt_token' => $_iap_receipt_token,
						);
						$completed_order    = $this->completing_order( $order_args );

						if ( ! is_wp_error( $completed_order ) && in_array( $completed_order->order_status, array( 'subscribed', 'completed' ), true ) ) {
							// Update Order Register user order custom status.
							// @todo always execute this after running completeing_order to avoid infinite loop.
							$this->update_meta( $order->id, '_custom_order_status', $completed_order->order_status );
						} else {
							$iap_error = is_wp_error( $completed_order ) ? $completed_order->get_error_message() : __( 'Something went wrong.', 'buddyboss-app' );
							/* translators: %s Order completing error. */
							$this->add_history( $order->id, esc_html( 'iap-error' ), sprintf( '%s %s', __( 'Completing Error :', 'buddyboss-app' ), $iap_error ) );
							$this->update_meta( $order->id, '_custom_order_status', 'failed' );
						}
					}
				}
			}
		}
	}

	/**
	 * When create new user update order user email.
	 *
	 * @param object $signup   Signup object.
	 * @param object $response Response object.
	 * @param object $request  Request object.
	 */
	public function bp_rest_signup_create_item_with_order_id( $signup, $response, $request ) {
		$order_id = $request->get_param( 'iap_order_id' );

		if ( ! empty( $order_id ) ) {
			$this->update_order(
				$order_id,
				array(
					'user_email' => $signup->user_email,
				)
			);
		}
	}

	/**
	 * When create new user update order user email.
	 *
	 * @param int    $user_id User id.
	 * @param object $request Request object.
	 */
	public function bbapp_auth_rest_user_registration_after_with_order_id( $user_id, $request ) {
		$order_id = $request->get_param( 'iap_order_id' );

		if ( ! empty( $order_id ) ) {
			$email = $request->get_param( 'email' );
			$this->update_order(
				$order_id,
				array(
					'user_email' => $email,
				)
			);
		}
	}

	/**
	 * Revoke order when user delete.
	 *
	 * @param int $user_id User id.
	 */
	public function revoke_order_for_delete_user( $user_id ) {
		$order_args = array(
			'order_status' => 'completed',
			'user_id'      => $user_id,
		);

		$orders_obj = $this->get_orders( $order_args );

		if ( ! empty( $orders_obj ) ) {
			foreach ( $orders_obj as $order ) {
				/* translators: %s: user id. */
				$success = self::instance()->cancel_order( $order->id, sprintf( __( 'Order has been cancelled. Because this user #%s is deleted.', 'buddyboss-app' ), $user_id ) );

				if ( ! is_wp_error( $success ) ) {
					// Store cancel detail on meta.
					self::instance()->update_meta( $order->id, 'order_cancelled_manually', 1 );
					self::instance()->update_meta( $order->id, 'order_cancelled_user_id', $user_id );
				}
			}
		}
	}

	/**
	 * Creating the order.
	 * This Function is responsible for creating order.
	 * This function validates the all supported stores order receipt.
	 * This function keeps the order into pending state when product is purchased in none user authentication mode.
	 *
	 * @param array  $order_args Order arguments.
	 * @param object $iap        IAP object.
	 *
	 * @return array|mixed|WP_Error
	 */
	public function creating_order( $order_args, $iap ) {
		$bbapp_product_id       = $order_args['bbapp_product_id'];
		$order_store_product_id = $order_args['store_product_id'];
		$device_platform        = $order_args['device_platform'];
		$blog_id                = $order_args['blog_id'];
		$iap_receipt_token      = $order_args['iap_receipt_token'];
		$user_id                = $order_args['user_id'];
		$user_email             = $order_args['user_email'];
		$test_mode              = $order_args['test_mode'];
		$is_store_kit_2         = $order_args['is_store_kit_2'];
		$purchase_id            = $order_args['purchase_id'];
		$bbapp_product          = $iap->get_product( $bbapp_product_id );
		$store_data             = maybe_unserialize( $bbapp_product['store_data'] );
		$store_product_ids      = $store_data['store_product_ids'];
		$store_product_id       = ! empty( $store_product_ids[ $device_platform ] ) ? $store_product_ids[ $device_platform ] : '';
		$store_product_types    = $store_data['store_product_types'];
		$store_product_type     = $store_product_types[ $device_platform ];
		$bbapp_product_type     = $store_data['bbapp_product_type'];
		$is_recurring           = Helpers::is_recurring_type( $store_product_type );

		// Check if iap product id is not available OR Check if iap product is not trashed.
		if ( empty( $bbapp_product ) || ( isset( $bbapp_product['status'] ) && 'trash' === $bbapp_product['status'] ) ) {
			return new WP_Error( 'purchase_not_available', __( 'Product not found.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		// Check if order has already exists for this product id.
		$get_active_order = ProductHelper::get_active_order( $bbapp_product, $user_id );

		if ( false !== $get_active_order ) {
			return $get_active_order;
		}

		if ( ! empty( $order_store_product_id ) && ! empty( $store_product_id ) && $store_product_id !== $order_store_product_id && 'ios' === $device_platform ) {
			$store_product_id = $order_store_product_id;
		}

		$integration_data = maybe_unserialize( $bbapp_product['integration_data'] );
		$misc_settings    = maybe_unserialize( $bbapp_product['misc_settings'] );

		// NOTE : For multi-site, we are storing data as [:blog-id][some_key_index]. Eg : integration_type or misc_settings.
		if ( bbapp()->is_network_activated() ) {
			$misc_settings    = $misc_settings[ $blog_id ];
			$integration_data = $integration_data[ $blog_id ];
		}

		$integration_types       = array();
		$item_ids                = array();
		$active_integration_slug = $misc_settings['integration_type'];
		$integration_slugs       = array_keys( $integration_data );
		$integration             = false;

		foreach ( $integration_slugs as $integration_slug ) {
			if ( $active_integration_slug === $integration_slug ) {
				if ( isset( bbapp_iap()->integrations[ $integration_slug ] ) ) {
					$integration_details = bbapp_iap()->integrations[ $integration_slug ];
					$integration_type    = $integration_details['type'];
					$integration         = Helpers::get_integration_instance( $integration_slug );

					if ( bbapp()->is_network_activated() ) {
						$integration_types[ $blog_id ] = $integration_type;
						$item_ids[ $blog_id ]          = array( $integration_type => maybe_serialize( $integration_data[ $integration_slug ] ) );
					} else {
						$integration_types[]           = $integration_type;
						$item_ids[ $integration_type ] = maybe_serialize( $integration_data[ $integration_slug ] );
					}
				}
			}
		}

		if ( ! $integration ) {
			return new WP_Error( 'purchase_not_available', __( "We don't have any supported integration available.", 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		if ( ! empty( $user_id ) ) {
			$userdata   = get_userdata( $user_id );
			$user_email = $userdata->user_email;
		}

		// Create a new order.
		$order = $this->create_order(
			array(
				'bbapp_product_id'  => $bbapp_product_id,
				'device_platform'   => $device_platform,
				'store_product_id'  => $store_product_id,
				'integration_types' => maybe_serialize( $integration_types ),
				'item_ids'          => maybe_serialize( $item_ids ),
				'is_recurring'      => $is_recurring,
				'user_id'           => $user_id,
				'user_email'        => $user_email,
				'blog_id'           => $blog_id,
			)
		);

		if ( empty( $order ) || is_wp_error( $order ) ) {
			$error = $order->get_error_message();

			return new WP_Error(
				'error_creating_order',
				/* translators: %s: Order creating error. */
				sprintf( __( 'There is an error while creating purchase :- %s.', 'buddyboss-app' ), $error ),
				array(
					'status' => 500,
					'more'   => 'simple purchase creation failed',
				)
			);
		}

		$this->update_meta( $order->id, '_iap_receipt_token', $iap_receipt_token ); // Update Order Basic Information.
		$this->update_meta( $order->id, '_device_platform', $device_platform ); // Eg : ios or android.
		$this->update_meta( $order->id, '_store_product_type', $store_product_type ); // Eg : auto-renewable or further.
		$this->update_meta( $order->id, '_bbapp_product_type', $bbapp_product_type ); // Eg : free-# or paid-#.
		$this->update_meta( $order->id, '_store_product_id', $store_product_id ); // Eg : com.example.ios-# or com.example.android-#.
		$this->update_meta( $order->id, '_integration_types', maybe_serialize( $integration_types ) ); // Eg : learndash-course and/or memberpress and/or wc-memberships.
		$this->update_meta( $order->id, '_transaction_data_test_mode', $test_mode ); // Eg: testmode enable or disable.
		$verify = $iap->_process_payment( // Process Token Validation Tests.
			array(
				'order_id'           => $order->id,
				'test_mode'          => $test_mode,
				'iap_receipt_token'  => $iap_receipt_token,
				'bbapp_product_id'   => $bbapp_product_id,
				'store_product_id'   => $store_product_id,
				'store_product_type' => $store_product_type,
				'bbapp_product_type' => $bbapp_product_type,
				'is_store_kit_2'     => $is_store_kit_2,
				'purchase_id'        => $purchase_id,
				'integration_types'  => maybe_serialize( $integration_types ),
				'item_ids'           => maybe_serialize( $item_ids ),
			)
		);

		if ( is_wp_error( $verify ) || empty( $verify ) ) {
			/* translators: %s: Purchase validate error message. */
			$error = sprintf( __( 'Error while validating purchase : %s', 'buddyboss-app' ), $verify->get_error_message() );
			$this->add_history( $order->id, 'iap-error', $error, true );
			/* translators: %s: Purchase validate error message. */

			$error_data = $verify->get_error_data();
			if ( ! empty( $error_data ) ) {
				$error = sprintf( '%1$s %2$s', esc_html__( 'Subscription Notice:', 'buddyboss-app' ), $verify->get_error_data() );
				$this->add_history( $order->id, 'iap-error', $error, true );
			}

			$bbap_error_msg = __( 'There is an error while processing purchase.', 'buddyboss-app' );

			if ( 'error_iap_sub_validation' === $verify->get_error_code() ) {
				$bbap_error_msg = $verify->get_error_message();
			}

			// Update status of order failed.
			$this->update_order(
				$order->id,
				array( 'order_status' => 'failed' )
			);

			return new WP_Error(
				'error_processing_order',
				$bbap_error_msg,
				array(
					'status' => 500,
					'more'   => 'verify-payment have issues, purchase ID is : ' . $order->id,
				)
			);
		}

		if ( empty( $user_id ) ) {
			// Update Order Non register user order custom status.
			$this->update_meta( $order->id, '_custom_order_status', 'pending_complete' );
		}

		// update meta information.
		$this->update_meta( $order->id, '_transaction_id', $verify['transaction_id'] );
		$this->update_meta( $order->id, '_parent_transaction_id', $verify['parent_transaction_id'] );

		if ( isset( $verify['transaction_date'] ) ) {
			$this->update_meta( $order->id, '_transaction_date', $verify['transaction_date'] );
			$this->update_meta( $order->id, '_transaction_date_ms', $verify['transaction_date_ms'] );
		} else {
			$this->update_meta( $order->id, '_transaction_date', gmdate( 'Y-m-d H:i:s' ) );
			$this->update_meta( $order->id, '_transaction_date_ms', strtotime( gmdate( 'Y-m-d H:i:s' ) ) * 1000 );
		}

		// Store extras data from processPayment.
		foreach ( $verify['data'] as $key => $value ) {
			$this->update_meta( $order->id, "_transaction_data_{$key}", $value );
		}

		$expire_at = null;

		if ( ! empty( $verify['expire_at'] ) ) {
			$expire_at = $verify['expire_at'];
		}

		// 1+ day from expire date. so we be sure renew process has correctly done.
		$next_order_validation = gmdate( 'Y-m-d H:i:s', strtotime( '+1 day', strtotime( $expire_at ) ) );

		// sandbox check should be instant so we will not keep safe margin.
		if ( isset( $verify['data']['sandbox'] ) && $verify['data']['sandbox'] ) {
			$next_order_validation = gmdate( 'Y-m-d H:i:s', strtotime( '+1 minute', strtotime( $expire_at ) ) );
		}

		$this->update_meta( $order->id, '_next_order_check', $next_order_validation );

		// Add expire-at when available. usually it's available on recurring order type.
		if ( null !== $expire_at ) {
			$this->update_order(
				$order->id,
				array(
					'expire_at' => gmdate( 'Y-m-d H:i:s', strtotime( $expire_at ) ),
				)
			);
		}

		return $order;
	}

	/**
	 * Process to complete the order with payment.
	 * This includes steps where users get the permissions of integrated integration in product.
	 *
	 * @param array $order_args Order arguments.
	 *
	 * @return null|array|object|void|WP_Error
	 */
	public function completing_order( $order_args ) {
		$order_id          = $order_args['order_id'];
		$iap_receipt_token = $order_args['iap_receipt_token'];
		$user_id           = $order_args['user_id'];

		/**
		 * Check user logged in permissions.
		 */
		if ( ! is_user_logged_in() ) {
			return RestErrors::instance()->user_not_logged_in();
		}

		// Get order.
		$order = self::instance()->get_by_id( $order_id );

		if ( is_wp_error( $order ) ) {
			return $order;
		}

		if ( ! isset( bbapp_iap()->iap[ $order->device_platform ] ) ) {
			/* translators: %s: Device platform.  */
			return new WP_Error( 'invalid_device_platform', sprintf( __( 'InAppPurchase for Platform(%s) you requested is currently not available.', 'buddyboss-app' ), $order->device_platform ), array( 'status' => 404 ) );
		}

		if ( empty( $order->bbapp_product_id ) ) {
			return new WP_Error( 'iap_product_id_missing', __( "InAppPurchase bbapp_product_id shouldn't be empty.", 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		/**
		 * Store abstract data.
		 *
		 * @var $iap StoreAbstract
		 */
		// Order Basic Information.
		$_iap_receipt_token = $this->get_meta( $order->id, '_iap_receipt_token' );

		// Database token and provided token verify.
		if ( $iap_receipt_token !== $_iap_receipt_token ) {
			return new WP_Error( 'error_receipt_token', __( 'Error while purchase : Purchase receipt token mismatch.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		// if order status is not pending and order status is subscribed or completed return the error.
		if ( ! is_user_logged_in() && $user_id !== $order->user_id ) {
			return new WP_Error( 'error_order_status', __( 'Error while purchase: Purchase user mismatch.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		if ( empty( $order->user_id ) ) {
			// Get current user login data and update order data and meta data.
			$userdata = get_userdata( $user_id );

			// if order status is not pending and order status is subscribed or completed return the error.
			if ( $userdata->user_email !== $order->user_email ) {
				return new WP_Error( 'error_order_status', __( 'Error while purchase: Purchase user email mismatch.', 'buddyboss-app' ), array( 'status' => 500 ) );
			}

			$this->update_order(
				$order->id,
				array(
					'user_id'    => $user_id,
					'user_email' => $userdata->user_email,
				)
			);
		}

		// run order status hook on integration.
		$complete_order = $this->complete_order( $order->id );

		if ( is_wp_error( $complete_order ) ) {
			$error = $complete_order->get_error_message();
			/* translators: %s: Order completing error. */
			$this->add_history( $order->id, 'iap-error', sprintf( __( 'Error completing purchase : %s', 'buddyboss-app' ), $error ), $complete_order );

			return new WP_Error(
				'error_processing_order',
				$error,
				array(
					'status' => 500,
					'more'   => 'complete order had issues',
				)
			);
		}

		$this->add_history( $order->id, 'success', __( 'Purchase has been completed successfully.', 'buddyboss-app' ) );

		return $complete_order;
	}

	/**
	 * If user has already order return the order if not so return false.
	 *
	 * @param string $iap_receipt_token IAP receipt token.
	 * @param int    $user_id           User id.
	 * @param string $user_email        User email.
	 *
	 * @return false|mixed
	 */
	public function user_already_has_order( $iap_receipt_token, $user_id, $user_email = '' ) {

		if ( ! empty( $user_id ) && empty( $user_email ) ) {
			// Get current user login data and update order data and meta data.
			$userdata   = get_userdata( $user_id );
			$user_email = $userdata->user_email;
		}

		// Get order is which same receipt token.
		$order_ids = self::instance()->get_orders_ids_by_meta( '_iap_receipt_token', $iap_receipt_token );

		// Filter orders which is pending and user id is zero.
		$order_status = array(
			'subscribed',
			'completed',
		);

		if ( ! is_user_logged_in() ) {
			$order_status[] = 'pending';
		}

		$orders = $this->get_orders(
			array(
				'id'           => $order_ids,
				'order_status' => $order_status,
				'user_id'      => $user_id,
				'user_email'   => $user_email,
			)
		);

		// Pending order should be complete first.
		if ( isset( $orders ) && ! empty( $orders ) ) {
			if ( is_array( $orders ) ) {
				foreach ( $orders as $order ) {
					return $order;
				}
			}
		}

		return false;
	}

	/**
	 * Complete Order.
	 *
	 * @param int $order_id Order id.
	 *
	 * @return WP_Error | Object $order
	 */
	public function complete_order( $order_id ) {
		$orders = $this->get_orders(
			array(
				'id' => $order_id,
			)
		);

		if ( empty( $orders ) ) {
			return new WP_Error( 'error_completing_order', __( 'Error while completing purchase : Purchase not found.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		$order = $orders[0];

		/**
		 * Store data.
		 *
		 * @var $iap StoreAbstract
		 */
		$iap           = bbapp_iap()->iap[ $order->device_platform ];
		$bbapp_product = $iap->get_product( $order->bbapp_product_id );
		$expired       = false;

		if ( ! empty( $order->is_recurring ) && strtotime( $order->expire_at ) <= strtotime( gmdate( 'Y-m-d H:i:s' ) ) ) {
			$expired = true;
		}

		$user_id              = get_current_user_id();
		$same_group_order_ids = ProductHelper::get_group_active_order_id_except( $bbapp_product['iap_group'], $user_id, $order->id );

		// Cancel Active order of same group if exists.
		if ( ! empty( $bbapp_product['iap_group'] ) && ! empty( $same_group_order_ids ) ) {
			$group_active_order = $this->cancel_group_order( $bbapp_product, $order );
			$this->update_meta( $order->id, '_old_reference_order', $group_active_order );
			$this->update_meta( $group_active_order, '_new_reference_order', $order->id );
		}

		/**
		 * This function checks user pending orders as soon as possible when they login & assign it.
		 * This is useful when user has purchased the product when user is not logged in phase.
		 * We find user orders by matching their email to iap unassigned orders and then we
		 * change product status to complete.
		 *
		 * Reference - check order IAP without user authentication.
		 */
		$custom_order_status = $this->get_meta( $order->id, '_custom_order_status' );
		if ( 'pending_complete' === $custom_order_status ) {
			$expired = false;
		}

		$integration_types = maybe_unserialize( $order->integration_types ); // NOTE : For multi-site, we are storing data as [:blog-id][some_key_index]. Eg : integration_type or misc_settings.

		foreach ( $integration_types as $integration_type ) {
			$integration = bbapp_iap()->integration[ $integration_type ];

			if ( ! $integration ) {
				/* translators: %s: Integration type. */
				return new WP_Error( 'error_completing_order', sprintf( __( "Error while completing purchase : '{%s}' Integration not found.", 'buddyboss-app' ), $integration_type ), array( 'status' => 500 ) );
			}

			// NOTE : Triggers on order completed/expired hook on/from integration.
			if ( $expired ) {
				$integration->_on_order_expired( $order );
			} else {
				$integration->_on_order_completed( $order );
			}
		}

		$order_status = 'completed';

		if ( $order->is_recurring ) {
			$order_status = $expired ? 'expired' : 'subscribed';
		}

		// Update status of order to complete.
		$update_status = $this->update_order(
			$order->id,
			array(
				'order_status' => $order_status,
			)
		);

		if ( ! $update_status ) {
			return new WP_Error( 'error_completing_order', __( 'Error while completing purchase : Error while updating purchase status.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		$order->order_status = $order_status;

		return $order;
	}

	/**
	 * Cancel the order.
	 *
	 * @param int  $order_id Order id.
	 * @param bool $note     Add note on cancel order.
	 *
	 * @return WP_Error
	 */
	public function cancel_order( $order_id, $note = false ) {
		$orders = $this->get_orders(
			array(
				'id' => $order_id,
			)
		);

		if ( empty( $orders ) ) {
			return new WP_Error( 'error_canceling_order', __( 'Error while cancelling purchase : Purchase not found.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		$order = $orders[0];

		if ( ! in_array( $order->order_status, array( 'subscribed', 'completed' ), true ) ) {
			return new WP_Error( 'error_canceling_order', __( 'Error while cancelling purchase: Purchase not subscribed(active).', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		$integration_types = null;

		// Verify if it is serialized.
		if ( is_serialized( $order->integration_types ) ) {
			$integration_types = maybe_unserialize( $order->integration_types );
		}

		// Verify if it is unserialized correctly.
		if ( is_array( $integration_types ) ) {
			foreach ( $integration_types as $integration_type ) {
				$integration = false;

				if ( isset( bbapp_iap()->integration[ $integration_type ] ) ) {
					// Integration.
					$integration = bbapp_iap()->integration[ $integration_type ];
				}

				if ( ! $integration ) {
					/* translators: %s: Integration type. */
					return new WP_Error( 'error_canceling_order', sprintf( __( "Error while cancelling purchase : '{%s}' Integration not found.", 'buddyboss-app' ), $integration_type ), array( 'status' => 500 ) );
				}

				// trigger on order cancelled hook on integration.
				$integration->_on_order_cancelled( $order );
			}
		} else {
			$no_integration_found = __( 'No integration found.', 'buddyboss-app' );
			$this->add_history( $order_id, 'info', $no_integration_found );
		}

		$order_status = 'cancelled';

		// update status of order to cancel.
		$update_status = $this->update_order(
			$order->id,
			array(
				'order_status' => $order_status,
			)
		);

		if ( $note ) {
			$this->add_history( $order_id, 'info', $note );
		}

		if ( ! $update_status ) {
			return new WP_Error( 'error_canceling_order', __( 'Error while cancelling purchase : Error while updating purchase status.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		$order->order_status = $order_status;

		return $order;
	}

	/**
	 * Cancel group activeorder.
	 *
	 * @param object $new_product New Product.
	 * @param object $new_order   New order.
	 *
	 * @return bool|int|WP_Error
	 */
	public function cancel_group_order( $new_product, $new_order ) {
		$user_id         = get_current_user_id();
		$active_order_id = ProductHelper::get_group_active_order_id( $new_product['iap_group'], $user_id );

		if ( ! empty( $active_order_id ) ) {
			$success = $this->cancel_order( $active_order_id, sprintf( __( "Purchase has been cancelled due to same group product: %1\$s purchased, Purchase: <a target='_blank' href=%2\$s>%3\$d</a>", 'buddyboss-app' ), $new_product['name'], 'admin.php?page=bbapp-iap&action=view_order&order_id=' . $new_order->id, $new_order->id ) );

			if ( ! is_wp_error( $success ) ) {
				// Store cancel detail on meta.
				self::instance()->update_meta( $active_order_id, 'order_upgrade_downgrade', 1 );

				return $active_order_id;
			}
		}

		return false;
	}

	/**
	 * Creates the order in db.
	 *
	 * @param array $order Order data.
	 *
	 * @return WP_Error|array
	 */
	public function create_order( $order ) {
		global $wpdb;

		$default = array(
			'blog_id'           => get_current_blog_id(),
			'user_id'           => 0,
			'bbapp_product_id'  => 0,
			'device_platform'   => false,
			'order_status'      => 'pending',
			'integration_types' => false,
			'item_ids'          => '',
			'is_recurring'      => 0,
			'secondary_id'      => 0,
			'date_created'      => current_time( 'mysql', 1 ),
			'date_updated'      => current_time( 'mysql', 1 ),
		);
		$order   = array_merge( $default, $order );

		if ( ! $order['device_platform'] ) {
			return new WP_Error( 'missing_param', __( 'Please specify purchase type.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		if ( empty( $order['order_status'] ) ) {
			return new WP_Error( 'missing_param', __( 'Please specify purchase status.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		if ( empty( $order['integration_types'] ) ) {
			return new WP_Error( 'missing_param', __( 'Please specify integration type(s).', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		if ( empty( $order['item_ids'] ) ) {
			return new WP_Error( 'missing_param', __( 'Please specify item IDs.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		$create    = $wpdb->insert( $this->table_name, $order ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$get_order = false;

		if ( $create ) {
			$get_order = $this->get_orders( array( 'id' => $wpdb->insert_id ) );
		}

		if ( empty( $create ) || ! $get_order ) {
			return new WP_Error( 'error_creating', __( 'There is an error while creating the purchase.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		return $get_order[0];
	}

	/**
	 * Update order.
	 *
	 * @param int   $order_id Order id.
	 * @param array $data     Order data.
	 *
	 * @return false|int
	 */
	public function update_order( $order_id, $data ) {
		global $wpdb;

		$columns = array(
			'blog_id'           => 1,
			'user_id'           => 1,
			'device_platform'   => 1,
			'order_status'      => 1,
			'integration_types' => '',
			'item_ids'          => '',
			'secondary_id'      => 1,
			'date_created'      => 1,
			'date_updated'      => 1,
			'user_email'        => 1,
			'expire_at'         => 1,
		);

		$new_vals = array();

		foreach ( $data as $k => $v ) {
			if ( isset( $columns[ $k ] ) ) {
				$new_vals[ $k ] = $v;
			}
		}

		$update = $wpdb->update( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$this->table_name,
			$new_vals,
			array( 'id' => $order_id )
		);

		return ( false !== $update ) ? true : false;
	}

	/**
	 * Get orders.
	 *
	 * @param array $where provide option to get results by meta.
	 *
	 * @param array $args  provide options to orderby limit the results.
	 *
	 * @return array|bool|null|object
	 */
	public function get_orders( $where = array(), $args = array() ) {
		global $wpdb;

		$default_args = array(
			'device_platform' => 'any',
			'per_page'        => false,
			'current_page'    => 1,
			'orderby'         => 'id',
			'order'           => 'desc',
		);

		$args = array_merge( $default_args, $args );

		$columns = array(
			'id'                => false,
			'blog_id'           => false,
			'user_id'           => false,
			'bbapp_product_id'  => false,
			'device_platform'   => false,
			'order_status'      => false,
			'integration_types' => false,
			'item_ids'          => '',
			'secondary_id'      => false,
			'date_created'      => false,
			'date_updated'      => false,
			'user_email'        => false,
		);

		$where_clause = array();

		foreach ( $where as $column => $value ) {
			if ( ! isset( $columns[ $column ] ) ) {
				return false;
			}

			if ( ! empty( $value ) ) {
				if ( is_array( $value ) ) {
					$value_str      = "'" . implode( "','", $value ) . "'";
					$where_clause[] = "{$column} IN ({$value_str})";
				} else {
					$where_clause[] = $wpdb->prepare( "{$column}=%s", $value ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				}
			}
		}

		if ( empty( $where_clause ) ) {
			$where_clause = '';
		} else {
			$where_clause = 'WHERE ' . implode( ' AND ', $where_clause );
		}

		$limit_clause = '';

		if ( $args['per_page'] ) {
			$args['per_page']     = (int) $args['per_page'];
			$args['current_page'] = (int) $args['current_page'];
			$limit_clause         = " LIMIT {$args["per_page"]} ";
			$limit_clause        .= ' OFFSET ' . ( $args['current_page'] - 1 ) * $args['per_page'];
		}

		$order = '';

		if ( ! empty( $args['orderby'] ) ) {
			$order .= ' ORDER BY ' . esc_sql( $args['orderby'] );
			$order .= ! empty( $args['order'] ) ? ' ' . esc_sql( $args['order'] ) : ' ASC';
		}

		$results = $wpdb->get_results( "SELECT * FROM {$this->table_name} {$where_clause} {$order} {$limit_clause}" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! $results ) {
			return null;
		}

		return $results;
	}

	/**
	 * Return total orders count.
	 *
	 * @param string $order_status Order status.
	 * @param string $iap_product Product id.
	 *
	 * @return string|null
	 */
	public function get_total_orders_count( $order_status = '', $iap_product = '' ) {
		global $wpdb;

		$search_by_email = isset( $_GET['search_by_email'] ) ? sanitize_email( wp_unslash( $_GET['search_by_email'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		$table_name   = bbapp_iap()->get_global_dbprefix() . 'bbapp_iap_orders';
		$where_clause = ' WHERE 1=1';

		if ( ! empty( $order_status ) ) {
			$where_clause .= ' AND order_status= ' . "'" . $order_status . "'";
		}

		if ( ! empty( $iap_product ) ) {
			$where_clause .= ' AND bbapp_product_id= ' . $iap_product;
		}

		if ( ! empty( $search_by_email ) ) {
			$where_clause .= ' AND user_email= ' . "'" . $search_by_email . "'";
		}

		return $wpdb->get_var( "SELECT COUNT(*) FROM {$table_name}{$where_clause}" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Get order/
	 *
	 * @param int $order_id Order meta.
	 *
	 * @return WP_Error|array|null|object|void
	 */
	public function get_by_id( $order_id ) {
		global $wpdb;

		$get = $wpdb->get_row( $wpdb->prepare( "SELECT *FROM {$this->table_name} WHERE id=%d", $order_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! $get ) {
			return new WP_Error( 'not_found', __( 'No purchase found.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		return $get;
	}

	/**
	 * Upodate order meta.
	 *
	 * @param int    $order_id   Order id.
	 * @param string $meta_key   Meta key.
	 * @param string $meta_value Meta value.
	 *
	 * @return bool|false|int
	 */
	public function update_meta( $order_id, $meta_key, $meta_value ) {
		global $wpdb;

		$get_meta   = $this->get_meta( $order_id, $meta_key, false );
		$meta_value = maybe_serialize( $meta_value );

		if ( empty( $get_meta ) ) {

			$insert = $wpdb->insert( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$this->table_meta_name,
				array(
					'meta_key'   => $meta_key, //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
					'meta_value' => $meta_value, //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value
					'order_id'   => $order_id,
				)
			);

			if ( false !== $insert ) {
				return $meta_value;
			}
		} else {
			$update = $wpdb->update( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$this->table_meta_name,
				array(
					'meta_value' => $meta_value, //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value
				),
				array(
					'order_id' => $order_id,
					'meta_key' => $meta_key, //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
				)
			);

			if ( false !== $update ) {
				return $meta_value;
			}
		}

		return false;
	}

	/**
	 * Get order meta.
	 *
	 * @param int    $order_id   Order id.
	 * @param string $meta_key   Meta key.
	 * @param bool   $only_value Only value.
	 *
	 * @return array|bool|mixed|null|object|void
	 */
	public function get_meta( $order_id, $meta_key, $only_value = true ) {
		global $wpdb;

		$get = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $this->table_meta_name WHERE order_id=%s AND meta_key=%s", $order_id, $meta_key ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! empty( $get ) ) {
			$get->meta_value = maybe_unserialize( $get->meta_value );

			if ( $only_value ) {
				return $get->meta_value;
			}

			return $get;
		}

		return false;
	}

	/**
	 * Get orders by meta.
	 *
	 * @param string $meta_key   Meta key.
	 * @param string $meta_value Meta value.
	 *
	 * @return WP_Error|array|null|object|void
	 */
	public function get_orders_ids_by_meta( $meta_key, $meta_value ) {
		global $wpdb;

		$get = $wpdb->get_col( $wpdb->prepare( "SELECT order_id FROM {$this->table_meta_name} WHERE meta_key=%s AND meta_value=%s ORDER BY order_id ASC", $meta_key, $meta_value ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! empty( $get ) ) {
			return $get;
		}

		return array();
	}

	/**
	 * Add history.
	 *
	 * @param int    $order_id Order id.
	 * @param string $type     Eg :info | success | warning | error.
	 * @param string $text     History text.
	 * @param bool   $debug    Whether debug.
	 *
	 * @return array|bool|mixed|null|object|void
	 */
	public function add_history( $order_id, $type, $text, $debug = false ) {
		if ( empty( $type ) ) {
			$type = 'info';
		}

		$history = $this->get_history( $order_id );

		if ( $debug ) {
			$debug = print_r( $debug, true ); //phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r
		}

		$history[] = array(
			'time'  => time(),
			'type'  => $type,
			'text'  => $text,
			'debug' => $debug,
		);

		$this->update_meta( $order_id, '_order_history', $history );

		return $history;
	}

	/**
	 * Get order history.
	 *
	 * @param int $order_id Order id.
	 *
	 * @return array|bool|mixed|null|object|void
	 */
	public function get_history( $order_id ) {
		$history = $this->get_meta( $order_id, '_order_history' );

		if ( ! is_array( $history ) ) {
			$history = array();
		}

		return $history;
	}

	/**
	 * Update order next validation retry time.
	 *
	 * @param int    $order_id    Order id.
	 * @param string $string_time Time text.
	 * @param bool   $from        Check from specific data.
	 */
	public function update_order_next_validation_retry( $order_id, $string_time = '+2 hours', $from = false ) {
		if ( ! $from ) {
			$gmt_timestamp = strtotime( gmdate( 'Y-m-d H:i:s' ) );
		} else {
			$gmt_timestamp = strtotime( $from );
		}

		$date = gmdate( 'Y-m-d H:i:s', strtotime( $string_time, $gmt_timestamp ) );
		self::instance()->update_meta( $order_id, '_next_order_check', $date );
	}

	/**
	 * This handles the IAP s2s notification events from apple server.
	 */
	public function handle_notification() {
		$is_notification = filter_input( INPUT_GET, 'ios_s2s_notification', FILTER_SANITIZE_NUMBER_INT );

		// Check request is for ios_s2s_notification.
		if ( ! empty( $is_notification ) ) {
			$json            = file_get_contents( 'php://input' );
			$data            = json_decode( $json );
			$device_platform = 'ios';

			// Check required data get from S2S notification info.
			if ( ! empty( $data ) && ! empty( $data->notification_type ) && isset( bbapp_iap()->iap[ $device_platform ] ) && ! empty( $data->auto_renew_product_id ) ) {
				$store_product_id = $data->auto_renew_product_id;
				$iap              = bbapp_iap()->iap[ $device_platform ];

				// Handle CANCEL and REFUND notification.
				if ( in_array( $data->notification_type, array( 'CANCEL', 'REFUND' ), true ) ) {
					// Check active order from latest_receipt_info.
					if ( ! empty( $data->unified_receipt->latest_receipt_info ) ) {
						// Loop through latest_receipt_info.
						foreach ( $data->unified_receipt->latest_receipt_info as $key => $purchase ) {
							// Check Receipt info is for store_product_id.
							if ( $purchase->product_id === $store_product_id ) {
								if ( IAP_LOG ) {
									Logger::instance()->add( 'iap_log', 'BuddyBossApp\InAppPurchases\Order->handle_notification(),Store Product ID matched' );
								}

								// Check if it doesn't exist.
								$iap_exists = $iap->do_transaction_exists( $purchase->original_transaction_id );

								// If transaction_id is already present on db.
								if ( ! empty( $iap_exists ) ) {
									// Loop for all iap exists.
									foreach ( $iap_exists as $order ) {
										// Check if order is active or not.
										if ( in_array(
											$order->order_status,
											array(
												'subscribed',
												'completed',
											),
											true
										) ) {
											// found one valid 'subscribed' order. Cancel it.
											$this->cancel_order( $order->id, 'Purchase cancelled by Apple Server Notification' );
										} // Check if order is active or not
									} // Loop for all iap exists
								} // If transaction_id is already present on db.
							} // Check Receipt info is for store_product_id
						} // Loop through latest_receipt_info
					} // Check active order from latest_receipt_info
				} // Handle CANCEL and REFUND notification
			} // // Check required data get from S2S notification info.
		} // Check request is for ios_s2s_notification
	}

	/**
	 * Install Database on plugin activate.
	 */
	public function on_activation() {
		global $wpdb;

		$charset_collate = $wpdb->get_charset_collate();

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$sql1 = "CREATE TABLE {$this->table_name} (
        id bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
        blog_id bigint(20) NOT NULL,
        user_id bigint(20) NOT NULL,
        bbapp_product_id bigint(20) NOT NULL,
        device_platform varchar(20) NOT NULL,
        order_status varchar(20) NOT NULL,
        store_product_id varchar(150) NOT NULL,
        integration_types longtext NOT NULL,
        item_ids longtext NOT NULL,
        is_recurring tinyint(1) NOT NULL,
        secondary_id bigint(20) NOT NULL,
        date_created datetime DEFAULT '0000-00-00 00:00:00' NULL,
        date_updated datetime DEFAULT '0000-00-00 00:00:00' NULL,
        expire_at datetime DEFAULT '0000-00-00 00:00:00' NULL,
        user_email varchar(255) NOT NULL,
        KEY  blog_id (blog_id),
        KEY  user_id (user_id),
        KEY  bbapp_product_id (bbapp_product_id),
        KEY  device_platform (device_platform),
        KEY  order_status (order_status),
        KEY  store_product_id (store_product_id),
        KEY  is_recurring (is_recurring),
        KEY  secondary_id (secondary_id),
        KEY  date_created (date_created),
        KEY  date_updated (date_updated),
        KEY  expire_at (expire_at)
        ) {$charset_collate}";

		dbDelta( $sql1 );

		$sql2 = "CREATE TABLE {$this->table_meta_name} (
		meta_id bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
		order_id bigint(20) NOT NULL,
		meta_key varchar(255) NOT NULL,
		meta_value longtext NOT NULL
		) {$charset_collate}";

		dbDelta( $sql2 );
	}
}
