<?php
/**
 * Holds IAP related functionality for IOS.
 *
 * @package BuddyBossApp\InAppPurchases
 */

namespace BuddyBossApp\InAppPurchases;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\Admin\Configure;
use BuddyBossApp\Admin\InAppPurchases\Helpers;
use BuddyBossApp\AppStores\Apple;
use BuddyBossApp\Library\Composer;
use BuddyBossApp\Tools\Logger;
use Exception;
use WP_Error as WP_Error;

/**
 * IOS class for IAP.
 */
final class Ios extends StoreAbstract {

	/**
	 * Class instance.
	 *
	 * @var null $instance
	 */
	private static $instance = null;

	/**
	 * Ios constructor.
	 */
	private function __construct() {
		// ... leave empty, see Singleton below.
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return Controller|null
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Initialize Integration
	 */
	public function init() {
		// Device Platform Type.
		$platform_type       = 'ios';
		$store_product_types = Helpers::platform_store_product_types( $platform_type );

		// Register StoreAbstract Type.
		parent::set_up( $platform_type, __( 'iOS', 'buddyboss-app' ), $store_product_types );

		bbapp_iap()->iap[ $platform_type ]         = $this::instance(); // Register Instance.
		bbapp_iap()->integration[ $platform_type ] = $store_product_types; // Register Integration.
	}

	/**
	 * For rendering product settings
	 *
	 * @param string $integration Integration name.
	 * @param array  $item        Item data.
	 */
	public function render_product_settings( $integration, $item ) {
	}

	/**
	 * For saving product settings
	 *
	 * @param string $integration Integration name.
	 * @param array  $item        Item data.
	 */
	public function save_product_settings( $integration, $item ) {
	}

	/**
	 * Validates order payment status
	 *
	 * @param array $data Purchase data.
	 *
	 * @return mixed
	 * @throws \GuzzleHttp\Exception\GuzzleException | \GuzzleHttp\Exception\GuzzleException Guzzle exception error.
	 */
	public function validate( $data ) {
		if ( IAP_LOG ) {
			Logger::instance()->add( 'iap_log', 'BuddyBossApp\InAppPurchases\Ios::validate()' );
		}

		$transaction_data = array(
			'status'                => '',
			'transaction_date'      => '',
			'parent_transaction_id' => '',
			'transaction_id'        => '',
			'data'                  => array(),
			'expire_at'             => '',
			'history'               => array(),
		);

		$order_id              = absint( $data['order_id'] );
		$store_product_type    = trim( $data['store_product_type'] );
		$store_product_id      = trim( $data['store_product_id'] );
		$parent_transaction_id = trim( $data['parent_transaction_id'] );
		$is_production         = $data['is_production'];
		$get_pending_renewals  = $this->get_subscription_statuses( $parent_transaction_id, $is_production );

		if ( ! is_wp_error( $get_pending_renewals ) && ! empty( $get_pending_renewals->getData() ) && is_array( $get_pending_renewals->getData() ) ) {
			Orders::instance()->add_history( $order_id, 'info', __( 'Started validation with StoreKit 2', 'buddyboss-app' ) );

			$subscription_renewal_info = null;
			$found_usable              = false;

			foreach ( $get_pending_renewals->getData() as $subscription_group_identifier ) {
				foreach ( $subscription_group_identifier->getLastTransactions() as $last_transaction ) {
					$renewal_info = $last_transaction->getRenewalInfo();
					$expires_date = $last_transaction->getTransactionInfo()->getExpiresDate() / 1000;
					$current_time = strtotime( gmdate( 'Y-m-d H:i:s' ) );

					if ( $renewal_info->getProductId() === $store_product_id ) {
						$subscription_renewal_info = $last_transaction;
					}

					if ( $expires_date >= $current_time ) {
						$found_usable = true;
						break 2; // Exits both the inner and outer loops
					}
				}
			}

			if ( IAP_LOG ) {
				Logger::instance()->add( 'iap_log', 'Store Product ID matched' );
			}

			if ( ! $subscription_renewal_info ) {
				// If we didn't find any subscription renewal info.
				$transaction_data['status']  = 'expired';
				$transaction_data['history'] = array( __( 'Subscription has expired.', 'buddyboss-app' ) );

				return $transaction_data;
			}

			$transaction_info = $subscription_renewal_info->getTransactionInfo();
			$renewal_info     = $subscription_renewal_info->getRenewalInfo();

			$transaction_data = array(
				'expire_at'             => gmdate( 'Y-m-d H:i:s', $transaction_info->getExpiresDate() / 1000 ),
				'transaction_id'        => $transaction_info->getTransactionId(),
				'parent_transaction_id' => $transaction_info->getOriginalTransactionId(),
				'data'                  => array(
					'sandbox' => ( 'Sandbox' === $transaction_info->getEnvironment() ),
				),
				'status'                => $found_usable ? 'subscribed' : 'expired',
			);

			if ( ! empty( $transaction_info->getPurchaseDate() ) ) {
				$transaction_data['transaction_date'] = gmdate( 'Y-m-d H:i:s', $transaction_info->getPurchaseDate() / 1000 );
			}

			if ( 1 === (int) $transaction_info->getOfferType() ) {
				$transaction_data['data'] += array(
					'trial_period'        => true,
					'transaction_history' => __( 'Transaction started as free trial.', 'buddyboss-app' ),
					'had_trial'           => true,
				);
			}

			if ( ! $found_usable ) {
				$transaction_data['history'][] = __( 'Subscription has expired.', 'buddyboss-app' );

				if ( $renewal_info->getExpirationIntent() ) {
					$expire_reason                                           = $this->get_expiration_intent_text( $renewal_info->getExpirationIntent() );
					$transaction_data['history'][]                           = sprintf( __( 'Expire Reason: %s', 'buddyboss-app' ), $expire_reason );
					$transaction_data['data']['last_expiration_reason_code'] = $renewal_info->getExpirationIntent();
				}

				if ( 1 === (int) $renewal_info->getIsInBillingRetryPeriod() ) {
					$transaction_data['status'] = 'retrying';
				}
			}

			return $transaction_data;
		} else {
			/**
			 * Sandbox will work on production mode also.
			 * ReceiptValidator tries with sandbox env when production fails on Sandbox Token.
			 * Check - https://github.com/aporat/store-receipt-validator/blob/master/src/iTunes/Validator.php#L213/
			 */
			$iap_env            = Composer::instance()->receipt_validator_instance()->validator_endpoint_production();
			$validator          = Composer::instance()->receipt_validator_instance()->receipt_validator( $iap_env );
			$iap_receipt_token  = trim( $data['iap_receipt_token'] );
			$shared_secret      = Configure::instance()->option( 'publish.ios.shared_secret' );

			// ReceiptValidator Throw Exception.
			try {
				// Auto-renewable requires shared secret.
				if ( in_array( $store_product_type, array( 'auto_renewable' ), true ) ) {
					$validator->setSharedSecret( $shared_secret );
				}

				$response = $validator->setReceiptData( $iap_receipt_token )->validate();

				if ( method_exists( $response, 'isValid' ) && $response->isValid() ) {
					$purchases = $response->getPurchases();

					// Auto-renewable(auto_renewable) has purchases in latest receipts.
					if ( in_array( $store_product_type, array( 'auto_renewable' ), true ) ) {
						// NOTE : Below method returns an array of array (NOT arrays of PurchaseItem Object).
						$purchases = (array) $response->getLatestReceiptInfo();

						// Note : Need to loop in reverse.
						foreach ( array_reverse( $purchases ) as $key => $purchase ) {
							// Match the product we are looking for.
							if ( $purchase['product_id'] === $store_product_id ) {
								if ( IAP_LOG ) {
									Logger::instance()->add( 'iap_log', 'BuddyBossApp\InAppPurchases\Ios->validate(),Store Product ID matched' );
								}

								// Meta-data for  : Auto-Renewable Subscription(auto_renewable).
								$transaction_data['expire_at'] = $purchase['expires_date'];

								// Check if it doesn't exist.
								$iap_exists = $this->do_transaction_exists( $purchase['original_transaction_id'] );

								/**
								 * Allow Duplicate transaction_id Flow.
								 * non_auto_renewable - same transaction_id is never allowed (same as consumable).
								 */

								// If transaction_id is not present on db.
								if ( empty( $iap_exists ) ) {
									continue;
								}

								$found_usable = true;
								// Storing transaction id in db to avoid granting access twice.
								$transaction_data['transaction_id']        = $purchase['transaction_id'];
								$transaction_data['parent_transaction_id'] = $purchase['original_transaction_id'];
								$transaction_data['data']['sandbox']       = $response->isSandbox();

								// Storing all transaction related data.
								if ( ! empty( $purchase['purchase_date'] ) ) {
									$transaction_data['transaction_date'] = $purchase['purchase_date'];
								}

								// Trial Period if applicable.
								if ( isset( $purchase['is_trial_period'] ) ) {
									if ( is_string( $purchase['is_trial_period'] ) ) {
										$purchase['is_trial_period'] = ( 'true' === (string) $purchase['is_trial_period'] ) ? true : false;
									}

									$transaction_data['data']['trial_period'] = $purchase['is_trial_period'];

									if ( $transaction_data['data']['trial_period'] ) {
										$transaction_data['data']['transaction_history'] = esc_html__( 'Transaction started as free trial.', 'buddyboss-app' );
										$transaction_data['data']['had_trial']           = true;
									}
								}

								// NOTE : Found usable don't mean it is not expired. Sandbox is unreliable.
								if ( strtotime( $purchase['expires_date'] ) < strtotime( gmdate( 'Y-m-d H:i:s' ) ) ) {
									$found_usable = false;
								}

								if ( $found_usable ) {
									// No need to continue the loop as we found the transaction.
									break;
								}
							}
						}
					} else {
						// NOTE : Apple returns more than one purchases,  we have to find the one we are looking for and assume.
						foreach ( $purchases as $purchase ) {
							// Match the product we are looking for.
							if ( $purchase->getProductId() === $store_product_id ) {
								if ( IAP_LOG ) {
									Logger::instance()->add( 'iap_log', 'BuddyBossApp\InAppPurchases\Ios->validate(),Store Product ID matched' );
								}

								$found_usable = true;
								// We will store transaction id on db so we don't grant two access with on transaction ID.
								$transaction_data['transaction_id']        = $purchase->getTransactionId();
								$transaction_data['parent_transaction_id'] = $purchase->getOriginalTransactionId();
								$transaction_data['data']['sandbox']       = $response->isSandbox();
								$raw_response                              = $purchase->getRawResponse();

								// Check for free trial thing.
								if ( isset( $raw_response['is_trial_period'] ) ) {
									if ( is_string( $raw_response['is_trial_period'] ) ) {
										$raw_response['is_trial_period'] = ( 'true' === (string) $raw_response['is_trial_period'] ) ? true : false;
									}

									$transaction_data['data']['trial_period'] = $raw_response['is_trial_period'];

									if ( $transaction_data['data']['trial_period'] ) {
										$transaction_data['data']['transaction_history'] = esc_html__( 'Transaction is renewed as free trial.', 'buddyboss-app' );
									}
								}

								if ( ! empty( $purchase->getPurchaseDate() ) ) {
									$transaction_data['transaction_date'] = gmdate( 'Y-m-d H:i:s', $purchase->getPurchaseDate()->getTimestamp() );
								}

								// NOTE : Found usable don't mean it is not expired. Sandbox is unreliable.
								if ( strtotime( $purchase->getExpiresDate() ) < strtotime( gmdate( 'Y-m-d H:i:s' ) ) ) {
									$found_usable = false;
								}
								break;
							}
						}
					}

					// Return data as it's found.
					if ( isset( $found_usable ) && $found_usable ) {
						$transaction_data['status'] = 'subscribed';
					} else {
						// as we don't found any valid purchase we will expire the order.
						$transaction_data['history'][] = __( 'Subscription has expired.', 'buddyboss-app' );

						// Get pending info of subscription.
						$subscription_renewal_info = false;
						$get_pending_renewals      = $response->getPendingRenewalInfo();

						foreach ( $get_pending_renewals as $pending_renewal ) {
							if ( $pending_renewal['product_id'] === $store_product_id ) {
								$subscription_renewal_info = $pending_renewal;
							}
						}

						// Log Expire Reason.
						if ( $subscription_renewal_info ) {
							if ( isset( $subscription_renewal_info['expiration_intent'] ) ) {
								$expire_reason = $this->get_expiration_intent_text( $subscription_renewal_info['expiration_intent'] );
								/* translators: %s: Expiration reason. */
								$transaction_data['history'][]                           = sprintf( __( 'Expire Reason : %s', 'buddyboss-app' ), $expire_reason );
								$transaction_data['data']['last_expiration_reason_code'] = $subscription_renewal_info['expiration_intent'];
							}
						}

						// Check if apple is still retrying for payments. than give some grace time.
						if ( $subscription_renewal_info ) {
							if ( isset( $subscription_renewal_info['is_in_billing_retry_period'] ) && '1' === (string) $subscription_renewal_info['is_in_billing_retry_period'] ) {
								$transaction_data['status'] = 'retrying';

								return $transaction_data;
							}
						}

						// If there is no retrying happening finally release order as expired.
						$transaction_data['status'] = 'expired';
					}

					return $transaction_data;
				} else {
					$status_code = '00000';

					if ( method_exists( $response, 'getResultCode' ) ) {
						$status_code = $response->getResultCode();
					}

					return new WP_Error( 'error_iap_validation', $this->error_by_apple_status_code( $status_code ) );
				}
			} catch ( Exception $e ) {
				return new WP_Error( 'error_iap_validation', __( 'Error while validating order, could be invalid receipt token.', 'buddyboss-app' ) );
			}
		}
	}

	/**
	 * Process actual payment here
	 *
	 * @param array $data Purchase data.
	 *
	 * @return array | WP_Error
	 * @throws \GuzzleHttp\Exception\GuzzleException Guzzle exception error.
	 * @uses StoreAbstract::_process_payment()
	 */
	public function process_payment( $data ) {
		$bbapp_product_id   = $data['bbapp_product_id']; // BuddyBossAppProduct ID.
		$store_product_type = $data['store_product_type']; // Store Product Type.
		$store_product_id   = $data['store_product_id']; // Store Product ID.
		$iap_receipt_token  = trim( $data['iap_receipt_token'] ); // Receipt Token.
		$store_kit_2        = $data['is_store_kit_2']; // If the request is for storekit2.
		$purchase_id        = $data['purchase_id']; // If transaction id found.
		$order_id           = $data['order_id']; // Order id.

		// Define transaction data.
		$transaction_data = array(
			'transaction_date'      => null,
			'parent_transaction_id' => null,
			'transaction_id'        => null,
			'data'                  => array(),
			'expire_at'             => false, // should be given if product is recurring.
		);

		if ( $store_kit_2 && ! empty( $purchase_id ) && ! empty( $iap_receipt_token ) ) {
			Orders::instance()->add_history( $order_id, 'info', __( 'StoreKit 2 payment flow started', 'buddyboss-app' ) );

			// Get transaction info.
			$purchase = $this->is_production( $iap_receipt_token ) ? $this->get_transaction_info( $purchase_id ) : $this->get_transaction_info( $purchase_id, false );

			if ( ! is_wp_error( $purchase ) ) {
				// Check if it doesn't exist in DB.
				$iap_exists = $this->do_transaction_exists( $purchase->getOriginalTransactionId() );

				if ( $store_product_type === 'auto_renewable' ) {
					return $this->process_auto_renewable( $purchase, $iap_exists, $store_product_id, $data );
				} else {
					return $this->process_non_auto_renewable( $purchase, $iap_exists, $store_product_id, $store_product_type, $data );
				}
			} else {
				return new WP_Error( 'error_fetching_transaction_info', $purchase->get_error_message() );
			}
		} else {
			/**
			 * Sandbox will work on production mode also.
			 * ReceiptValidator tries with sandbox env when production fails on Sandbox Token.
			 * Check - https://github.com/aporat/store-receipt-validator/blob/master/src/iTunes/Validator.php#L213/
			 */
			$iap_env       = Composer::instance()->receipt_validator_instance()->validator_endpoint_production();
			$validator     = Composer::instance()->receipt_validator_instance()->receipt_validator( $iap_env );
			$shared_secret = Configure::instance()->option( 'publish.ios.shared_secret' );

			// ReceiptValidator Throw Exception.
			try {
				// NOTE : Safe to put shared secret on all receipts check, else only auto_renewable requires it.
				$validator->setSharedSecret( $shared_secret );
				$response                 = $validator->setReceiptData( $iap_receipt_token )->validate();
				$error_iap_sub_validation = __( 'Matching product(s) found, but no usable transaction found.', 'buddyboss-app' );

				if ( method_exists( $response, 'isValid' ) && $response->isValid() ) {
					// Auto renewable(auto_renewable) has purchases in latest receipts.
					if ( in_array( $store_product_type, array( 'auto_renewable' ), true ) ) {
						// NOTE : Below method returns an array of array (NOT arrays of PurchaseItem Object).
						$purchases = (array) $response->getLatestReceiptInfo();

						// Note : Need to loop in reverse.
						foreach ( $purchases as $key => $purchase ) {
							// Check if it doesn't exists.
							$iap_exists = $this->do_transaction_exists( $purchase['original_transaction_id'] );
							// Match the product we are looking for.
							if ( $purchase['product_id'] === $store_product_id || $iap_exists ) {
								if ( IAP_LOG ) {
									Logger::instance()->add( 'iap_log', 'BuddyBossApp\InAppPurchases\Ios->process_payment(),Store Product ID matched' );
								}

								// Meta-data for  : Auto-Renewable Subscription(auto_renewable).
								$transaction_data['expire_at'] = $purchase['expires_date'];

								/**
								 * Allow Duplicate transaction_id Flow.
								 * non_auto_renewable -Order has been renewed successfully. same transaction_id is never allowed (same as consumable).
								 */

								// If transaction_id is already present on db.
								if ( ! empty( $iap_exists ) ) {
									$do_skip               = true;
									$subscription_order    = false; // holds false until we found any useful order in existing order.
									$purchase_product_data = $this->get_product( $data['bbapp_product_id'] );
									$matched_order         = false;

									foreach ( $iap_exists as $order ) {
										if ( in_array( $order->order_status, array(
											'subscribed',
											'completed'
										), true ) ) {
											$order_product_data = $this->get_product( $order->bbapp_product_id );
											$subscription_order = true; // found one valid 'subscribed' order.
											$matched_order      = $order;

											// Cancel the order if it's from the same group & current user. so the upgrade of the purchase can be performed.
											if ( $store_product_id !== $order->store_product_id && get_current_user_id() === (int) $order->user_id && $order_product_data['iap_group'] === $purchase_product_data['iap_group'] ) {
												Orders::instance()->cancel_order( $order->id );

												$new_order_id   = absint( $data['order_id'] );
												$new_order_link = bbapp_get_admin_url( 'admin.php?page=bbapp-iap&action=view_order&order_id=' . $new_order_id );

												/* translators: %1$s: New order link, %2$s: New order id. */
												Orders::instance()->add_history( $order->id, 'info', sprintf( __( 'Upgraded/Downgraded to order #<a href="%1$s" target="_blank">%2$s</a>', 'buddyboss-app' ), esc_url( $new_order_link ), esc_html( $new_order_id ) ) );
												Orders::instance()->update_meta( $new_order_id, '_upgrade_downgrade_from', $order->id );
												$matched_order      = false;
												$subscription_order = false;
											}
										}
									}

									if ( ! $subscription_order ) {
										// if no subscribed order found don't this purchase.
										$do_skip = false;
									}

									if ( $do_skip ) {
										$error_iap_sub_validation = __( 'There is already an active subscription for this product in your account.', 'buddyboss-app' );
										/* translators: %1$s: New order link, %2$s: New order id. */
										if ( ! empty( $matched_order ) ) {
											$error_iap_sub_validation_data = sprintf(
												'%1$s <a href="%2$s" target="_blank">#%3$s</a>',
												__( 'You already have an active subscription for this product. Subscription ID:', 'buddyboss-app' ),
												bbapp_get_admin_url( 'admin.php?page=bbapp-iap&action=view_order&order_id=' . $matched_order->id ),
												esc_html( $matched_order->id )
											);
										}

										continue;
									}

									// disable $iapExists here. as we didn't skip current purchase one.
									$iap_exists = false;
								}

								$found_usable = true;
								// Storing transaction id in db to avoid granting access twice.
								$transaction_data['transaction_id']        = $purchase['transaction_id'];
								$transaction_data['parent_transaction_id'] = $purchase['original_transaction_id'];
								$transaction_data['data']['sandbox']       = $response->isSandbox();

								// Storing all transaction related data.
								if ( ! empty( $purchase['purchase_date'] ) ) {
									$transaction_data['transaction_date']    = $purchase['purchase_date'];
									$transaction_data['transaction_date_ms'] = ! empty( $purchase['transaction_date_ms'] ) ? $purchase['transaction_date_ms'] : strtotime( $purchase['purchase_date'] ) * 1000;
								}

								// Trial Period if applicable.
								if ( isset( $purchase['is_trial_period'] ) ) {
									if ( is_string( $purchase['is_trial_period'] ) ) {
										$purchase['is_trial_period'] = ( 'true' === (string) $purchase['is_trial_period'] ) ? true : false;
									}

									$transaction_data['data']['trial_period'] = $purchase['is_trial_period'];

									if ( $transaction_data['data']['trial_period'] ) {
										$transaction_data['data']['transaction_history'] = esc_html__( 'Transaction started as free trial.', 'buddyboss-app' );
										$transaction_data['data']['had_trial']           = true;
									}
								}

								if ( $found_usable ) {
									// No need to continue the loop as we found the transaction.
									break;
								}
							}
						}
					} else {
						// NOTE : getPurchases() returns Arrays of PurchaseItem Object.
						$purchases = $response->getPurchases();

						// NOTE : Apple returns more than one purchases,  we have to find the one we are looking for and assume.
						foreach ( $purchases as $key => $purchase ) {
							// Match the product we are looking for.
							if ( $purchase->getProductId() === $store_product_id ) {
								if ( IAP_LOG ) {
									Logger::instance()->add( 'iap_log', 'BuddyBossApp\InAppPurchases\Ios->process_payment(),Store Product ID matched' );
								}

								// Check if it doesn't exist in DB.
								$iap_exists = $this->do_transaction_exists( $purchase->getOriginalTransactionId() );

								// If transaction_id is already present on db.
								if ( ! empty( $iap_exists ) ) {
									$do_skip = true;

									/**
									 * Allow Duplicate transaction_id Flow.
									 * non_consumable - we will allow same transaction_id if previous transaction user is same.
									 * consumable - same transaction_id is never allowed.
									 */

									/**
									 * On none consumable product type. we allow one to be restore if user account is same.
									 * Else they will use one none consumable to redeem same course on multiple WordPress user's account using same apple ID.
									 */
									if ( in_array( $store_product_type, array( 'non_consumable' ), true ) ) {
										$found_invalid = false;

										foreach ( $iap_exists as $order ) {
											if ( get_current_user_id() !== $order->user_id ) {
												$found_invalid = true;
											}
										}

										if ( ! $found_invalid ) {
											// if none invalid order found don't skip this purchase.
											$do_skip = false;
										}
									}

									if ( $do_skip ) {
										continue;
									}

									// Flagging it as false as we didn't skipped current purchase one.
									$iap_exists = false;
								}

								$found_usable = true;
								// Storing transaction id in db to avoid granting access twice.
								$transaction_data['transaction_id']        = $purchase->getTransactionId();
								$transaction_data['parent_transaction_id'] = $purchase->getOriginalTransactionId();
								$transaction_data['data']['sandbox']       = $response->isSandbox();

								if ( in_array( $store_product_type, array( 'non_auto_renewable' ), true ) ) {
									// Meta-data for : Non-Renewing Subscription(non_auto_renewable).
									$subscription_duration                     = $data['subscription_duration'];
									$transaction_data['subscription_duration'] = $subscription_duration;
									$transaction_data['expire_at']             = $this->get_subscription_expiry( $subscription_duration );
								}

								if ( ! empty( $purchase->getPurchaseDate() ) ) {
									$transaction_data['transaction_date']    = gmdate( 'Y-m-d H:i:s', $purchase->getPurchaseDate()->getTimestamp() );
									$transaction_data['transaction_date_ms'] = $purchase->getPurchaseDate()->getTimestamp();
								}

								$raw_response = $purchase->getRawResponse();

								if ( isset( $raw_response['is_trial_period'] ) ) {
									if ( is_string( $raw_response['is_trial_period'] ) ) {
										$raw_response['is_trial_period'] = ( 'true' === (string) $raw_response['is_trial_period'] ) ? true : false;
									}

									$transaction_data['data']['trial_period'] = $raw_response['is_trial_period'];

									if ( $transaction_data['data']['trial_period'] ) {
										$transaction_data['data']['transaction_history'] = esc_html__( 'Transaction started as free trial.', 'buddyboss-app' );
										$transaction_data['data']['had_trial']           = true;
									}
								}

								if ( $found_usable ) {
									// No need to continue the loop as we found the transaction.
									break;
								}
							}
						}
					}

					// Return data as it's found.
					if ( isset( $found_usable ) ) {
						return $transaction_data;
					}

					// When we didn't find any store_product_id having unused transaction_id we will throw here.
					if ( ! empty( $iap_exists ) ) {
						$data = '';
						if ( ! empty( $error_iap_sub_validation_data ) ) {
							$data = $error_iap_sub_validation_data;
						}

						return new WP_Error( 'error_iap_sub_validation', $error_iap_sub_validation, $data );
					}

					return new WP_Error( 'error_iap_validation', __( 'No usable transaction found.', 'buddyboss-app' ) );
				} else {
					$status_code = '00000';

					if ( method_exists( $response, 'getResultCode' ) ) {
						$status_code = $response->getResultCode();
					}

					return $this->error_by_apple_status_code( $status_code );
				}
			} catch ( Exception $e ) {
				return new WP_Error( 'error_iap_validation', __( 'Error while reading InAppPurchase token, could be invalid receipt token.', 'buddyboss-app' ) );
			}
		}
	}

	/**
	 * Error handling
	 *
	 * @param string $status_code Status code.
	 *
	 * @return WP_Error
	 */
	public function error_by_apple_status_code( $status_code ) {
		switch ( $status_code ) {
			case '21000':
				/* translators: %s: Apple error */
				$err_msg = new WP_Error( 'error_iap_validation', sprintf( __( 'The App Store could not read the JSON object you provided. Error Code #%s.', 'buddyboss-app' ), $status_code ) );
				break;
			case '21002':
				/* translators: %s: Apple error */
				$err_msg = new WP_Error( 'error_iap_validation', sprintf( __( 'The data in the receipt-data property was malformed or missing. Error Code #%s.', 'buddyboss-app' ), $status_code ) );
				break;
			case '21003':
				/* translators: %s: Apple error */
				$err_msg = new WP_Error( 'error_iap_validation', sprintf( __( 'The receipt could not be authenticated. Error Code #%s.', 'buddyboss-app' ), $status_code ) );
				break;
			case '21004':
				/* translators: %s: Apple error */
				$err_msg = new WP_Error( 'error_iap_validation', sprintf( __( 'The shared secret you provided does not match the shared secret on file for your account. Error Code #%s.', 'buddyboss-app' ), $status_code ) );
				break;
			case '21005':
				/* translators: %s: Apple error */
				$err_msg = new WP_Error( 'error_iap_validation', sprintf( __( 'The receipt server is not currently available. Error Code #%s.', 'buddyboss-app' ), $status_code ) );
				break;
			case '21006':
				/* translators: %s: Apple error */
				$err_msg = new WP_Error( 'error_iap_validation', sprintf( __( 'This receipt is valid but the subscription has expired. Error Code #%s.', 'buddyboss-app' ), $status_code ) );
				break;
			case '21007':
				/* translators: %s: Apple error */
				$err_msg = new WP_Error( 'error_iap_validation', sprintf( __( 'This receipt is from the test environment, but it was sent to the production environment for verification. Send it to the test environment instead. Error Code #%s.', 'buddyboss-app' ), $status_code ) );
				break;
			case '21008':
				/* translators: %s: Apple error */
				$err_msg = new WP_Error( 'error_iap_validation', sprintf( __( 'This receipt is from the production environment, but it was sent to the test environment for verification. Send it to the production environment instead. Error Code #%s.', 'buddyboss-app' ), $status_code ) );
				break;
			case '21010':
				/* translators: %s: Apple error */
				$err_msg = new WP_Error( 'error_iap_validation', sprintf( __( 'This receipt could not be authorized. Treat this as if a purchase was never made. Error Code #%s.', 'buddyboss-app' ), $status_code ) );
				break;
			default:
				/* translators: %s: Apple error */
				$err_msg = new WP_Error( 'error_iap_validation', sprintf( __( 'Error while reading IAP token. Response found to be invalid. Error Code  #%s', 'buddyboss-app' ), $status_code ) );
				break;
		}

		return $err_msg;
	}

	/**
	 * Return Text Reason for expiration intent.
	 *
	 * @param string $code Code number.
	 *
	 * @return mixed
	 */
	public function get_expiration_intent_text( $code ) {
		switch ( $code ) {
			case '1':
				$expiration_txt = __( 'Customer cancelled their subscription.', 'buddyboss-app' );
				break;
			case '2':
				$expiration_txt = __( "Billing error: for example, customer's payment information is no longer valid.", 'buddyboss-app' );
				break;
			case '3':
				$expiration_txt = __( 'Customer did not agree to a recent price increase.', 'buddyboss-app' );
				break;
			case '4':
				$expiration_txt = __( 'Product was not available for purchase at the time of renewal.', 'buddyboss-app' );
				break;
			default:
				$expiration_txt = __( 'Unknown error.', 'buddyboss-app' );
				break;
		}

		return $expiration_txt;
	}

	/**
	 * Sync store product with local data.
	 *
	 * @return array
	 */
	public function sync_store_product() {
		$products = Apple::instance()->sync_store_product();

		update_option( "bbapp_{$this->type}_store_product", $products );

		return $products;
	}

	/**
	 * Get the App Store Server API client.
	 *
	 * This function checks the required fields and validates them before creating the API client.
	 * If any of the necessary fields or configurations are missing, it returns a WP_Error.
	 *
	 * @param bool $is_production Whether to use the production environment or not.
	 *
	 * @since 2.3.80
	 */
	public function get_api_client( $is_production = true ) {
		// Validate if the iOS connect fields are properly set.
		if ( ! Apple::instance()->verify_ios_connect_fields() ) {
			return new WP_Error( 'error_api_client_fields_missing', __( 'One or more required fields are missing to get the API client for App Store Server API.', 'buddyboss-app' ) );
		}

		// Check if the release bundle ID is set.
		if ( empty( Apple::instance()->get_release_bundle_id() ) ) {
			return new WP_Error( 'error_api_client_release_bundle_missing', __( 'Release bundle id missing.', 'buddyboss-app' ) );
		}

		// Get and validate the private key file.
		$private_key      = Apple::instance()->get_private_key();
		$private_key_path = bbapp_get_upload_full_path( $private_key );

		if ( ! file_exists( $private_key_path ) ) {
			return new WP_Error(
				'error_api_client_private_key_missing',
				__( 'Error while connecting to your Apple Developer account. Please make sure you have configured correct Apple API keys.', 'buddyboss-app' )
			);
		}

		// Retrieve required parameters to create the API client.
		$issuer_id         = Apple::instance()->get_issuer_id();
		$release_bundle_id = Apple::instance()->get_release_bundle_id();
		$key_id            = Apple::instance()->get_key_id();
		$secret_content    = file_get_contents( $private_key_path );
		$environment       = Composer::instance()->appstore_server_api_instance()->get_environment( $is_production );

		return Composer::instance()->appstore_server_api_instance()->AppStoreServerAPI( $environment, $issuer_id, $release_bundle_id, $key_id, $secret_content );
	}

	/**
	 * Retrieve transaction info based on the purchase ID.
	 *
	 * @param string $purchase_id The purchase ID for the transaction.
	 * @param bool $is_production Whether to use the production environment or not.
	 *
	 * @since 2.3.80
	 * @return object|WP_Error Transaction info on success or WP_Error on failure.
	 */
	public function get_transaction_info( $purchase_id, $is_production = true ) {
		// Get the App Store Server API client.
		$api_client = $this->get_api_client( $is_production );

		// Check if the API client is valid, else return the error.
		if ( is_wp_error( $api_client ) ) {
			return $api_client; // Return the error generated from get_api_client().
		}

		try {
			// Attempt to fetch the transaction info from the API client.
			$transaction_response = $api_client->getTransactionInfo( $purchase_id );

			try {
				return $this->get_single_transaction_info( $transaction_response );
			} catch ( Exception $e ) {
				return new WP_Error( 'error_fetching_single_transaction_info', $e->getMessage() );
			}
		} catch ( Exception $e ) {
			// Catch and handle any errors during the transaction retrieval process.
			return new WP_Error( 'error_fetching_transaction_info', $e->getMessage() );
		}
	}

	/**
	 * Get single transaction info.
	 *
	 * @param object $transaction_response Transaction response.
	 *
	 * @return WP_Error | object
	 */
	public function get_single_transaction_info( $transaction_response ) {
		try {
			return $transaction_response->getTransactionInfo();
		} catch ( Exception $e ) {
			return new WP_Error( 'error_fetching_single_transaction_info', $e->getMessage() );
		}
	}

	/**
	 * Get subscription statuses.
	 *
	 * @param int $purchase_id Purchase ID.
	 *
	 * @return WP_Error
	 */
	public function get_subscription_statuses( $purchase_id, $is_production = true ) {
		$api_client = $this->get_api_client( $is_production );

		if ( is_wp_error( $api_client ) ) {
			return $api_client;
		}

		try {
			return $api_client->getAllSubscriptionStatuses( $purchase_id );
		} catch ( Exception $e ) {
			// Catch and handle any errors during the transaction retrieval process.
			return new WP_Error( 'error_fetching_subscription_statuses', $e->getMessage() );
		}
	}

	/**
	 * Process auto-renewable subscriptions.
	 *
	 * @param object $purchase The purchase object.
	 * @param array $iap_exists The existing transactions.
	 * @param string $store_product_id The store product ID.
	 * @param array $data The input data.
	 *
	 * @since 2.3.80
	 * @return array|WP_Error
	 */
	private function process_auto_renewable( $purchase, $iap_exists, $store_product_id, $data ) {
		$transaction_data = [];

		// Check if the product matches and process accordingly
		if ( ! $this->is_product_match( $purchase, $store_product_id, $iap_exists ) ) {
			return $transaction_data; // Early return if the product doesn't match
		}

		// Set expiration date in transaction data
		$transaction_data['expire_at'] = gmdate( 'Y-m-d H:i:s', $purchase->getExpiresDate() / 1000 );

		// Handle existing subscriptions if any
		if ( ! empty( $iap_exists ) ) {
			$do_skip = $this->handle_existing_subscription( $iap_exists, $purchase, $store_product_id, $data, $transaction_data );

			// Return early if there's an error or validation issue
			if ( is_wp_error( $do_skip ) ) {
				return $do_skip;
			}

			if ( $do_skip ) {
				return new WP_Error( 'error_iap_sub_validation', 'Validation error.' );
			}
		}

		// Store transaction data and return
		return $this->store_transaction_data( $purchase, $transaction_data );
	}

	/**
	 * Process non-auto-renewable subscriptions.
	 *
	 * @param object $purchase The purchase object.
	 * @param array $iap_exists The existing transactions.
	 * @param string $store_product_id The store product ID.
	 * @param array $data The input data.
	 *
	 * @since 2.3.80
	 * @return array|WP_Error
	 */
	private function process_non_auto_renewable( $purchase, $iap_exists, $store_product_id, $store_product_type, $data ) {
		$transaction_data = array();
		$do_skip          = false;

		if ( ! empty( $iap_exists ) ) {
			$do_skip = $this->handle_non_consumable( $iap_exists, $purchase, $store_product_id );
		}

		if ( $do_skip ) {
			return new WP_Error( 'error_iap_sub_validation', 'Validation error.' );
		}

		$transaction_data = $this->store_transaction_data( $purchase, $transaction_data );

		if ( in_array( $store_product_type, array( 'non_auto_renewable' ), true ) ) {
			$transaction_data['expire_at'] = $this->get_subscription_expiry( $data['subscription_duration'] );
		}

		return $transaction_data;
	}

	/**
	 * Check if the product matches the store product or if the IAP already exists.
	 *
	 * @param object $purchase The purchase object.
	 * @param string $store_product_id The store product ID.
	 * @param array $iap_exists The existing transactions.
	 *
	 * @since 2.3.80
	 * @return bool
	 */
	private function is_product_match( $purchase, $store_product_id, $iap_exists ) {
		return $purchase->getProductId() === $store_product_id || $iap_exists;
	}

	/**
	 * Handle existing subscriptions and perform necessary actions like skipping or canceling old orders.
	 *
	 * @param array $iap_exists The existing transactions.
	 * @param object $purchase The purchase object.
	 * @param string $store_product_id The store product ID.
	 * @param array $data The input data.
	 * @param array $transaction_data The transaction data array.
	 *
	 * @since 2.3.80
	 * @return bool | WP_Error Whether to skip the transaction.
	 */
	private function handle_existing_subscription( $iap_exists, $purchase, $store_product_id, $data, &$transaction_data ) {
		$do_skip       = true;
		$matched_order = null;

		// Check if there's an active subscription order that can be upgraded
		foreach ( $iap_exists as $order ) {
			if ( in_array( $order->order_status, [ 'subscribed', 'completed' ], true ) ) {
				$matched_order = $order;

				if ( $this->can_upgrade_order( $order, $purchase, $store_product_id, $data ) ) {
					$this->upgrade_order( $order, $purchase, $data );
					$matched_order = null;
					$do_skip       = false;
					break; // No need to continue if we've upgraded an order
				}
			}
		}

		// If no active subscription order was found, don't skip the purchase
		if ( $matched_order === null ) {
			$do_skip = false;
		}

		// If we should skip, return an error with the appropriate message
		if ( $do_skip && $matched_order !== null ) {
			$error_iap_sub_validation      = __( 'There is already an active subscription for this product in your account.', 'buddyboss-app' );
			$error_iap_sub_validation_data = sprintf(
				'%1$s <a href="%2$s" target="_blank">#%3$s</a>',
				__( 'You already have an active subscription for this product. Subscription ID:', 'buddyboss-app' ),
				bbapp_get_admin_url( 'admin.php?page=bbapp-iap&action=view_order&order_id=' . $matched_order->id ),
				esc_html( $matched_order->id )
			);

			return new WP_Error( 'error_iap_sub_validation', $error_iap_sub_validation, $error_iap_sub_validation_data );
		}

		return $do_skip;
	}

	/**
	 * Check if an order can be upgraded.
	 *
	 * @param object $order The order object.
	 * @param object $purchase The purchase object.
	 * @param string $store_product_id The store product ID.
	 *
	 * @since 2.3.80
	 * @return bool Whether the order can be upgraded.
	 */
	private function can_upgrade_order( $order, $purchase, $store_product_id, $data ) {
		$order_product_data    = $this->get_product( $order->bbapp_product_id );
		$purchase_product_data = $this->get_product( $data['bbapp_product_id'] );

		return $store_product_id !== $order->store_product_id && get_current_user_id() === (int) $order->user_id && $order_product_data['iap_group'] === $purchase_product_data['iap_group'];
	}

	/**
	 * Perform the upgrade for an order.
	 *
	 * @param object $order The order object.
	 * @param object $purchase The purchase object.
	 * @param array $data The input data.
	 *
	 * @since 2.3.80
	 * @return void
	 */
	private function upgrade_order( $order, $purchase, $data ) {
		Orders::instance()->cancel_order( $order->id );

		$new_order_id   = absint( $data['order_id'] );
		$new_order_link = bbapp_get_admin_url( 'admin.php?page=bbapp-iap&action=view_order&order_id=' . $new_order_id );

		Orders::instance()->add_history( $order->id, 'info', sprintf( __( 'Upgraded/Downgraded to order #<a href="%1$s" target="_blank">%2$s</a>', 'buddyboss-app' ), esc_url( $new_order_link ), esc_html( $new_order_id ) ) );
		Orders::instance()->update_meta( $new_order_id, '_upgrade_downgrade_from', $order->id );
	}

	/**
	 * Store the transaction data in the database.
	 *
	 * @param object $purchase The purchase object.
	 * @param array $transaction_data The transaction data array.
	 *
	 * @since 2.3.80
	 * @return array The updated transaction data.
	 */
	private function store_transaction_data( $purchase, $transaction_data ) {
		$transaction_data['transaction_id']        = $purchase->getTransactionId();
		$transaction_data['parent_transaction_id'] = $purchase->getOriginalTransactionId();
		$transaction_data['data']['sandbox']       = ( ! empty( $purchase->getEnvironment() ) && 'Sandbox' === $purchase->getEnvironment() );

		if ( ! empty( $purchase->getPurchaseDate() ) ) {
			$transaction_data['transaction_date']    = gmdate( 'Y-m-d H:i:s', $purchase->getPurchaseDate() / 1000 );
			$transaction_data['transaction_date_ms'] = $purchase->getPurchaseDate();
		}

		// Trial Period if applicable.
		if ( 1 === (int) $purchase->getOfferType() ) {
			$transaction_data['data']['trial_period'] = (bool) $purchase->getOfferType();

			if ( $transaction_data['data']['trial_period'] ) {
				$transaction_data['data']['transaction_history'] = esc_html__( 'Transaction started as free trial.', 'buddyboss-app' );
				$transaction_data['data']['had_trial']           = true;
				$transaction_data['data']['offer_type']          = $purchase->getOfferDiscountType();
			}
		}

		return $transaction_data;
	}

	/**
	 * Get the subscription expiry date based on the duration.
	 *
	 * @param string $subscription_duration The subscription duration.
	 *
	 * @since 2.3.80
	 * @return string The expiry date.
	 */
	private function get_subscription_expiry( $subscription_duration ) {
		switch ( $subscription_duration ) {
			case '1-week':
				return gmdate( 'Y-m-d H:i:s', strtotime( '+1 week', time() ) );
			case '1-month':
				return gmdate( 'Y-m-d H:i:s', strtotime( '+1 month', time() ) );
			case '2-month':
				return gmdate( 'Y-m-d H:i:s', strtotime( '+2 month', time() ) );
			case '3-month':
				return gmdate( 'Y-m-d H:i:s', strtotime( '+3 month', time() ) );
			case '6-month':
				return gmdate( 'Y-m-d H:i:s', strtotime( '+6 month', time() ) );
			default:
				return gmdate( 'Y-m-d H:i:s', strtotime( '+365 day', time() ) );
		}
	}

	/**
	 * Handle non-consumable product logic.
	 *
	 * @param array $iap_exists The existing transactions.
	 * @param object $purchase The purchase object.
	 * @param string $store_product_id The store product ID.
	 *
	 * @since 2.3.80
	 * @return bool Whether to skip the transaction.
	 */
	private function handle_non_consumable( $iap_exists, $purchase, $store_product_id ) {
		$found_invalid = false;

		foreach ( $iap_exists as $order ) {
			if ( get_current_user_id() !== $order->user_id ) {
				$found_invalid = true;
			}
		}

		return ! $found_invalid;
	}

	/**
	 * Check whether current transaction is from production or sandbox environment.
	 *
	 * @param string $iap_receipt_token Purchase receipt token.
	 *
	 * @since 2.3.80
	 * @return WP_Error | bool The product data.
	 */
	public function is_production( $iap_receipt_token ) {
		return Composer::instance()->appstore_server_api_instance()->isProduction( $iap_receipt_token );
	}
}
