<?php
/**
 * Abstract class for integrations.
 *
 * @package BuddyBossApp\InAppPurchases
 */

namespace BuddyBossApp\InAppPurchases;


if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Integration abstract class.
 */
abstract class IntegrationAbstract {

	/**
	 * Integration slug.
	 *
	 * @var bool $integration_slug
	 */
	protected $integration_slug = false;

	/**
	 * Integration type.
	 *
	 * @var bool $integration_type
	 */
	protected $integration_type = false;

	/**
	 * Integration label.
	 *
	 * @var bool $integration_label
	 */
	protected $integration_label = false;

	/**
	 * Item item label.
	 *
	 * @var bool $item_label
	 */
	protected $item_label = false;

	/**
	 * IntegrationAbstract constructor.
	 */
	private function __construct() {
		// ... leave empty, see Singleton below.
	}

	/**
	 * Integration slug.
	 *
	 * @return string
	 */
	public function get_slug() {
		return $this->integration_slug;
	}

	/**
	 * Integration type.
	 *
	 * @return string
	 */
	public function get_type() {
		return $this->integration_type;
	}

	/**
	 * Integration label.
	 *
	 * @return string
	 */
	public function get_label() {
		return $this->integration_label;
	}

	/**
	 * Integration item label.
	 *
	 * @return string
	 */
	public function get_item_label() {
		return $this->item_label;
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return void
	 */
	public static function instance() {
	}

	/**
	 * Function to be overridden in sub-class.
	 *
	 * @param string $integration_type  Integration type.
	 * @param string $integration_label Integration label.
	 */
	public function set_up( $integration_type, $integration_label ) {
		$this->integration_type  = $integration_type;
		$this->integration_label = $integration_label;
		$this->item_label        = '';
		$this->hooks();
	}

	/**
	 * Function to add actions/filters
	 */
	private function hooks() {
		add_filter( 'bbapp_iap_get_registered_integrations', array( $this, 'register_integration' ) );
		add_filter( 'bbapp_ajax_iap_linking_options', array( $this, '_iap_linking_options' ), 10, 2 );
		add_filter( 'bbapp_ajax_iap_integration_ids', array( $this, '_iap_integration_ids' ), 10, 3 ); // Remove filter on  [PROD-2938].
		add_filter( 'bbapp_iap_integration_item_id_permalink', array( $this, '_item_id_permalink' ), 10, 3 );
		add_filter( 'bbapp_iap_is_purchase_available', array( $this, '_is_purchase_available' ), 10, 4 );
		add_filter( 'bbapp_iap_is_has_access', array( $this, '_has_access' ), 10, 4 );
	}

	/**
	 * Register current instance iap type for easy access using the filter.
	 *
	 * @param array $registered_integrations Registered integration.
	 *
	 * @return mixed
	 */
	public function register_integration( $registered_integrations ) {
		$registered_integrations[ $this->integration_type ] = $this->integration_label;

		return $registered_integrations;
	}

	/**
	 * Triggers when order is completed.
	 *
	 * @param object $order Order data.
	 */
	public function _on_order_completed( $order ) { //phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore
		// NOTE : For multi-site, we are storing data as [:blog-id][some_key_index]. Eg : integration_type or misc_settings.
		$all_item_ids = maybe_unserialize( $order->item_ids );

		if ( bbapp()->is_network_activated() ) {
			$all_item_ids = $all_item_ids[ get_current_blog_id() ];
		}

		$item_ids = maybe_unserialize( $all_item_ids[ $this->integration_type ] );

		if ( empty( $item_ids ) ) {
			Orders::instance()->add_history( $order->id, 'warning', __( 'Admin Error : Integration does not have any item mapped in backend still invoked from front-end', 'buddyboss-app' ) );
		} else {
			$this->on_order_completed( $item_ids, $order );
		}

		/**
		 * Trigger when iap order is completed.
		 *
		 * @param object $order Order data.
		 * @param object $this  Class reference.
		 */
		do_action( 'bbapp_iap_order_completed', $order, $this );
	}

	/**
	 * Triggers when order is completed.
	 *
	 * @param object $order Order data.
	 */
	public function _on_order_activate( $order ) { //phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore
		// NOTE : For multi-site, we are storing data as [:blog-id][some_key_index]. Eg : integration_type or misc_settings.
		$all_item_ids = maybe_unserialize( $order->item_ids );

		if ( bbapp()->is_network_activated() ) {
			$all_item_ids = $all_item_ids[ get_current_blog_id() ];
		}

		$item_ids = maybe_unserialize( $all_item_ids[ $this->integration_type ] );

		if ( empty( $item_ids ) ) {
			Orders::instance()->add_history( $order->id, 'warning', __( 'Admin Error : Integration does not have any item mapped in backend still invoked from front-end', 'buddyboss-app' ) );
		} else {
			$this->on_order_activate( $item_ids, $order );
		}

		/**
		 * Trigger when iap order is activate.
		 *
		 * @param object $order Order data.
		 * @param object $this  Class reference.
		 */
		do_action( 'bbapp_iap_order_activate', $order, $this );
	}

	/**
	 * Triggers when order is expired
	 *
	 * @param object $order Order data.
	 */
	public function _on_order_expired( $order ) { //phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore
		$all_item_ids = maybe_unserialize( $order->item_ids );

		if ( bbapp()->is_network_activated() ) {
			$all_item_ids = $all_item_ids[ get_current_blog_id() ];
		}

		$item_ids = maybe_unserialize( $all_item_ids[ $this->integration_type ] );

		if ( empty( $item_ids ) ) {
			Orders::instance()->add_history( $order->id, 'warning', __( 'Admin Error : Integration does not have any item mapped in backend still invoked from front-end', 'buddyboss-app' ) );
		} else {
			$this->on_order_expired( $item_ids, $order );
		}

		/**
		 * Trigger when iap order is expired.
		 *
		 * @param object $order Order data.
		 * @param object $this  Class reference.
		 */
		do_action( 'bbapp_iap_order_expired', $order, $this );
	}

	/**
	 * Triggers when order is cancelled
	 *
	 * @param object $order Order data.
	 */
	public function _on_order_cancelled( $order ) { //phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore
		$all_item_ids = maybe_unserialize( $order->item_ids );

		if ( bbapp()->is_network_activated() ) {
			$all_item_ids = $all_item_ids[ get_current_blog_id() ];
		}

		$item_ids = maybe_unserialize( $all_item_ids[ $this->integration_type ] );

		if ( empty( $item_ids ) ) {
			Orders::instance()->add_history( $order->id, 'warning', __( 'Admin Error : Integration does not have any item mapped in backend still invoked from front-end', 'buddyboss-app' ) );
		} else {
			$this->on_order_cancelled( $item_ids, $order );
		}

		/**
		 * Trigger when iap order is cancelled.
		 *
		 * @param object $order Order data.
		 * @param object $this  Class reference.
		 */
		do_action( 'bbapp_iap_order_cancelled', $order, $this );
	}

	/**
	 * User has access or not.
	 *
	 * @param bool   $return              Hase accees or not.
	 * @param array  $integrated_item_ids Integration ids.
	 * @param string $integration_type    Integration type.
	 * @param int    $user_id             User id.
	 *
	 * @return mixed
	 */
	public function _has_access( $return, $integrated_item_ids, $integration_type, $user_id ) { //phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore
		if ( ! empty( $integrated_item_ids ) && $integration_type === $this->integration_slug ) {
			$return = $this->has_access( $integrated_item_ids, $user_id );
		}

		return $return;
	}

	/**
	 * Handle bbapp ajax iap_linking_options for this integration
	 *
	 * @param array  $results levels data.
	 * @param string $slug    Search term.
	 *
	 * @return mixed
	 */
	public function _iap_linking_options( $results, $slug ) { //phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore
		if ( $slug === $this->integration_slug ) {
			return $this->iap_linking_options( $results );
		}

		return $results;
	}

	/**
	 * Handle bbapp ajax iap_integration_ids for this integration
	 * it's return items label with id.
	 *
	 * @param array     $results         levels data.
	 * @param int|array $integration_ids selected integration id.
	 * @param string    $slug            integration's slug.
	 *
	 * @return mixed
	 */
	public function _iap_integration_ids( $results, $integration_ids, $slug ) { //phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore
		if ( $slug === $this->integration_slug ) {
			return $this->iap_integration_ids( $results, $integration_ids );
		}

		return $results;
	}

	/**
	 * Get item edit link to show on order page.
	 *
	 * @param string $link    Edit link.
	 * @param string $item_id Item id.
	 * @param string $type    Integration type.
	 *
	 * @return string
	 */
	public function _item_id_permalink( $link, $item_id, $type ) { //phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore
		if ( $type === $this->integration_type ) {
			return $this->item_id_permalink( $link, $item_id );
		}

		return $link;
	}

	/**
	 * Function to check given $forItemId is connect directly/indirectly with given $integration_item_id
	 *
	 * Tells whether purchase is available on given iap type. Purchase availability depends on configuration and active settings.
	 * Warning this only tells depending on configuration.
	 * You have to check bbapp_product_id is available or not.
	 *
	 * @param bool   $is_available        Flag for product is available or not.
	 * @param int    $item_id             Post id or membership id. it's directly or indirectly [ through integration's item id ] connected with bbapp product.
	 * @param int    $integration_item_id Integration item id which is connected with bbapp product.
	 * @param string $slug                Integration slug.
	 *
	 * @return bool
	 */
	public function _is_purchase_available( $is_available, $item_id, $integration_item_id, $slug ) { //phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore
		if ( $slug === $this->integration_slug ) {
			return $this->is_purchase_available( $is_available, $item_id, $integration_item_id );
		}

		return $is_available;
	}

	/**
	 * Hook function get triggers when order is completed.
	 *
	 * @param array  $item_ids Item ids.
	 * @param object $order    Order data.
	 *
	 * @return mixed
	 */
	abstract public function on_order_completed( $item_ids, $order );

	/**
	 * Fires on order activate.
	 *
	 * @param array  $item_ids Item ids.
	 * @param object $order    Order data.
	 *
	 * @return mixed
	 */
	abstract public function on_order_activate( $item_ids, $order );

	/**
	 * Fires on order cancel.
	 *
	 * @param array  $item_ids Item ids.
	 * @param object $order    Order data.
	 *
	 * @return mixed
	 */
	abstract public function on_order_cancelled( $item_ids, $order );

	/**
	 * Fires on order expire.
	 *
	 * @param array  $item_ids Item ids.
	 * @param object $order    Order data.
	 *
	 * @return mixed
	 */
	abstract public function on_order_expired( $item_ids, $order );

	/**
	 * Function to get IAP linking options.
	 *
	 * @param array $results Membership levels.
	 *
	 * @return mixed
	 */
	abstract public function iap_linking_options( $results );

	/**
	 * Function to IAP integration ids.
	 *
	 * @param array $results         Array of membership levels.
	 * @param array $integration_ids Array of integration ids.
	 *
	 * @return mixed
	 */
	abstract public function iap_integration_ids( $results, $integration_ids );

	/**
	 * Product id link.
	 *
	 * @param string $link    Product link.
	 * @param int    $item_id Item id.
	 *
	 * @return mixed
	 */
	abstract public function item_id_permalink( $link, $item_id );

	/**
	 * Function to check given $forItemId is connect directly/indirectly with given $integration_item_id
	 *
	 * @param bool $is_available        Flag for product is available or not.
	 * @param int  $item_id             Item id.
	 * @param int  $integration_item_id Integration item id.
	 *
	 * @return mixed
	 */
	abstract public function is_purchase_available( $is_available, $item_id, $integration_item_id );

	/**
	 * Check if user has access.
	 *
	 * @param array $item_ids item ids.
	 * @param int   $user_id  User id.
	 *
	 * @return mixed
	 */
	abstract public function has_access( $item_ids, $user_id );
}
