<?php
/**
 * Holds in-app purchases controller functionality.
 *
 * @package BuddyBossApp\InAppPurchases
 */

namespace BuddyBossApp\InAppPurchases;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\Admin\InAppPurchases\Helpers;
use BuddyBossApp\Admin\Settings;
use BuddyBossApp\Api\InAppPurchases\Main;
use BuddyBossApp\AppSettings;
use BuddyBossApp\Integrations\LearndashCourse\IAP as LearnDashCourseIAP;
use BuddyBossApp\Integrations\LearndashGroup\IAP as LearnDashGroupIAP;
use BuddyBossApp\Integrations\Memberium\IAP as MemberiumIAP;
use BuddyBossApp\Integrations\MemberPress\IAP as MemberPressIAP;
use BuddyBossApp\Integrations\PmProMembership\IAP as PmProMembershipIAP;
use BuddyBossApp\Integrations\RestrictContentPro\IAP as RestrictContentProIAP;
use BuddyBossApp\Integrations\S2Member\IAP as S2MemberIAP;
use BuddyBossApp\Integrations\WcMembership\IAP as WcMembershipIAP;
use BuddyBossApp\Integrations\WishlistMember\IAP as WishlistMemberIAP;
use BuddyBossApp\Integrations\TutorLMSCourse\IAP as TutorLMSCourseIAP;

/**
 * IAP controller class.
 */
final class Controller {

	/**
	 * IAP products array.
	 *
	 * @var array $iap
	 */
	public $iap = array();

	/**
	 * Array of integration.
	 *
	 * @var array $integration
	 * todo: this should be refactor.
	 */
	public $integration = array();

	/**
	 * Array of integrations.
	 *
	 * @var array $integrations
	 */
	public $integrations = array();

	/**
	 * Option name.
	 *
	 * @var string $option_setting
	 */
	public $option_setting = 'bbapp_iap';

	/**
	 * Classs instance.
	 *
	 * @var null $instance
	 */
	private static $instance = null;

	/**
	 * LearnDash group slug identifier.
	 *
	 * @var string $learndash_group_slug
	 */
	public static $learndash_group_slug = 'learndash';

	/**
	 * LearnDash course slug identifier.
	 *
	 * @var string $learndash_course_slug
	 */
	public static $learndash_course_slug = 'learndash-course';

	/**
	 * MemberPress plugin slug identifier.
	 *
	 * @var string $memberpress_slug
	 */
	public static $memberpress_slug = 'memberpressproduct';

	/**
	 * WooCommerce Membership plugin slug identifier.
	 *
	 * @var string $wc_membership_slug
	 */
	public static $wc_membership_slug = 'wc_membership_plan';

	/**
	 * Paid Membership Pro plugin slug identifier.
	 *
	 * @var string $pmpro_membership_slug
	 */
	public static $pmpro_membership_slug = 'pmpro_membership_levels';

	/**
	 * WishList Membership plugin slug identifier.
	 *
	 * @var string $wishlist_membership_slug
	 */
	public static $wishlist_membership_slug = 'wishlist_member_level';

	/**
	 * Restrict Content Pro plugin slug identifier.
	 *
	 * @var string $restrict_content_pro_slug
	 */
	public static $restrict_content_pro_slug = 'restrict_content_pro';

	/**
	 * Memberium plugin slug identifier.
	 *
	 * @var string $memberium_slug
	 */
	public static $memberium_slug = 'memberium_level';

	/**
	 * Imember360 plugin slug identifier.
	 *
	 * @var string $imember360_slug
	 */
	public static $imember360_slug = 'imember360_level';

	/**
	 * S2 member plugin slug identifier.
	 *
	 * @var string $s2member_slug
	 */
	public static $s2member_slug = 's2member_level';

	/**
	 * TutorLMS course slug identifier.
	 *
	 * @var string $tutor_lms_course_slug
	 */
	public static $tutor_lms_course_slug = 'tutor-lms-course';

	/**
	 * Controller constructor.
	 */
	private function __construct() {
		// ... leave empty, see Singleton below.
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return Controller|null
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			// include functions(utils) file.
			require_once dirname( __FILE__ ) . '/functions.php';

			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->setup_globals();
			self::$instance->load_classes();
			self::$instance->setup_actions();
			self::$instance->setup_filters();
		}

		return self::$instance;
	}

	/**
	 * Setup Globals.
	 */
	protected function setup_globals() {
		$this->integrations = array(
			self::$learndash_course_slug     => array(
				'type'    => 'learndash-course',
				'label'   => __( 'LearnDash Course', 'buddyboss-app' ),
				'enabled' => false,
				'class'   => null,
			),
			self::$learndash_group_slug      => array(
				'type'    => 'learndash-group',
				'label'   => __( 'LearnDash Group', 'buddyboss-app' ),
				'enabled' => false,
				'class'   => null,
				'group'   => 'membership',
			),
			self::$memberium_slug            => array(
				'type'    => 'memberium',
				'label'   => __( 'Memberium', 'buddyboss-app' ),
				'enabled' => false,
				'class'   => null,
				'group'   => 'membership',
			),
			self::$memberpress_slug          => array(
				'type'    => 'memberpress',
				'label'   => __( 'MemberPress', 'buddyboss-app' ),
				'enabled' => false,
				'class'   => null,
				'group'   => 'membership',
			),
			self::$pmpro_membership_slug     => array(
				'type'    => 'pm-pro-membership',
				'label'   => __( 'Paid Memberships Pro', 'buddyboss-app' ),
				'enabled' => false,
				'class'   => null,
				'group'   => 'membership',
			),
			self::$restrict_content_pro_slug => array(
				'type'    => 'restrict-content-pro',
				'label'   => __( 'Restrict Content Pro', 'buddyboss-app' ),
				'enabled' => false,
				'class'   => null,
				'group'   => 'membership',
			),
			self::$s2member_slug             => array(
				'type'    => 's2member',
				'label'   => __( 'S2Member', 'buddyboss-app' ),
				'enabled' => false,
				'class'   => null,
				'group'   => 'membership',
			),
			self::$wishlist_membership_slug  => array(
				'type'    => 'wishlist-member',
				'label'   => __( 'Wishlist Member', 'buddyboss-app' ),
				'enabled' => false,
				'class'   => null,
				'group'   => 'membership',
			),
			self::$wc_membership_slug        => array(
				'type'    => 'woo-membership',
				'label'   => __( 'WooCommerce Memberships', 'buddyboss-app' ),
				'enabled' => false,
				'class'   => null,
				'group'   => 'membership',
			),
			self::$tutor_lms_course_slug     => array(
				'type'    => 'tutor-lms-course',
				'label'   => __( 'TutorLMS Course', 'buddyboss-app' ),
				'enabled' => false,
				'class'   => null,
			),
		);
	}

	/**
	 * Hooks here.
	 */
	protected function setup_actions() {
		add_action( 'init', array( $this, 'load_integrations' ), 1 );
	}

	/**
	 * Initiate the required classes.
	 */
	protected function load_classes() {
		if ( ! defined( 'IAP_LOG' ) ) {
			$settings = Settings::instance()->get_settings();
			if ( bbapp_is_active( 'iap' ) && ( isset( $settings['logger.iap_log.enabled'] ) && $settings['logger.iap_log.enabled'] ) ) {
				define( 'IAP_LOG', true );
			} else {
				define( 'IAP_LOG', false );
			}
		}

		SubscriptionGroupTaxonomy::instance();
		Main::instance();
		Orders::instance();
		Ios::instance();
		Android::instance();
		Ajax::instance();
		LearnDashIapRest::instance();
		TutorLmsIapRest::instance();
	}

	/**
	 * Load the integrations.
	 */
	public function load_integrations() {
		if ( bbapp_is_learndash_enabled() ) {
			LearnDashCourseIAP::instance()->set_up( $this->integrations[ self::$learndash_course_slug ]['type'], $this->integrations[ self::$learndash_course_slug ]['label'] );
			$this->integrations[ self::$learndash_course_slug ] ['enabled'] = true;
			$this->integrations[ self::$learndash_course_slug ] ['class']   = LearnDashCourseIAP::class;

			LearnDashGroupIAP::instance()->set_up( $this->integrations[ self::$learndash_group_slug ]['type'], $this->integrations[ self::$learndash_group_slug ]['label'] );
			$this->integrations[ self::$learndash_group_slug ] ['enabled'] = true;
			$this->integrations[ self::$learndash_group_slug ] ['class']   = LearnDashGroupIAP::class;
		}

		if ( function_exists( 'wc_memberships' ) ) {
			WcMembershipIAP::instance()->set_up( $this->integrations[ self::$wc_membership_slug ]['type'], $this->integrations[ self::$wc_membership_slug ]['label'] );
			$this->integrations[ self::$wc_membership_slug ] ['enabled'] = true;
			$this->integrations[ self::$wc_membership_slug ] ['class']   = WcMembershipIAP::class;
		}

		if ( class_exists( 'MeprTransaction' ) ) {
			MemberPressIAP::instance()->set_up( $this->integrations[ self::$memberpress_slug ]['type'], $this->integrations[ self::$memberpress_slug ]['label'] );
			$this->integrations[ self::$memberpress_slug ] ['enabled'] = true;
			$this->integrations[ self::$memberpress_slug ] ['class']   = MemberPressIAP::class;
		}

		if ( function_exists( 'pmpro_changeMembershipLevel' ) ) {
			PmProMembershipIAP::instance()->set_up( $this->integrations[ self::$pmpro_membership_slug ]['type'], $this->integrations[ self::$pmpro_membership_slug ]['label'] );
			$this->integrations[ self::$pmpro_membership_slug ] ['enabled'] = true;
			$this->integrations[ self::$pmpro_membership_slug ] ['class']   = PmProMembershipIAP::class;
		}

		if ( class_exists( 'WishListMember' ) ) {
			WishlistMemberIAP::instance()->set_up( $this->integrations[ self::$wishlist_membership_slug ]['type'], $this->integrations[ self::$wishlist_membership_slug ]['label'] );
			$this->integrations[ self::$wishlist_membership_slug ] ['enabled'] = true;
			$this->integrations[ self::$wishlist_membership_slug ] ['class']   = WishlistMemberIAP::class;
		}

		if ( class_exists( 'Restrict_Content_Pro' ) ) {
			RestrictContentProIAP::instance()->set_up( $this->integrations[ self::$restrict_content_pro_slug ]['type'], $this->integrations[ self::$restrict_content_pro_slug ]['label'] );
			$this->integrations[ self::$restrict_content_pro_slug ] ['enabled'] = true;
			$this->integrations[ self::$restrict_content_pro_slug ] ['class']   = RestrictContentProIAP::class;
		}

		if ( defined( 'MEMBERIUM_SKU' ) ) {
			MemberiumIAP::instance()->set_up( $this->integrations[ self::$memberium_slug ]['type'], $this->integrations[ self::$memberium_slug ]['label'] );
			$this->integrations[ self::$memberium_slug ] ['enabled'] = true;
			$this->integrations[ self::$memberium_slug ] ['class']   = MemberiumIAP::class;
		}

		if ( defined( 'WS_PLUGIN__S2MEMBER_VERSION' ) ) {
			S2MemberIAP::instance()->set_up( $this->integrations[ self::$s2member_slug ]['type'], $this->integrations[ self::$s2member_slug ]['label'] );
			$this->integrations[ self::$s2member_slug ] ['enabled'] = true;
			$this->integrations[ self::$s2member_slug ] ['class']   = S2MemberIAP::class;
		}

		if ( bbapp_is_tutor_lms_plugins_active() ) {
			TutorLMSCourseIAP::instance()->set_up( $this->integrations[ self::$tutor_lms_course_slug ]['type'], $this->integrations[ self::$tutor_lms_course_slug ]['label'] );
			$this->integrations[ self::$tutor_lms_course_slug ] ['enabled'] = true;
			$this->integrations[ self::$tutor_lms_course_slug ] ['class']   = TutorLMSCourseIAP::class;
		}
	}

	/**
	 * Get all Integrations
	 *
	 * @return array Integrations list
	 */
	public function get_integrations() {
		return $this->integrations;
	}

	/**
	 * Get all Active Integrations.
	 *
	 * @return array
	 */
	public function get_enabled_integrations() {
		$all_integrations    = $this->integrations;
		$active_integrations = array();

		if ( ! empty( $all_integrations ) ) {
			foreach ( $all_integrations as $key => $integration ) {
				if ( ! empty( $integration['enabled'] ) && ! empty( $integration['label'] ) ) {
					$active_integrations[ $key ] = $integration;
				}
			}
		}

		return $active_integrations;
	}

	/**
	 * Get DB global prefix.
	 *
	 * @return mixed
	 */
	public function get_global_dbprefix() {
		return bbapp_get_global_db_prefix();
	}

	/**
	 * Return the rest Settings.
	 *
	 * @return array
	 */
	public function get_rest_settings() {
		$products                 = bbapp_is_active( 'iap' ) ? Helpers::get_products() : array();
		$lock_app                 = bbapp_is_active( 'iap' ) ? AppSettings::instance()->get_setting_value( 'iap.lock_app' ) : false;
		$purchase_before_register = bbapp_is_active( 'iap' ) ? AppSettings::instance()->get_setting_value( 'iap.purchase_before_register' ) : false;
		$register_show_products   = bbapp_is_active( 'iap' ) ? AppSettings::instance()->get_setting_value( 'iap.register.show_products' ) : '';
		$register_show_products   = ! empty( $register_show_products ) ? $register_show_products : 'before_registration';
		$terms_page_id            = bbapp_is_active( 'iap' ) ? AppSettings::instance()->get_setting_value( 'iap.terms' ) : null;
		$policy_page_id           = bbapp_is_active( 'iap' ) ? AppSettings::instance()->get_setting_value( 'iap.policy' ) : null;
		$review_version_ios       = bbapp_is_active( 'iap' ) ? AppSettings::instance()->get_setting_value( 'iap.review_version.ios' ) : null;
		$subscribe_message        = bbapp_is_active( 'iap' ) ? AppSettings::instance()->get_setting_value( 'iap.subscribe_message' ) : null;
		$has_access               = true;

		if ( ! empty( $products ) ) {
			$has_access = false;
			foreach ( $products as $product ) {
				if ( $product['has_access'] ) {
					$has_access = true;
					break;
				}
			}
		}

		$has_access = bbapp_is_active( 'iap' ) ? $has_access : false;

		/**
		 * When Access Control Module is Active we will override the lock_app to false.
		 * It's recommended the site admin should use access controls module to lock the app further.
		 */
		$lock_app   = bbapp_is_active( 'access_controls' ) ? false : 1 === (int) $lock_app;
		$properties = array(
			'has_access'               => $has_access,
			'lock_app'                 => $lock_app,
			'purchase_before_register' => ( 1 === $purchase_before_register ) ? true : false,
			'register_show_products'   => ( 1 === (int) $purchase_before_register ) ? $register_show_products : false,
			'terms_page_id'            => (int) $terms_page_id,
			'policy_page_id'           => (int) $policy_page_id,
			'review_version_ios'       => ! empty( $review_version_ios ) ? $review_version_ios : null,
			'subscribe_message'        => $subscribe_message,
			'products'                 => $products,
		);

		return $properties;
	}

	/**
	 * When BuddyBoss App plugin is activated
	 */
	public function on_activation() {
		global $wpdb;

		$global_prefix   = \bbapp_iap()->get_global_dbprefix();
		$charset_collate = $wpdb->get_charset_collate();
		$table_name      = "{$global_prefix}bbapp_iap_products";
		$meta_table_name = "{$global_prefix}bbapp_iap_productmeta";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$sql = "CREATE TABLE {$table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
            blog_id bigint(20) NOT NULL,
            product_author_id bigint(20) NOT NULL,
            name varchar(100) NOT NULL,
            tagline longtext NOT NULL,
            description longtext NOT NULL,
            store_data longtext NOT NULL,
            misc_settings longtext NOT NULL,            
            integration_data longtext NOT NULL,
            status varchar(20) NOT NULL DEFAULT 'published',
            date_created datetime DEFAULT '0000-00-00 00:00:00' NULL,
            date_updated datetime DEFAULT '0000-00-00 00:00:00' NULL,
            iap_group bigint(20) NOT NULL DEFAULT 0,
            menu_order int(11) NOT NULL default '0',
            KEY  blog_id (blog_id),
            KEY  product_author_id (product_author_id),
            KEY  date_created (date_created),
            KEY  date_updated (date_updated)
		    ) {$charset_collate}";
		dbDelta( $sql );

		$meta_sql = "CREATE TABLE {$meta_table_name} (
		    meta_id bigint(20) unsigned NOT NULL auto_increment,
			iap_id bigint(20) unsigned NOT NULL default '0',
			meta_key varchar(255) default NULL,
			meta_value longtext,
			PRIMARY KEY  (meta_id),
			KEY iap_id (iap_id),
        	KEY meta_key (meta_key(191))
		) {$charset_collate}";
		dbDelta( $meta_sql );
	}

	/**
	 * When BuddyBoss App plugin is deactivated
	 */
	public function on_plugin_deactivate() {
	}

	/**
	 * Define all filter hooks.
	 *
	 * @since 1.3.7
	 *
	 * @return void
	 */
	protected function setup_filters() {
		// Unlock the privacy and terms page When the Web and APP is private mode.
		add_filter( 'bp_private_network_pre_check', array( $this, 'unlock_restricted_page' ) );
	}

	/**
	 * Unlock the privacy and term page for IAP products.
	 *
	 * @param bool $retval Access privacy mode for terms and privacy content.
	 *
	 * @since 1.3.7
	 *
	 * @return bool
	 */
	public function unlock_restricted_page( $retval ) {
		global $wp_query;

		$current_page_object = $wp_query->get_queried_object();
		$page_id             = isset( $current_page_object->ID ) ? $current_page_object->ID : get_the_ID();
		$page_id             = ( ! empty( $page_id ) ) ? $page_id : 0;

		if ( empty( $page_id ) ) {
			return $retval;
		}

		$terms_page_id  = bbapp_is_active( 'iap' ) ? AppSettings::instance()->get_setting_value( 'iap.terms' ) : '';
		$policy_page_id = bbapp_is_active( 'iap' ) ? AppSettings::instance()->get_setting_value( 'iap.policy' ) : '';

		if ( ! empty( $terms_page_id ) && ( (int) $page_id === (int) $terms_page_id ) ) {
			return true;
		}

		if ( ! empty( $policy_page_id ) && ( (int) $page_id === (int) $policy_page_id ) ) {
			return true;
		}

		return $retval;
	}
}
