<?php
/**
 * Holds Android in-app purchase functionality.
 *
 * @package BuddyBossApp\InAppPurchases
 */

namespace BuddyBossApp\InAppPurchases;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\Admin\Configure;
use BuddyBossApp\Admin\InAppPurchases\Helpers;
use BuddyBossApp\Library\Composer;
use BuddyBossApp\Tools\Logger;
use Google_Service_AndroidPublisher_SubscriptionPurchasesAcknowledgeRequest;
use WP_Error as WP_Error;

/**
 * Android In-app purchase class.
 */
final class Android extends StoreAbstract {

	/**
	 * Class instance.
	 *
	 * @var null $instance
	 */
	private static $instance = null;

	/**
	 * Android constructor.
	 */
	private function __construct() {
		// ... leave empty, see Singleton below
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return Controller|null
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Initialize IntegrationAbstract
	 */
	public function init() {
		// Device platform type.
		$platform_type       = 'android';
		$store_product_types = Helpers::platform_store_product_types( $platform_type );

		// Register StoreAbstract Type.
		parent::set_up( $platform_type, __( 'Android', 'buddyboss-app' ), $store_product_types );
		// Register instance.
		bbapp_iap()->iap[ $platform_type ] = $this::instance();
		// Register integration.
		bbapp_iap()->integration[ $platform_type ] = $store_product_types;
	}

	/**
	 * For rendering product settings.
	 *
	 * @param string $integration Integration name.
	 * @param object $item        Product data.
	 */
	public function render_product_settings( $integration, $item ) {
	}

	/**
	 * For saving product settings
	 *
	 * @param string $integration Integration name.
	 * @param object $item        Product data.
	 */
	public function save_product_settings( $integration, $item ) {
	}

	/**
	 * Returns Android Publisher.
	 *
	 * @return \Google\Service\AndroidPublisher|\Google_Service_AndroidPublisher|WP_Error
	 * @throws \Google\Exception Google Exception.
	 */
	public function get_android_publisher() {
		$file_name                   = Configure::instance()->option( 'publish.android.account_key' );
		$google_service_account_json = bbapp_get_upload_full_path( $file_name );
		$google_service_account_name = __( 'Play Publisher', 'buddyboss-app' );

		if ( empty( $file_name ) || ! file_exists( $google_service_account_json ) ) {
			return new WP_Error( 'google_iap_not_configured', __( 'Google In-App Purchase setup is not configured. Please configure it on admin settings.', 'buddyboss-app' ) );
		}

		$google_client = Composer::instance()->google_instance()->Google_Client();
		$google_client->setApplicationName( $google_service_account_name );
		$google_client->setScopes( 'https://www.googleapis.com/auth/androidpublisher' );
		$google_client->setAuthConfig( $google_service_account_json );

		$google_android_publisher = Composer::instance()->google_instance()->Google_Service_AndroidPublisher( $google_client );

		return $google_android_publisher;
	}

	/**
	 * Should be override by iap type.
	 *
	 * @param array $data Purchase data.
	 *
	 * @return array|\Google_Service_AndroidPublisher|WP_Error
	 */
	public function process_payment( $data ) {
		$bbapp_product_id   = $data['bbapp_product_id']; // BuddyBossAppProduct ID.
		$store_product_type = $data['store_product_type']; // Store Product Type.
		$store_product_id   = trim( $data['store_product_id'] ); // Store Product ID.
		$iap_receipt_token  = trim( $data['iap_receipt_token'] ); // Receipt Token.

		if ( IAP_LOG ) {
			Logger::instance()->add( 'iap_log', sprintf( 'BuddyBossApp\InAppPurchases\Android->process_payment(),BuddyBossAppProductId: %s, storeProductType: %s, storeProductId: %s ', $bbapp_product_id, $store_product_type, $store_product_id ) );
		}

		// NOTE : Getting appnamespace(package name) stored in _options table.
		$app_namespace = Configure::instance()->option( 'publish.android.namespace' );

		if ( ! isset( $app_namespace ) || empty( $app_namespace ) ) {
			return new WP_Error( 'configuration_error', __( 'No app configuration found. <a href="admin.php?page=bbapp-build">here</a>', 'buddyboss-app' ) );
		}

		$transaction_data = array(
			'transaction_date'      => null,
			'parent_transaction_id' => null,
			'transaction_id'        => null,
			'data'                  => array(),
			'expire_at'             => false, // should be given if product is recurring.
		);

		$google_android_publisher = $this->get_android_publisher();

		if ( is_wp_error( $google_android_publisher ) ) {
			return $google_android_publisher;
		}

		try {
			/**
			 * Managed Product have below types :
			 * 1. Consumable (We're not supporting it)
			 * 2. Non-Consumable.
			 */

			if ( in_array( $store_product_type, array( 'auto_renewable' ), true ) ) {
				// Acknowledge the purchase.
				$acknowledge_developer_payload = Composer::instance()->google_instance()->Google_Service_AndroidPublisher_SubscriptionPurchasesAcknowledgeRequest();
				$google_android_publisher->purchases_subscriptions->acknowledge( $app_namespace, $store_product_id, $iap_receipt_token, $acknowledge_developer_payload );

				$product_response = $google_android_publisher->purchases_subscriptions->get(
					$app_namespace,
					$store_product_id,
					$iap_receipt_token
				);
			} else {
				// Acknowledge the purchase.
				$acknowledge_developer_payload = Composer::instance()->google_instance()->Google_Service_AndroidPublisher_ProductPurchasesAcknowledgeRequest();
				$google_android_publisher->purchases_products->acknowledge( $app_namespace, $store_product_id, $iap_receipt_token, $acknowledge_developer_payload );

				$product_response = $google_android_publisher->purchases_products->get(
					$app_namespace,
					$store_product_id,
					$iap_receipt_token
				);
			}

			// Validate if response is correct.
			if ( empty( $product_response->getOrderId() ) ) {
				if ( isset( $data['order_id'] ) ) {
					$ref_order_id           = $data['order_id'];
					$product_response_debug = '';
					if ( defined( 'WP_DEBUG' ) && WP_DEBUG === true ) {
						// phpcs:disable WordPress.PHP.DevelopmentFunctions
						$product_response_debug = print_r( $product_response, true );
						// phpcs:enable
					}

					bbapp_iap_add_logs( 'bbapp_iap_google', "Invalid Google InAppPurchase Response - \n\n " . $product_response_debug, $ref_order_id );
				}

				return new WP_Error( 'error_iap_validation', __( 'Error while reading In-App Purchase Token. Response found to be invalid.', 'buddyboss-app' ) );
			}

			$order_id        = $product_response->getOrderId();
			$parent_order_id = $order_id;

			if ( in_array( $store_product_type, array( 'auto_renewable' ), true ) ) {
				// remove postfix if available to get original order_id.
				$parent_order_id               = explode( '..', $parent_order_id );
				$parent_order_id               = $parent_order_id[0];
				$transaction_data['expire_at'] = gmdate( 'Y-m-d H:i:s', $product_response->getExpiryTimeMillis() / 1000 );

				// Validate Expiry.
				$time = strtotime( gmdate( 'Y-m-d H:i:s' ) );

				if ( strtotime( $transaction_data['expire_at'] ) < $time ) {
					return new WP_Error( 'error_iap_validation', __( 'Google purchase has expired.', 'buddyboss-app' ) );
				}
			}

			// Check if it's nt exists.
			$iap_exists = $this->do_transaction_exists( $parent_order_id );

			if ( ! empty( $iap_exists ) ) {
				// On none consumable product type. we allow one to be restore if user account is same.
				if ( ! in_array( $store_product_type, array( 'non_consumable', 'auto_renewable' ), true ) ) {
					return new WP_Error( 'error_iap_validation', __( 'Matching product(s) found, but no usable transaction found.', 'buddyboss-app' ) );
				}

				if ( in_array( $store_product_type, array( 'auto_renewable' ), true ) ) {
					$subscription_order = false; // holds false until we found any useful order in existing order.

					foreach ( $iap_exists as $order ) {
						if ( in_array( $order->order_status, array( 'subscribed', 'completed' ), true ) ) {
							$subscription_order = true; // found one valid 'subscribed' order.
						}
					}

					// looks like subscription is already being used by another order.
					if ( $subscription_order ) {
						return new WP_Error( 'error_iap_validation', __( 'Subscription is already being used on another order.', 'buddyboss-app' ) );
					}
				}
			}

			$transaction_data['data']['sandbox']       = ( ! empty( $product_response->getPurchaseType() ) && '0' === (string) $product_response->getPurchaseType() ) ? 1 : 0;
			$transaction_data['transaction_id']        = $order_id;
			$transaction_data['parent_transaction_id'] = $parent_order_id;

			// Extra data for future purpose.
			$transaction_data['data']['purchase_kind'] = $product_response->getKind();
			$transaction_data['data']['trial_period']  = false;

			if ( in_array( $store_product_type, array( 'auto_renewable' ), true ) ) {
				$transaction_data['transaction_date']    = gmdate( 'Y-m-d H:i:s', $product_response->getStartTimeMillis() / 1000 );
				$transaction_data['transaction_date_ms'] = $product_response->getStartTimeMillis();
				if ( ! empty( $product_response->getPaymentState() ) ) {
					$transaction_data['data']['payment_state'] = $product_response->getPaymentState();
				}
				if ( ! empty( $product_response->getProfileId() ) ) {
					$transaction_data['data']['profileId'] = $product_response->getProfileId();
				}
				if ( ! empty( $product_response->getProfileName() ) ) {
					$transaction_data['data']['profile_name'] = $product_response->getProfileName();
				}
				if ( ! empty( $product_response->getLinkedPurchaseToken() ) ) {
					$transaction_data['data']['linked_purchase_token'] = $product_response->getLinkedPurchaseToken();
				}
				if ( ! empty( $product_response->getPaymentState() ) && '2' === $product_response->getPaymentState() ) {
					$transaction_data['data']['trial_period'] = true;
					$transaction_data['data']['had_trial']    = true;
				}
			} else {
				$transaction_data['transaction_date']    = gmdate( 'Y-m-d H:i:s', $product_response->getPurchaseTimeMillis() / 1000 );
				$transaction_data['transaction_date_ms'] = $product_response->getPurchaseTimeMillis();
				if ( ! empty( $product_response->getPurchaseState() ) ) {
					$transaction_data['data']['purchase_state'] = $product_response->getPurchaseState();
				}
			}

			return $transaction_data;
		} catch ( \Exception $e ) {
			$json   = json_decode( $e->getMessage() );
			$reason = '';

			if ( isset( $json->error ) && isset( $json->error->errors ) && isset( $json->error->errors[0] ) ) {
				/**
				 * Possible/Known Reason
				 * purchaseTokenDoesNotMatchProductId || purchaseTokenDoesNotMatchSubscriptionId
				 */
				$reason = $json->error->errors[0]->reason;
			}

			if ( isset( $data['order_id'] ) ) {
				$ref_id = $data['order_id'];
				bbapp_iap_add_logs( 'bbapp_iap_google', "Error while validation error - \n\n " . $reason, $ref_id );
			}

			/* translators: %s: Error reason. */
			return new WP_Error( 'error_iap_validation', sprintf( __( 'Error while reading InAppPurchase token, could be invalid receipt token. Reason : %s.', 'buddyboss-app' ), $reason ) );
		}
	}

	/**
	 * Validates order payment status
	 *
	 * @param array $data Purchase data.
	 *
	 * @return array|\Google_Service_AndroidPublisher|mixed|WP_Error
	 */
	public function validate( $data ) {
		if ( IAP_LOG ) {
			Logger::instance()->add( 'iap_log', 'BuddyBossApp\InAppPurchases\Android->validate()' );
		}

		$bbapp_product_id   = $data['bbapp_product_id']; // BuddyBossAppProduct ID.
		$iap_receipt_token  = trim( $data['iap_receipt_token'] );
		$store_product_type = trim( $data['store_product_type'] );
		$store_product_id   = trim( $data['store_product_id'] );

		$transaction_data = array(
			'status'                => '',
			'transaction_date'      => '',
			'parent_transaction_id' => '',
			'transaction_id'        => '',
			'data'                  => array(),
			'expire_at'             => '',
			'history'               => array(),
		);

		$google_android_publisher = $this->get_android_publisher();

		if ( is_wp_error( $google_android_publisher ) ) {
			return $google_android_publisher;
		}

		// NOTE : Getting appnamespace(package name) stored in _options table.
		$app_namespace = Configure::instance()->option( 'publish.android.namespace' );

		if ( ! isset( $app_namespace ) || empty( $app_namespace ) ) {
			return new WP_Error( 'configuration_error', __( 'Error while retrieving app namespace, Android namespace not found(or configured). Configure <a href="admin.php?page=bbapp-build">here</a>', 'buddyboss-app' ) );
		}

		try {
			if ( ! in_array( $store_product_type, array( 'non_consumable', 'auto_renewable' ), true ) ) {
				return new WP_Error( 'error_iap_validation', __( 'Error while reading InAppPurchase token, Invalid Product Type.', 'buddyboss-app' ) );
			}

			$product_response = $google_android_publisher->purchases_subscriptions->get( $app_namespace, $store_product_id, $iap_receipt_token );

			// validate the response is correct.
			if ( empty( $product_response->getOrderId() ) ) {
				return new WP_Error( 'error_iap_validation', __( 'Error while reading InAppPurchase token, Response found to be invalid.', 'buddyboss-app' ) );
			}

			$order_id        = $product_response->getOrderId();
			$parent_order_id = $order_id;

			if ( in_array( $store_product_type, array( 'auto_renewable' ), true ) ) {
				// remove postfix if available to get original order_id.
				$parent_order_id               = explode( '..', $parent_order_id );
				$parent_order_id               = $parent_order_id[0];
				$transaction_data['expire_at'] = gmdate( 'Y-m-d H:i:s', $product_response->getExpiryTimeMillis() / 1000 );
			}

			$transaction_data['data']['sandbox'] = ( ! empty( $product_response->getPurchaseType() ) && '0' === (string) $product_response->getPurchaseType() ) ? 1 : 0;

			$transaction_data['transaction_id']        = $order_id;
			$transaction_data['parent_transaction_id'] = $parent_order_id;

			if ( is_null( $product_response->getCancelReason() ) && 0 !== $product_response->getPaymentState() ) {
				// Acknowledge the purchase.
				$acknowledge_developer_payload = Composer::instance()->google_instance()->Google_Service_AndroidPublisher_SubscriptionPurchasesAcknowledgeRequest();
				$google_android_publisher->purchases_subscriptions->acknowledge( $app_namespace, $store_product_id, $iap_receipt_token, $acknowledge_developer_payload );
			}

			if ( in_array( $store_product_type, array( 'auto_renewable' ), true ) ) {
				$transaction_data['transaction_date'] = gmdate( 'Y-m-d H:i:s', $product_response->getStartTimeMillis() / 1000 );
				// Extra data for future purpose.
				if ( ! empty( $product_response->getStartTimeMillis() ) ) {
					$transaction_data['data']['transaction_date_ms'] = $product_response->getStartTimeMillis();
				}
				if ( ! empty( $product_response->getPaymentState() ) ) {
					$transaction_data['data']['payment_state'] = $product_response->getPaymentState();
				}
			} else {
				$transaction_data['transaction_date'] = gmdate( 'Y-m-d H:i:s', $product_response->getPurchaseTimeMillis() / 1000 );
				// Extra data for future purpose.
				if ( ! empty( $product_response->getPurchaseTimeMillis() ) ) {
					$transaction_data['data']['transaction_date_ms'] = $product_response->getPurchaseTimeMillis();
				}
				if ( ! empty( $product_response->getPurchaseState() ) ) {
					$transaction_data['data']['purchase_state'] = $product_response->getPurchaseState();
				}
			}

			// Validate Expiry.
			$time = strtotime( gmdate( 'Y-m-d H:i:s' ) );

			if ( strtotime( $transaction_data['expire_at'] ) < $time ) {
				// Looks like transaction is expired.
				$transaction_data['status'] = 'expired';

				// Check if it's not really cancel then give some retry.
				if ( ! empty( $product_response->getAutoRenewing() ) && ! is_int( $product_response->getCancelReason() ) ) {
					$transaction_data['status'] = 'retrying';

					return $transaction_data;
				}

				$expire_reason = $this->get_cancel_text_reason( $product_response->getCancelReason() );
				/* translators: %s: Expire reason. */
				$transaction_data['history'][]                           = sprintf( __( 'Expire Reason : %s', 'buddyboss-app' ), $expire_reason );
				$transaction_data['data']['last_expiration_reason_code'] = $product_response->getCancelReason();

				if ( ! empty( $product_response->getUserCancellationTimeMillis() ) ) {
					$transaction_data['data']['userCancellationTimeMillis'] = $product_response->getUserCancellationTimeMillis();
				}
			} else {
				// Trial stuff.
				$transaction_data['data']['trial_period'] = false;

				if ( ! empty( $product_response->getPaymentState() ) && '2' === (string) $product_response->getPaymentState() ) {
					$transaction_data['data']['trial_period']        = true;
					$transaction_data['data']['transaction_history'] = __( 'Transaction is renewed as free trial.', 'buddyboss-app' );
				}

				$transaction_data['status'] = 'subscribed';
			}

			return $transaction_data;
		} catch ( \Exception $e ) {
			$json   = json_decode( $e->getMessage() );
			$reason = '';

			if ( isset( $json->error ) && isset( $json->error->errors ) && isset( $json->error->errors[0] ) ) {
				/**
				 * Possible/Known Reason
				 * purchaseTokenDoesNotMatchProductId || purchaseTokenDoesNotMatchSubscriptionId
				 */
				$reason = $json->error->errors[0]->reason;
			}

			/* translators: %s. Reason. */

			return new WP_Error( 'error_iap_validation', sprintf( __( 'Error while reading InAppPurchase token, could be invalid receipt token. Reason : %s.', 'buddyboss-app' ), $reason ) );
		}
	}

	/**
	 * Returns reason of cancel in text.
	 *
	 * @param string $code Code number.
	 *
	 * @return mixed
	 */
	public function get_cancel_text_reason( $code ) {
		$reason = __( 'Unknown', 'buddyboss-app' );
		switch ( $code ) {
			case '0':
				$reason = __( 'User cancelled the subscription.', 'buddyboss-app' );
				break;
			case '1':
				$reason = __( 'Subscription was cancelled by the system. For example, due to a billing problem.', 'buddyboss-app' );
				break;
			case '2':
				$reason = __( 'Subscription was replaced with a new subscription.', 'buddyboss-app' );
				break;
			case '3':
				$reason = __( 'Subscription was cancelled by the developer.', 'buddyboss-app' );
				break;
			default:
				$reason = __( 'Unknown', 'buddyboss-app' );
				break;
		}
		return $reason;
	}

	/**
	 * Sync store product with local data.
	 */
	public function sync_store_product() {

		$products = \BuddyBossApp\AppStores\Android::instance()->sync_store_product();

		update_option( "bbapp_{$this->type}_store_product", $products );

		return $products;
	}
}
