<?php
/**
 * Holds Abstract class functionality related deepliking types.
 *
 * @package BuddyBossApp\DeepLinking\Type
 */

namespace BuddyBossApp\DeepLinking\Type;

/**
 * Abstract class for the different types of deppliking.
 */
abstract class TypeAbstract {

	/**
	 * URL data.
	 *
	 * @var array $url_data
	 */
	private static $url_data = array();

	/**
	 * Construct function.
	 */
	public function __construct() {
		add_filter( 'bbapp_deeplinking_details', array( $this, 'get_details' ), 10, 2 );
	}

	/**
	 * Get url parse data.
	 *
	 * @param object $response Response.
	 * @param string $url      URL to get details.
	 *
	 * @return mixed
	 */
	public function get_details( $response, $url ) {
		if ( empty( $response ) ) {
			$response = $this->parse( $url );
		}

		return $response;
	}

	/**
	 * Parse url.
	 *
	 * @param string $url URL to parse.
	 *
	 * @return mixed
	 */
	abstract public function parse( $url );

	/**
	 * Get embed object
	 *
	 * @param string $rest_endpoint Rest endpoint.
	 * @param array  $args          Arguments.
	 *
	 * @return false|string
	 */
	public function get_embed( $rest_endpoint, $args = array() ) {
		$request  = new \WP_REST_Request( 'GET', $rest_endpoint );
		$response = rest_do_request( $request );

		$request->set_query_params( $args );

		$server = rest_get_server();
		$data   = $server->response_to_data( $response, false );

		return $data;
	}

	/**
	 * Get url data for WordPress core features.
	 * For getting URL data we are mating given url with WordPress rewrite rule one by one.
	 * we fetch query variable form match rewrite rule and used that variable to figure our url refer to which object
	 * We refer `url_to_postid` function to get data of given url
	 *
	 * @param string $url URL to get the data.
	 *
	 * @return array
	 */
	public function get_url_data( $url ) {
		global $wp_rewrite;

		$parse_url    = wp_parse_url( $url );
		$query_params = array();

		if ( ! empty( $parse_url['query'] ) ) {
			parse_str( $parse_url['query'], $query_params );
		}

		// Search category only from the root url. Root url validation `'/' === $parse_url['path']`.
		if ( ! empty( $query_params['cat'] ) && '/' === $parse_url['path'] ) {
			$cat_url = get_category_link( (int) $query_params['cat'] );
			$url     = ! empty( $cat_url ) ? $cat_url : $url;
		}

		// Check whether statics data already set or not.
		if ( empty( self::$url_data[ $url ] ) ) {
			// Check whether Transient Cache set or not.
			$hash_url   = md5( $url );
			$cache_data = get_transient( '_bbapp_deeplink_wp_url_data' );
			if ( empty( $cache_data ) || ! isset( $cache_data ) ) {
				$cache_data = array();
			}

			if ( empty( $cache_data ) || ( is_array( $cache_data ) && ! isset( $cache_data[ $hash_url ] ) ) ) {
				// Get rid of the #anchor.
				$url_split = explode( '#', $url );
				$url       = $url_split[0];

				// Get rid of URL ?query=string.
				$url_split = explode( '?', $url );
				$url       = $url_split[0];

				// Set the correct URL scheme.
				$scheme = wp_parse_url( home_url(), PHP_URL_SCHEME );
				$url    = set_url_scheme( $url, $scheme );

				// Add 'www.' if it is absent and should be there.
				if ( false !== strpos( home_url(), '://www.' ) && false === strpos( $url, '://www.' ) ) {
					$url = str_replace( '://', '://www.', $url );
				}

				// Strip 'www.' if it is present and shouldn't be.
				if ( false === strpos( home_url(), '://www.' ) ) {
					$url = str_replace( '://www.', '://', $url );
				}

				// Strip 'index.php/' if we're not using path info permalinks.
				if ( ! $wp_rewrite->using_index_permalinks() ) {
					$url = str_replace( $wp_rewrite->index . '/', '', $url );
				}

				if ( false !== strpos( trailingslashit( $url ), home_url( '/' ) ) ) {
					// Chop off http://domain.com/[path].
					$url = str_replace( home_url(), '', $url );
				} else {
					// Chop off /path/to/blog.
					$home_path = wp_parse_url( home_url( '/' ) );
					$home_path = isset( $home_path['path'] ) ? $home_path['path'] : '';
					$url       = preg_replace( sprintf( '#^%s#', preg_quote( $home_path, null ) ), '', trailingslashit( $url ) );
				}

				// Trim leading and lagging slashes.
				$url     = trim( $url, '/' );
				$request = $url;

				// Look for matches.
				$request_match = $request;

				// Check to see if we are using rewrite rules.
				$rewrite = $wp_rewrite->wp_rewrite_rules();

				// Not using rewrite rules, and 'p=N' and 'page_id=N' methods failed, so we're out of options.
				if ( ! empty( $rewrite ) ) {

					foreach ( (array) $rewrite as $match => $query ) {

						// If the requesting file is the anchor of the match, prepend it
						// to the path info.
						if ( ! empty( $url ) && ( $url !== $request ) && 0 === ( strpos( $match, $url ) ) ) {
							$request_match = $url . '/' . $request;
						}

						if ( preg_match( "#^$match#", $request_match, $matches ) ) {

							if ( $wp_rewrite->use_verbose_page_rules && preg_match( '/pagename=\$matches\[([0-9]+)\]/', $query, $varmatch ) ) {
								// This is a verbose page match, let's check to be sure about it.
								$page = get_page_by_path( $matches[ $varmatch[1] ] );
								if ( ! $page ) {
									continue;
								}

								$post_status_obj = get_post_status_object( $page->post_status );

								if ( ! $post_status_obj->public && ! $post_status_obj->protected && ! $post_status_obj->private && $post_status_obj->exclude_from_search ) {
									continue;
								}
							}

							// Got a match.
							$matched_rule = $match;
							break;
						}
					}

					if ( isset( $matched_rule ) ) {
						// Trim the query of everything up to the '?'.
						$query = preg_replace( '!^.+\?!', '', $query );

						// Substitute the substring matches into the query.
						$query = addslashes( \WP_MatchesMapRegex::apply( $query, $matches ) );

						// Parse the query.
						parse_str( $query, $perma_query_vars );

						// set transient data.
						$cache_data[ $hash_url ] = $perma_query_vars;
						set_transient( '_bbapp_deeplink_wp_url_data', $cache_data, 12 * HOUR_IN_SECONDS );

						// set data to static variable.
						self::$url_data[ $url ] = $perma_query_vars;
					}
				}
			} else {
				// set transient data to static variable.
				self::$url_data[ $url ] = $cache_data[ $hash_url ];
			}
		}

		// Check Data is set or empty.
		if ( ! empty( self::$url_data[ $url ] ) ) {
			return self::$url_data[ $url ];
		}

		return array();
	}
}
