<?php
/**
 * DeepLinking for BB app core type.
 *
 * @package BuddyBossApp\DeepLinking\Type
 */

namespace BuddyBossApp\DeepLinking\Type;

use BuddyBossApp\Admin\InAppPurchases\Helpers;
use BuddyBossApp\AppSettings;
use BuddyBossApp\DeepLinking\Screens;

/**
 * Class CoreType
 *
 * @package BuddyBossApp\DeepLinking\Type
 */
class CoreType extends TypeAbstract {

	/**
	 * Parse Term urls.
	 *
	 * @param string $url Deeplink url.
	 *
	 * @return array|mixed|null
	 */
	public function parse( $url ) {
		$data = $this->get_url_data( $url );
		if ( ! empty( $data ) ) {
			return $data;
		}

		/**
		 * Parse homepage URL.
		 */
		if ( trim( $url, '/' ) === get_home_url() ) {
			/**
			 * If BuddyBoss Platform is active, activity component is enabled,
			 * and activity component is set as front page, then open home screen.
			 */
			$action = ( defined( 'BP_PLATFORM_VERSION' ) &&
			            bp_is_active( 'activity' ) &&
			            bp_is_component_front_page( 'activity' ) ) ? 'open_home' : 'open_page';

			return array(
				'action'    => $action,
				'namespace' => 'core',
				'url'       => $url,
			);
		}

		/**
		 * Parse Posts Page URL.
		 */
		$page_for_posts = get_option( 'page_for_posts' );
		$link           = get_permalink( $page_for_posts );
		if ( untrailingslashit( $url ) === untrailingslashit( $link ) ) {
			return array(
				'action'    => 'open_archive',
				'namespace' => 'core',
				'url'       => $url,
				'post_type' => 'post',
			);
		}

		/**
		 * Parse Posts URL From year/month/day link format.
		 */
		$query_args = TypeAbstract::get_url_data( $url );
		$has_date   = array_intersect( array_keys( $query_args ), array( 'year', 'monthnum', 'day' ) );
		$url_id     = url_to_postid( $url );

		if ( empty( $url_id ) && ! empty( $has_date ) && empty( $query_args['category_name'] ) ) {
			$response = array(
				'action'    => 'open_archive',
				'namespace' => 'core',
				'url'       => $url,
				'post_type' => 'post',
			);

			if ( ! empty( $query_args['year'] ) ) {
				$response['year'] = $query_args['year'];
			}

			if ( ! empty( $query_args['monthnum'] ) ) {
				$response['monthnum'] = $query_args['monthnum'];
			}

			if ( ! empty( $query_args['day'] ) ) {
				$response['day'] = $query_args['day'];
			}

			return $response;
		}

		// Check if it's forgot password page.
		if ( isset( $query_args['name'] ) && 'wp-login.php' === $query_args['name'] ) {
			$parse_url = wp_parse_url( $url );

			if ( ! empty( $parse_url['query'] ) ) {
				parse_str( $parse_url['query'], $query_params );

				if ( 'rp' === $query_params['action'] ) {
					$response = array(
						'action'    => 'open_reset_password',
						'namespace' => 'core',
						'url'       => $url,
					);

					if ( isset( $query_params['key'] ) ) {
						$response['key'] = $query_params['key'];
					}

					if ( isset( $query_params['login'] ) ) {
						$response['login'] = $query_params['login'];
					}

					if ( isset( $query_params['wp_lang'] ) ) {
						$response['wp_lang'] = $query_params['wp_lang'];
					}

					return $response;
				}
			}
		}

		// Check if it's search page.
		if ( strpos( $url, '?s=' ) !== false || strpos( $url, '&s=' ) !== false ) {
			$response = array(
				'action'    => 'inapp',
				'namespace' => 'url',
				'url'       => $url,
			);
			return $response;
		}

		if ( AppSettings::instance()->get_setting_value( 'app_sharing.enabled' ) ) {
			$page_slug = trim( wp_parse_url( $url, PHP_URL_PATH ), '/' );

			if ( 'dl' === $page_slug ) {
				return array(
					'action'    => 'outapp',
					'namespace' => 'external_url',
					'url'       => $url,
				);
			}
		}

		return null;
	}

	/**
	 * Get url data.
	 *
	 * @param string $url Deeplink url.
	 *
	 * @return array
	 */
	public function get_url_data( $url ) {
		$url_data   = array();
		$url_params = Screens::instance()->get_url_core_params( $url );
		if ( ! empty( $url_params ) && 'bbapp' === Screens::instance()->get_core_namespace( $url_params ) ) {
			$url_data         = array(
				'action'    => 'open_404',
				'namespace' => 'core',
				'item_id'   => '404',
				'url'       => $url,
			);
			$core_screen_name = Screens::instance()->get_core_screen( $url_params );
			if ( ! empty( $core_screen_name ) && in_array( $core_screen_name, Screens::instance()->bbapp_deeplinking_screens(), true ) ) {
				$url_data = $this->bbapp_prepare_core_data( $core_screen_name, $url_params, $url_data );
			}
		}

		// App install on mobile we run deeplinking on app side.
		if ( ! empty( $url_params ) && isset( $url_params['bbapp_installer_screen'] ) && 1 === (int) $url_params['bbapp_installer_screen'] ) {
			$url_data = array(
				'action'    => 'outapp',
				'namespace' => 'url',
				'url'       => $url,
			);
		}

		return $url_data;
	}

	/**
	 * Prepare Core data
	 *
	 * @param string $screen_name Screen name.
	 * @param array  $url_params  Url parameters.
	 * @param array  $url_data    Url dta.
	 *
	 * @return mixed
	 */
	public function bbapp_prepare_core_data( $screen_name, $url_params, $url_data ) {
		$url_data['action']    = $this->get_action( $screen_name );
		$url_data['namespace'] = 'core';
		$url_data['item_id']   = Screens::instance()->get_core_screen_item_id( $url_params );
		$id                    = Screens::instance()->get_core_screen_id( $url_params );
		if ( ! empty( $id ) ) {
			$url_data['id'] = $id;
		}
		if ( in_array( $screen_name, apply_filters( 'bbapp_deep_screen_allow_user_embed_data', array( 'profile', 'settings' ) ), true ) ) {
			$url_data['user_id']      = get_current_user_id();
			$url_data['_link_action'] = class_exists( 'BuddyPress' ) ? 'buddypress_member' : 'wp_user';
		}

		if ( 'open_app_page' === $url_data['action'] ) {
			$url_data['_link_action'] = 'app_page';
		}

		if ( bbapp_is_active( 'iap' ) && in_array( $screen_name, array( 'screen' ), true ) && 'iap_products' === $url_data['item_id'] ) {
			$product_args   = array( 'global_subscription' => true );
			$bbapp_products = Helpers::get_products( $product_args );
			if ( ! empty( $bbapp_products ) ) {
				$url_data['iap_products'] = array_column( $bbapp_products, 'product_id' );
			}
		}

		return apply_filters( 'bbapp_prepare_core_data', $url_data, $url_params );
	}

	/**
	 * Allow actions.
	 *
	 * @param string $screen_name Screen name.
	 *
	 * @return string
	 */
	public function get_action( $screen_name ) {
		$action         = 'open_';
		$current_screen = '404';
		if ( 'screen' === $screen_name ) {
			$current_screen = 'screen';
		}
		if ( 'page' === $screen_name ) {
			$current_screen = 'app_page';
		}
		if ( 'settings' === $screen_name ) {
			$current_screen = 'settings';
		}
		if ( 'products' === $screen_name ) {
			$current_screen = 'iap_product';
		}
		if ( 'profile' === $screen_name ) {
			$current_screen = 'profile';
		}

		return $action . $current_screen;
	}
}
