<?php
/**
 * DeepLinking Screens.
 *
 * @package BuddyBossApp\DeepLinking
 */

namespace BuddyBossApp\DeepLinking;

use BuddyBossApp\ClientCommon;

/**
 * Class Screens
 *
 * @package BuddyBossApp\DeepLinking
 */
class Screens {

	/**
	 * Class instance.
	 *
	 * @var bool $instance
	 */
	private static $instance;

	/**
	 * Using Singleton, see instance()
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return Screens
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Load hooks.
	 */
	public function load() {
		add_action( 'template_redirect', array( $this, 'load_deep_linking_screen' ), 9 );
		add_filter( 'bbapp_prepare_core_data', array( $this, 'filter_bbapp_prepare_core_data' ), 10, 2 );
	}

	/**
	 * Load and identify deeplinking screen.
	 */
	public function load_deep_linking_screen() {
		global $wp;
		if ( ! empty( $wp->request ) ) {
			$url_params = $this->get_url_core_params( $wp->request );
			if ( ! empty( $url_params ) && 'bbapp' === $this->get_core_namespace( $url_params ) ) {
				$core_screen_name = $this->get_core_screen( $url_params );
				if ( ! empty( $core_screen_name ) && in_array( $core_screen_name, $this->bbapp_deeplinking_screens(), true ) ) {
					wp_safe_redirect( home_url() );
					exit();
				}
			}
		}
	}

	/**
	 * Body Class.
	 *
	 * @param array $classes Screen body class.
	 *
	 * @return mixed
	 */
	public function body_class( $classes ) {
		$classes[] = 'bbapp-app-deeplinking-page';

		return $classes;
	}

	/**
	 * Deeplinking screens.
	 *
	 * @return string[]
	 */
	public function bbapp_deeplinking_screens() {
		return array(
			'page',
			'products',
			'screen',
			'settings',
			'profile',
		);
	}

	/**
	 * Get url core params.
	 *
	 * @param string $url Deeplink url.
	 *
	 * @return false|string[]
	 */
	public function get_url_core_params( $url ) {
		$url       = str_replace( get_home_url(), '', $url );
		$url       = wp_parse_url( $url );
		$url_param = array();
		if ( isset( $url['path'] ) ) {
			$url_param = explode( '/', $url['path'] );
			$url_param = array_filter( $url_param );
		}

		if ( isset( $url['query'] ) ) {
			parse_str( $url['query'], $query );

			return $query;
		}

		return array_values( $url_param );
	}

	/**
	 * Get core namespace.
	 *
	 * @param array $url_params Deeplink url parameters.
	 *
	 * @return mixed|string
	 */
	public function get_core_namespace( $url_params ) {
		return isset( $url_params[0] ) ? $url_params[0] : '';
	}

	/**
	 * Get Screen name.
	 *
	 * @param array $url_params Deeplink url parameters.
	 *
	 * @return mixed|string
	 */
	public function get_core_screen( $url_params ) {
		return isset( $url_params[1] ) ? $url_params[1] : '';
	}

	/**
	 * Get screen item id.
	 *
	 * @param array $url_params Deeplink url parameters.
	 *
	 * @return mixed|string
	 */
	public function get_core_screen_item_id( $url_params ) {
		return isset( $url_params[2] ) ? $url_params[2] : '';
	}

	/**
	 * Get screen id.
	 *
	 * @param array $url_params Deeplink url parameters.
	 *
	 * @return mixed|string
	 */
	public function get_core_screen_id( $url_params ) {
		return isset( $url_params[3] ) ? $url_params[3] : '';
	}

	/**
	 * Deeplinking url component checking.
	 *
	 * @param array $url_data   Url data.
	 * @param array $url_params Deeplink url parameters.
	 *
	 * @return mixed
	 */
	public function filter_bbapp_prepare_core_data( $url_data, $url_params ) {
		$common = ClientCommon::instance();
		switch ( $url_data['action'] ) {
			case 'open_screen':
				$url_data = $this->bbapp_menu_screens( $url_data );
				break;

			case 'open_iap_product':
				if ( ! $common->is_active_logic( array( 'bbapp_iap' ) ) || function_exists( 'bbapp_iap_get_product' ) && false === bbapp_iap_get_product( $url_data['item_id'] ) ) {
					$url_data['action']  = 'open_404';
					$url_data['item_id'] = '404';
				}
				break;

			case 'open_app_page':
				$post = get_post( $url_data['item_id'] );
				if ( empty( $post ) || is_wp_error( $post ) || 'publish' !== $post->post_status ) {
					$url_data['action']  = 'open_404';
					$url_data['item_id'] = '404';
				}
				break;
		}
		return $url_data;
	}

	/**
	 * Menu screens filter.
	 *
	 * @param array $url_data Deeplink url data.
	 *
	 * @return mixed
	 */
	public function bbapp_menu_screens( $url_data ) {
		$common = ClientCommon::instance();

		if ( bbapp_is_lite_live_app() && in_array( $url_data['item_id'], bbapp_lite_app_license_locked_pages(), true ) ) {
			return array(
				'action'    => 'open_restricted',
				'namespace' => 'core',
				'item_id'   => '403',
				'url'       => ( ! empty( $url_data['url'] ) ) ? $url_data['url'] : '',
			);
		} else {
			switch ( $url_data['item_id'] ) {
				case 'activity':
					if ( ! $common->is_active_logic( array( 'buddypress', 'buddypress_activity' ) ) ) {
						$url_data['action']  = 'open_404';
						$url_data['item_id'] = '404';
					}

					break;

				case 'members':
					if ( ! $common->is_active_logic( array( 'buddypress', 'buddypress_members' ) ) ) {
						$url_data['action']  = 'open_404';
						$url_data['item_id'] = '404';
					}

					break;

				case 'profile':
					if ( ! $common->is_active_logic( array( 'buddypress', 'buddypress_profile' ) ) ) {
						$url_data['action']  = 'open_404';
						$url_data['item_id'] = '404';
					}

					break;

				case 'messages':
					if ( ! $common->is_active_logic( array( 'buddypress', 'buddypress_messages' ) ) ) {
						$url_data['action']  = 'open_404';
						$url_data['item_id'] = '404';
					}
					break;

				case 'documents':
					if ( ! $common->is_active_logic( array( 'buddypress', 'buddypress_documents' ) ) ) {
						$url_data['action']  = 'open_404';
						$url_data['item_id'] = '404';
					}
					break;

				case 'videos':
					if ( ! $common->is_active_logic( array( 'buddypress', 'buddypress_videos' ) ) ) {
						$url_data['action']  = 'open_404';
						$url_data['item_id'] = '404';
					}
					break;

				case 'photos':
					if ( ! $common->is_active_logic( array( 'buddypress', 'buddypress_photos' ) ) ) {
						$url_data['action']  = 'open_404';
						$url_data['item_id'] = '404';
					}
					break;

				case 'topics':
				case 'forums':
					if ( ! $common->is_active_logic( array( 'bbapp_bbpress_api', 'bbpress' ) ) ) {
						$url_data['action']  = 'open_404';
						$url_data['item_id'] = '404';
					}

					break;

				case 'notifications':
					if ( ! $common->is_active_logic( array( 'buddypress', 'buddypress_notifications' ) ) ) {
						$url_data['action']  = 'open_404';
						$url_data['item_id'] = '404';
					}
					break;

				case 'groups':
					if ( ! $common->is_active_logic( array( 'buddypress', 'buddypress_groups' ) ) ) {
						$url_data['action']  = 'open_404';
						$url_data['item_id'] = '404';
					}
					break;
				case 'courses':
				case 'courses_all':
				case 'course_certificates':
				case 'courses_category':
					if ( ! $common->is_active_logic( array( 'bbapp_learner_api' ) ) && ! $common->is_active_logic( array( 'bbapp_tutor_lms_api' ) ) && ! $common->is_active_logic( array( 'memberpress_lms_api' ) ) ) {
						$url_data['action']  = 'open_404';
						$url_data['item_id'] = '404';
					}
					break;
				case 'my_library':
					if ( ! $common->is_active_logic( array( 'bbapp_learner_course_downloading' ) ) && ! $common->is_active_logic( array( 'memberpress_lms_api' ) ) ) {
						$url_data['action']  = 'open_404';
						$url_data['item_id'] = '404';
					}
					break;
				case 'iap_products':
					if ( ! $common->is_active_logic( array( 'bbapp_iap' ) ) ) {
						$url_data['action']  = 'open_404';
						$url_data['item_id'] = '404';
					}
					break;

				case 'app_page':
				case 'page':
					$post = get_post( $url_data['id'] );
					if ( empty( $post ) || is_wp_error( $post ) || 'publish' !== $post->post_status ) {
						$url_data['action']  = 'open_404';
						$url_data['item_id'] = '404';
						unset( $url_data['id'] );
					}
					break;
			}
		}

		return $url_data;
	}
}
