<?php
/**
 * DeepLinking Rest Endpoint.
 *
 * @package BuddyBossApp\DeepLinking
 */

namespace BuddyBossApp\DeepLinking;

use BuddyBossApp\Api\DeepLinking\Main;
use BuddyBossApp\DeepLinking\Type;
use Exception;
use WP_REST_Request;

/**
 * Class DeepLinking
 *
 * @package BuddyBossApp\DeepLinking
 */
class DeepLinking {

	/**
	 * Class instance.
	 *
	 * @var object $instance .
	 */
	private static $instance;

	/**
	 * API Namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'buddyboss-app/core/v1';

	/**
	 * DeepLinking constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return DeepLinking
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Load deep linking types
	 */
	public function load() {
		add_action( 'rest_api_init', array( $this, 'load_types' ), 99 );
		Main::instance();
		Screens::instance();
		add_action( 'init', array( $this, 'inline_page_embed' ) );
		add_action( 'wp_footer', array( $this, 'inline_page_embed' ) );
		add_action( 'wp_ajax_bbapp_linkd_embed_data', array( $this, 'bbapp_linkd_embed_data' ) );
		add_action( 'wp_ajax_nopriv_bbapp_linkd_embed_data', array( $this, 'bbapp_linkd_embed_data' ) );

		// Groups and member deeplinking url.
		add_filter( 'bbapp_deeplinking_buddypress_groups', array( $this, 'deeplinking_buddypress_groups' ), 10, 2 );
		add_filter( 'bbapp_deeplinking_buddypress_members', array( $this, 'deeplinking_buddypress_members' ), 10, 2 );
	}

	/**
	 * Load all Deep Linking URLs Types.
	 */
	public function load_types() {
		if ( function_exists( 'bp_is_active' ) ) {
			new \BuddyBossApp\Integrations\BuddyBossPlatform\DeepLinking();
		}

		if ( function_exists( 'bbpress' ) ) {
			new \BuddyBossApp\Integrations\BuddyBossPlatform\BbPressDeepLinking();
		}

		if ( bbapp_is_learndash_enabled() ) {
            new \BuddyBossApp\Integrations\Learndash\DeepLinking();
		}

		if ( bbapp_is_tutor_lms_plugins_active() ) {
			new \BuddyBossApp\Integrations\TutorLMS\DeepLinking();
		}

		if ( bbapp_is_memberpress_lms_enabled() ) {
			new \BuddyBossApp\Integrations\MemberPressCourses\DeepLinking();
		}

		new Type\CoreType();
		new Type\CptType();
		new Type\TaxonomyType();
		new Type\TermType();
		new Type\AuthorType();
		new Type\AppPageType();
	}

	/**
	 * Output the deeplink embeds into the page using js template tag.
	 *
	 * @return bool
	 */
	public function inline_page_embed() {

		if ( ! bbapp_is_loaded_from_inapp_browser() || is_admin() ) {
			return false;
		}

		if ( 'init' === current_action() ) {
			ob_start();
		}

		if ( 'wp_footer' === current_action() ) {

			$content         = ob_get_contents();
			$web_page_urls   = $this->bbapp_get_urls_from_content( $content );
			$deep_links_data = self::instance()->bbapp_linkd_embed_data( $web_page_urls );
			?>
			<script>
				if ( typeof window.bbapp_embed_data === 'undefined' ) {
					window.bbapp_webpage_links = [];
					window.bbapp_embed_data = [];
				}

				window.bbapp_webpage_links = <?php echo wp_json_encode( $web_page_urls ); ?>;
				window.bbapp_embed_data = <?php echo wp_json_encode( $deep_links_data ); ?>;
				window.bbapp_embed_last_anchor_count = 0;

				jQuery( document ).ready( function() {
					bbappAddEmbedTags();

					window.bbapp_embed_last_anchor_count = jQuery( 'a' ).length;

					setInterval( function() {
						if ( window.bbapp_embed_last_anchor_count !== jQuery( 'a' ).length ) {
							window.bbapp_embed_last_anchor_count = jQuery( 'a' ).length;
							bbappAddEmbedTags();
							bbappLoadEmbedsData();
						}
					}, 5000 );
				} );

				/**
				 * Load new links find using ajax.
				 */
				function bbappLoadEmbedsData() {
					jQuery.ajax( {
						type: 'POST',
						url: window.ajaxurl,
						data: {
							action: 'bbapp_linkd_embed_data',
							urls: window.bbapp_webpage_links,
						},
						success( response ) {
							// successful request; do something with the data
							window.bbapp_embed_data = response;
							bbappAddEmbedTags(); // now again mark all links with hasembed tag.
						},
						error( jqxhr, exception ) {
							// eslint-disable-next-line no-console
							console.log( 'error:' + jqxhr + ' ! ' + exception );
						},
					} );
				}

				/**
				 * Tag the links with hasembed tag.
				 */
				function bbappAddEmbedTags() {
					// loop through each link in window.bbapp_embed_data and add their link in document with attr hasembed=1
					jQuery( 'a' ).each( function() {
						if ( jQuery.inArray( jQuery( this ).attr( 'href' ), window.bbapp_webpage_links ) !== -1 ) {
							jQuery( this ).addClass( 'has-deep' );
						} else if ( bbappIsValidUrl( jQuery( this ).attr( 'href' ) ) === 1 ) {
							window.bbapp_webpage_links.push( jQuery( this ).attr( 'href' ) );
						}
					} );
				}

				/**
				 * Check if the URL is valid.
				 *
				 * @param {string} url - URL to check.
				 *
				 * @return {number} - 1 if valid, -1 if not.
				 */
				function bbappIsValidUrl( url ) {
					if ( /^(http|https|ftp):\/\/[a-z0-9]+([\-\.]{1}[a-z0-9]+)*\.[a-z]{2,5}(:[0-9]{1,5})?(\/.*)?$/i.test( url ) ) {
						return 1;
					}

					return -1;
				}
			</script>
			<?php
		}

	}

	/**
	 * Get all urls from content.
	 *
	 * @param string $content HTML content.
	 *
	 * @return array
	 */
	public function bbapp_get_urls_from_content( $content ) {
		$dom = new \DOMDocument();
		libxml_use_internal_errors( true );
		$dom->loadHTML( $content );
		libxml_use_internal_errors( false );

		$xpath = new \DOMXPath( $dom );
		$hrefs = $xpath->evaluate( '//a[@href]' );
		$urls  = array();

		foreach ( $hrefs as $href ) {
			$url = $href->getAttribute( 'href' );
			$url = filter_var( $url, FILTER_SANITIZE_URL );
			// Use function for URL validation.
			if ( bbapp_is_valid_url( $url ) ) {
				$urls[] = $url;
			}
		}

		return $urls;
	}

	/**
	 * Lets BuddyBoss App Server know which that app client is configured properly.
	 *
	 * @param array $urls Deep link urls.
	 *
	 * @return array
	 */
	public function bbapp_linkd_embed_data( $urls = array() ) {
		$is_ajax = false;
		if ( ! empty( $_POST['urls'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
			$urls    = bbapp_input_clean( wp_unslash( $_POST['urls'] ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Missing
			$is_ajax = true;
		} elseif ( empty( $urls ) ) {
			return array();
		}
		$request_curl = new WP_REST_Request( 'POST', '/buddyboss-app/core/v1/bulk-url-details' );
		$request_curl->set_param( 'urls', $urls );
		if ( isset( $_GET['_bbapp_embed'] ) && true === (bool) bbapp_input_clean( wp_unslash( $_GET['_bbapp_embed'] ) ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$query_params           = $request_curl->get_query_params();
			$query_params['_embed'] = 'true';
			$request_curl->set_query_params( $query_params );
		}

		$server   = rest_get_server();
		$response = $server->dispatch( $request_curl );

		$status = $response->get_status();

		$deep_links_data = array();
		if ( 200 === $status ) {
			$deep_links_data = $response->get_data();
		}
		if ( true === $is_ajax ) {
			wp_send_json( $deep_links_data );
		} else {
			return $deep_links_data;
		}
	}

	/**
	 * Groups deeplinking url update.
	 * /me/groups url update to groups url.
	 *
	 * @param \WP_REST_Response $response The response data.
	 * @param array             $data     Group data.
	 *
	 * @return mixed
	 */
	public function deeplinking_buddypress_groups( $response, $data ) {
		if ( empty( $response['url'] ) ) {
			return $response;
		}

		$response['url'] = preg_replace( '/me\?/', '', $response['url'] );

		return $response;
	}

	/**
	 * Members deeplinking url update.
	 * /me url update to members url.
	 *
	 * @param \WP_REST_Response $response The response data.
	 * @param array             $data     Members data.
	 *
	 * @return mixed
	 */
	public function deeplinking_buddypress_members( $response, $data ) {
		$parse_url = wp_parse_url( $response['url'] );
		if ( '/me' === $parse_url['path'] ) {
			$response['url'] = str_replace( '/me', '/members', $response['url'] );
		}

		return $response;
	}
}
