<?php
/**
 * Holds version update methods.
 *
 * @package BuddyBossApp\DBUpdate
 */

namespace BuddyBossApp\DBUpdate;

use BuddyBossApp\Admin\Configure;
use BuddyBossApp\AppSettings;
use BuddyBossApp\Bookmark;
use BuddyBossApp\Branding;
use BuddyBossApp\Helpers\BBAPP_File;
use BuddyBossApp\Jobs;
use BuddyBossApp\ManageApp;
use BuddyBossApp\NativeAppPage\Deprecated_Content;
use BuddyBossApp\Styling;
use BuddyBossApp\Tools\Logger\BgProcessLog;

/**
 * DB versions class.
 */
class Versions {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Construct method.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return Versions
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Remove Configuration setting for Buddyboss app 1.0.4 version.
	 */
	public function update_4() {
		$remove_options_list = array(
			// iOS.
			'publish.ios.namespace', // Release App Bundle ID.
			'publish.ios.dev.namespace', // Test App Bundle ID.
			'publish.ios.signing_certificates_automatic', // Generate Certificates.
			'publish.ios.signing_certificate', // Release App Signing Certificate - file.
			'publish.ios.signing_certificate_id', // Release App Signing Certificate ID.
			'publish.ios.signing_certificate_password', // Release App Signing Certificate Password.
			'publish.ios.dev.signing_certificate', // Test App Signing Certificate - file.
			'publish.ios.dev.signing_certificate_password', // Test App Signing Certificate Password.
			'publish.ios.provisioning_profile_automatic', // Generate Profiles.
			'publish.ios.provisioning_profile', // Release App Provisioning Profile - file.
			'publish.ios.dev.provisioning_profile', // Test App Provisioning Profile - file.
			// Android.
			'publish.android.namespace', // Application ID.
			'publish.android.namespace.registered', // Register Application ID.
			'publish.android.keystore', // KeyStore - file.
			'publish.android.keystore_pwd', // KeyStore Password.
			'publish.android.keystore_alias', // KeyStore Alias.
			'publish.android.keystore_key_pwd', // KeyStore Key Password.
		);
		$file_keys           = array(
			'publish.ios.signing_certificate', // Release App Signing Certificate - file.
			'publish.ios.dev.signing_certificate', // Test App Signing Certificate - file.
			'publish.ios.provisioning_profile', // Release App Provisioning Profile - file.
			'publish.ios.dev.provisioning_profile', // Test App Provisioning Profile - file.
			'publish.android.keystore', // KeyStore - file.
		);
		$upload_dir          = wp_upload_dir();
		$settings            = Configure::instance()->get_settings();
		foreach ( $file_keys as $file_key ) {
			if ( isset( $settings[ $file_key ] ) ) {
				$file = $upload_dir['basedir'] . $settings[ $file_key ];
				if ( file_exists( $file ) ) {
					wp_delete_file( $file );
				}
			}
		}
		foreach ( $remove_options_list as $remove_option ) {
			$settings[ $remove_option ] = '';
		}

		ManageApp::instance()->update_app_settings( $settings );
	}

	/**
	 * Remove Performance table for update primary key of performance table.
	 */
	public function update_5() {
		global $wpdb;

		$table_name = "{$wpdb->prefix}bb_performance_cache";
		if ( null !== $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) ) ) { //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->query( "DROP TABLE  {$table_name};" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}
	}

	/**
	 * IAP product migration for product type.
	 */
	public function update_6() {
		// IAP terms page make empty for app page to wp pages changes.
		$bbapp_settings = ManageApp::instance()->get_app_settings();
		$settings       = ( isset( $bbapp_settings ) ) ? $bbapp_settings : array();
		if ( isset( $settings['iap.terms'] ) ) {
			$settings['iap.terms'] = 0;
		}
		ManageApp::instance()->update_app_settings( $settings );

		if ( bbapp_is_active( 'iap' ) && function_exists( 'bbapp_iap_get_products' ) ) {
			// IAP product migration.
			$results = bbapp_iap_get_products( array() );
			foreach ( $results as $result ) {
				$store_data = maybe_unserialize( $result['store_data'] );
				$is_paid    = false;
				if ( $store_data['store_product_ids']['ios'] ) {
					$is_paid = true;
				}
				if ( $store_data['store_product_ids']['android'] ) {
					$is_paid = true;
				}
				unset( $store_data['bbapp_product_types'] );
				$store_data['bbapp_product_type'] = ( true === $is_paid ) ? 'paid' : 'free';
				$result['store_data']             = maybe_serialize( $store_data );
				$id                               = $result['id'];
				bbapp_iap_update_product( $id, $result );
			}
		}
	}

	/**
	 * Auto Connect Apple Account if detail filled.
	 */
	public function update_7() {
		\BuddyBossApp\AppStores\Apple::instance()->is_connected( true );
	}

	/**
	 * Branding login setting copy for registration screens.
	 */
	public function update_8() {
		// Settings migration code.
		$clone_fields = array(
			'styles' => array(
				'styles.colors.authBgColor'         => 'styles.colors.regBgColor',
				'styles.colors.authTextColor'       => 'styles.colors.regTextColor',
				'styles.colors.authFieldBgColor'    => 'styles.colors.regFieldBgColor',
				'styles.colors.authFieldTextColor'  => 'styles.colors.regFieldTextColor',
				'styles.colors.authButtonBgColor'   => 'styles.colors.regButtonBgColor',
				'styles.colors.authButtonTextColor' => 'styles.colors.regButtonTextColor',
			),
			'images' => array(
				'login_background_img' => 'register_background_img',
			),
		);
		$styles       = Styling::instance()->get_options();
		if ( isset( $styles['styles'] ) && ! empty( $styles['styles'] ) ) {
			foreach ( $clone_fields['styles'] as $original => $copy ) {
				if ( ! isset( $styles['styles'][ $copy ] ) ) {
					$styles['styles'][ $copy ] = $styles['styles'][ $original ];
				}
			}
			Styling::instance()->set_options( $styles );
		}

		// Files.
		$branding         = Branding::instance();
		$branding_options = $branding->get_options();
		$fields           = $branding->get_app_branding_fields();
		$files_dir        = $branding->get_branding_upload_dir();
		if ( isset( $branding_options['uploads_hash'] ) ) {
			foreach ( $clone_fields['images'] as $original => $copy ) {
				if ( isset( $branding_options['uploads_hash'][ $original . '.png' ] ) ) {
					$branding_options['uploads_hash'][ $copy . '.' . $fields[ $copy ]['format'] ] = $branding_options['uploads_hash'][ $original . '.' . $fields[ $original ]['format'] ];

					$original_filename = $original . '.' . $fields[ $original ]['format'];
					$copy_filename     = $copy . '.' . $fields[ $copy ]['format'];
					if ( ( file_exists( $files_dir . '/' . $original_filename ) ) ) {
						$original_to_file_path = trailingslashit( $files_dir ) . $original_filename;
						$copy_to_file_path     = trailingslashit( $files_dir ) . $copy_filename;
						BBAPP_File::copy_file( $original_to_file_path, $copy_to_file_path );
					}
				}
			}
			$branding->set_options( $branding_options );
		}
	}

	/**
	 * Product meta table create.
	 */
	public function update_9() {
		\BuddyBossApp\InAppPurchases\Controller::instance()->on_activation();
	}

	/**
	 * Update to version 1.4.2
	 * Update bbapp_notifications_devices table to bbapp_user_devices.
	 *
	 * @since 1.4.2
	 */
	public function update_11() {
		global $wpdb;

		$old_table_name = bbapp_get_network_table( 'bbapp_notifications_devices' );

		if ( null !== $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $old_table_name ) ) ) { //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$new_table_name = bbapp_get_network_table( 'bbapp_user_devices' );
			$wpdb->query( "ALTER TABLE  {$old_table_name} RENAME TO {$new_table_name};" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}
	}

	/**
	 * Update to version 1.4.3
	 * Increase the varchar number for 'store_product_id' field in bbapp_iap_orders table.
	 *
	 * @since 1.4.3
	 */
	public function update_12() {
		global $wpdb;

		$order_table = $wpdb->prefix . 'bbapp_iap_orders';
		$has_table   = $wpdb->query(
			$wpdb->prepare(
				'show tables like %s',
				$order_table
			)
		); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

		if ( empty( $has_table ) ) {
			return;
		}

		// Get the current database name.
		$current_db = $wpdb->dbname;

		$column_exists = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE table_schema = %s AND table_name = %s AND column_name = 'store_product_id'",
				$current_db,
				$order_table
			)
		); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		if ( empty( $column_exists ) ) {
			$wpdb->query( "ALTER TABLE $order_table CHANGE store_product_id store_product_id VARCHAR(150) NOT NULL" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}
	}

	/**
	 * Update to version 1.4.4
	 *
	 * @since 1.4.4
	 */
	public function update_13() {
		$this->prepare_duplicate_iap_orders_jobs_13();
		$this->bbapp_migrate_registration_products();
	}

	/**
	 * Update to iap orders.
	 *
	 * @parent function update_13
	 * @since  1.4.4
	 */
	public function prepare_duplicate_iap_orders_jobs_13() {
		global $wpdb;
		$table_name = bbapp_get_network_table( 'bbapp_iap_orders' );

		$order_results = $wpdb->get_results( "SELECT id,bbapp_product_id,user_id,order_status FROM {$table_name} WHERE `order_status` IN ('completed','subscribed') ORDER BY `date_created` ASC" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		if ( ! empty( $order_results ) ) {
			$order_results_chunks = array_chunk( $order_results, 100 );

			if ( ! empty( $order_results_chunks ) ) {
				foreach ( $order_results_chunks as $order_chunks_items ) {
					Jobs::instance()->add( 'rm_dup_iap_orders_13', $order_chunks_items, 1 );
				}
				Jobs::instance()->start();
			}
		}
	}

	/**
	 * Function to migrate registration products.
	 *
	 * @parent function update_13
	 * @since  1.4.4
	 *
	 * @return false|void
	 */
	public function bbapp_migrate_registration_products() {
		global $wpdb;

		$bbapp_settings = ManageApp::instance()->get_app_settings();
		$get_components = bbapp_get_network_option( 'bbapp_active_components' );
		$get_components = is_array( $get_components ) ? $get_components : array();

		if ( empty( $get_components ) || ! isset( $get_components['iap'] ) || 1 !== (int) $get_components['iap'] || 1 !== $bbapp_settings['iap.purchase_before_register'] ) {
			return false;
		}

		$product_table_name = $wpdb->prefix . 'bbapp_iap_products';
		$iaps               = $wpdb->get_results( "SELECT * FROM {$product_table_name}" ); //phpcs:ignore

		if ( ! empty( $iaps ) ) {
			foreach ( $iaps as $iap ) {
				if ( ! empty( $iap->misc_settings ) ) {
					$misc_unserialize = maybe_unserialize( $iap->misc_settings );
					if ( 1 === (int) $misc_unserialize['global_subscription'] ) {
						$wpdb->iapmeta = $wpdb->prefix . 'bbapp_iap_productmeta';
						update_metadata( 'iap', $iap->id, 'product_visibility_registration', 1 );
					}
				}
			}
		}
	}

	/**
	 * Update to version 1.4.4
	 *
	 * @since 1.4.5
	 */
	public function update_14() {
		// Update deprecated quick link block.
		if ( class_exists( '\BuddyBossApp\NativeAppPage\Deprecated_Content' ) ) {
			$args        = array(
				'post_type'      => 'app_page',
				'post_status'    => 'publish',
				'posts_per_page' => - 1,
			);
			$posts_query = new \WP_Query();
			$app_pages   = $posts_query->query( $args );
			foreach ( $app_pages as $app_page ) {
				$content         = $app_page->post_content;
				$updated_content = Deprecated_Content::instance()->bbapp_get_block_content( $content );
				if ( ! empty( $updated_content ) ) {
					$app_page_args = array(
						'ID'           => $app_page->ID,
						'post_content' => $updated_content,
					);
					wp_update_post( $app_page_args );
				}
			}
		}
	}

	/**
	 * Update to version 1.4.6
	 *
	 * @since 1.4.6
	 */
	public function update_15() {
		// If key is exists on database then it will run on db version update.
		if ( get_option( 'bbapp_db_version' ) ) {
			$global_settings                              = \BuddyBossApp\Admin\Settings::instance()->get_settings();
			$global_settings['cleartext_traffic.enabled'] = 1;
			\BuddyBossApp\Admin\Settings::instance()->save_global_settings( $global_settings );
		}
	}

	/**
	 * Update to version 1.4.8
	 *
	 * @since 1.5.1
	 */
	public function update_16() {
		global $wpdb;

		$user_devices = $wpdb->prefix . 'bbapp_user_devices';
		$has_table    = $wpdb->query(
			$wpdb->prepare(
				'show tables like %s',
				$user_devices
			)
		); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

		if ( empty( $has_table ) ) {
			return;
		}

		// Get the current database name.
		$current_db = $wpdb->dbname;

		$column_exists = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE table_schema = %s AND table_name = %s AND column_name = 'device_id' OR column_name = 'device_model'",
				$current_db,
				$user_devices
			)
		); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( empty( $column_exists ) ) {
			$wpdb->query( "ALTER TABLE $user_devices CHANGE device_id device_id VARCHAR(100) NOT NULL" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$wpdb->query( "ALTER TABLE $user_devices CHANGE device_model device_model VARCHAR(100) NOT NULL" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}
	}

	/**
	 * Update to next version
	 *
	 * @since 1.5.5
	 */
	public function update_17() {
		// Install Access Group tables.
		\BuddyBossApp\AccessControls::instance()->on_activation();

		// Old menus to new nav menu migration.
		$delete_menu = false;
		$app_menus   = get_option( 'bbapp_menus' );
		if ( isset( $app_menus ) && ! empty( $app_menus ) ) {
			foreach ( $app_menus as $key => $menus ) {
				if ( 'more' === $key || 'tabbar' === $key ) {
					if ( ! get_option( "bbapp_menus_default-{$key}" ) ) {
						$delete_menu = true;
						update_option( "bbapp_menus_default-{$key}", $menus );
					}
				}
			}
			update_option( 'bbapp_menus_db_15', $app_menus );
			if ( $delete_menu ) {
				delete_option( 'bbapp_menus' );
			}
		}
	}

	/**
	 * Update to next version
	 *
	 * @since 1.6.3
	 */
	public function update_18() {
		$this->get_stored_wp_category_access_rules();
		$this->get_stored_ld_category_access_rules();
	}

	/**
	 * Function to get the store category access rule.
	 *
	 * @param array $taxonomies Taxonomies array.
	 *
	 * @since 1.6.3
	 */
	public function get_stored_wp_category_access_rules( $taxonomies = array() ) {
		if ( empty( $taxonomies ) ) {
			$taxonomies     = array();
			$reg_taxonomies = get_object_taxonomies( \BuddyBossApp\AccessControls\Core\Settings\Posts::instance()->post_type, 'objects' );
			if ( $reg_taxonomies ) {
				foreach ( $reg_taxonomies as $key => $taxonomy ) {
					// By pass post_format taxonomy.
					if ( 'post_format' === $key ) {
						continue;
					}
					$taxonomies[] = $key;
				}
			}
		}

		$item_type = \BuddyBossApp\AccessControls\Core\Settings\CategoryRule::instance()->item_type;

		$wp_category = $this->get_store_terms_by_item_type( $taxonomies, $item_type );
		if ( ! empty( $wp_category ) ) {
			foreach ( $wp_category as $key => $order_item ) {
				update_term_meta( $order_item, "bb_access_control_{$item_type}_menu_order", absint( $key ) + ( 1 ) );
			}
		}
	}

	/**
	 * Function to get the store category access rule.
	 *
	 * @param array $taxonomies Taxonomies array.
	 *
	 * @since 1.6.3
	 */
	public function get_stored_ld_category_access_rules( $taxonomies = array() ) {
		if ( empty( $taxonomies ) ) {
			$taxonomies     = array();
			$reg_taxonomies = get_object_taxonomies( \BuddyBossApp\Integrations\Learndash\AccessControls\Course::instance()->post_type, 'objects' );
			if ( $reg_taxonomies ) {
				foreach ( $reg_taxonomies as $key => $taxonomy ) {
					// By pass post_format taxonomy.
					if ( 'post_format' === $key ) {
						continue;
					}
					$taxonomies[] = $key;
				}
			}
		}

		$item_type = \BuddyBossApp\Integrations\Learndash\AccessControls\LDCategoryRule::instance()->item_type;

		$ld_category = $this->get_store_terms_by_item_type( $taxonomies, $item_type );
		if ( ! empty( $ld_category ) ) {
			foreach ( $ld_category as $key => $order_item ) {
				update_term_meta( $order_item, "bb_access_control_{$item_type}_menu_order", absint( $key ) + ( 1 ) );
			}
		}
	}

	/**
	 * Function to get the store category access rule.
	 *
	 * @param array  $taxonomies Taxonomies array.
	 * @param string $item_type  Item type.
	 *
	 * @since 1.6.3
	 * @return int|int[]|string|string[]|\WP_Term[]
	 */
	public function get_store_terms_by_item_type( $taxonomies, $item_type ) {
		$access_rules = \BuddyBossApp\AccessControls\AccessRule::instance()->get_access_rules(
			array(
				'include_item_types' => $item_type,
				'only_count'         => true,
			)
		);

		if ( ! empty( $access_rules['count'] ) ) {
			$args  = array(
				'taxonomy'   => $taxonomies,
				'fields'     => 'ids',
				'hide_empty' => false,
				'orderby'    => 'meta_value_num',
				'order'      => 'DESC',
				'meta_query' => array( //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
					'relation' => 'OR',
					array(
						'key'     => "bb_access_control_{$item_type}_menu_order",
						'value'   => '',
						'compare' => '!=',
					),
				),
			);
			$terms = new \WP_Term_Query( $args );

			return $terms->get_terms();
		}

		return array();
	}

	/**
	 * Update to next version
	 *
	 * @since 1.6.3
	 */
	public function update_19() {
		// Display plugin update notice.
		update_option( '_bbapp_is_update', true );
	}

	/**
	 * This db version used for migrate new icon to new icons in menus.
	 *
	 * @since 1.7.0
	 */
	public function update_20() {
		// Get all menus which is store on db.
		$main_menus = bbapp_get_network_option( 'bbapp_menus' );
		if ( empty( $main_menus ) ) {
			$main_menus['tabbar']['default-tabbar'] = array();
			$main_menus['more']['default-more']     = array();
		}

		// get old icon style from db.
		$apps_settings     = get_option( 'bbapp_settings' );
		$tabbar_icon_style = ! empty( $apps_settings['app_menu.tab_bar_icon_style'] ) ? $apps_settings['app_menu.tab_bar_icon_style'] : 'outlined';
		$more_icon_style   = ! empty( $apps_settings['app_menu.more_icon_style'] ) ? $apps_settings['app_menu.more_icon_style'] : 'outlined';
		$is_tabbar_update  = false;
		$is_more_update    = false;
		// Fetch all tabbar menus for migrated with new icons.
		if ( ! empty( $main_menus['tabbar'] ) ) {
			foreach ( $main_menus['tabbar'] as $tabbar_menu_id => $tabbar_menu ) {
				$tabar_menu_items = bbapp_get_network_option( "bbapp_menus_{$tabbar_menu_id}" );
				$menu_items_old   = $tabar_menu_items;
				if ( ! empty( $tabar_menu_items ) ) {
					foreach ( $tabar_menu_items as $menu_item_key => $menu_item ) {
						if ( isset( $menu_item['icon']['uri'] ) ) {
							// Prepare old icon to new icon data.
							$tabar_menu_items[ $menu_item_key ]['icon'] = $this->prepare_migrated_icon_data( $menu_item, $tabbar_icon_style );
							// Set flag if tabbar menu updated.
							$is_tabbar_update = true;
						}
					}
				}
				// if tabbar has updated then store update menus in db.
				if ( true === $is_tabbar_update ) {
					bbapp_set_network_option( "bbapp_menus_{$tabbar_menu_id}_20", $menu_items_old );
					bbapp_set_network_option( "bbapp_menus_{$tabbar_menu_id}", $tabar_menu_items );
				}
			}
		}
		// Fetch all more menus for migrated with new icons.
		if ( ! empty( $main_menus['more'] ) ) {
			foreach ( $main_menus['more'] as $tabbar_menu_id => $tabbar_menu ) {
				$more_menu_items = bbapp_get_network_option( "bbapp_menus_{$tabbar_menu_id}" );
				$menu_items_old  = $more_menu_items;
				if ( ! empty( $more_menu_items ) ) {
					foreach ( $more_menu_items as $menu_item_key => $menu_item ) {
						if ( isset( $menu_item['section'] ) ) {
							foreach ( $menu_item['section'] as $section_menu_item_key => $section_menu_item ) {
								if ( isset( $section_menu_item['icon']['uri'] ) ) {
									// Prepare old icon to new icon data for section.
									$more_menu_items[ $menu_item_key ]['section'][ $section_menu_item_key ]['icon'] = $this->prepare_migrated_icon_data(
										$section_menu_item,
										$more_icon_style
									);

									// Set flag if more menu updated.
									$is_more_update = true;
								}
							}
						} elseif ( isset( $menu_item['icon']['uri'] ) ) {
								// Prepare old icon to new icon data.
								$more_menu_items[ $menu_item_key ]['icon'] = $this->prepare_migrated_icon_data( $menu_item, $more_icon_style );
								// Set flag if more menu updated.
								$is_more_update = true;
						}
					}
				}
				// if more has updated then store update menus in db.
				if ( true === $is_more_update ) {
					bbapp_set_network_option( "bbapp_menus_{$tabbar_menu_id}_20", $menu_items_old );
					bbapp_set_network_option( "bbapp_menus_{$tabbar_menu_id}", $more_menu_items );
				}
			}
		}
	}

	/**
	 * Prepare migrating icon data.
	 *
	 * @param array  $menu_item  Menu item.
	 * @param string $icon_style Icon style.
	 *
	 * @since 1.7.0
	 *
	 * @return array
	 */
	public function prepare_migrated_icon_data( $menu_item, $icon_style ) {
		$menu_icon = $menu_item['icon'];

		$type    = 'legacy';
		$icon_id = 'file';
		if ( strpos( $menu_icon['uri'], 'custom/' ) !== false ) {
			$type    = 'custom';
			$icon_id = basename( str_replace( 'custom/', '', $menu_icon['uri'] ), '.png' );
		} elseif ( strpos( $menu_item['icon']['uri'], 'bbapp/' ) !== false ) {
			$icon_id = str_replace( 'bbapp/', '', $menu_icon['uri'] );
		}
		// Monochrome Color.
		$monochrome_setting = isset( $menu_icon['monochrome_setting'] ) ? $menu_icon['monochrome_setting'] : '';
		$monochrome_setting = str_replace( '\\', '', $monochrome_setting );
		$setting            = ( is_string( $monochrome_setting ) ) ? json_decode( $monochrome_setting, ARRAY_A ) : '';

		$color      = 'default';
		$fill_color = true;
		if ( ! empty( $setting['icon_monochrome_checkbox'] ) ) {
			if ( 'custom' === $setting['monochrome_option'] ) {
				$color = $setting['icon_monochrome_color'];
			}
			if ( 'yes' !== $setting['icon_monochrome_checkbox'] && 'custom' === $type ) {
				$fill_color = false;
			}
		}

		$style     = 'outlined' === $icon_style ? 'lined' : $icon_style;
		$box_style = '';

		if ( 'boxed' === $icon_style ) {
			$style     = 'lined';
			$box_style = 'box';
		}

		return array(
			'id'         => $icon_id,
			'type'       => $type,
			'style'      => $style,
			'box_style'  => $box_style,
			'color'      => $color,
			'fill_color' => $fill_color,
			'extra'      => array(
				'uniq_id' => $menu_item['id'],
			),
		);
	}

	/**
	 * This db version used for migrate new icon to new icons in menus.
	 *
	 * @since 1.7.4
	 */
	public function update_21() {
		Bookmark::instance()->on_activation();
		if ( method_exists( '\BuddyBoss\Performance\Cache', 'purge_by_group' ) ) {
			\BuddyBoss\Performance\Cache::instance()->purge_by_group( 'blog-post' );
		}
	}

	/**
	 * This db version used to drop push notification meta table.
	 *
	 * @since 1.7.6
	 */
	public function update_22() {
		global $wpdb;

		// Drop push notification meta Table.
		$table_name = "{$wpdb->prefix}bbapp_push_notifications_meta";
		if ( null !== $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) ) ) { //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->query( "DROP TABLE  {$table_name};" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}
	}

	/**
	 * Store app related notices on array.
	 *
	 * @since 1.8.80
	 * @return void
	 */
	public function update_23() {
		global $wpdb;

		$option_name       = 'bbapp-dismissed-notice-';
		$dismissed_rows    = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT option_name, option_value FROM {$wpdb->prefix}options WHERE option_name LIKE %s",
				$option_name . '%'
			)
		); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$dismissed_notices = get_option( 'bbapp-dismissed-notices', array() );

		if ( ! empty( $dismissed_rows ) ) {
			foreach ( $dismissed_rows as $dismissed_row ) {
				if ( ! array_key_exists( $dismissed_row->option_name, $dismissed_notices ) ) {
					$dismissed_notices[ $dismissed_row->option_name ] = $dismissed_row->option_value;

					delete_option( $dismissed_row->option_name );
				}
			}

			if ( ! empty( $dismissed_notices ) ) {
				update_option( 'bbapp-dismissed-notices', $dismissed_notices );
			}
		}
	}

	/**
	 * Remove duplicate user ids.
	 *
	 * @since 2.0.51
	 * @return void
	 */
	public function update_24() {
		global $wpdb;

		$table_name        = bbapp_get_network_table( 'bbapp_user_devices' );
		$duplicate_devices = $wpdb->get_results( "SELECT user_id, device_id FROM {$table_name} WHERE user_id != 0 GROUP BY user_id, device_id HAVING COUNT(device_id) > 1" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! empty( $duplicate_devices ) ) {
			$chunked_devices = array_chunk( $duplicate_devices, 20 );

			if ( ! empty( $chunked_devices ) ) {
				foreach ( $chunked_devices as $chunked_device ) {
					Jobs::instance()->add( 'rm_dup_devices_24', $chunked_device, 1 );
				}

				// Start job.
				Jobs::instance()->start();
			}
		}
	}

	/**
	 * Recreate BG process lag table.
	 *
	 * @since 2.0.90
	 * @return void
	 */
	public function update_26() {
		global $wpdb;

		if ( class_exists( 'BuddyBossApp\Tools\Logger\BgProcessLog' ) ) {
			$log_table_name  = "{$wpdb->prefix}bb_background_process_logs";
			$indices_columns = $wpdb->get_col(
				$wpdb->prepare(
					'SELECT COLUMN_NAME FROM information_schema.statistics WHERE table_schema = %s AND table_name = %s',
					$wpdb->__get( 'dbname' ),
					$log_table_name
				)
			); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

			if ( empty( $indices_columns ) ) {
				return;
			}

			$pre_indices  = array( 'id', 'process_start_date_gmt' );
			$diff_indices = array_diff( $indices_columns, $pre_indices );

			if ( empty( $diff_indices ) ) {
				return;
			}

			// Delete the existing table.
			$wpdb->query( "DROP TABLE IF EXISTS {$log_table_name}" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

			// Create a new table again.
			BgProcessLog::instance()->create_table();
		}
	}

	/**
	 * Create headerbars.
	 *
	 * @since 2.1.90
	 */
	public function update_27() {
		if ( ! in_array( trailingslashit( bbapp()->plugin_dir ) . 'include/Admin/Menus/functions.php', get_included_files(), true ) ) {
			require_once trailingslashit( bbapp()->plugin_dir ) . 'include/Admin/Menus/functions.php';
		}

		if ( ! function_exists( 'get_db_main_menus' ) || ! function_exists( 'update_db_main_menus' ) ) {
			return;
		}

		$main_menus = \get_db_main_menus();

		if ( ! empty( $main_menus['tabbar'] ) ) {
			foreach ( $main_menus['tabbar'] as $tabbar_menu_id => $tabbar_menu ) {
				if ( ! $this->bbapp_headerbar_from_tabbar( $tabbar_menu_id ) ) {
					if ( 'default-tabbar' === $tabbar_menu_id ) {
						$headerbar_menu_data = array(
							'id'                 => 'default-headerbar',
							'label'              => $tabbar_menu['label'],
							'menu_type'          => 'headerbar',
							'login_state'        => $tabbar_menu['login_state'],
							'access_groups'      => $tabbar_menu['access_groups'],
							'priority'           => $tabbar_menu['priority'],
							'is_default'         => ( ! empty( $tabbar_menu['is_default'] ) ) ? $tabbar_menu['is_default'] : false,
							'title_position'     => 'left',
							'more_menu_position' => 'tabbar',
							'tabbar_id'          => $tabbar_menu_id,
						);

						if ( isset( $tabbar_menu['is_default'] ) ) {
							$headerbar_menu_data['is_default'] = ( ! empty( $tabbar_menu['is_default'] ) ) ? $tabbar_menu['is_default'] : false;
						}

						$main_menus['headerbar']['default-headerbar']            = $headerbar_menu_data;
						$main_menus['tabbar'][ $tabbar_menu_id ]['headerbar_id'] = 'default-headerbar';

						update_db_main_menus( $main_menus );
					} else {
						$nav_id              = uniqid( time() );
						$headerbar_menu_data = array(
							'id'                 => $nav_id,
							'label'              => $tabbar_menu['label'] . '/Header bar',
							'menu_type'          => 'headerbar',
							'login_state'        => $tabbar_menu['login_state'],
							'access_groups'      => $tabbar_menu['access_groups'],
							'priority'           => $tabbar_menu['priority'],
							'title_position'     => 'left',
							'more_menu_position' => 'tabbar',
							'tabbar_id'          => $tabbar_menu_id,
						);

						if ( isset( $tabbar_menu['is_default'] ) ) {
							$headerbar_menu_data['is_default'] = ( ! empty( $tabbar_menu['is_default'] ) ) ? $tabbar_menu['is_default'] : false;
						}

						$main_menus['headerbar'][ $nav_id ]                      = $headerbar_menu_data;
						$main_menus['tabbar'][ $tabbar_menu_id ]['headerbar_id'] = $nav_id;
					}
				}
			}

			update_db_main_menus( $main_menus );

			unset( $main_menus, $tabbar_menu_id, $tabbar_menu, $headerbar_menu_data, $nav_id );
		}

		// Update more menu icon type.
		update_option( 'more_menu_icon_type', 'avatar' );
	}

	/**
	 * Function to check headerbar menu exists.
	 *
	 * @param string $tabbar_id Tabbar id.
	 *
	 * @since 2.4.10
	 * @return bool
	 *
	 * @todo  : Added this because update_27 creating conflict with update_30.
	 */
	public function bbapp_headerbar_from_tabbar( $tabbar_id ) {
		// Load the functions file if not already loaded to avoid redeclaring functions.
		if ( ! function_exists( 'get_db_main_menus' ) ) {
			require_once bbapp()->plugin_dir . 'include/Admin/Menus/functions.php';
		}

		if ( ! function_exists( 'get_db_main_menus' ) ) {
			return;
		}

		$main_menus = get_db_main_menus();

		if ( ! empty( $main_menus['headerbar'] ) ) {
			foreach ( $main_menus['headerbar'] as $key => $main_menu ) {
				if ( $main_menu['tabbar_id'] === $tabbar_id ) {
					return $key;
				}
			}
		}

		unset( $main_menus, $tabbar_id, $key, $main_menu );

		return false;
	}

	/**
	 * Get the latest information of the app.
	 *
	 * @since 2.2.80
	 */
	public function update_28() {
		$app_info = ManageApp::instance()->get_app_info( true );

		if ( ! is_wp_error( $app_info ) && ! empty( $app_info ) && ! empty( $app_info['app_licence_type'] ) && ! empty( $app_info['license_data'] ) ) {
			ManageApp::instance()->update_license_data(
				array(
					'license_type'      => $app_info['app_licence_type'],
					'license_transient' => $app_info['app_licence_type'],
					'license_token'     => $app_info['license_data'],
				)
			);
		}
	}

	/**
	 * Update the bb_performance_cache table structure to add all required indexes.
	 *
	 * @since 2.3.70
	 * @return void
	 */
	public function update_29() {
		global $wpdb;

		$table_name = "{$wpdb->prefix}bb_performance_cache";
		$table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) );

		if ( $table_exists ) {
			// Get all existing indexes on the table
			$existing_indexes = $wpdb->get_col( $wpdb->prepare(
				"SELECT index_name FROM information_schema.statistics WHERE table_schema = %s AND table_name = %s GROUP BY index_name",
				$wpdb->dbname,
				$table_name
			) );

			// Define all required indexes
			$required_indexes = array(
				'cache_name'   => array('cache_name'),
				'cache_group'  => array('cache_group'),
				'cache_expire' => array('cache_expire'),
				'user_group'   => array('user_id', 'cache_group'),
				'blog_id'      => array('blog_id'),
				'expiry_group' => array('cache_expire', 'cache_group'),
				'user_blog_id' => array('user_id', 'blog_id')
			);

			// Check if MySQL version requires limiting varchar index length
			$mysql_server_info = '';
			$mysql_version     = '';
			$is_mariadb        = false;

			if ( $wpdb->use_mysqli && function_exists( 'mysqli_get_server_info' ) ) {
				$mysql_server_info = $wpdb->db_server_info();

				if ( strpos( strtolower( $mysql_server_info ), 'maria' ) !== false ) {
					$is_mariadb = true;
				}

				if ( function_exists( 'mysqli_get_server_version' ) ) {
					$mysql_version = mysqli_get_server_version( $wpdb->dbh );
				}
			}

			$needs_limited_index = !empty( $mysql_version ) &&
				( ( $is_mariadb && $mysql_version < 100300 ) ||
				( !$is_mariadb && $mysql_version < 80000 ) );

			// Add any missing indexes
			foreach ( $required_indexes as $index_name => $columns ) {
				if ( !in_array( $index_name, $existing_indexes ) && $index_name !== 'PRIMARY' ) {
					$column_definitions = array();

					foreach ( $columns as $column ) {
						if ( $needs_limited_index && in_array( $column, array('cache_name', 'cache_group') ) ) {
							// For older MySQL/MariaDB versions, limit varchar index length
							$column_definitions[] = $column . '(191)';
						} else {
							$column_definitions[] = $column;
						}
					}

					$column_string = implode( ', ', $column_definitions );
					$wpdb->query( "ALTER TABLE {$table_name} ADD INDEX {$index_name} ({$column_string})" );
				}
			}
		}
	}

	/**
	 * Update the menu system to use the new database tables.
	 *
	 * @since 2.4.10
	 */
	public function update_30() {
		$this->update_menus_30();
		$this->update_string_translations_30();
	}

	/**
	 * Update the menu system to use the new database tables.
	 *
	 * @since 2.4.10
	 */
	public function update_menus_30() {
		global $wpdb;

		// Check if tables already exist.
		$menus_table      = $wpdb->prefix . 'bbapp_menus';
		$menu_items_table = $wpdb->prefix . 'bbapp_menu_items';

		$tables_exist = $wpdb->get_var( "SHOW TABLES LIKE '$menus_table'" ) === $menus_table &&  // phpcs:ignore
		                $wpdb->get_var( "SHOW TABLES LIKE '$menu_items_table'" ) === $menu_items_table; // phpcs:ignore

		// If tables don't exist, call the on_activation method to create them and set up the menu system.
		if ( ! $tables_exist ) {
			\BuddyBossApp\Menu::instance()->on_activation();
		}

		// Load the functions file if not already loaded to avoid redeclaring functions.
		if ( ! function_exists( 'get_db_main_menus' ) ) {
			require_once bbapp()->plugin_dir . 'include/Admin/Menus/functions.php';
		}

		// Check if we already have any menu data in the new tables to avoid duplicate migration.
		$existing_menus_count = $wpdb->get_var( "SELECT COUNT(*) FROM {$menus_table}" ); // phpcs:ignore
		if ( $existing_menus_count > 0 ) {
			return;
		}

		// Get all existing menu data from options using the global function get_db_main_menus().
		$main_menus = \get_db_main_menus();

		// Get existing menu items for each menu and store them in the new menu items table.
		if ( ! empty( $main_menus ) ) {
			// Migrate menus data to the new table.
			$menu_map     = array(); // Store old menu ID to new menu ID mapping.
			$site_id      = get_current_blog_id();
			$current_time = current_time( 'mysql' );

			// First, process all menus to create the base records in the bbapp_menus table.
			foreach ( $main_menus as $menu_type => $menus ) {
				foreach ( $menus as $old_menu_id => $menu_data ) {
					// Prepare menu data for insertion into the new table.
					$login_state = ( isset( $menu_data['login_state'] ) && 'logged-in' === $menu_data['login_state'] ) ? 1 : 0;
					$menu_name   = ! empty( $menu_data['label'] ) ? $menu_data['label'] : 'Menu ' . $old_menu_id;
					if ( 'headerbar' === $menu_type ) {
						$menu_name = str_replace( '/Header bar', '', $menu_name ); // Remove '/Header bar' from the menu name if present.
					}

					// Make sure access_groups is an array or null if not set.
					$access_groups = ! empty( $menu_data['access_groups'] ) ? $menu_data['access_groups'] : array();
					if ( ! is_array( $access_groups ) ) {
						$access_groups = array();
					}

					// Get app selected language code.
					$app_language_code = bbapp_wp_locale_to_app_locale();

					// Insert into bbapp_menus table with the new structure.
					$wpdb->insert( // phpcs:ignore
						$menus_table,
						array(
							'site_id'       => $site_id,
							'menu_type'     => $menu_type,
							'menu_name'     => $menu_name,
							'login_state'   => $login_state,
							'access_groups' => maybe_serialize( $access_groups ),
							'language_code' => $app_language_code, // Default language code for now, can be changed later.
							'data'          => maybe_serialize( $menu_data ), // Store all original data for reference if needed.
							'created_at'    => $current_time,
							'updated_at'    => $current_time,
						),
						array(
							'%d', // site_id.
							'%s', // menu_type.
							'%s', // menu_name.
							'%d', // login_state.
							'%s', // access_groups.
							'%s', // language_code.
							'%s', // data.
							'%s', // created_at.
							'%s', // updated_at.
						)
					);

					$new_menu_id              = $wpdb->insert_id;
					$menu_map[ $old_menu_id ] = $new_menu_id;

					// Set priority based on menu name, login state, and existing priority.
					$priority = 0; // Default priority for custom menus.

					// Check if this is a Default menu (highest priority).
					if ( 'Default' === $menu_name ) {
						$priority = 9999;
					} // Check if this is a Logged-out menu (second highest priority).
					elseif ( 'Logged-out' === $menu_name || ( isset( $menu_data['login_state'] ) && 'logged-out' === $menu_data['login_state'] ) ) {
						$priority = 9998;
					} // Otherwise use the existing priority from the options table if available.
					elseif ( isset( $menu_data['priority'] ) ) {
						// Make sure custom menu priority is below 9998 to avoid conflicts with default menus.
						$priority = min( intval( $menu_data['priority'] ), 9997 );
					}

					// Update the menu with the determined priority value.
					$wpdb->update( // phpcs:ignore
						$menus_table,
						array( 'priority' => $priority ),
						array( 'id' => $new_menu_id ),
						array( '%d' ),
						array( '%d' )
					);
				}
			}

			// Now process menu items after all menus have been created and we have their new IDs.
			foreach ( $main_menus as $menu_type => $menus ) {
				foreach ( $menus as $old_menu_id => $menu_data ) {
					// Skip if this menu wasn't mapped properly to a new ID.
					if ( ! isset( $menu_map[ $old_menu_id ] ) ) {
						continue;
					}

					$new_menu_id = $menu_map[ $old_menu_id ];

					// Get menu items for this menu using the global function get_db_menu_items().
					$menu_items = \get_db_menu_items( $old_menu_id );

					if ( ! empty( $menu_items ) ) {
						$section_map = array(); // For mapping section IDs to new item IDs.

						// First pass to create all items and record section mappings if needed.
						foreach ( $menu_items as $order => $item ) {
							$item_type = ! empty( $item['type'] ) ? $item['type'] : '';

							if ( empty( $item_type ) ) {
								continue;
							}

							// Make sure we have proper data structure for item data.
							if ( ! isset( $item['data'] ) || ! is_array( $item['data'] ) ) {
								$item['data'] = array();
							}

							// Prepare icon data for item.
							$icon_data = ! empty( $item['icon'] ) ? $item['icon'] : array();

							// Determine parent_id for this item.
							$parent_id = 0;
							if ( ! empty( $item['data']['parent'] ) ) {
								if ( isset( $section_map[ $item['data']['parent'] ] ) ) {
									$parent_id = $section_map[ $item['data']['parent'] ];
								}
							}

							$item_slug = ! empty( $item['object'] ) ? $item['object'] : $item_type;
							if ( 'headerbar' === $menu_type && 'headerbar-action' === $item_type ) {
								$item_type = ! empty( $item['object'] ) ? $item['object'] : $item_type;
								$item_slug = ! empty( $item['type'] ) ? $item['type'] : '';
							}

							// Set up item data for insertion into the new table.
							$item_data = array(
								'menu_id'        => $new_menu_id,
								'label'          => ! empty( $item['label'] ) ? $item['label'] : '',
								'item_id'        => ! empty( $item['data']['id'] ) ? $item['data']['id'] : 0,
								'item_type'      => $item_type,
								'item_slug'      => $item_slug, // Store object as slug if available.
								'item_icon_data' => ! empty( $icon_data ) ? maybe_serialize( $icon_data ) : null,
								'item_data'      => ! empty( $item['data'] ) ? maybe_serialize( $item['data'] ) : null,
								'item_link'      => ! empty( $item['data']['link'] ) ? $item['data']['link'] : null,
								'parent_id'      => $parent_id,
								'menu_order'     => $order,
								'created_at'     => $current_time,
								'updated_at'     => $current_time,
							);

							// Insert menu item into the new table.
							$wpdb->insert( // phpcs:ignore
								$menu_items_table,
								$item_data,
								array(
									'%d', // menu_id.
									'%s', // label.
									'%d', // item_id.
									'%s', // item_type.
									'%s', // item_slug.
									'%s', // item_icon_data.
									'%s', // item_data.
									'%s', // item_link.
									'%d', // parent_id.
									'%d', // menu_order.
									'%s', // created_at.
									'%s', // updated_at.
								)
							);

							$inserted_item_id = $wpdb->insert_id;

							// If this is a section, store its mapping for later.
							if ( 'section' === $item_type && isset( $item['id'] ) ) {
								$section_map[ $item['id'] ] = $inserted_item_id;
							}

							// Handle section items (more menu has section array format).
							if ( ! empty( $item['section'] ) && is_array( $item['section'] ) ) {
								$section_index = 0;
								foreach ( $item['section'] as $section_item ) {
									if ( empty( $section_item ) || empty( $section_item['type'] ) ) {
										continue;
									}

									// Prepare section item icon data for insertion.
									$section_icon_data = ! empty( $section_item['icon'] ) ? $section_item['icon'] : array();

									// Make sure we have proper data structure for section item data.
									if ( ! isset( $section_item['data'] ) || ! is_array( $section_item['data'] ) ) {
										$section_item['data'] = array();
									}

									// Section item always has the current item as parent since it's a child of the section.
									$section_item_data = array(
										'menu_id'        => $new_menu_id,
										'label'          => ! empty( $section_item['label'] ) ? $section_item['label'] : '',
										'item_id'        => ! empty( $section_item['data']['id'] ) ? $section_item['data']['id'] : 0,
										'item_type'      => $section_item['type'],
										'item_slug'      => ! empty( $section_item['object'] ) ? $section_item['object'] : $section_item['type'], // Store object as slug if available.
										'item_icon_data' => ! empty( $section_icon_data ) ? maybe_serialize( $section_icon_data ) : null,
										'item_data'      => ! empty( $section_item['data'] ) ? maybe_serialize( $section_item['data'] ) : null,
										'item_link'      => ! empty( $section_item['data']['link'] ) ? $section_item['data']['link'] : null,
										'parent_id'      => $inserted_item_id, // Parent is the section we just inserted above.
										'menu_order'     => $section_index++,
										'created_at'     => $current_time,
										'updated_at'     => $current_time,
									);

									// Insert section menu item into the new table.
									$wpdb->insert( // phpcs:ignore
										$menu_items_table,
										$section_item_data,
										array(
											'%d', // menu_id.
											'%s', // label.
											'%d', // item_id.
											'%s', // item_type.
											'%s', // item_slug.
											'%s', // item_icon_data.
											'%s', // item_data.
											'%s', // item_link.
											'%d', // parent_id.
											'%d', // menu_order.
											'%s', // created_at.
											'%s', // updated_at.
										)
									);
								}
							}

							// Handle headerbar-action items with icons array (headerbar action menu items).
							if ( isset( $item['type'] ) && 'headerbar-action' === $item['type'] && is_array( $item['icons'] ) && ! empty( $item['icons'] ) ) {
								$headerbar_icon_index = 0;
								foreach ( $item['icons'] as $icon_key => $icon_item ) {
									if ( empty( $icon_item ) || empty( $icon_item['type'] ) ) {
										continue;
									}

									// Prepare icon item data for insertion.
									$icon_item_icon_data = ! empty( $icon_item['icon'] ) ? $icon_item['icon'] : array();

									// Set item id based on available data or default to 0.
									$action_item_id = 0;
									if ( ! empty( $icon_item['id'] ) ) {
										$action_item_id = $icon_item['id'];
									}

									// Get label from icon data if available or use the icon key as default.
									$action_item_label = $icon_key;
									if ( ! empty( $icon_item['icon']['extra']['label'] ) ) {
										$action_item_label = $icon_item['icon']['extra']['label'];
									}

									// Icon item always has the current headerbar-action item as parent since it's a child of the action item.
									$icon_item_data = array(
										'menu_id'        => $new_menu_id,
										'label'          => $action_item_label,
										'item_id'        => 0, // Always use numeric ID for menu items.
										'item_type'      => $icon_item['type'],
										'item_slug'      => ! empty( $icon_item['id'] ) ? $icon_item['id'] : 'headerbar', // Store string ID in item_slug if available.
										'item_icon_data' => ! empty( $icon_item_icon_data ) ? maybe_serialize( $icon_item_icon_data ) : null,
										'item_data'      => ! empty( $icon_item['data'] ) ? maybe_serialize( $icon_item['data'] ) : null,
										'item_link'      => null,
										'parent_id'      => $inserted_item_id, // Parent is the headerbar-action item we just inserted above.
										'menu_order'     => $headerbar_icon_index++,
										'created_at'     => $current_time,
										'updated_at'     => $current_time,
									);

									// Insert headerbar action menu item into the new table.
									$wpdb->insert( // phpcs:ignore
										$menu_items_table,
										$icon_item_data,
										array(
											'%d', // menu_id.
											'%s', // label.
											'%d', // item_id (now always numeric).
											'%s', // item_type.
											'%s', // item_slug (contains string ID like 'headerbar-action-create-post').
											'%s', // item_icon_data.
											'%s', // item_data.
											'%s', // item_link.
											'%d', // parent_id.
											'%d', // menu_order.
											'%s', // created_at.
											'%s', // updated_at.
										)
									);
								}
							}
						}
						// We don't need the second pass anymore since we're handling parent IDs correctly in the first pass now.
					}
				}
			}

			// Update headerbar and tabbar relationships in the database if they exist.
			foreach ( $main_menus as $menu_type => $menus ) {
				if ( 'tabbar' === $menu_type ) {
					foreach ( $menus as $tabbar_id => $tabbar_data ) {
						if ( ! empty( $tabbar_data['headerbar_id'] ) && isset( $menu_map[ $tabbar_id ] ) && isset( $menu_map[ $tabbar_data['headerbar_id'] ] ) ) {
							// Store relationships in the database for tabbar and headerbar.
							$wpdb->update( // phpcs:ignore
								$menus_table,
								array(
									'data' => maybe_serialize(
										array(
											'related_headerbar_id' => $menu_map[ $tabbar_data['headerbar_id'] ],
											'original_data'        => $tabbar_data,
										)
									),
								),
								array( 'id' => $menu_map[ $tabbar_id ] ),
								array( '%s' ),
								array( '%d' )
							);

							$wpdb->update( // phpcs:ignore
								$menus_table,
								array(
									'data' => maybe_serialize(
										array(
											'related_tabbar_id' => $menu_map[ $tabbar_id ],
											'title_position' => isset( $main_menus['headerbar'][ $tabbar_data['headerbar_id'] ]['title_position'] ) ? $main_menus['headerbar'][ $tabbar_data['headerbar_id'] ]['title_position'] : 'left',
											'more_menu_position' => isset( $main_menus['headerbar'][ $tabbar_data['headerbar_id'] ]['more_menu_position'] ) ? $main_menus['headerbar'][ $tabbar_data['headerbar_id'] ]['more_menu_position'] : 'tabbar',
											'priority' => isset( $main_menus['headerbar'][ $tabbar_data['headerbar_id'] ]['priority'] ) ? $main_menus['headerbar'][ $tabbar_data['headerbar_id'] ]['priority'] : 0,
											'original_data' => isset( $main_menus['headerbar'][ $tabbar_data['headerbar_id'] ] ) ? $main_menus['headerbar'][ $tabbar_data['headerbar_id'] ] : array(),
										)
									),
								),
								array( 'id' => $menu_map[ $tabbar_data['headerbar_id'] ] ),
								array( '%s' ),
								array( '%d' )
							);
						}
					}
				}
			}
		}
	}

	/**
	 * Migrate existing languages from bbapp_languages option to bbapp_string_translations table.
	 *
	 * @since 2.4.10
	 * @return void
	 */
	public function update_string_translations_30() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'bbapp_string_translations';

		// Check if the table exists.
		$table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

		if ( empty( $table_exists ) ) {
			// Table doesn't exist, so we need to create it first.
			\BuddyBossApp\AppLanguages::instance()->on_activation();
		}

		// Check if import_languages() process is already in progress.
		$import_in_progress = get_transient( 'bbapp_import_languages_in_progress' );
		if ( $import_in_progress ) {
			return;
		}

		// Set flag to indicate import process is starting.
		set_transient( 'bbapp_import_languages_in_progress', true, 10 * MINUTE_IN_SECONDS );

		try {
			\BuddyBossApp\AppLanguages::instance()->import_languages(); // This will trigger the import on the next admin page load.
		} finally {
			// Clear the flag when process completes or fails.
			delete_transient( 'bbapp_import_languages_in_progress' );
		}

		// Check if migration has already been performed.
		$migration_completed = get_option( 'bbapp_languages_migrated', false );

		// Skip if migration was already completed.
		if ( $migration_completed ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'BuddyBoss App: Skipping language migration as it was already performed previously.' ); // phpcs:ignore
			}

			return;
		}

		// Get the existing languages from option table.
		$existing_languages = get_option( 'bbapp_languages', array() );

		if ( empty( $existing_languages ) || ! is_array( $existing_languages ) ) {
			// Mark migration as completed even if there's nothing to migrate.
			update_option( 'bbapp_languages_migrated', true );

			return; // No languages to migrate.
		}

		$site_id        = get_current_blog_id();
		$current_time   = current_time( 'mysql' );
		$migrated_count = 0;

		// Begin transaction.
		$wpdb->query( 'START TRANSACTION' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery

		try {
			foreach ( $existing_languages as $language_code => $translations ) {
				if ( ! is_array( $translations ) ) {
					continue;
				}

				if ( 'en_US' === $language_code ) {
					$language_code = 'en';
				}

				foreach ( $translations as $string_handle => $translation_text ) {
					// Check if this is a build string (starts with 'bbAppBuildTime.').
					$is_build_string = 0;
					if ( 0 === strpos( $string_handle, 'bbAppBuildTime.' ) ) {
						$is_build_string = 1;
					}

					// Check if this string already exists in the translations table.
					$existing = $wpdb->get_var( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
						$wpdb->prepare( "SELECT id FROM {$table_name} WHERE language_code = %s AND string_handle = %s", $language_code, $string_handle ) // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
					);

					if ( ! $existing ) {
						// Insert new translation.
						$result = $wpdb->insert( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
							$table_name,
							array(
								'site_id'          => $site_id,
								'string_handle'    => $string_handle,
								'default_string'   => $translation_text,
								'language_code'    => $language_code,
								'updated_string'   => $translation_text,
								'is_custom_string' => 1,
								'is_build_string'  => $is_build_string,
								'created_at'       => $current_time,
								'updated_at'       => $current_time,
							)
						);

						if ( $result ) {
							++$migrated_count;
						}
					} else {
						// For existing records, only update the updated_string field.
						$result = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
							$table_name,
							array(
								'updated_string'   => $translation_text,
								'is_custom_string' => 0,
								'updated_at'       => $current_time,
							),
							array( 'id' => $existing )
						);

						if ( $result ) {
							++$migrated_count;
						}
					}
				}
			}

			// Commit the transaction if everything went well.
			$wpdb->query( 'COMMIT' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery

			// Log the migration if debug is enabled.
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( sprintf( 'BuddyBoss App: Migrated %d language strings from bbapp_languages option to bbapp_string_translations table', $migrated_count ) ); // phpcs:ignore
			}

			// We'll keep the original option for backward compatibility
			// but we could consider adding a flag to indicate it's been migrated.
			update_option( 'bbapp_languages_migrated', true );
		} catch ( \Exception $e ) {
			// Rollback on error.
			$wpdb->query( 'ROLLBACK' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery

			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'BuddyBoss App: Error migrating languages: ' . $e->getMessage() ); // phpcs:ignore
			}
		}
	}

	/**
	 * Deletes the specific app setting for access controls.
	 *
	 * @since 2.4.31
	 *
	 * @return void
	 */
	public function update_31() {
		AppSettings::instance()->delete_setting_value( 'labs.access_controls' );
		AppSettings::instance()->delete_setting_value( 'react_native_library.enable' );
	}

	/**
	 * Add item_data column to bbapp_menu_items and backfill from legacy option menus.
	 *
	 * @since 2.4.40
	 * @return void
	 */
	public function update_32() {
		global $wpdb;

		$menu_items_table = $wpdb->prefix . 'bbapp_menu_items';

		// Ensure the table exists.
		$table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $menu_items_table ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		if ( empty( $table_exists ) ) {
			return;
		}

		// Get the current database name.
		$current_db = $wpdb->dbname;

		// Add the column if missing.
		$column_exists = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE table_schema = %s AND table_name = %s AND column_name = 'item_data'",
				$current_db,
				$menu_items_table
			)
		); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( empty( $column_exists ) ) {
			$wpdb->query( "ALTER TABLE {$menu_items_table} ADD COLUMN item_data longtext NULL AFTER item_icon_data" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}

		// If there are already any non-null item_data rows, assume backfill ran.
		$needs_backfill = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$menu_items_table} WHERE item_data IS NOT NULL" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		if ( $needs_backfill > 0 ) {
			return;
		}

		// Load helper functions to read legacy option menus.
		if ( ! function_exists( 'get_db_main_menus' ) ) {
			require_once bbapp()->plugin_dir . 'include/Admin/Menus/functions.php';
		}
		if ( ! function_exists( 'get_db_main_menus' ) || ! function_exists( 'get_db_menu_items' ) ) {
			return;
		}

		$main_menus = get_db_main_menus();
		if ( empty( $main_menus ) ) {
			return;
		}

		// Build a map of new menu IDs by using data column stored during update_30.
		$menus_table = $wpdb->prefix . 'bbapp_menus';
		$menus = $wpdb->get_results( "SELECT id, data FROM {$menus_table}" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		if ( empty( $menus ) ) {
			return;
		}

		$menu_id_map = array();
		foreach ( $menus as $menu_row ) {
			$original = maybe_unserialize( $menu_row->data );
			if ( is_array( $original ) ) {
				if ( ! empty( $original['original_data'] ) && is_array( $original['original_data'] ) && ! empty( $original['original_data']['id'] ) ) {
					$menu_id_map[ $original['original_data']['id'] ] = (int) $menu_row->id;
				}
			}
		}

		// Also map known default IDs to their new IDs if missing from original_data.
		foreach ( array( 'default-tabbar', 'default-more', 'default-headerbar' ) as $default_id ) {
			if ( ! isset( $menu_id_map[ $default_id ] ) ) {
				$maybe_id = (int) $wpdb->get_var( $wpdb->prepare( "SELECT id FROM {$menus_table} WHERE data LIKE %s LIMIT 1", '%' . $wpdb->esc_like( $default_id ) . '%' ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				if ( $maybe_id ) {
					$menu_id_map[ $default_id ] = $maybe_id;
				}
			}
		}

		// For each legacy menu, iterate items and update rows' item_data using menu_id + menu_order + parent mapping.
		foreach ( $main_menus as $menu_type => $menus_data ) {
			foreach ( $menus_data as $legacy_menu_id => $legacy_menu ) {
				if ( empty( $menu_id_map[ $legacy_menu_id ] ) ) {
					continue;
				}
				$new_menu_id = (int) $menu_id_map[ $legacy_menu_id ];
				$items       = get_db_menu_items( $legacy_menu_id );
				if ( empty( $items ) || ! is_array( $items ) ) {
					continue;
				}

				$section_id_to_row = array();
				$order            = 0;
				foreach ( $items as $item ) {
					if ( empty( $item['type'] ) ) {
						$order++;
						continue;
					}
					$data = isset( $item['data'] ) && is_array( $item['data'] ) ? $item['data'] : array();
					$serialized = ! empty( $data ) ? maybe_serialize( $data ) : null;

					// Update the top-level row for this order.
					$wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
						$menu_items_table,
						array( 'item_data' => $serialized ),
						array( 'menu_id' => $new_menu_id, 'menu_order' => $order, 'parent_id' => 0 ),
						array( '%s' ),
						array( '%d', '%d', '%d' )
					);

					// Store section DB id to map children parent.
					$row_id = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM {$menu_items_table} WHERE menu_id=%d AND menu_order=%d AND parent_id=0", $new_menu_id, $order ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
					if ( $row_id && isset( $item['type'] ) && 'section' === $item['type'] && isset( $item['id'] ) ) {
						$section_id_to_row[ $item['id'] ] = (int) $row_id;
					}

					// Children inside section.
					if ( ! empty( $item['section'] ) && is_array( $item['section'] ) ) {
						$child_index = 0;
						foreach ( $item['section'] as $child ) {
							$child_data = isset( $child['data'] ) && is_array( $child['data'] ) ? $child['data'] : array();
							$child_serialized = ! empty( $child_data ) ? maybe_serialize( $child_data ) : null;
							$wpdb->update( // phpcs:ignore
								$menu_items_table,
								array( 'item_data' => $child_serialized ),
								array( 'menu_id' => $new_menu_id, 'parent_id' => (int) $row_id, 'menu_order' => $child_index ),
								array( '%s' ),
								array( '%d', '%d', '%d' )
							);
							$child_index++;
						}
					}

					// Headerbar-action icons children.
					if ( isset( $item['type'] ) && 'headerbar-action' === $item['type'] && ! empty( $item['icons'] ) && is_array( $item['icons'] ) ) {
						$child_index = 0;
						foreach ( $item['icons'] as $icon_item ) {
							$icon_data = isset( $icon_item['data'] ) && is_array( $icon_item['data'] ) ? $icon_item['data'] : array();
							$icon_serialized = ! empty( $icon_data ) ? maybe_serialize( $icon_data ) : null;
							$wpdb->update( // phpcs:ignore
								$menu_items_table,
								array( 'item_data' => $icon_serialized ),
								array( 'menu_id' => $new_menu_id, 'parent_id' => (int) $row_id, 'menu_order' => $child_index ),
								array( '%s' ),
								array( '%d', '%d', '%d' )
							);
							$child_index++;
						}
					}

					$order++;
				}
			}
		}
	}

	/**
	 * Fill locale gaps using bbapp_wp_locale_to_app_locale mapping.
	 * Migrates regional locale menus to canonical locales only when target locale is missing.
	 * Includes menu_name matching for precision.
	 *
	 * @since 2.8.0
	 */
	public function update_32_1() {
		global $wpdb;

		$menu_table = $wpdb->prefix . 'bbapp_menus';

		// Get the locale mapping from bbapp_wp_locale_to_app_locale.
		$locale_mappings = array(
			// English variants.
			'en_us'          => 'en',
			'en_au'          => 'en',
			'en_gb'          => 'en',
			'en_ca'          => 'en',
			'en_nz'          => 'en',
			'en_za'          => 'en',
			'en_ie'          => 'en',
			'en_in'          => 'en',
			'en_sg'          => 'en',
			'en_my'          => 'en',
			'en_ph'          => 'en',
			'en_jm'          => 'en',

			// French variants.
			'fr_ca'          => 'fr',
			'fr_fr'          => 'fr',
			'fr_be'          => 'fr',
			'fr_ch'          => 'fr',

			// German variants.
			'de_de'          => 'de',
			'de_ch'          => 'de',
			'de_at'          => 'de',
			'de_de_formal'   => 'de',
			'de_ch_informal' => 'de',

			// Spanish variants.
			'es_es'          => 'es',
			'es_mx'          => 'es',
			'es_ar'          => 'es',
			'es_co'          => 'es',
			'es_cl'          => 'es',
			'es_pe'          => 'es',
			'es_ve'          => 'es',
			'es_gt'          => 'es',
			'es_cr'          => 'es',
			'es_do'          => 'es',
			'es_ec'          => 'es',
			'es_uy'          => 'es',

			// Portuguese variants.
			'pt_br'          => 'pt',
			'pt_pt'          => 'pt',
			'pt_ao'          => 'pt',

			// Italian variants.
			'it_it'          => 'it',
			'it_ch'          => 'it',

			// Dutch variants.
			'nl_nl'          => 'nl',
			'nl_be'          => 'nl',

			// Chinese variants.
			'zh_cn'          => 'zh',
			'zh_tw'          => 'zh',
			'zh_hk'          => 'zh',

			// Arabic variants.
			'ar'             => 'ar',
			'ar_sa'          => 'ar',
			'ar_eg'          => 'ar',
			'ar_ma'          => 'ar',
			'ar_tn'          => 'ar',
			'ar_dz'          => 'ar',
			'ar_jo'          => 'ar',
			'ar_iq'          => 'ar',
			'ar_kw'          => 'ar',
			'ar_lb'          => 'ar',
			'ar_sy'          => 'ar',
			'ar_ye'          => 'ar',
			'ar_ae'          => 'ar',
			'ar_qa'          => 'ar',
			'ar_bh'          => 'ar',

			// Russian variants.
			'ru_ru'          => 'ru',
			'ru_ua'          => 'ru',

			// Japanese.
			'ja'             => 'ja',

			// Korean.
			'ko_kr'          => 'ko',

			// Norwegian variants.
			'nb_no'          => 'nb',
			'nn_no'          => 'nb',
			'no'             => 'nb',

			// Swedish variants.
			'sv_se'          => 'sv',
			'sv_fi'          => 'sv',

			// Danish.
			'da_dk'          => 'da',

			// Finnish.
			'fi'             => 'fi',
			'fi_fi'          => 'fi',

			// Polish.
			'pl_pl'          => 'pl',

			// Czech.
			'cs_cz'          => 'cs',

			// Slovak.
			'sk_sk'          => 'sk',

			// Hungarian.
			'hu_hu'          => 'hu',

			// Romanian.
			'ro_ro'          => 'ro',

			// Bulgarian.
			'bg_bg'          => 'bg',

			// Croatian.
			'hr'             => 'hr',
			'hr_hr'          => 'hr',

			// Serbian.
			'sr_rs'          => 'sr',

			// Slovenian.
			'sl_si'          => 'sl',

			// Estonian.
			'et'             => 'et',
			'et_ee'          => 'et',

			// Latvian.
			'lv'             => 'lv',
			'lv_lv'          => 'lv',

			// Lithuanian.
			'lt_lt'          => 'lt',

			// Greek.
			'el'             => 'el',
			'el_gr'          => 'el',

			// Turkish.
			'tr_tr'          => 'tr',

			// Hebrew.
			'he_il'          => 'he',

			// Hindi.
			'hi_in'          => 'hi',

			// Thai.
			'th'             => 'th',
			'th_th'          => 'th',

			// Vietnamese.
			'vi'             => 'vi',
			'vi_vn'          => 'vi',

			// Indonesian.
			'id_id'          => 'id',

			// Malay.
			'ms_my'          => 'ms',

			// Filipino.
			'fil'            => 'fil',

			// Ukrainian.
			'uk'             => 'uk',
			'uk_ua'          => 'uk',

			// Belarusian.
			'be_by'          => 'be',

			// Catalan.
			'ca'             => 'ca',
			'ca_es'          => 'ca',

			// Basque.
			'eu'             => 'eu',
			'eu_es'          => 'eu',

			// Galician.
			'gl_es'          => 'gl',

			// Welsh.
			'cy'             => 'cy',
			'cy_gb'          => 'cy',

			// Irish.
			'ga_ie'          => 'ga',

			// Scots Gaelic.
			'gd'             => 'gd',
			'gd_gb'          => 'gd',

			// Icelandic.
			'is_is'          => 'is',

			// Maltese.
			'mt_mt'          => 'mt',

			// Afrikaans.
			'af'             => 'af',
			'af_za'          => 'af',

			// Swahili.
			'sw'             => 'sw',
			'sw_ke'          => 'sw',

			// Amharic.
			'am'             => 'am',
			'am_et'          => 'am',

			// Bengali.
			'bn_bd'          => 'bn',
			'bn_in'          => 'bn',

			// Gujarati.
			'gu'             => 'gu',
			'gu_in'          => 'gu',

			// Punjabi.
			'pa_in'          => 'pa',

			// Tamil.
			'ta_in'          => 'ta',
			'ta_lk'          => 'ta',

			// Telugu.
			'te'             => 'te',
			'te_in'          => 'te',

			// Kannada.
			'kn'             => 'kn',
			'kn_in'          => 'kn',

			// Malayalam.
			'ml_in'          => 'ml',

			// Marathi.
			'mr'             => 'mr',
			'mr_in'          => 'mr',

			// Nepali.
			'ne_np'          => 'ne',

			// Urdu.
			'ur'             => 'ur',
			'ur_pk'          => 'ur',

			// Persian/Dari.
			'fa_ir'          => 'fa',

			// Azerbaijani.
			'az'             => 'az',
			'az_az'          => 'az',

			// Georgian.
			'ka_ge'          => 'ka',

			// Armenian.
			'hy'             => 'hy',
			'hy_am'          => 'hy',

			// Kazakh.
			'kk'             => 'kk',
			'kk_kz'          => 'kk',

			// Kyrgyz.
			'ky_kg'          => 'ky',

			// Mongolian.
			'mn'             => 'mn',
			'mn_mn'          => 'mn',

			// Uzbek.
			'uz_uz'          => 'uz',

			// Tajik.
			'tg'             => 'tg',
			'tg_tj'          => 'tg',

			// Kurdish.
			'ckb'            => 'ckb',
			'ku'             => 'ku',

			// Pashto.
			'ps'             => 'ps',
			'ps_af'          => 'ps',

			// Sindhi.
			'sd_pk'          => 'sd',

			// Sinhala.
			'si_lk'          => 'si',

			// Burmese.
			'my_mm'          => 'my',

			// Khmer.
			'km'             => 'km',
			'km_kh'          => 'km',

			// Lao.
			'lo'             => 'lo',
			'lo_la'          => 'lo',

			// Tibetan.
			'bo'             => 'bo',
			'bo_cn'          => 'bo',
		);

		foreach ( $locale_mappings as $regional_locale => $target_locale ) {
			// Case 1: Find cases where target locale menu is missing
			// 1. Regional locale menu exists (e.g., en_gb + "Default")
			// 2. Target locale menu is missing (e.g., en + "Default")
			// 3. Match on menu_type, login_state, menu_name, site_id.
			$gaps_to_fill = $wpdb->get_results( $wpdb->prepare(
				"SELECT regional.*
				FROM {$menu_table} regional
				WHERE regional.language_code = %s
				AND NOT EXISTS (
					SELECT 1 FROM {$menu_table} target
					WHERE target.menu_type = regional.menu_type
					AND target.login_state = regional.login_state
					AND target.menu_name = regional.menu_name
					AND target.site_id = regional.site_id
					AND target.language_code = %s
				)",
				$regional_locale,
				$target_locale
			) );

			// Update only the menus that fill gaps.
			foreach ( $gaps_to_fill as $menu ) {
				$wpdb->update(
					$menu_table,
					array( 'language_code' => $target_locale ),
					array( 'id' => $menu->id ),
					array( '%s' ),
					array( '%d' )
				);

				if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
					// Log the migration for debugging.
					error_log( "BBAPP Migration: Updated menu ID {$menu->id} from {$regional_locale} to {$target_locale} (type: {$menu->menu_type}, name: {$menu->menu_name}, login: {$menu->login_state})" );
				}
			}

			// Case 2: Find cases where both locales exist but target locale has no menu items
			// 1. Both regional and target locale menus exist
			// 2. Target locale menu has no items in bbapp_menu_items table
			// 3. Regional locale menu has actual items in bbapp_menu_items table.
			$menu_items_table = $wpdb->prefix . 'bbapp_menu_items';
			$empty_target_menus = $wpdb->get_results( $wpdb->prepare(
				"SELECT target.id as target_id, regional.id as regional_id, regional.menu_type, regional.menu_name, regional.login_state
				FROM {$menu_table} target
				INNER JOIN {$menu_table} regional ON (
					regional.menu_type = target.menu_type
					AND regional.login_state = target.login_state
					AND regional.menu_name = target.menu_name
					AND regional.site_id = target.site_id
				)
				WHERE target.language_code = %s
				AND regional.language_code = %s
				AND NOT EXISTS (
					SELECT 1 FROM {$menu_items_table} target_items
					WHERE target_items.menu_id = target.id
				)
				AND EXISTS (
					SELECT 1 FROM {$menu_items_table} regional_items
					WHERE regional_items.menu_id = regional.id
				)",
				$target_locale,
				$regional_locale
			) );

			// Replace empty target menus with regional content.
			foreach ( $empty_target_menus as $menu_pair ) {
				// Delete the empty target menu.
				$wpdb->delete(
					$menu_table,
					array( 'id' => $menu_pair->target_id ),
					array( '%d' )
				);

				// Update regional menu to target locale.
				$wpdb->update(
					$menu_table,
					array( 'language_code' => $target_locale ),
					array( 'id' => $menu_pair->regional_id ),
					array( '%s' ),
					array( '%d' )
				);

				if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
					// Log the replacement for debugging.
					error_log( "BBAPP Migration: Replaced empty menu ID {$menu_pair->target_id} ({$target_locale}) with content from menu ID {$menu_pair->regional_id} ({$regional_locale}) (type: {$menu_pair->menu_type}, name: {$menu_pair->menu_name}, login: {$menu_pair->login_state})" );
				}
			}
		}
	}
}
