<?php
/**
 * Holds cron jobs related fuctionality.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

use BuddyBossApp\InAppPurchases\Orders;
use BuddyBossApp\Tools\Logger;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Cron jobs class.
 */
final class CronJobs {

	/**
	 * Class instance.
	 *
	 * @var null $instance
	 */
	private static $instance = null;

	/**
	 * CronJobs constructor.
	 */
	private function __construct() {
		// ... leave empty, see Singleton below.
	}

	/**
	 * Get the instance of this class.
	 *
	 * @return object
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Filters/hooks here.
	 */
	private function hooks() {
		add_action( 'bbapp_on_plugin_activate', array( $this, 'on_plugin_activate' ) );
		add_action( 'init', array( $this, 'cron_health_checkup' ) );
		add_filter( 'cron_schedules', array( $this, 'on_cron_schedules' ), 8888, 1 );

		/**
		 * IAP Recurring Order Process.
		 */
		add_filter( 'bbapp_every_10min', array( $this, 'recurring_orders_process' ), 9999 );

		/**
		 * Table data cleanup.
		 */
		add_filter( 'bbapp_every_week', array( $this, 'cleanup_tables' ), 9999 );

		/**
		 * Run the connection health check in every 5 hours.
		 */
		add_action(
			'bbapp_every_5hours',
			array(
				\BuddyBossApp\Admin\ManageApp::instance(),
				'verify_apps_connection',
			)
		);

		/**
		 * Run the apple and android account connection attempt every 30 min.
		 */
		add_action(
			'bbapp_every_30min',
			array(
				\BuddyBossApp\Admin\ManageApp::instance(),
				'attempt_to_reconnect_account',
			)
		);

		/**
		 * Verify the client app with App Center every hour.
		 */
		add_action( 'bbapp_every_hour', array( ManageApp::instance(), 'verify_app' ) );

		/**
		 * Update Google/Apple connectivity to App Center every 30 min.
		 */
		add_action( 'bbapp_every_30min', array( ManageApp::instance(), 'sync_store_connectivity_with_appcenter' ) );
	}

	/**
	 *    To make sure cron-job is running, we can't keep reactivating plugin to keep enable it again
	 *    `init` is action defined by WordPress.
	 *
	 * @see : https://codex.wordpress.org/Plugin_API/Action_Reference/init
	 */
	public function cron_health_checkup() {
		$last_checked = get_transient( 'bbapp_cron_job_health_flag' );

		if ( empty( $last_checked ) ) {
			$this->schedule_jobs();

			// Check schedule every 0.5 hours(30 min).
			set_transient( 'bbapp_cron_job_health_flag', 'done', 0.5 * HOUR_IN_SECONDS );
		}
	}

	/**
	 *    Triggers/Runs when cron_schedules(filter) happens
	 *    `cron_schedules` is filter defined by WordPress.
	 *
	 * @see : https://codex.wordpress.org/Plugin_API/Filter_Reference/cron_schedules
	 *
	 * @param array $schedules Cron schedules.
	 *
	 * @return mixed
	 */
	public function on_cron_schedules( $schedules ) {
		// Related to ItemMeta and further.
		if ( ! isset( $schedules['bbapp_every_day'] ) ) {
			$schedules['bbapp_every_day'] = array(
				'interval' => MINUTE_IN_SECONDS * 60 * 24, // 60 * 60 * 24; Eg : 60 (seconds) * 60  * 24  => 24 hrs
				'display'  => __( 'BuddyBoss App ItemMeta and other tasks every day', 'buddyboss-app' ),
			);
		}

		if ( ! isset( $schedules['bbapp_every_10min'] ) ) {
			$schedules['bbapp_every_10min'] = array(
				'interval' => MINUTE_IN_SECONDS * 10, // 60 * 10; Eg : 60 (seconds) * 10 => 10 minutes
				'display'  => __( 'BuddyBoss App InAppPurchases Every 10 Minutes', 'buddyboss-app' ),
			);
		}

		if ( ! isset( $schedules['bbapp_every_5hours'] ) ) {
			$schedules['bbapp_every_5hours'] = array(
				'interval' => MINUTE_IN_SECONDS * 60 * 5, // 60 * 60 * 5; Eg : 60 (seconds) * 60  * 5  => 5 hrs
				'display'  => __( 'BuddyBoss App AppCenter Every 5 Hours', 'buddyboss-app' ),
			);
		}

		// Related to Push Notifications.
		if ( ! isset( $schedules['bbapp_every_5min'] ) ) {
			$schedules['bbapp_every_5min'] = array(
				'interval' => MINUTE_IN_SECONDS * 5, // 60 * 5; Eg : 60 (seconds) * 5 => 5 minutes
				'display'  => __( 'BuddyBoss App Push Notification Every 5 minutes', 'buddyboss-app' ),
			);
		}

		// Related to Sync Data from App Store to Appcenter.
		if ( ! isset( $schedules['bbapp_every_30min'] ) ) {
			$schedules['bbapp_every_30min'] = array(
				'interval' => MINUTE_IN_SECONDS * 30, // Every half an hour.
				'display'  => __( 'BuddyBoss App Publish sync.', 'buddyboss-app' ),
			);
		}

		// Related to BuddyBossApp Cleaner.
		if ( ! isset( $schedules['bbapp_weekly'] ) ) {
			$schedules['bbapp_weekly'] = array(
				'interval' => MINUTE_IN_SECONDS * 60 * 24 * 7,
				// * 60 * 24 * 14, // 60 * 60 * 7; Eg : 60 (seconds) * 60  * 24  * 7 => 14 days
				'display'  => __( 'BuddyBoss App Cleaner Weekly', 'buddyboss-app' ),
			);
		}

		// BG process log clear.
		if ( ! isset( $schedules['bbapp_every_hour'] ) ) {
			$schedules['bbapp_every_hour'] = array(
				'interval' => HOUR_IN_SECONDS,
				'display'  => __( 'BuddyBoss App Background Process Log Cleaner Hourly', 'buddyboss-app' ),
			);
		}

		return $schedules;
	}

	/**
	 *    Triggers/Runs when bbapp_on_plugin_activate(action) happens.
	 *    `bbapp_on_plugin_activate` is custom action defined by BuddyBossApp
	 *
	 * @see : buddyboss-app.php->on_plugin_activate() or find do_action("bbapp_on_plugin_activate") in source code
	 */
	public function on_plugin_activate() {
		$this->schedule_jobs();
	}

	/**
	 * Schedules(Install) the jobs.
	 */
	public function schedule_jobs() {
		// NOTE : To avoid more than 1 cron schedule irrespective of # of times plugin is activated.
		wp_clear_scheduled_hook( 'bbapp_every_day' );
		wp_clear_scheduled_hook( 'bbapp_every_10min' );
		wp_clear_scheduled_hook( 'bbapp_every_5hours' );
		wp_clear_scheduled_hook( 'bbapp_every_5min' );
		wp_clear_scheduled_hook( 'bbapp_every_30min' );
		wp_clear_scheduled_hook( 'bbapp_every_week' );
		wp_clear_scheduled_hook( 'bbapp_every_hour' );

		// NOTE : Cron every 10 min.
		if ( ! wp_next_scheduled( 'bbapp_every_10min' ) ) {
			wp_schedule_event( time(), 'bbapp_every_10min', 'bbapp_every_10min', array() );
		}

		// NOTE : Cron every 5 hours.
		if ( ! wp_next_scheduled( 'bbapp_every_5hours' ) ) {
			wp_schedule_event( time(), 'bbapp_every_5hours', 'bbapp_every_5hours', array() );
		}

		// NOTE : Cron every 5 min.
		if ( ! wp_next_scheduled( 'bbapp_every_5min' ) ) {
			wp_schedule_event( time(), 'bbapp_every_5min', 'bbapp_every_5min', array() );
		}

		// NOTE : Cron every day.
		if ( ! wp_next_scheduled( 'bbapp_every_30min' ) ) {
			wp_schedule_event( time(), 'bbapp_every_30min', 'bbapp_every_30min', array() );
		}

		// NOTE : Cron One hour.
		if ( ! wp_next_scheduled( 'bbapp_every_hour' ) ) {
			wp_schedule_event( time(), 'bbapp_every_hour', 'bbapp_every_hour', array() );
		}

		// NOTE : Cron every day.
		if ( ! wp_next_scheduled( 'bbapp_every_day' ) ) {
			wp_schedule_event( time(), 'bbapp_every_day', 'bbapp_every_day', array() );
		}

		// NOTE : Cron every week.
		if ( ! wp_next_scheduled( 'bbapp_weekly' ) ) {
			wp_schedule_event( time(), 'bbapp_weekly', 'bbapp_every_week', array() );
		}
	}

	/**
	 * Process Recurring Orders...
	 * Triggers/Runs when bbapp_every_10min(action) happens.
	 * `bbapp_every_10min` is custom action defined by BuddyBossApp
	 */
	public function recurring_orders_process() {
		if ( bbapp_is_active( 'iap' ) ) {
			// now utc time.
			$now = gmdate( 'Y-m-d H:i:s' );

			if ( IAP_LOG ) {
				Logger::instance()->add( 'iap_log', sprintf( 'BuddyBossApp\CronJob->recurringOrdersProcess() at: %s', $now ) );
			}

			global $wpdb;

			$lock_key = 'bbapp_iap_validation_task_lock';

			set_transient( $lock_key, '1', MINUTE_IN_SECONDS * 10 );

			$table      = Orders::instance()->table_name;
			$meta_table = Orders::instance()->table_meta_name;

			/**
			 * Filter to use to change 10 validations per batch.
			 *
			 * @param int $per_batch Per batch.
			 *
			 * @since 1.7.7
			 */
			$validations_per_batch = (int) apply_filters( 'bbapp_validation_per_batch', 10 );
			/**
			 * We are checking below things :
			 * 1. expire_at(from bbapp_iap_orders) should be older then UTC NOW.
			 * 2. _next_order_check(from bbapp_iap_ordermeta) should be older than UTC NOW.
			 *
			 * When order is created initially we store `_next_order_check` time little(+1 minute) greater then real expiry for allowing renewing time period.
			 * NOTE : `_next_order_check` time gap vary based on sandbox & production.
			 */
			$query   = $wpdb->prepare( "SELECT *FROM {$table} as o, {$meta_table} as m WHERE o.id=m.order_id AND m.meta_key='_next_order_check' AND m.meta_value < %s AND o.expire_at < %s AND o.is_recurring=1 AND o.order_status='subscribed'", $now, $now ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$results = $wpdb->get_results( $query ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
			// NOTE : Should we have this log stored in DB forever?
			$orders_to_verify = count( $results );

			if ( IAP_LOG && $orders_to_verify > 0 ) {
				Logger::instance()->add( 'iap_log', sprintf( 'BuddyBossApp\CronJob->recurringOrdersProcess(), Number of orders on which revoke/renew operations are required : %s', $orders_to_verify ) );
			}

			$i = 0;

			foreach ( $results as $result ) {
				if ( $i > $validations_per_batch ) {
					break;
				}

				if ( isset( bbapp_iap()->iap[ $result->device_platform ] ) ) {
					$iap = bbapp_iap()->iap[ $result->device_platform ];
					$iap->_validate( $result->id );
				} else {
					// NOTE : Schedule next revoke/renew check after 24 hours.
					Orders::instance()->update_order_next_validation_retry( $result->id, '+24 hours' );
				}

				++$i;
			}

			// Once task is finished release the lock.
			delete_transient( $lock_key );
		}
	}

	/**
	 * Cleanup(delete) logs table and etc
	 * Triggers/Runs when bbapp_cleaner(action) happens.
	 * `bbapp_cleaner` is custom action defined by BuddyBossApp
	 */
	public function cleanup_tables() {
		global $wpdb;

		$table  = bbapp_get_network_table( Logger::instance()->table );
		$query  = $wpdb->prepare( "DELETE FROM {$table} WHERE `created` <= %s ", gmdate( 'Y-m-d', strtotime( ' -7 day' ) ) . ' 23:59:59' ); //phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$status = $wpdb->query( $query ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		// now utc time.
		$now = gmdate( 'Y-m-d H:i:s' );

		if ( ! empty( $status ) ) {
			Logger::instance()->add( 'info_log', sprintf( 'BuddyBossApp\CronJob->cleanupTables() attempt at: %s, status is: %s ', $now, $status ) );
		}
	}
}
