<?php
/**
 * Holds component register functionality.
 *
 * @package BuddyBossApp\Components
 */

namespace BuddyBossApp\Components;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * This class is used to add the components into BuddyBossApp.
 * Class Extend
 *
 * @package BuddyBossApp\Notification
 */
abstract class Extend {

	/**
	 * Class instance.
	 *
	 * @var array $instances
	 */
	private static $instances = array();

	/**
	 * Component name.
	 *
	 * @var bool $name
	 */
	private $name = false;

	/**
	 * Component id.
	 *
	 * @var bool $id
	 */
	private $id = false;

	/**
	 * Component description.
	 *
	 * @var string $desc
	 */
	private $desc = '';

	/**
	 * If component is enabled by default.
	 *
	 * @var bool $enabled_default
	 */
	private $enabled_default = false;

	/**
	 * If component is required.
	 *
	 * @var bool $required
	 */
	private $required = false;

	/**
	 * Hide from components list.
	 *
	 * @var bool $hide_from_list
	 */
	private $hide_from_list = false;

	/**
	 * Extend constructor.
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return mixed
	 */
	public static function instance() {
		$class = get_called_class();

		if ( ! isset( self::$instances[ $class ] ) ) {
			self::$instances[ $class ] = new $class();
			self::$instances[ $class ]->components_load();
		}

		return self::$instances[ $class ];
	}

	/**
	 * Filters/hooks here.
	 */
	public function components_load() {
		$this->load();
		if ( $this->id ) {
			add_filter( 'bbapp_get_components', array( $this, 'bbapp_get_components' ) );
			add_action( 'bbapp_components_loaded', array( $this, 'components_loaded' ), 99 );
		}
	}

	/**
	 * This function is must needed to call.
	 *
	 * @param string $id                 Components ID.
	 * @param string $name               Components Name ( Will used as Component Label or Title on Components Settings ).
	 * @param string $desc               Description.
	 * @param bool   $enabled_by_default Enabled by default.
	 * @param bool   $required           If the component is required.
	 *
	 * @return void
	 */
	public function start( $id, $name, $desc = '', $enabled_by_default = false, $required = false ) {
		if ( ! empty( $id ) && ! empty( $name ) ) {
			$this->id              = $id;
			$this->name            = $name;
			$this->desc            = $desc;
			$this->enabled_default = $enabled_by_default;
			$this->required        = $required;
		}
	}

	/**
	 * Function to hide component from the components listing.
	 *
	 * @since 1.5.2.1
	 */
	public function hide_from_list() {
		$this->hide_from_list = true;
	}

	/**
	 * Returns the component id.
	 *
	 * @return bool
	 */
	public function get_component_id() {
		return $this->component_name;
	}

	/**
	 * Returns the component name.
	 *
	 * @return bool
	 */
	public function get_component_name() {
		return $this->component_name;
	}

	/**
	 * Abstract load method.
	 *
	 * @return mixed
	 */
	abstract public function load();

	/**
	 * This class shouldn't be override.
	 *
	 * @param array $components Components.
	 *
	 * @return mixed
	 */
	public function bbapp_get_components( $components ) {
		$components[ $this->id ] = array(
			'id'              => $this->id,
			'name'            => $this->name,
			'desc'            => $this->desc,
			'enabled_default' => $this->enabled_default,
			'required'        => $this->required,
			'instance'        => $this,
			'hide_from_list'  => $this->hide_from_list,
		);

		return $components;
	}

	/**
	 * Provide setting links by components.
	 *
	 * @return mixed
	 */
	abstract public function settings_links();

	/**
	 * Return the settings link.
	 *
	 * @return array
	 */
	public function get_settings_link() {
		$setting_links = array();

		if ( method_exists( $this, 'settings_links' ) ) {
			$setting_links = $this->settings_links();

			if ( ! is_array( $setting_links ) ) {
				$setting_links = array();
			}
		}

		return $setting_links;
	}

	/**
	 * Safe point to load all codes.
	 *
	 * @return mixed
	 */
	abstract public function components_loaded();
}

