<?php
/**
 * Holds components class functionality.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Components class.
 */
class Components {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	public static $instance;

	/**
	 * Get the instance of class
	 *
	 * @return Components
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Components constructor.
	 */
	public function __construct() {
	}

	/**
	 * Returns all available components.
	 */
	public function get_components() {
		$components = apply_filters( 'bbapp_get_components', array() );

		return $components;
	}

	/**
	 * Get active components
	 *
	 * @return array
	 */
	public function get_active_components() {
		static $get_components;

		if ( ! isset( $get_components ) ) {
			$get_components = bbapp_get_network_option( 'bbapp_active_components' );
			$get_components = is_array( $get_components ) ? $get_components : array();
		}

		$all_components    = $this->get_components();
		$active_components = array();

		foreach ( $all_components as $component_id => $value ) {
			if ( isset( $get_components[ $component_id ] ) && $get_components[ $component_id ] ) {
				$active_components[ $component_id ] = true;
			}

			// if component is required make it enable.
			if ( isset( $all_components[ $component_id ] ) && $all_components[ $component_id ]['required'] ) {
				$active_components[ $component_id ] = true;
			}

			// Enable default component.
			if ( isset( $all_components[ $component_id ] ) && ! isset( $get_components[ $component_id ] ) && $all_components[ $component_id ]['enabled_default'] ) {
				$active_components[ $component_id ] = $all_components[ $component_id ]['enabled_default'];
			}
		}

		return $active_components;
	}

	/**
	 * Return active components.
	 *
	 * @return array
	 */
	public function get_inactive_components() {
		static $inactive_components;

		if ( ! isset( $inactive_components ) ) {
			$all_components      = $this->get_components();
			$active_components   = $this->get_active_components();
			$inactive_components = array();

			foreach ( $all_components as $component_id => $component ) {
				if ( ! isset( $active_components[ $component_id ] ) || ! $active_components[ $component_id ] ) {
					$inactive_components[ $component_id ] = false;
				}

				if ( isset( $all_components[ $component_id ] ) && $all_components[ $component_id ]['required'] ) {
					unset( $inactive_components[ $component_id ] );
				}
			}
		}

		return $inactive_components;
	}

	/**
	 * Return required components.
	 *
	 * @return array
	 */
	public function get_required_components() {
		$all_components      = $this->get_components();
		$required_components = array();

		foreach ( $all_components as $component_id => $component ) {
			if ( $component['required'] ) {
				$required_components[ $component_id ] = true;
			}
		}

		return $required_components;
	}

	/**
	 * Enable Components
	 *
	 * @param array $components Components.
	 *
	 * @return bool
	 */
	public function activate_components( $components = array() ) {
		$active_components = bbapp_get_network_option( 'bbapp_active_components' );
		$active_components = ( is_array( $active_components ) ) ? $active_components : array();
		$all_components    = $this->get_components();

		foreach ( $components as $component_id ) {
			if ( isset( $all_components[ $component_id ] ) ) {
				$active_components[ $component_id ] = true;
			}
		}

		$option_saved = bbapp_set_network_option( 'bbapp_active_components', $active_components );

		/**
		 * Allow to add custom code on components activate
		 *
		 * @param array $active_components Components.
		 */
		do_action( 'bbapp_components_activate', $active_components );

		return $option_saved;
	}

	/**
	 * Disable components.
	 *
	 * @param array $components Components.
	 *
	 * @return bool
	 */
	public function deactivate_components( $components = array() ) {
		$active_components = bbapp_get_network_option( 'bbapp_active_components' );
		$active_components = ( is_array( $active_components ) ) ? $active_components : array();
		$all_components    = $this->get_components();

		foreach ( $components as $component_id ) {
			if ( isset( $all_components[ $component_id ] ) && ! $all_components[ $component_id ]['required'] ) {
				$active_components[ $component_id ] = false;
			}
		}

		$option_saved = bbapp_set_network_option( 'bbapp_active_components', $active_components );

		/**
		 * Allow to add custom code on components deactivate.
		 *
		 * @param array $active_components Active components.
		 */
		do_action( 'bbapp_components_deactivate', $active_components );

		return $option_saved;
	}

	/**
	 * Return if given component is active or not.
	 *
	 * @param string $component_id Component id.
	 *
	 * @return bool
	 */
	public function is_active( $component_id ) {
		$active_components = $this->get_active_components();

		if ( isset( $active_components[ $component_id ] ) && $active_components[ $component_id ] ) {
			return true;
		}

		return false;
	}
}
