<?php
/**
 * Holds icon picker functionality.
 *
 * @package BuddyBossApp\Common
 */

namespace BuddyBossApp\Common;

// Exit if accessed directly.
use BuddyBossApp\Helpers\BBAPP_File;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Icon picker class.
 */
class IconPicker {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	public static $instance;

	/**
	 * IconPicker constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return IconPicker
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		$this->hooks();
	}

	/**
	 * Define all Hooks.
	 */
	public function hooks() {
		add_action( 'wp_ajax_bbapp_icon_picker_upload', array( $this, 'icon_picker_upload' ) );
		add_action( 'wp_ajax_bbapp_delete_custom_icon', array( $this, 'delete_custom_icon' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_head' ), 9 );
	}

	/**
	 * Load scripts on admin side.
	 */
	public function admin_head() {
		global $pagenow, $post_type;

		$setting  = ! empty( $_GET['setting'] ) ? sanitize_text_field( wp_unslash( $_GET['setting'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$page_tab = ! empty( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( 'bbapp-menus' === $page_tab ) {
			$setting = ! empty( $setting ) ? $setting : 'tabbar';
		}

		$allowed_page_now = array(
			'post-new.php',
			'edit.php',
			'post.php',
			'edit.php',
		);

		$allow_page_tab = array( 'bbapp-appearance', 'bbapp-menus' );

		$allowed_setting_tab = array(
			'app_menu',
			'branding',
			'tabbar',
			'more',
			'headerbar',
		);

		if ( ( in_array( $pagenow, $allowed_page_now, true ) && 'app_page' === $post_type ) || ( 'admin.php' === $pagenow && in_array( $page_tab, $allow_page_tab, true ) && in_array( $setting, $allowed_setting_tab, true ) ) ) {
			add_action( 'admin_footer', array( $this, 'load_icon_picker_code' ) );
		}
	}

	/**
	 * Icon Picker Related Code
	 */
	public function load_icon_picker_code() {

		// Include AppPage Footer Code.
		include bbapp()->plugin_dir . 'views/icon-picker/rendor-icons.php';
	}

	/**
	 * Ajax handler to upload custom icons into WordPress.
	 *
	 * @since 1.7.0
	 *
	 * @return mixed
	 */
	public function icon_picker_upload() {
		$icons_size = 200;

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'You don\'t have permission to upload a custom icon', 'buddyboss-app' ) );
		}

		if ( isset( $_POST['nonce'] ) && wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'bbapp-upload-custom-icon' ) ) { //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$dir = $this->icon_picker_custom_icon_dir();
			$url = $this->icon_picker_custom_icon_url();

			if ( empty( $_FILES['icon'] ) ) {
				wp_send_json_error( __( 'Icon not found on file object.', 'buddyboss-app' ) );
			}

			$file_data = map_deep( $_FILES['icon'], 'sanitize_text_field' ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
			$file_data = BBAPP_File::read_file( $file_data['tmp_name'] );

			if ( empty( $file_data ) ) {
				wp_send_json_error( __( 'Icon you provided is invalid.', 'buddyboss-app' ) );
			}

			$file_hash = uniqid() . '.png';
			$im        = imagecreatefromstring( $file_data );

			if ( ! $im ) {
				wp_send_json_error( __( 'Icon you provided is invalid.', 'buddyboss-app' ) );
			}

			$width  = imagesx( $im );
			$height = imagesy( $im );

			if ( $width !== $icons_size || $height !== $icons_size ) {
				wp_send_json_error( __( 'Please provide an icon with correct size.', 'buddyboss-app' ) );
			}

			$out_location = "{$dir}/{$file_hash}";

			// Save the image.
			$new = imagecreatetruecolor( $width, $height );
			imagecolortransparent( $new, imagecolorallocatealpha( $new, 0, 0, 0, 127 ) );
			imagealphablending( $new, false );
			imagesavealpha( $new, true );
			imagecopyresampled( $new, $im, 0, 0, 0, 0, $icons_size, $icons_size, $width, $height );
			imagepng( $new, $out_location );
			imagedestroy( $im );

			$name = "custom/{$file_hash}";

			wp_send_json_success(
				array(
					'url'  => "{$url}/{$file_hash}",
					'name' => $name,
					'id'   => basename( $name, '.png' ),
				)
			);
		}
	}

	/**
	 * Delete Custom Icon selected.
	 */
	public function delete_custom_icon() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( "You don't have permission to delete a custom icon", 'buddyboss-app' ) );
		}

		if ( isset( $_POST['nonce'] ) && wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'bbapp-delete-custom-icon' ) ) { //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$task      = ! empty( $_POST['task'] ) ? sanitize_text_field( wp_unslash( $_POST['task'] ) ) : '';
			$icon_data = ! empty( $_POST['elm1_data_val'] ) ? sanitize_text_field( wp_unslash( $_POST['elm1_data_val'] ) ) : '';

			if ( empty( $icon_data ) ) {
				wp_send_json_error( __( 'Please choose custom icon to delete.', 'buddyboss-app' ) );
			}

			$icon_data = explode( '/', $icon_data );
			$icon      = isset( $icon_data[1] ) ? $icon_data[1] : '';

			if ( empty( $icon ) ) {
				wp_send_json_error( __( 'Please choose custom icon to delete.', 'buddyboss-app' ) );
			}

			if ( 'delete_custom_icon' === $task ) {
				$dir       = $this->icon_picker_custom_icon_dir();
				$icon_path = "{$dir}/{$icon}.png";

				if ( file_exists( $icon_path ) ) {
					wp_delete_file( $icon_path );
					wp_send_json_success( __( 'Icon deleted.', 'buddyboss-app' ) );
				}
				wp_send_json_error( __( 'There is no icon for deleting.', 'buddyboss-app' ) );
			}
			wp_send_json_error( __( 'There was a problem deleting the icon.', 'buddyboss-app' ) );
		}
		wp_send_json_error( __( 'Security check failed.', 'buddyboss-app' ) );
	}

	/**
	 * App Icon picker dir path.
	 *
	 * @return string
	 */
	public function icon_picker_app_icon_dir() {
		return trailingslashit( bbapp()->plugin_dir . 'assets/app-icons/' );
	}

	/**
	 * App Icon picker url path.
	 *
	 * @return string
	 */
	public function icon_picker_app_icon_url() {
		return trailingslashit( bbapp()->plugin_url . 'assets/app-icons/' );
	}

	/**
	 * App Icon picker dir path.
	 *
	 * @since 1.7.0
	 * @return string
	 */
	public function icon_picker_bb_icon_dir() {
		return trailingslashit( bbapp()->plugin_dir . 'assets/icons/bb-icons/' );
	}

	/**
	 * App Icon picker url path.
	 *
	 * @since 1.7.0
	 * @return string
	 */
	public function icon_picker_bb_icon_url() {
		return trailingslashit( bbapp()->plugin_url . 'assets/icons/bb-icons/' );
	}

	/**
	 * App icon list.
	 *
	 * @return array[]
	 */
	public function ap_icon_picker_app_icon_list() {
		$bpapp_icon_list = array(
			'activity'              => _x( 'Activity', 'Tab Bar Icon', 'buddyboss-app' ),
			'blog'                  => _x( 'Blog', 'Tab Bar Icon', 'buddyboss-app' ),
			'book-open'             => _x( 'Courses', 'Tab Bar Icon', 'buddyboss-app' ),
			'achievement'           => _x( 'Achievement', 'Tab Bar Icon', 'buddyboss-app' ),
			'certificates'          => _x( 'Certificates', 'Tab Bar Icon', 'buddyboss-app' ),
			'category'              => _x( 'Category', 'Tab Bar Icon', 'buddyboss-app' ),
			'file'                  => _x( 'Page', 'Tab Bar Icon', 'buddyboss-app' ),
			'forums'                => _x( 'Forums', 'Tab Bar Icon', 'buddyboss-app' ),
			'groups'                => _x( 'Groups', 'Tab Bar Icon', 'buddyboss-app' ),
			'home'                  => _x( 'Home', 'Tab Bar Icon', 'buddyboss-app' ),
			'library'               => _x( 'Library', 'Tab Bar Icon', 'buddyboss-app' ),
			'messages'              => _x( 'Messages', 'Tab Bar Icon', 'buddyboss-app' ),
			'notification'          => _x( 'Notifications', 'Tab Bar Icon', 'buddyboss-app' ),
			'people'                => _x( 'Members', 'Tab Bar Icon', 'buddyboss-app' ),
			'photos'                => _x( 'Photos', 'Tab Bar Icon', 'buddyboss-app' ),
			'products'              => _x( 'Products', 'Tab Bar Icon', 'buddyboss-app' ),
			'profile'               => _x( 'Profile', 'Tab Bar Icon', 'buddyboss-app' ),
			'settings'              => _x( 'Settings', 'Tab Bar Icon', 'buddyboss-app' ),
			'topics'                => _x( 'Discussion', 'Tab Bar Icon', 'buddyboss-app' ),
			'web-link'              => _x( 'Web Link', 'Tab Bar Icon', 'buddyboss-app' ),
			'documents'             => _x( 'Documents', 'Tab Bar Icon', 'buddyboss-app' ),
			'email'                 => _x( 'Email', 'Tab Bar Icon', 'buddyboss-app' ),
			'folder'                => _x( 'Folder', 'Tab Bar Icon', 'buddyboss-app' ),
			'folders'               => _x( 'Folders', 'Tab Bar Icon', 'buddyboss-app' ),
			'graduation-cap'        => _x( 'Graduation Cap', 'Tab Bar Icon', 'buddyboss-app' ),
			'bar-chart'             => _x( 'Bar Chart', 'Tab Bar Icon', 'buddyboss-app' ),
			'hashtag'               => _x( 'Hashtag', 'Tab Bar Icon', 'buddyboss-app' ),
			'health'                => _x( 'Health', 'Tab Bar Icon', 'buddyboss-app' ),
			'list-square'           => _x( 'List Square', 'Tab Bar Icon', 'buddyboss-app' ),
			'map'                   => _x( 'Map', 'Tab Bar Icon', 'buddyboss-app' ),
			'map-marker'            => _x( 'Map Marker', 'Tab Bar Icon', 'buddyboss-app' ),
			'chat'                  => _x( 'Chat', 'Tab Bar Icon', 'buddyboss-app' ),
			'audio'                 => _x( 'Audio', 'Tab Bar Icon', 'buddyboss-app' ),
			'photo-edit'            => _x( 'Photo Edit', 'Tab Bar Icon', 'buddyboss-app' ),
			'pinterest'             => _x( 'Pinterest', 'Tab Bar Icon', 'buddyboss-app' ),
			'playlist'              => _x( 'Playlist', 'Tab Bar Icon', 'buddyboss-app' ),
			'lock'                  => _x( 'Lock', 'Tab Bar Icon', 'buddyboss-app' ),
			'heart'                 => _x( 'Heart', 'Tab Bar Icon', 'buddyboss-app' ),
			'reward'                => _x( 'Reward', 'Tab Bar Icon', 'buddyboss-app' ),
			'videos'                => _x( 'Videos', 'Tab Bar Icon', 'buddyboss-app' ),
			'airbnb'                => _x( 'Airbnb', 'Tab Bar Icon', 'buddyboss-app' ),
			'bookmark'              => _x( 'Bookmark', 'Tab Bar Icon', 'buddyboss-app' ),
			'browser'               => _x( 'Browser', 'Tab Bar Icon', 'buddyboss-app' ),
			'build'                 => _x( 'Build', 'Tab Bar Icon', 'buddyboss-app' ),
			'business'              => _x( 'Business', 'Tab Bar Icon', 'buddyboss-app' ),
			'calendar'              => _x( 'Calendar', 'Tab Bar Icon', 'buddyboss-app' ),
			'camera'                => _x( 'Camera', 'Tab Bar Icon', 'buddyboss-app' ),
			'pie-chart'             => _x( 'Pie Chart', 'Tab Bar Icon', 'buddyboss-app' ),
			'call'                  => _x( 'Call', 'Tab Bar Icon', 'buddyboss-app' ),
			'clock'                 => _x( 'Clock', 'Tab Bar Icon', 'buddyboss-app' ),
			'content'               => _x( 'Content', 'Tab Bar Icon', 'buddyboss-app' ),
			'credit-card'           => _x( 'Credit Card', 'Tab Bar Icon', 'buddyboss-app' ),
			'dashboard'             => _x( 'Dashboard', 'Tab Bar Icon', 'buddyboss-app' ),
			'dribbble'              => _x( 'Dribbble', 'Tab Bar Icon', 'buddyboss-app' ),
			'dropbox'               => _x( 'Dropbox', 'Tab Bar Icon', 'buddyboss-app' ),
			'facebook'              => _x( 'Facebook', 'Tab Bar Icon', 'buddyboss-app' ),
			'google-drive'          => _x( 'Google Drive', 'Tab Bar Icon', 'buddyboss-app' ),
			'instagram'             => _x( 'Instagram', 'Tab Bar Icon', 'buddyboss-app' ),
			'briefcase'             => _x( 'Briefcase', 'Tab Bar Icon', 'buddyboss-app' ),
			'list'                  => _x( 'List', 'Tab Bar Icon', 'buddyboss-app' ),
			'medium'                => _x( 'Medium', 'Tab Bar Icon', 'buddyboss-app' ),
			'menu'                  => _x( 'Menu', 'Tab Bar Icon', 'buddyboss-app' ),
			'paypal'                => _x( 'PayPal', 'Tab Bar Icon', 'buddyboss-app' ),
			'slack'                 => _x( 'Slack', 'Tab Bar Icon', 'buddyboss-app' ),
			'soundcloud'            => _x( 'SoundCloud', 'Tab Bar Icon', 'buddyboss-app' ),
			'spotify'               => _x( 'Spotify', 'Tab Bar Icon', 'buddyboss-app' ),
			'telegram'              => _x( 'Telegram', 'Tab Bar Icon', 'buddyboss-app' ),
			'tumblr'                => _x( 'Tumblr', 'Tab Bar Icon', 'buddyboss-app' ),
			'twitter'               => _x( 'Twitter', 'Tab Bar Icon', 'buddyboss-app' ),
			'vk'                    => _x( 'VK', 'Tab Bar Icon', 'buddyboss-app' ),
			'web-page'              => _x( 'Web Page', 'Tab Bar Icon', 'buddyboss-app' ),
			'whatsapp'              => _x( 'WhatsApp', 'Tab Bar Icon', 'buddyboss-app' ),
			'youtube'               => _x( 'YouTube', 'Tab Bar Icon', 'buddyboss-app' ),
			'android'               => _x( 'Android', 'Tab Bar Icon', 'buddyboss-app' ),
			'apple'                 => _x( 'Apple', 'Tab Bar Icon', 'buddyboss-app' ),
			'apple-watch'           => _x( 'Apple Watch', 'Tab Bar Icon', 'buddyboss-app' ),
			'award-rating'          => _x( 'Award Rating', 'Tab Bar Icon', 'buddyboss-app' ),
			'shopping-bag'          => _x( 'Shopping Bag', 'Tab Bar Icon', 'buddyboss-app' ),
			'bar-graph'             => _x( 'Bar Graph', 'Tab Bar Icon', 'buddyboss-app' ),
			'download'              => _x( 'Download', 'Tab Bar Icon', 'buddyboss-app' ),
			'fingerprint'           => _x( 'Fingerprint', 'Tab Bar Icon', 'buddyboss-app' ),
			'github'                => _x( 'Github', 'Tab Bar Icon', 'buddyboss-app' ),
			'google-chrome'         => _x( 'Google Chrome', 'Tab Bar Icon', 'buddyboss-app' ),
			'google'                => _x( 'Google', 'Tab Bar Icon', 'buddyboss-app' ),
			'linkedin'              => _x( 'Linkedin', 'Tab Bar Icon', 'buddyboss-app' ),
			'skype'                 => _x( 'Skype', 'Tab Bar Icon', 'buddyboss-app' ),
			'snapchat'              => _x( 'SnapChat', 'Tab Bar Icon', 'buddyboss-app' ),
			'about'                 => _x( 'About', 'Tab Bar Icon', 'buddyboss-app' ),
			'email-preferences'     => _x( 'Email Preferences', 'Tab Bar Icon', 'buddyboss-app' ),
			'export-data'           => _x( 'Export Data', 'Tab Bar Icon', 'buddyboss-app' ),
			'feedback'              => _x( 'Feedback', 'Tab Bar Icon', 'buddyboss-app' ),
			'privacy-settings'      => _x( 'Privacy Settings', 'Tab Bar Icon', 'buddyboss-app' ),
			'profile-connections'   => _x( 'Profile Connections', 'Tab Bar Icon', 'buddyboss-app' ),
			'profile-email-invites' => _x( 'Invites', 'Tab Bar Icon', 'buddyboss-app' ),
			'profile-groups'        => _x( 'Profile Groups', 'Tab Bar Icon', 'buddyboss-app' ),
			'profile-timeline'      => _x( 'Profile Activity', 'Tab Bar Icon', 'buddyboss-app' ),
			'push-notifications'    => _x( 'Push notifications', 'Tab Bar Icon', 'buddyboss-app' ),
		);

		asort( $bpapp_icon_list );

		return array(
			'bbapp' => $bpapp_icon_list,
		);
	}

	/**
	 * Returns the directory of custom icons.
	 *
	 * @return string
	 */
	public function icon_picker_custom_icon_dir() {
		$upload_dir = wp_upload_dir();
		$dir        = $upload_dir['basedir'] . '/bbapp/custom-icons';
		if ( ! file_exists( $dir ) ) {
			BBAPP_File::create_dir( dirname( $dir ) );
			BBAPP_File::create_dir( $dir );
		}

		return $dir;
	}

	/**
	 * Returns the url of custom icons directory.
	 *
	 * @return string
	 */
	public function icon_picker_custom_icon_url() {
		$upload_dir = wp_upload_dir();
		$dir        = $upload_dir['baseurl'] . '/bbapp/custom-icons';

		return $dir;
	}

	/**
	 * Return the list of custom icons uploaded.
	 *
	 * @return array
	 */
	public function icon_picker_custom_icon_list() {
		$dir   = self::icon_picker_custom_icon_dir();
		$icons = glob( $dir . '/*.png' );
		$list  = array();

		if ( ! empty( $icons ) ) {
			foreach ( $icons as $icon ) {
				$icon   = basename( $icon, '.png' );
				$list[] = "custom/{$icon}"; // /custom is a identifier.
			}
		}

		return $list;
	}

	/**
	 * Render category.
	 *
	 * @since 1.7.0
	 */
	public function render_category_dropdown() {
		$font_map = $this->bb_icon_font_map( 'groups' );
		?>
		<select class="bbapp-icon-category-filter-select">
			<option value="all"><?php esc_html_e( 'All Categories', 'buddyboss-app' ); ?></option>
			<?php
			if ( ! empty( $font_map ) ) {
				foreach ( $font_map as $font_map_cat ) {
					printf( '<option value="%1$s">%2$s</option>', esc_attr( $font_map_cat['id'] ), esc_html( $font_map_cat['label'] ) );
				}
			}
			?>
		</select>
		<?php
	}

	/**
	 * Fetch bb icons data.
	 *
	 * @param string $key Array key.
	 *
	 * @since 1.7.3
	 *
	 * @return array
	 */
	public function bb_icon_font_map( $key = '' ) {
		global $bbapp_icons;
		include bbapp()->plugin_dir . 'assets/icons/bb-icons/font-map.php';

		return ! empty( $key ) ? ( isset( $bbapp_icons[ $key ] ) ? $bbapp_icons[ $key ] : false ) : $bbapp_icons;
	}

	/**
	 * Prepare menu item icon argument.
	 *
	 * @param array $menu_item Menu items.
	 *
	 * @since 1.7.0
	 *
	 * @return array
	 */
	public function prepare_menu_icon_argument( $menu_item ) {

		$menu_icon_url = '';
		$icon_id       = ! empty( $menu_item['icon']['id'] ) ? $menu_item['icon']['id'] : 'file';
		$type          = ! empty( $menu_item['icon']['type'] ) ? $menu_item['icon']['type'] : 'legacy';
		$style         = ! empty( $menu_item['icon']['style'] ) ? $menu_item['icon']['style'] : 'lined';
		$color         = ! empty( $menu_item['icon']['color'] ) ? $menu_item['icon']['color'] : 'default';
		$fg_color      = ! empty( $menu_item['icon']['fg_color'] ) ? $menu_item['icon']['fg_color'] : 'transparent';
		$fill_color    = isset( $menu_item['icon']['fill_color'] ) ? $menu_item['icon']['fill_color'] : true;
		$box_style     = ! empty( $menu_item['icon']['box_style'] ) ? $menu_item['icon']['box_style'] : '';

		if ( 'custom' === $type && false === $fill_color ) {
			$color = '';
		}

		if ( 'custom' !== $type ) {
			$fill_color = true;
		}

		$is_default_icon = false;
		if ( 'legacy' === $type ) {
			$menu_icon_url = bbapp_tabbar_get_selected_icon( "bbapp/{$icon_id}", $style );
		} elseif ( 'custom' === $type ) {
			$menu_icon_url = bbapp_tabbar_get_selected_icon( "custom/{$icon_id}.png" );
			if ( false !== strpos( $menu_icon_url, 'bbapp/file' ) ) {
				$is_default_icon = true;
			}
		} elseif ( 'buddyboss' === $type ) {
			$menu_icon_url = self::bbapp_get_icon_url( $icon_id, $style, $box_style );
		}

		if ( 'filled' === $style && ( 'box' === $box_style || 'round' === $box_style ) && empty( $fg_color ) ) {
			$fg_color = 'transparent';
		}

		if ( true === $is_default_icon ) {
			return array(
				'id'         => 'file',
				'type'       => 'buddyboss',
				'style'      => 'lined',
				'box_style'  => '',
				'color'      => 'default',
				'fg_color'   => 'transparent',
				'fill_color' => true,
				'extra'      => array(
					'uniq_id' => $menu_item['id'],
					'img_url' => self::bbapp_get_icon_url( 'file', 'lined', '' ),
				),
			);
		}

		return array(
			'id'         => $icon_id,
			'type'       => $type,
			'style'      => $style,
			'box_style'  => $box_style,
			'color'      => $color,
			'fg_color'   => $fg_color,
			'fill_color' => $fill_color,
			'extra'      => array(
				'uniq_id' => $menu_item['id'],
				'img_url' => $menu_icon_url,
			),
		);
	}

	/**
	 * Get icon for menu api.
	 *
	 * @param array  $menu        Menu array.
	 * @param string $menu_screen Tabbar or More.
	 *
	 * @since 1.7.0
	 *
	 * @return array
	 */
	public function get_icon_for_api( $menu, $menu_screen, $include_legacy_settings = false ) {
		global $post;

		$default_color = ! empty( $post ) ? bbapp_get_default_color( 'primaryColor' ) : bbapp_get_default_color( 'bottomTabsColor' );
		$menu_icon     = ( ! isset( $menu['icon'] ) ) ? array() : $menu['icon'];
		$color         = ! empty( $menu_icon['color'] ) ? $menu_icon['color'] : 'default';
		$color         = ( 'default' !== $color ) ? $color : $default_color;
		$fg_color      = ! empty( $menu_icon['fg_color'] ) ? $menu_icon['fg_color'] : '#00000000';
		$icon_id       = ! empty( $menu_icon['id'] ) ? $menu_icon['id'] : 'file';
		$uri_active    = null;
		$style         = ! empty( $menu_icon['style'] ) ? $menu_icon['style'] : 'lined';
		$fill_color    = isset( $menu_icon['fill_color'] ) ? $menu_icon['fill_color'] : true;
		$icon_style    = 'outlined';
		$box_style     = ! empty( $menu_icon['box_style'] ) ? $menu_icon['box_style'] : 'none';
		$type          = ! empty( $menu_icon['type'] ) ? $menu_icon['type'] : 'legacy';

		if ( 'lined' !== $style ) {
			$icon_style = $menu_icon['style'];
		}

		if ( 'custom' === $type ) {
			$style = '';
			if ( false === $fill_color ) {
				$color = '';
			}
		}

		$menu_icon_url   = '';
		$is_default_icon = false;
		if ( 'legacy' === $type ) {
			$menu_icon_url = bbapp_tabbar_get_selected_icon( "bbapp/{$icon_id}", $icon_style );
		} elseif ( 'custom' === $type ) {
			$menu_icon_url = bbapp_tabbar_get_selected_icon( "custom/{$icon_id}.png" );
			if ( false !== strpos( $menu_icon_url, 'bbapp/file' ) ) {
				$is_default_icon = true;
			}
		} elseif ( 'buddyboss' === $type ) {
			$menu_icon_url = self::bbapp_get_icon_url( $icon_id, $style, $box_style );
			if ( 'lined' === $style ) {
				$uri_active = self::bbapp_get_icon_url( $icon_id, 'filled', $box_style );
			}
		}

		// find token on header.
		$header = \BuddyBossApp\Auth\Common::instance()->getallheaders();
		if ( isset( $header['Appcodeversion'] ) && $header['Appcodeversion'] < '1.9.0' ) {
			if ( 'buddyboss' === $type ) {
				$menu_icon_url = $this->bbapp_get_backward_compatibility_icon( $icon_id, $style, $box_style );
			}
		}

		// If icon is filled and box style is box or round and fg color is empty or transparent then set transparent color code.
		if ( 'filled' === $style && ( 'box' === $box_style || 'round' === $box_style ) ) {
			if ( empty( $fg_color ) || 'transparent' === $fg_color ) {
				$fg_color = '#00000000'; // transparent color code.
			}
		} else {
			$fg_color = '';
		}

		// if default icon then set default icon object.
		if ( true === $is_default_icon ) {
			$icon_obj = array(
				'id'         => 'file',
				'uri'        => self::bbapp_get_icon_url( 'file', 'lined', '' ),
				'uri_active' => null,
				'type'       => 'buddyboss',
				'style'      => 'lined',
				'box_style'  => 'none',
				'color'      => $default_color,
				'fg_color'   => '',
				'fill_color' => true,
			);

			if ( true === $include_legacy_settings ) {
				$icon_obj['tint_color'] = $default_color; // @deprecated.
				$icon_obj['monochrome'] = true; // @deprecated.
				$icon_obj['icon_style'] = 'outlined'; // @deprecated.
			}

			return $icon_obj;
		}

		$icon_obj = array(
			'id'         => $icon_id,
			'uri'        => $menu_icon_url,
			'uri_active' => $uri_active,
			'type'       => $type,
			'style'      => $style,
			'box_style'  => $box_style,
			'color'      => $color,
			'fg_color'   => $fg_color,
			'fill_color' => $fill_color,
		);

		if ( true === $include_legacy_settings ) {
			$icon_obj['tint_color'] = $color; // @deprecated.
			$icon_obj['monochrome'] = $fill_color; // @deprecated.
			$icon_obj['icon_style'] = $icon_style; // @deprecated.
		}

		return $icon_obj;
	}

	/**
	 * Default monochrome setting.
	 *
	 * @since 1.5.2.1
	 * @return string|array
	 */
	public function default_monochrome_setting() {
		return wp_json_encode(
			array(
				'icon_monochrome_checkbox' => 'yes',
				'monochrome_option'        => 'default',
				'icon_monochrome_color'    => bbapp_get_default_color( 'bottomTabsColor' ),
			)
		);
	}

	/**
	 * Get icon url.
	 *
	 * @param string $icon_id   Icon id.
	 * @param string $style     style of icon.
	 * @param string $box_style box style of icon.
	 *
	 * @since 1.7.0
	 *
	 * @return string
	 */
	public function bbapp_get_icon_url( $icon_id, $style, $box_style = '' ) {
		$bb_icons_path = self::icon_picker_bb_icon_url();
		$bb_icons_url  = self::icon_picker_bb_icon_url();

		$path      = ! empty( $box_style ) && 'none' !== $box_style ? "{$box_style}-{$style}" : $style;
		$icon_path = "{$bb_icons_path}svg/{$path}/{$icon_id}.svg";
		if ( ! file_exists( $icon_path ) ) {
			$icon_url = "{$bb_icons_url}svg/{$path}/{$icon_id}.svg";
		} else {
			$icon_url = "{$bb_icons_url}svg/{$path}/file.svg";
		}

		return $icon_url;
	}

	/**
	 * Icon backward compatibility for app core version  equal or less 1.9.0.
	 *
	 * @param string $icon_id   Icon id.
	 * @param string $style     style of icon.
	 * @param string $box_style box style of icon.
	 *
	 * @since 1.7.1
	 *
	 * @return mixed
	 */
	public function bbapp_get_backward_compatibility_icon( $icon_id, $style, $box_style ) {

		$path               = ! empty( $box_style ) && 'none' !== $box_style ? "{$box_style}-{$style}" : $style;
		$file_dir_path      = "/bb-icons/img/{$path}/";
		$icon_name          = "{$icon_id}.png";
		$upload             = wp_upload_dir();
		$file_location_path = $this->bbapp_backward_compatibility_icon_dir( $file_dir_path, $upload );
		$file_location_url  = $this->bbapp_backward_compatibility_icon_url( $file_dir_path, $upload );

		if ( file_exists( $file_location_path . $icon_name ) ) {
			$bb_icons_url = $file_location_url . $icon_name;
		} else {
			$assets_url = 'https://buddyboss-app-files.s3.amazonaws.com/assets/icons';
			$do_request = bbapp_remote_get( "{$assets_url}{$file_dir_path}{$icon_name}" );
			if ( is_wp_error( $do_request ) || empty( $do_request ) ) {
				$bb_icons_url = "https://buddyboss-app-files.s3.amazonaws.com/assets/icons/{$file_dir_path}/{$icon_name}";
			} else {
				$body = wp_remote_retrieve_body( $do_request );
				if ( ! empty( $body ) && 200 === wp_remote_retrieve_response_code( $do_request ) ) {
					$file_name     = str_replace( 'inline; filename=', '', $icon_name );
					$file_uploaded = BBAPP_File::file_handler( $file_location_path . $file_name, $body );
					if ( ! empty( $file_uploaded ) || ! is_wp_error( $file_uploaded ) ) {
						$bb_icons_url = $file_location_url . $file_name;
					} else {
						$bb_icons_url = "https://buddyboss-app-files.s3.amazonaws.com/assets/icons/{$file_dir_path}/{$icon_name}";
					}
				} else {
					$bb_icons_url = "https://buddyboss-app-files.s3.amazonaws.com/assets/icons/{$file_dir_path}/{$icon_name}";
				}
			}
		}

		return $bb_icons_url;
	}

	/**
	 * Returns the directory of backward compatibility icons.
	 *
	 * @param string $path       Dir path.
	 * @param array  $upload_dir wp_upload_dir.
	 *
	 * @since 1.7.1
	 *
	 * @return string
	 */
	public function bbapp_backward_compatibility_icon_dir( $path = '', $upload_dir = '' ) {
		if ( empty( $upload_dir ) ) {
			$upload_dir = wp_upload_dir();
		}
		$dir = $upload_dir['basedir'] . '/bbapp/backward-compatibility-icons';
		if ( ! file_exists( $dir . $path ) ) {
			BBAPP_File::create_dir( dirname( $dir . $path ) );
			BBAPP_File::create_dir( $dir . $path );
		}

		return $dir . $path;
	}

	/**
	 * Returns the url of custom icons directory.
	 *
	 * @param string $path       Dir path.
	 * @param array  $upload_dir wp_upload_dir.
	 *
	 * @since 1.7.1
	 *
	 * @return string
	 */
	public function bbapp_backward_compatibility_icon_url( $path, $upload_dir = '' ) {
		if ( empty( $upload_dir ) ) {
			$upload_dir = wp_upload_dir();
		}
		$dir = $upload_dir['baseurl'] . '/bbapp/backward-compatibility-icons';

		return $dir . $path;
	}
}
