<?php
/**
 * Holds client common functionality.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

use BuddyBossApp\Admin\Settings;
use WP_REST_Request;
use WP_REST_Response;

/**
 * Client common class.
 *
 * @todo - Methods and vars needs to have PSR-4 standards. By Ketan, May-2019
 * @FYI - old file name was : class.bbapp_common.php
 */
class ClientCommon {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * ClientCommon constructor.
	 */
	public function __construct() {
		$this->hooks();
	}

	/**
	 * Get the single instance of class
	 *
	 * @return ClientCommon|null
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Filters/hooks here.
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'rest_api_init' ) );

		// Param support @look https://github.com/WP-API/rest-filter/.
		add_action( 'rest_api_init', array( $this, 'rest_api_filter_add_filters' ) );
		add_filter( 'init', array( $this, 'hide_admin_bar' ), 99 );
		add_filter( 'body_class', array( $this, 'custom_body_classes' ) );

		/**
		 * WordPress Default show only user who contribute into rest enable posttype
		 * User rest endpoint `rest_user_cannot_view` error fixed
		 * */
		add_filter( 'rest_request_before_callbacks', array( $this, 'rest_request_before_callbacks' ), 10, 3 );
		add_filter( 'rest_request_after_callbacks', array( $this, 'rest_request_after_callbacks' ), 10, 3 );
	}

	/**
	 * Rest API init.
	 */
	public function rest_api_init() {
		add_filter( 'get_avatar_data', array( $this, 'get_avatar_data_fix_url_scheme' ), 99 );
	}

	/**
	 * Merge default app settings into db of app settings when.
	 * 1. App Added
	 * 2. Plugin Updated
	 */
	public function merge_default_app_settings() {
		$bbapp_settings = ManageApp::instance()->get_app_settings();

		// Make sure settings key is there.
		if ( ! is_array( $bbapp_settings ) ) {
			$bbapp_settings = array();
		}

		foreach ( bbapp()->app_settings_defaults() as $key => $value ) {
			if ( ! isset( $bbapp_settings[ $key ] ) ) {
				$bbapp_settings[ $key ] = $value;
			}
		}

		ManageApp::instance()->update_app_settings( $bbapp_settings );
	}

	/**
	 * Merges the auth settings default into db if they not exists.
	 */
	public function merge_default_auth_settings() {
		$bbapp_settings = ManageApp::instance()->get_app_settings();

		foreach ( bbapp()->bbapp_default_settings() as $key => $value ) {
			if ( ! isset( $bbapp_settings[ $key ] ) ) {
				$bbapp_settings[ $key ] = $value;
			}
		}

		ManageApp::instance()->update_app_settings( $bbapp_settings );
	}

	/**
	 * Fix gravatar URL scheme issues over API.
	 *
	 * @param array $args Arguments.
	 *
	 * @return mixed
	 */
	public function get_avatar_data_fix_url_scheme( $args ) {
		$args['url'] = bbapp_fix_url_scheme( $args['url'] );

		return $args;
	}

	/**
	 * Hide WP admin bar when ?bbapp_nobar is set.
	 * Used in Webview Content on App.
	 */
	public function hide_admin_bar() {
		if ( is_admin() ) {
			return false;
		}

		if ( isset( $_GET['bbapp_noabar'] ) || isset( $_COOKIE['bbapp_noabar'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( ! isset( $_COOKIE['bbapp_noabar'] ) ) {
				/**
				 * Set as session cookies so it'll be deleted when browser closed
				 */
				setcookie( 'bbapp_noabar', '1', 0 );
			}

			show_admin_bar( false );
		}
	}

	/**
	 * Add Body call in template if page loaded form mobile app.
	 *
	 * @param array $classes Body Classes.
	 *
	 * @return mixed
	 */
	public function custom_body_classes( $classes ) {
		if ( bbapp_is_loaded_from_inapp_browser() || isset( $_GET['mobile-view-content'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$classes[] = 'in-bbapp';
		}

		return $classes;
	}

	/**
	 * If the site is loaded from in-app browser.
	 *
	 * @deprecated don't use this function direct anywhere.
	 *
	 * @return bool
	 */
	public function is_site_loaded_from_inapp_browser() {
		return bbapp_is_loaded_from_inapp_browser();
	}

	/**
	 * Return the pagemode type from header.
	 *
	 * @param bool $set_cookie If needs to set cookie.
	 *
	 * @return bool|mixed|string
	 */
	public function get_page_mode( $set_cookie = false ) {
		/**
		 * App sends pagemode info using header.
		 */
		$template_type = $this->capture_header( 'pagemode' );

		/**
		 * App also sends pagemode info in header so consider it if header one is not present.
		 */
		if ( empty( $template_type ) ) {
			if ( ! empty( $_SERVER['HTTP_USER_AGENT'] ) ) {
				// Capture it from useragent.
				preg_match( '/pagemode\/(.*)/', sanitize_text_field( $_SERVER['HTTP_USER_AGENT'] ), $useragent_output ); //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
				if ( isset( $useragent_output[1] ) ) {
					$template_type = trim( $useragent_output[1] );
				}
			}
		}

		if ( ! empty( $template_type ) && ( ! isset( $_COOKIE['bbapp_page_mode'] ) || $template_type !== $_COOKIE['bbapp_page_mode'] ) ) {
			/**
			 * Set as session cookies so it'll be deleted when browser closed
			 */
			if ( $set_cookie ) {
				$secure = is_ssl();
				setcookie( 'bbapp_page_mode', $template_type, 0, COOKIEPATH, COOKIE_DOMAIN, $secure, true );
			}
		}

		if ( empty( $template_type ) && ! empty( $_COOKIE['bbapp_page_mode'] ) ) {
			$template_type = sanitize_text_field( wp_unslash( $_COOKIE['bbapp_page_mode'] ) );
		}

		return $template_type;
	}

	/**
	 * Returns the center api url.
	 *
	 * @param string $version Version.
	 * @param string $request Request type.
	 *
	 * @return string
	 */
	public function get_center_api_url( $version = 'v1', $request = 'api-get' ) {
		$api = 'https://appcenter.buddyboss.com';

		if ( defined( 'bbapp_center_api_url' ) && ! filter_var( bbapp_center_api_url, FILTER_VALIDATE_URL ) === false ) {
			$api = bbapp_center_api_url;
		}

		$api = trailingslashit( $api ) . "wp-json/center/{$version}/{$request}";

		return $api;
	}

	/**
	 * Add the necessary filter to each post type
	 *
	 * @credit https://github.com/WP-API/rest-filter/
	 **/
	public function rest_api_filter_add_filters() {
		foreach ( get_post_types( array( 'show_in_rest' => true ), 'objects' ) as $post_type ) {
			add_filter(
				'rest_' . $post_type->name . '_query',
				array(
					$this,
					'rest_api_filter_add_filter_param',
				),
				10,
				2
			);
		}
	}

	/**
	 * Add the filter parameter
	 *
	 * @credit https://github.com/WP-API/rest-filter/
	 *
	 * @param array           $args    The query arguments.
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return array $args.
	 **/
	public function rest_api_filter_add_filter_param( $args, $request ) {
		// Bail out if no filter parameter is set.
		if ( empty( $request['filter'] ) || ! is_array( $request['filter'] ) ) {
			return $args;
		}

		$filter = $request['filter'];

		if ( isset( $filter['posts_per_page'] ) && ( (int) $filter['posts_per_page'] >= 1 && (int) $filter['posts_per_page'] <= 100 ) ) {
			$args['post_per_page'] = $filter['posts_per_page'];
		}

		global $wp;

		$vars = apply_filters( 'query_vars', $wp->public_query_vars );

		foreach ( $vars as $var ) {
			if ( isset( $filter[ $var ] ) ) {
				$args[ $var ] = $filter[ $var ];
			}
		}

		return $args;
	}

	/**
	 * Function helps to capture header information
	 *
	 * @param string $header_name Case insensitive.
	 *
	 * @return bool|mixed
	 */
	public function capture_header( $header_name ) {
		$headers = $this->getallheaders();
		$return  = false;

		foreach ( $headers as $name => $value ) {
			$name = strtolower( $name );

			if ( strtolower( $name ) === strtolower( $header_name ) ) {
				$return = $value;
				break;
			}
		}

		return $return;
	}

	/**
	 * Returns all headers available on incoming http request.
	 *
	 * @return array
	 */
	public function getallheaders() {
		if ( function_exists( 'getallheaders' ) ) {
			return getallheaders();
		}

		if ( ! is_array( $_SERVER ) ) {
			return array();
		}

		$headers = array();

		foreach ( $_SERVER as $name => $value ) {
			if ( 'HTTP_' === substr( $name, 0, 5 ) ) {
				$headers[ str_replace( ' ', '-', ucwords( strtolower( str_replace( '_', ' ', substr( $name, 5 ) ) ) ) ) ] = $value;
			}
		}

		return $headers;
	}

	/**
	 * Retuns true false depending on checklist
	 *
	 * @param array $checklist Check list.
	 *
	 * @return bool
	 */
	public function is_active_logic( $checklist ) {
		// Will Keep All Checks in is_active.
		$is_active = array();
		// BuddyPress.
		$is_active['buddypress'] = class_exists( '\BuddyPress' );
		// bbPress.
		$is_active['bbpress'] = function_exists( 'bbpress' );
		// BuddyBossApp BuddyPress API.
		$is_active['bbapp_buddypress_api'] = class_exists( '\BuddyBossApp\Api\BuddyPress\Main' );
		// BuddyBossApp bbPress API.
		$is_active['bbapp_bbpress_api'] = class_exists( '\BuddyBossApp\Api\BbPress\Main' );
		// BuddyBossApp Media API.
		$is_active['bbapp_bb_media_api'] = class_exists( 'BuddyBossApp\Integrations\BuddyBossMedia' );
		// BuddyBossApp Auth API.
		$is_active['bbapp_auth_api'] = function_exists( 'bbapp_auth' );
		// BuddyPress Groups (Social Groups).
		$is_active['buddypress_groups'] = ( $is_active['buddypress'] && bp_is_active( 'groups' ) );
		// BuddyPress Activity (Activity Feeds).
		$is_active['buddypress_activity'] = ( $is_active['buddypress'] && bp_is_active( 'activity' ) );
		// BuddyPress Message (Message).
		$is_active['buddypress_messages'] = ( $is_active['buddypress'] && bp_is_active( 'messages' ) );
		// BuddyPress Members (Members).
		$is_active['buddypress_members'] = ( $is_active['buddypress'] && bp_is_active( 'members' ) );
		// BuddyPress XProfile (Profile).
		$is_active['buddypress_profile'] = ( $is_active['buddypress'] && bp_is_active( 'xprofile' ) );
		// BuddyPress Documents.
		$bp_is_group_document_support_enabled   = ( function_exists( 'bp_is_group_document_support_enabled' ) && bp_is_group_document_support_enabled() );
		$bp_is_profile_document_support_enabled = ( function_exists( 'bp_is_profile_document_support_enabled' ) && bp_is_profile_document_support_enabled() );
		$bp_is_forums_document_support_enabled  = ( function_exists( 'bp_is_forums_document_support_enabled' ) && bp_is_forums_document_support_enabled() );
		$is_active['buddypress_documents']      = ( $is_active['buddypress'] && bp_is_active( 'media' ) && bp_is_active( 'document' ) && ( $bp_is_group_document_support_enabled || $bp_is_profile_document_support_enabled || $bp_is_forums_document_support_enabled ) );
		// BuddyPress Videos.
		$bp_is_group_video_support_enabled   = ( function_exists( 'bp_is_group_video_support_enabled' ) && bp_is_group_video_support_enabled() );
		$bp_is_profile_video_support_enabled = ( function_exists( 'bp_is_profile_video_support_enabled' ) && bp_is_profile_video_support_enabled() );
		$bp_is_forums_video_support_enabled  = ( function_exists( 'bp_is_forums_video_support_enabled' ) && bp_is_forums_video_support_enabled() );
		$is_active['buddypress_videos']      = ( $is_active['buddypress'] && bp_is_active( 'media' ) && bp_is_active( 'video' ) && ( $bp_is_group_video_support_enabled || $bp_is_profile_video_support_enabled || $bp_is_forums_video_support_enabled ) );
		// BuddyPress Photos.
		$is_active['buddypress_photos'] = ( $is_active['buddypress'] && bp_is_active( 'media' ) );
		// BuddyPress Forums & Topics (Forum Discussions).
		$is_active['buddypress_forums'] = ( $is_active['buddypress'] && bp_is_active( 'forums' ) );
		// BuddyBoss platform active or not. If active then check Notification Component enabled or not.
		$is_active['buddypress_notifications'] = bbapp_is_active( 'push_notification' );
		// BuddyBossApp Learner api.
		$is_active['bbapp_learner_api']                = ( bbapp_is_learndash_enabled() ) && class_exists( '\BuddyBossApp\Api\LearnDash\Main' );
		$is_active['bbapp_tutor_lms_api']              = ( bbapp_is_tutor_lms_plugins_active() );
		$settings                                      = Settings::instance()->get_settings( true );
		$is_active['bbapp_learner_course_downloading'] = $is_active['bbapp_learner_api'] && ( isset( $settings['learndash_course_downloading'] ) ? $settings['learndash_course_downloading'] : 0 );
		// Memberpress LMS.
		$is_active['memberpress_lms_api'] = bbapp_is_memberpress_lms_enabled();

		if ( ! is_user_logged_in() && $is_active['bbapp_learner_api'] ) {
			$reader_app_compatibility       = isset( $settings['learndash_reader_app_compatibility'] ) ? $settings['learndash_reader_app_compatibility'] : 0;
			$is_active['bbapp_learner_api'] = ! $reader_app_compatibility;
		}

		// IAP Component enabled not.
		$is_active['bbapp_iap'] = bbapp_is_active( 'iap' );
		$is_active              = apply_filters( 'bbapp_is_active_logic', $is_active );
		$return                 = true;

		foreach ( $checklist as $check ) {
			if ( isset( $is_active[ $check ] ) && empty( $is_active[ $check ] ) ) {
				$return = false;
				break;
			}
		}

		return $return;
	}

	/**
	 * By WordPress Default shows only users publicly who has contribute into post type enabled into rest endpoints.
	 * Blow functions adds users a temporary cap `list_users` to allow them to list all users on rest.
	 * This feature only get enabled while content is being requested from rest api.
	 *
	 * @param WP_REST_Response $response Rest response.
	 * @param array            $handler  API handler.
	 * @param WP_REST_Request  $request  Rest request.
	 *
	 * @return mixed
	 */
	public function rest_request_before_callbacks( $response, $handler, $request ) {
		// Elementor editor endpoint not passing callback array.
		if ( ! is_array( $handler['callback'] ) || ! isset( $handler['callback'][0] ) || ! is_object( $handler['callback'][0] ) ) {
			return $response;
		}

		if ( is_a( $handler['callback'][0], 'WP_REST_Users_Controller' ) && 'get_items' === $handler['callback'][1] ) {
			add_filter( 'rest_user_query', array( $this, 'show_all_users' ) );
		}

		if ( is_a( $handler['callback'][0], 'WP_REST_Users_Controller' ) && 'get_item' === $handler['callback'][1] ) {
			add_filter( 'user_has_cap', array( $this, 'give_permissions_list_users' ), 10, 3 );
		}

		return $response;
	}

	/**
	 * WordPress Default show only user who contribute into rest enable posttype
	 * remove hook to Update `list_users` capability only for rest get user/users request.
	 *
	 * @param WP_REST_Response $response Rest response.
	 * @param array            $handler  API handler.
	 * @param WP_REST_Request  $request  Rest request.
	 *
	 * @return mixed
	 */
	public function rest_request_after_callbacks( $response, $handler, $request ) {
		// Elementor editor endpoint not passing callback array.
		if ( ! is_array( $handler['callback'] ) || ! isset( $handler['callback'][0] ) || ! is_object( $handler['callback'][0] ) ) {
			return $response;
		}

		if ( is_a( $handler['callback'][0], 'WP_REST_Users_Controller' ) && 'get_items' === $handler['callback'][1] ) {
			remove_filter( 'rest_user_query', array( $this, 'show_all_users' ) );
		}

		if ( is_a( $handler['callback'][0], 'WP_REST_Users_Controller' ) && 'get_item' === $handler['callback'][1] ) {
			remove_filter( 'user_has_cap', array( $this, 'give_permissions_list_users' ), 10, 3 );
		}

		return $response;
	}

	/**
	 * WordPress Default show only user who contribute into rest enable posttype
	 * Remove `has_published_posts` for get users endpoint
	 *
	 * @param array $prepared_args Prepare arguments.
	 *
	 * @return mixed
	 */
	public function show_all_users( $prepared_args ) {
		unset( $prepared_args['has_published_posts'] );

		return $prepared_args;
	}

	/**
	 * WordPress Default show only user who contribute into rest enable posttype
	 * Give `list_users` capability only for rest get user request.
	 *
	 * @param array $allcaps All capabilities.
	 * @param array $cap     Capabilities.
	 * @param array $args    Arguments.
	 *
	 * @return mixed
	 */
	public function give_permissions_list_users( $allcaps, $cap, $args ) {
		// Bail out if we're not asking about a post.
		if ( 'list_users' !== $args[0] ) {
			return $allcaps;
		}

		$allcaps[ $cap[0] ] = true;

		return $allcaps;
	}

	/**
	 * Get the language code from header or cookie to use for content
	 *
	 * @since 2.4.10
	 * @return string Language code (e.g. 'en', 'fr', 'hi')
	 */
	public function get_language_code_from_header() {
		// Try to get language code from headers.
		$language_code = null;

		// Check direct server variable first (faster).
		if ( ! empty( $_SERVER['HTTP_APPLANG'] ) ) {
			return sanitize_text_field( wp_unslash( $_SERVER['HTTP_APPLANG'] ) );
		}

		// Check for appLang header (primary).
		$app_lang = $this->capture_header( 'appLang' );
		if ( ! empty( $app_lang ) ) {
			$language_code = $app_lang;
		}

		// If no language found in headers, use site default.
		if ( empty( $language_code ) ) {
			$language_code = bbapp_wp_locale_to_app_locale();
		}

		// Map language codes like 'vn' to 'vi' for proper handling.
		if ( ! empty( $language_code ) ) {
			$language_code = \BuddyBossApp\AppLanguages::instance()->mapped_language_code( $language_code );
		}

		return $language_code;
	}
}
