<?php
/**
 * Holds functionality related build installation.
 *
 * @package BuddyBossApp\Builds
 */

namespace BuddyBossApp\Builds;

use BuddyBossApp\Build;
use BuddyBossApp\ManageApp;

/**
 * Build installer class.
 */
class InstallerScreen {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Build data.
	 *
	 * @var bool $build
	 */
	public $build = array();

	/**
	 * Using Singleton, see instance()
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return InstallerScreen
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Load method
	 */
	public function load() {
		add_action( 'init', array( $this, 'load_installer_screen' ), 99 );
	}

	/**
	 * Renders the app installer screen.
	 */
	public function load_installer_screen() {
		$installer_screen_get = ( ! empty( $_GET['bbapp_installer_screen'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['bbapp_installer_screen'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 1 === (int) $installer_screen_get ) {
			$build_id = $this->get_build_id();

			if ( empty( $build_id ) ) {
				wp_safe_redirect( home_url( '?bbapp-error-installing=1' ) );
				exit;
			}

			$builds       = Build::instance()->get_app_builds( '', '', array( $build_id ) );
			$fullfill_req = true;

			if ( ! isset( $builds['data'][0] ) || ! isset( $builds['data'][0]['env'] ) ) {
				$fullfill_req = false;
			} else {
				$this->build = $builds['data'][0];
			}

			add_theme_support( 'title-tag' );
			add_filter( 'wp_title', array( $this, 'wp_title' ), 99, 2 );
			remove_action( 'wp_head', '_wp_render_title_tag', 1 );
			add_filter( 'wp_head', array( $this, 'wp_render_title_tag' ), 99 );
			add_filter( 'body_class', array( $this, 'body_class' ) );
			show_admin_bar( false );

			$http_user_agent = ! empty( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';
			$ipod            = stripos( $http_user_agent, 'iPod' );
			$iphone          = stripos( $http_user_agent, 'iPhone' );
			$ipad            = stripos( $http_user_agent, 'iPad' );
			$safari          = stripos( $http_user_agent, 'Safari' );
			$android         = stristr( $http_user_agent, 'android' );

			if ( 'ios' === $this->get_app_platform() ) {
				if ( false === $safari || $android ) {
					$fullfill_req = false;
				}
			}

			if ( 'android' === $this->get_app_platform() ) {
				if ( ! $android || ! ( ! $ipod && ! $iphone && ! $ipad ) ) {
					$fullfill_req = false;
				}
			}
			wp_enqueue_style( 'jquery-googleapis-font', 'https://fonts.googleapis.com/css2?family=Roboto:wght@300&display=swap', array(), '1.6.6' );
			if ( ! $fullfill_req ) {
				$this->installer_downfall();
			} else {
				$this->installer_screen();
			}

			exit;
		}
	}

	/**
	 * Return the app icon url
	 */
	public function get_app_icon() {
		$app_icon = \BuddyBossApp\Admin\Appearance\Branding::instance()->get_upload_file_info( 'app_icon_ios.png' );

		if ( ! isset( $app_icon['fileurl'] ) ) {
			$app_icon['fileurl'] = false;
		}

		return $app_icon['fileurl'];
	}

	/**
	 * Function to override body class.
	 *
	 * @param array $classes Array of classes.
	 *
	 * @return mixed
	 */
	public function body_class( $classes ) {
		$classes[] = 'bbapp-app-installation-page';

		return $classes;
	}

	/**
	 * Change the installer screen title.
	 *
	 * @param string $title Override WP title.
	 * @param string $sep   Title separator.
	 *
	 * @return string
	 */
	public function wp_title( $title, $sep ) {
		$title .= get_bloginfo( 'name' );
		/* translators: %1$s: App platform, %2$s: App verion */
		$title_text = sprintf( __( 'Installing %1$s %2$s', 'buddyboss-app' ), $this->get_app_platform(), $this->get_app_version() );
		$title      = "$title $sep $title_text";

		return $title;
	}

	/**
	 * Change the installer screen title.
	 */
	public function wp_render_title_tag() {
		?>
		<title>
			<?php wp_title( '|', true, 'right' ); ?>
		</title>
		<?php
	}

	/**
	 * Installer Screen.
	 */
	public function installer_screen() {
		include bbapp()->plugin_dir . 'views/app-installer/install-screen.php';
		exit;
	}

	/**
	 * Function to get build install failed screen.
	 */
	public function installer_downfall() {
		include bbapp()->plugin_dir . 'views/app-installer/install-downfall.php';
		exit;
	}

	/**
	 * Return app platform
	 *
	 * @return bool
	 */
	public function get_app_platform() {
		return ( isset( $this->build ['type'] ) ) ? $this->build ['type'] : false;
	}

	/**
	 * Return app platform
	 *
	 * @return bool
	 */
	public function get_app_platform_label() {
		$labels = array(
			'android' => __( 'Android', 'buddyboss-app' ),
			'ios'     => __( 'iOS', 'buddyboss-app' ),
		);

		if ( isset( $labels[ $this->get_app_platform() ] ) ) {
			return $labels[ $this->get_app_platform() ];
		} else {
			return $this->get_app_platform();
		}
	}

	/**
	 * Return app platform
	 *
	 * @return bool
	 */
	public function get_build_id() {
		return (int) ( isset( $_GET['build_id'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['build_id'] ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	}

	/**
	 * Return app platform
	 *
	 * @return bool
	 */
	public function get_app_version() {
		return ( isset( $this->build ['build_version'] ) ) ? $this->build ['build_version'] : false;
	}

	/**
	 * Return app platform
	 *
	 * @return bool
	 */
	public function get_app_version_code() {
		if ( $this->build ) {
			return ( isset( $this->build ['build_version_code'] ) ) ? $this->build ['build_version_code'] : false;
		}

		return false;
	}

	/**
	 * Return App NAme.
	 *
	 * @return mixed|string
	 */
	public function get_app_name() {
		$get_app_settings = ManageApp::instance()->get_app_settings( true );

		return $get_app_settings['publish.general.appname'];
	}

	/**
	 * Return the build download URL.
	 */
	public function get_build_download_url() {
		if ( $this->build ) {
			if ( 'ios' === $this->get_app_platform() ) {
				return Installer::instance()->get_ios_plist_link( $this->build['id'] );
			}

			return $this->build['build_url'];
		}

		return false;
	}
}
