<?php
/**
 * Holds build installation functionality.
 *
 * @package BuddyBossApp\Builds
 */

namespace BuddyBossApp\Builds;

use BuddyBossApp\AppStores\Apple;
use BuddyBossApp\Build;
use BuddyBossApp\ClientCommon;
use BuddyBossApp\ManageApp;

/**
 * Build installer class.
 */
class Installer {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Build data.
	 *
	 * @var bool $build
	 */
	public $build = false;

	/**
	 * Using Singleton, see instance()
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return Installer
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Load
	 */
	public function load() {
		InstallerScreen::instance();
		add_action( 'init', array( $this, 'ios_plist_output' ), 99 );
		add_action( 'init', array( $this, 'register_device' ), 99 );
	}

	/**
	 * Register Device
	 */
	public function register_device() {
		if ( isset( $_GET['register_ios_device'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$uuid              = isset( $_GET['uuid'] ) ? sanitize_title( wp_unslash( $_GET['uuid'] ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$device_name       = isset( $_GET['device_name'] ) && ! empty( $_GET['device_name'] ) ? bbapp_input_clean( wp_unslash( $_GET['device_name'] ) ) : 'iPhone'; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$device_owner_name = isset( $_GET['device_owner_name'] ) && ! empty( $_GET['device_owner_name'] ) ? bbapp_input_clean( wp_unslash( $_GET['device_owner_name'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$bbapp_app_id      = isset( $_GET['bbapp_app_id'] ) ? bbapp_input_clean( wp_unslash( $_GET['bbapp_app_id'] ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$token             = isset( $_GET['bbapp_token'] ) ? bbapp_input_clean( wp_unslash( $_GET['bbapp_token'] ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( empty( $uuid ) || empty( $device_name ) || empty( $bbapp_app_id ) ) {
				die( esc_html__( '#1 There was an error while registering your device. Please try again later.', 'buddyboss-app' ) );
			}

			$app = ManageApp::instance()->get_app();

			if ( $app['bbapp_app_id'] !== $bbapp_app_id ) {
				die( esc_html__( '#2 There was an error while registering your device. Please try again later.', 'buddyboss-app' ) );
			}

			// Validate Auth.
			if ( ManageApp::instance()->get_auth_app_key() !== $token ) {
				die( esc_html__( '#3 There was an error while registering your device. Please try again later.', 'buddyboss-app' ) );
			}

			$device_name   = "$device_owner_name ($device_name)";
			$is_registered = Apple::instance()->register_development_device( $device_name, $uuid );

			// Create new provisioning profile.
			if ( $is_registered ) {
				Apple::instance()->update_provisioning_profile( true );
			}

			add_filter(
				'body_class',
				function ( $classes ) {
					$classes[] = 'bbapp-app-device-registered-page';

					return $classes;
				}
			);

			show_admin_bar( false );

			include bbapp()->plugin_dir . 'views/app-installer/device-registered.php';
			exit;
		}
	}

	/**
	 * Get pList Install Link.
	 *
	 * @param int $build_id Build id.
	 *
	 * @return string
	 */
	public function get_ios_plist_link( $build_id ) {

		$actionurl = str_replace( '&amp;', '&', home_url( "?build_id={$build_id}" ) );
		$plist     = esc_html( add_query_arg( 'bbapp_ios_plist', $this->create_nonce( 'bbapp_ios_plist' ), $actionurl ) );
		// make sure download URL is always https. because apple don't allow to install app from insecure connection.
		$plist = str_replace( 'http://', 'https://', $plist );
		// Encode for URL safe.
		$plist = str_replace( 'amp%3B', '', rawurlencode( $plist ) );

		return 'itms-services://?action=download-manifest&url=' . $plist;
	}

	/**
	 * Output the iOS install plist.
	 */
	public function ios_plist_output() {
		$nonce_get = ( ! empty( $_GET['bbapp_ios_plist'] ) ) ? sanitize_text_field( wp_unslash( $_GET['bbapp_ios_plist'] ) ) : '';

		if ( empty( $nonce_get ) ) {
			return false;
		}

		// security check.
		if ( ! $this->verify_nonce( $nonce_get, 'bbapp_ios_plist' ) ) {
			die( esc_html__( 'Installation Expired.', 'buddyboss-app' ) );
		}

		$app      = ManageApp::instance()->get_app();
		$build_id = isset( $_GET['build_id'] ) ? sanitize_text_field( wp_unslash( $_GET['build_id'] ) ) : false;

		if ( empty( $build_id ) ) {
			die( '404' );
		}

		$builds = Build::instance()->get_app_builds( '', '', array( $build_id ) );

		if ( ! isset( $builds['data'][0] ) || ! isset( $builds['data'][0]['env'] ) || 'dev' !== $builds['data'][0]['env'] ) {
			die( '404' );
		}

		$build = $builds['data'][0];

		$settings = ManageApp::instance()->get_app_settings( true );

		$app_icon = \BuddyBossApp\Admin\Appearance\Branding::instance()->get_upload_file_info( 'app_icon_ios.png' );

		$app_name = isset( $settings['publish.general.appname'] ) ? $settings['publish.general.appname'] : $app['bbapp_app_id'];

		$bundle_id = $this->get_build_bundle_id( $build_id );

		if ( false === $bundle_id ) {
			die( '404' );
		}

		if ( ! $build || ! isset( $build['build_url'] ) ) {
			die( '404' );
		}

		header( 'Content-Type:text/xml' );

		ob_start();
		?>
		<plist version="1.0">
			<dict>
				<key>items</key>
				<array>
					<dict>
						<key>assets</key>
						<array>
							<dict>
								<key>kind</key>
								<string>software-package</string>
								<key>url</key>
								<string><?php echo esc_html( $build['build_url'] ); ?></string>
							</dict>
							<dict>
								<key>kind</key>
								<string>display-image</string>
								<key>needs-shine</key>
								<false/>
								<key>url</key>
								<string><?php echo esc_html( $app_icon['fileurl'] ); ?></string>
							</dict>
						</array>
						<key>metadata</key>
						<dict>
							<key>bundle-identifier</key>
							<string><?php echo esc_html( $bundle_id ); ?></string>
							<key>bundle-version</key>
							<string><?php echo esc_html( $build['build_version_code'] ); ?></string>
							<key>subtitle</key>
							<string><?php echo esc_html( $build['build_version'] ); ?>
								(<?php echo esc_html( $build['build_version_code'] ); ?>)
							</string>
							<key>title</key>
							<string><?php echo esc_html( $app_name ); ?></string>
							<key>kind</key>
							<string>software</string>
						</dict>
					</dict>
				</array>
			</dict>
		</plist>
		<?php
		$xml = ob_get_contents();
		ob_end_clean();
		$xml                    = str_replace( array( '   ', "\r", "\n" ), '', $xml );
		$allowed_html           = wp_kses_allowed_html( 'post' );
		$allowed_html['plist']  = array( 'version' => array() );
		$allowed_html['dict']   = true;
		$allowed_html['key']    = true;
		$allowed_html['array']  = true;
		$allowed_html['string'] = true;
		$allowed_html['false']  = true;
		echo wp_kses( $xml, $allowed_html );
		exit;
	}

	/**
	 * Get bundle id by build id.
	 *
	 * @param int $build_id Build id.
	 *
	 * @return false|mixed
	 */
	public function get_build_bundle_id( $build_id ) {
		$args['bbapp_id']  = ManageApp::instance()->get_app_id();
		$args['bbapp_key'] = ManageApp::instance()->get_auth_app_key();
		$args['build_id']  = $build_id;
		$api_url           = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/get-build-details' );

		$request = bbapp_remote_get(
			$api_url,
			array(
				'timeout' => 15,
				'body'    => $args,
			)
		);

		$response         = wp_remote_retrieve_body( $request );
		$response         = json_decode( $response, true );
		$response_code    = wp_remote_retrieve_response_code( $request );
		$response_message = wp_remote_retrieve_response_message( $request );

		if (
			'OK' === $response_message
			&& 200 === $response_code
			&& ! empty( $response )
		) {
			if ( isset( $response['id'] ) && $response['id'] === $build_id ) {
				if ( isset( $response['ios'] ) ) {
					return isset( $response['ios']['bundleid'] ) ? $response['ios']['bundleid'] : false;
				}
			}
		}

		return false;
	}

	/**
	 * Get build install link.
	 *
	 * @param int $build_id Build id.
	 *
	 * @return null|string
	 */
	public function get_installer_link( $build_id ) {
		$link = home_url( "?bbapp_installer_screen=1&build_id={$build_id}" );

		return $link;
	}

	/**
	 * Register URL.
	 *
	 * @return string
	 */
	public function get_device_register_url() {
		$bbapp_id = ManageApp::instance()->get_app_id();
		$cc_url   = ClientCommon::instance()->get_center_api_url();
		$site_url = home_url();
		$site_url = str_replace( 'amp%3B', '', rawurlencode( $site_url ) );
		$link     = "{$cc_url}/device-enroll-start?client_url={$site_url}&bbapp_id={$bbapp_id}";

		return $link;
	}


	/**
	 * Creates a cryptographic token tied to a specific action,
	 * and window of time.
	 *
	 * @since 1.7.2
	 *
	 * @param string|int $action Scalar value to add context to the nonce.
	 * @return string The token.
	 */
	public function create_nonce( $action = - 1 ) {
		$salt = 'buddyboss-app';
		$i    = wp_nonce_tick( $action );

		return substr( wp_hash( $i . '|' . $salt . '|' . $action, 'nonce' ), - 12, 10 );
	}

	/**
	 * Verifies that a correct security nonce was used with time limit.
	 *
	 * A nonce is valid for 24 hours (by default).
	 *
	 * @since 1.7.2
	 *
	 * @param string     $nonce  Nonce value that was used for verification, usually via a form field.
	 * @param string|int $action Should give context to what is taking place and be the same when nonce was created.
	 * @return int|false 1 if the nonce is valid and generated between 0-12 hours ago,
	 *                   2 if the nonce is valid and generated between 12-24 hours ago.
	 *                   False if the nonce is invalid.
	 */
	public function verify_nonce( $nonce, $action = - 1 ) {
		$nonce = (string) $nonce;

		if ( empty( $nonce ) ) {
			return false;
		}
		$salt  = 'buddyboss-app';
		$i     = wp_nonce_tick( $action );

		// Nonce generated 0-12 hours ago.
		$expected = substr( wp_hash( $i . '|' . $salt . '|' . $action, 'nonce' ), - 12, 10 );
		if ( hash_equals( $expected, $nonce ) ) {
			return 1;
		}

		// Nonce generated 12-24 hours ago.
		$expected = substr( wp_hash( ( $i - 1 ) . '|' . $salt . '|' . $action, 'nonce' ), - 12, 10 );
		if ( hash_equals( $expected, $nonce ) ) {
			return 2;
		}

		// Invalid nonce.
		return false;
	}

}
