<?php
/**
 * Holds build related functionality.
 *
 * @package BuddyBossApp\Builds
 */

namespace BuddyBossApp;

use BuddyBossApp\Admin\Configure;
use BuddyBossApp\Builds\Installer;
use BuddyBossApp\Helpers\BBAPP_File;
use WP_Error;

/**
 * Build class.
 */
class Build {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Runtime cache for builds to prevent duplicate API calls within same request.
	 *
	 * @var array
	 */
	private static $runtime_builds_cache = array();

	/**
	 * Cache duration for stable builds (30 minutes).
	 *
	 * @var int
	 */
	const BUILDS_CACHE_DURATION = 1800; // 30 * 60

	/**
	 * Cache duration for active/in-progress builds (5 minutes).
	 *
	 * @var int
	 */
	const BUILDS_CACHE_DURATION_SHORT = 300; // 5 * 60

	/**
	 * Get the instance of the class.
	 *
	 * @return Build
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Build constructor.
	 */
	public function __construct() {
	}

	/**
	 * On Class Load.
	 */
	public function load() {
		Installer::instance();
		add_action( 'bbapp_queue_task_check_build_status', array( $this, 'queue_task_check_build_status' ) ); // Background Job.
		add_action( 'wp_ajax_bbapp_build_sync', array( $this, 'bbapp_build_sync' ) );
		add_action( 'wp_ajax_nopriv_bbapp_build_sync', array( $this, 'bbapp_build_sync' ) );
		add_action( 'wp_ajax_update_build_status_now', array( $this, 'bbapp_update_build_status_now' ) );
		add_action( 'wp_ajax_nupdate_build_status_now', array( $this, 'bbapp_update_build_status_now' ) );
		add_action( 'admin_notices', array( $this, 'app_core_version_admin_notice' ) );
	}

	/**
	 * Send Build Request.
	 *
	 * @param string $build_json Build json data.
	 * @param array  $files      Build files.
	 *
	 * @return bool | array
	 */
	public function send_build_request( $build_json, $files ) {
		$app = ManageApp::instance()->get_app();

		if ( empty( $app ) ) {
			return false;
		}

		/**
		 * Filter the build json data.
		 *
		 * @param array $build_json Build json data.
		 *
		 * @since 2.2.70
		 */
		$build_json = apply_filters( 'bbapp_build_request_json', $build_json );

		$api_url     = ClientCommon::instance()->get_center_api_url( 'v2' );
		$post_fields = array(
			'action'    => 'build-request',
			'data'      => $build_json,
			'bbapp_id'  => $app['bbapp_app_id'],
			'bbapp_key' => ManageApp::instance()->get_auth_app_key(),
		);
		// phpcs:disabled
		foreach ( $files as $file_key => $file ) {
			$url_parts                  = wp_parse_url( $file_key );
			$cleaned_url                = $url_parts['scheme'] . '://' . $url_parts['host'] . $url_parts['path'];
			$md5                        = md5( $cleaned_url );
			$post_fields["file_{$md5}"] = new \CURLFILE( $file );
		}

		$result = false;
		$curl   = curl_init();

		curl_setopt_array( $curl, array(
			CURLOPT_URL            => $api_url,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING       => '',
			CURLOPT_MAXREDIRS      => 10,
			CURLOPT_TIMEOUT        => 45,
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST  => 'POST',
			CURLOPT_POSTFIELDS     => $post_fields,
			CURLOPT_HTTPHEADER     => array(),
		) );

		$response = curl_exec( $curl );

		if ( ! curl_errno( $curl ) ) {
			switch ( $http_code = curl_getinfo( $curl, CURLINFO_HTTP_CODE ) ) {
				case 200:
					$result = $response;
					break;
				default:
					$result = new WP_Error( $http_code, curl_error( $curl ) );
			}
		}

		curl_close( $curl );

		if ( is_wp_error( $result ) ) {
			return false;
		}
        // phpcs:enabled
		return json_decode( $result, true );
	}

	/**
	 * Update lst build version.
	 *
	 * @param string  $env   Environment name.
	 * @param boolean $force When force is true it will forcefully check instead of skipping.
	 *
	 * @return array|false|mixed|WP_Error
	 */
	public function update_last_versions( $env = '', $force = false ) {
		$app       = ManageApp::instance()->get_app();
		$cache_key = '_bbapp_fetch_last_versions';

		if ( empty( $app ) ) {
			return array();
		}

		if ( ! $force ) {
			$cached = get_site_transient( $cache_key );

			if ( false !== $cached ) {
				return $cached;
			}
		}

		if ( 'live' === $env ) {
			$bundle_id_ios     = Configure::instance()->option( 'publish.ios.namespace' );
			$bundle_id_android = Configure::instance()->option( 'publish.android.namespace' );
		} else {
			$bundle_id_ios     = Configure::instance()->option( 'publish.ios.dev.namespace' );
			$bundle_id_android = Configure::instance()->option( 'publish.android.dev.namespace' );
		}

		$api_url    = ClientCommon::instance()->get_center_api_url( 'v2' );
		$do_request = bbapp_remote_post(
			$api_url,
			array(
				'method'  => 'POST',
				'timeout' => 15,
				'body'    => array(
					'action'           => 'app-versions',
					'android_bundleid' => $bundle_id_android,
					'ios_bundleid'     => $bundle_id_ios,
					'bbapp_id'         => $app['bbapp_app_id'],
					'bbapp_key'        => ManageApp::instance()->get_auth_app_key(),
				),
			)
		);

		if ( is_wp_error( $do_request ) || empty( $do_request ) ) {
			return false;
		}

		$body     = isset( $do_request['body'] ) ? $do_request['body'] : '';
		$response = (array) json_decode( $body, true );

		if ( ! isset( $response['success'] ) || ! isset( $response['success'] ) ) {
			return false;
		}

		$build_options = $this->get_build_options();

		// change the cast.
		$response['ios']     = (array) $response['ios'];
		$response['android'] = (array) $response['android'];

		// Update the versions info got from API.
		$build_options['app_version_ios']          = $response['ios']['version'];
		$build_options['app_version_android']      = $response['android']['version'];
		$build_options['app_version_code_ios']     = $response['ios']['version_code'];
		$build_options['app_version_code_android'] = $response['android']['version_code'];

		$this->update_build_options( $build_options );

		set_site_transient(
			$cache_key,
			$response,
			30 * 60
		); // Cache it for 30min.

		return $do_request;
	}

	/**
	 * Tells if build is in process.
	 *
	 * @param string $platform Platform name.
	 * @param string $env      Environment name.
	 *
	 * @return bool
	 */
	public function is_build_in_process( $platform, $env ) {
		$build_options   = $this->get_build_options();
		$build_requested = ( isset( $build_options['build_requested'] ) ? $build_options['build_requested'] : array() );

		if (
			isset( $build_requested[ $platform ]['build_status'] ) &&
			isset( $build_requested[ $platform ]['success'] ) &&
			isset( $build_requested[ $platform ]['type'] ) &&
			! empty( $build_requested ) &&
			$build_requested[ $platform ]['type'] === $env &&
			$build_requested[ $platform ]['success'] &&
			! empty( $build_requested[ $platform ]['build_id'] ) &&
			! in_array(
				$build_requested[ $platform ]['build_status'],
				array(
					'completed',
					'cancelled',
					'failed',
				),
				true
			)
		) {
			return true;
		}
	}

	/**
	 * Check if build is in process for specific platform.
	 *
	 * @param string $platform Platform name.
	 *
	 * @since [BBAPVERSION]
	 *
	 * @return bool
	 */
	public function is_build_in_process_for_platform( $platform ) {
		$build_options   = $this->get_build_options();
		$build_requested = ( isset( $build_options['build_requested'] ) ? $build_options['build_requested'] : array() );

		if ( ! empty( $build_requested ) &&
		     isset( $build_requested[ $platform ]['build_status'] ) &&
		     isset( $build_requested[ $platform ]['success'] ) &&
		     isset( $build_requested[ $platform ]['type'] ) &&
		     $build_requested[ $platform ]['success'] &&
		     ! empty( $build_requested[ $platform ]['build_id'] ) &&
		     ! in_array(
			     $build_requested[ $platform ]['build_status'],
			     array(
				     'completed',
				     'cancelled',
				     'failed',
			     ),
			     true
		     )
		) {
			return true;
		}

		return false;
	}

	/**
	 * Lock build for platform to prevent concurrent build requests.
	 *
	 * @param string $platform Platform type (ios or android).
	 *
	 * @since [BBAPVERSION]
	 * @return bool True if lock acquired successfully, false if platform is already locked.
	 */
	public function lock_build_for_platform( $platform ) {
		$transient_key = 'bbapp_build_lock_' . $platform;

		if ( get_transient( $transient_key ) ) {
			return false;
		}

		set_transient( $transient_key, true, HOUR_IN_SECONDS );

		return true;
	}

	/**
	 * Unlock build for platform.
	 *
	 * @param string $platform Platform type (ios or android).
	 *
	 * @since [BBAPVERSION]
	 * @return bool True on success, false on failure.
	 */
	public function unlock_build_for_platform( $platform ) {
		$transient_key = 'bbapp_build_lock_' . $platform;

		return delete_transient( $transient_key );
	}

	/**
	 * Background queue task for checking build status.
	 * 1. BuddyBossApp Server Pings WordPress to do build status check.
	 * 2. Plugin adds a background job to attempt it at background.
	 * 3. This function get triggered once job gets triggered.
	 *
	 * @param object $task BG queue task.
	 *
	 * @return bool
	 */
	public function queue_task_check_build_status( $task ) {
		\BuddyBossApp\Admin\Build\Build::instance()->update_build_status();
		$this->clear_all_build_transients();

		return true;
	}

	/**
	 * Checks for build status. & update the data accordingly.
	 *
	 * @param int $build_id Build number.
	 *
	 * @return bool
	 */
	public function cancel_build( $build_id ) {
		$app     = ManageApp::instance()->get_app();
		$api_url = ClientCommon::instance()->get_center_api_url( 'v2' );

		// Do the api call.
		$do_request = bbapp_remote_post(
			$api_url,
			array(
				'method'  => 'POST',
				'timeout' => 15,
				'body'    => array(
					'action'    => 'build-cancel',
					'build_id'  => $build_id,
					'bbapp_id'  => $app['bbapp_app_id'],
					'bbapp_key' => ManageApp::instance()->get_auth_app_key(),
				),
			)
		);

		if ( is_wp_error( $do_request ) || empty( $do_request ) ) {
			return false;
		}

		$response = isset( $do_request['body'] ) ? $do_request['body'] : '';
		$response = json_decode( $response, true );

		if ( ! isset( $response['status'] ) || ! isset( $response['sub_status'] ) ) {
			return false;
		}

		// fetch new status.
		self::instance()->get_build_status( $build_id, true );

		return true;
	}

	/**
	 * Checks for build status. & update the data accordingly.
	 *
	 * @param int  $build_id  Build id number.
	 * @param bool $use_cache Override cache.
	 *
	 * @return mixed|string
	 */
	public function get_build_status( $build_id, $use_cache = true ) {
		$app   = ManageApp::instance()->get_app();
		$cache = get_transient( 'get_build_status_cache_' . $build_id );

		if ( ! empty( $cache ) && $use_cache ) {
			return $cache;
		}

		$api_url    = ClientCommon::instance()->get_center_api_url( 'v2' );
		$req_body   = array(
			'action'    => 'build-status',
			'build_id'  => $build_id,
			'bbapp_id'  => $app['bbapp_app_id'],
			'bbapp_key' => ManageApp::instance()->get_auth_app_key(),
		);
		$do_request = bbapp_remote_post(
			$api_url,
			array(
				'method'  => 'POST',
				'timeout' => 15,
				'body'    => $req_body,
			)
		);

		if ( is_wp_error( $do_request ) || empty( $do_request ) ) {
			$status = 'failed';
		} else {
			$response = isset( $do_request['body'] ) ? $do_request['body'] : '';
			$response = json_decode( $response, true );
			// end api call.

			// in-case of error.
			if ( isset( $response['code'] ) && in_array( $response['code'], array( 'invalid_build' ), true ) ) {
				$status = 'failed';
			} elseif ( isset( $response['sub_status'] ) ) {
				$status = $response['sub_status'];
			} else {
				$status = 'failed';
			}
		}

		set_transient( 'get_build_status_cache_' . $build_id, $status, 60 * 30 ); // 30 minutes of cache.

		return $status;
	}

	/**
	 * Return build options
	 *
	 * @return mixed
	 */
	public function get_build_options() {
		$options = bbapp_get_network_option( 'bbapp_builds' );

		if ( ! is_array( $options ) ) {
			$options = array();
		}

		if ( ! isset( $options ) ) {
			$options = array();
		}

		return $options;
	}

	/**
	 * Update build options.
	 *
	 * @param array $app_options Options.
	 *
	 * @return bool
	 */
	public function update_build_options( $app_options ) {
		bbapp_set_network_option( 'bbapp_builds', $app_options );

		return true;
	}

	/**
	 * Update URL redirect for specific build.
	 *
	 * @param string $platform Platform name.
	 * @param int    $version  Version code.
	 * @param string $url      Site URl.
	 * @param bool   $status   Build status.
	 *
	 * @return bool
	 */
	public function update_build_url_redirect( $platform, $version, $url, $status = false ) {
		if ( ! bbapp_is_valid_url( $url ) ) {
			return false;
		}

		$options = bbapp_get_network_option( 'bbapp_build_url_redirects' );

		if ( ! is_array( $options ) ) {
			$options = array();
		}

		if ( ! isset( $options ) ) {
			$options = array();
		}

		$options[ $platform ][ $version ]['url']    = $url;
		$options[ $platform ][ $version ]['status'] = $status;

		bbapp_set_network_option( 'bbapp_build_url_redirects', $options );

		return $url;
	}

	/**
	 * Get app version.
	 *
	 * @param string $platform Platform name.
	 *
	 * @return string
	 */
	public function get_app_version( $platform ) {
		$build_options = $this->get_build_options();
		$app_version   = isset( $build_options[ "app_version_{$platform}" ] ) ? $build_options[ "app_version_{$platform}" ] : '';

		if ( empty( $app_version ) ) {
			// @todo: get this state value from customer center.
			$app_version = '0.0.0';
		}

		return $app_version;
	}

	/**
	 * Returns the app version code.
	 *
	 * @param string $platform Platform name.
	 *
	 * @return string
	 */
	public function get_app_version_code( $platform ) {
		$build_options = $this->get_build_options();
		$app_version   = isset( $build_options[ "app_version_code_{$platform}" ] ) ? $build_options[ "app_version_code_{$platform}" ] : '';

		if ( empty( $app_version ) ) {
			// @todo: get this state value from customer center.
			$app_version = '1';
		}

		return $app_version;
	}

	/**
	 * Gets the next possible version.
	 *
	 * @param int    $version  Version.
	 * @param string $platform Platform name.
	 *
	 * @return string
	 */
	public function get_next_version( $version, $platform ) {
		if ( ! empty( $version ) ) {
			$min_version = $version;
		} else {
			if ( 'ios' === $platform ) {
				$min_version = Configure::instance()->option( 'publish.general.ios.min_app' );
			} elseif ( 'android' === $platform ) {
				$min_version = Configure::instance()->option( 'publish.general.android.min_app' );
			}
		}

		if ( empty( $min_version ) ) {
			$min_version = '0.0.0';
		}

		$split_version = explode( '.', $min_version );
		$split_version[2] ++;

		if ( $split_version[2] > 9 ) {
			$split_version[1] ++;

			if ( $split_version[1] > 9 ) {
				$split_version[0] ++;
				$split_version[1] = 0;
			}

			$split_version[2] = 0;
		}

		return implode( '.', $split_version );
	}

	/**
	 * Return the next possible version code.
	 *
	 * @param int $version_code Version code.
	 *
	 * @return mixed
	 */
	public function get_next_version_code( $version_code ) {
		$version_code ++;

		return $version_code;
	}

	/**
	 * Check if provided version is valid or not.
	 *
	 * @param string $app_version App version.
	 *
	 * @return bool
	 */
	public function is_valid_app_version( $app_version ) {
		$allowed = array( '.' ); // you can add here more value, you want to allow.

		if ( ctype_digit( str_replace( $allowed, '', $app_version ) ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Return the toggle build min version support link.
	 *
	 * @param string $platform           Platform name.
	 * @param string $build_version_code Platform version code.
	 *
	 * @return string
	 */
	public function get_build_min_version_code_toggle_link( $platform, $build_version_code ) {
		return wp_nonce_url( bbapp_get_admin_url( "admin.php?page=bbapp-build&setting=all-builds&platform={$platform}&build_version_code={$build_version_code}&action=bbapp_min_version_build_toggle" ), 'bbapp_min_version_build_toggle_nonce', 'bbapp_min_version_build_toggle_nonce' );
	}

	/**
	 * Return the Min version support for build.
	 *
	 * @param string $platform Platform name.
	 *
	 * @return int|string
	 */
	public function get_build_min_version_code_support( $platform ) {
		if ( 'ios' === $platform ) {
			$min_version_code = Configure::instance()->option( 'publish.general.ios.min_app' );
		}

		if ( 'android' === $platform ) {
			$min_version_code = Configure::instance()->option( 'publish.general.android.min_app' );
		}

		return ! empty( $min_version_code ) ? (int) $min_version_code : 0;
	}

	/**
	 * Update the min version support in build.
	 *
	 * @param string $platform     Platform name.
	 * @param string $version_code Version code.
	 */
	public function update_build_min_version_code_support( $platform, $version_code ) {
		$bbapp_settings = Configure::instance()->get_settings();

		if ( 'ios' === $platform ) {
			$bbapp_settings['publish.general.ios.min_app'] = $version_code;
		}

		if ( 'android' === $platform ) {
			$bbapp_settings['publish.general.android.min_app'] = $version_code;
		}

		Configure::instance()->save_settings( $bbapp_settings );
	}

	/**
	 * Helper function to clear all build cache.
	 */
	public function clear_app_builds_cache() {
		update_option( 'bbapp_all_builds_unique_key', uniqid() );
	}

	/**
	 * Get app builds from API.
	 *
	 * @param string     $platform_type  Type of build.
	 * @param string     $platform_env   Platform environment.
	 * @param bool|array $build_ids      Build IDs.
	 * @param string     $status         Filter by build status.
	 * @param int        $page           Number of page.
	 * @param int        $per_page       Number of item per page.
	 * @param boolean    $override_cache Overwrite cache.
	 *
	 * @return array
	 */
	public function get_app_builds( $platform_type = '', $platform_env = '', $build_ids = false, $status = '', $page = 1, $per_page = 12, $override_cache = false ) {
        // Get connected app details.
		$app = \BuddyBossApp\ManageApp::instance()->get_app();

		if ( ! isset( $app['verified'] ) || ! $app['verified'] ) {
			return;
		}

		if ( empty( $per_page ) ) {
			$per_page = 12;
		}

		$args = array();

		if ( ! empty( $platform_type ) ) {
			if ( ! is_array( $platform_type ) ) {
				$platform_type = array( $platform_type );
			}

			$args['type'] = $platform_type;
		}

		if ( ! empty( $platform_env ) ) {
			if ( ! is_array( $platform_env ) ) {
				$platform_env = array( $platform_env );
			}

			$args['env'] = $platform_env;
		} else {
			$args['env']  = array( 'live', 'dev' );
			$platform_env = $args['env'];
		}

		if ( ! empty( $page ) ) {
			$args['page'] = $page;
		}

		if ( ! empty( $per_page ) ) {
			$args['per_page'] = $per_page;
		}

		if ( ! empty( $status ) ) {
			$args['status'] = $status;
		}

		if ( ! empty( $build_ids ) ) {
			foreach ( $build_ids as $build_id ) {
				$args['build_ids[]'] = $build_id;
			}
		}

		$publish_ios_namespace         = Configure::instance()->option( 'publish.ios.namespace' );
		$publish_ios_dev_namespace     = Configure::instance()->option( 'publish.ios.dev.namespace' );
		$publish_android_namespace     = Configure::instance()->option( 'publish.android.namespace' );
		$publish_android_dev_namespace = Configure::instance()->option( 'publish.android.dev.namespace' );

		if ( isset( $args['type'] ) && in_array( 'ios', $args['type'], true ) ) {
			if ( in_array( 'live', $args['env'], true ) && ! empty( $publish_ios_namespace ) ) {
				$args['ios_bundle_ids'][] = $publish_ios_namespace;
			}

			if ( in_array( 'dev', $args['env'], true ) && ! empty( $publish_ios_dev_namespace ) ) {
				$args['ios_bundle_ids'][] = $publish_ios_dev_namespace;
			}
		}

		if ( isset( $args['type'] ) && in_array( 'android', $args['type'], true ) ) {
			if ( in_array( 'live', $args['env'], true ) && ! empty( $publish_android_namespace ) ) {
				$args['android_bundle_ids'][] = $publish_android_namespace;
			}

			if ( in_array( 'dev', $args['env'], true ) && ! empty( $publish_android_dev_namespace ) ) {
				$args['android_bundle_ids'][] = $publish_android_dev_namespace;
			}
		}

		$args['bbapp_id']  = ManageApp::instance()->get_app_id();
		$args['bbapp_key'] = ManageApp::instance()->get_auth_app_key();

		// Runtime cache key to prevent duplicate API calls within same request.
		$runtime_cache_key = 'builds_' . md5( maybe_serialize( $args ) );

		// Check runtime cache first (fastest).
		if ( ! $override_cache && isset( self::$runtime_builds_cache[ $runtime_cache_key ] ) ) {
			return self::$runtime_builds_cache[ $runtime_cache_key ];
		}

		$cache_unique_key = get_option( 'bbapp_all_builds_unique_key' ); // this will help forcefully kill all group cache.
		$cache_key        = 'bbapp_all_builds_' . md5( maybe_serialize( $args ) . $cache_unique_key );
		$cached           = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			$retval = $cached;

			// Store in runtime cache.
			self::$runtime_builds_cache[ $runtime_cache_key ] = $retval;
		} else {
			$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/get-builds' );

			$request = bbapp_remote_get(
				$api_url,
				array(
					'timeout' => 15,
					'body'    => $args,
				)
			);

			$response         = wp_remote_retrieve_body( $request );
			$response         = json_decode( $response, true );
			$response_code    = wp_remote_retrieve_response_code( $request );
			$response_message = wp_remote_retrieve_response_message( $request );

			if (
				'OK' === $response_message
				&& 200 === $response_code
			) {
				$data['data']                  = $response;
				$data['headers']['total']      = wp_remote_retrieve_header( $request, 'Total' );
				$data['headers']['totalpages'] = wp_remote_retrieve_header( $request, 'TotalPages' );

				$retval = $data;

				// Use dynamic cache duration based on build status.
				$cache_duration = $this->get_builds_cache_duration( $retval );

				set_site_transient( $cache_key, $retval, $cache_duration );

				// Store in runtime cache.
				self::$runtime_builds_cache[ $runtime_cache_key ] = $retval;
			} elseif ( 400 !== $response_code ) {
				$retval = new WP_Error(
					str_replace( ' ', '_', strtolower( $response_message ) ),
					$response_message,
					array( 'status' => $response_code )
				);
			} else {
				$retval = new WP_Error(
					str_replace( ' ', '_', strtolower( $response['code'] ) ),
					$response['message'],
					array( 'status' => $response_code )
				);
			}
		}

		return $retval;
	}

	/**
	 * Get cache duration based on builds data.
	 *
	 * Returns shorter cache duration if any build is in progress,
	 * otherwise returns longer duration for stable data.
	 *
	 * @param array $builds_data The builds response data.
	 *
	 * @since 2.10.0
	 * @return int Cache duration in seconds.
	 */
	private function get_builds_cache_duration( $builds_data ) {
		// If any build is in progress, use shorter cache.
		if ( ! empty( $builds_data['data'] ) && is_array( $builds_data['data'] ) ) {
			foreach ( $builds_data['data'] as $build ) {
				if ( isset( $build['status'] ) && in_array( $build['status'], array( 'queued', 'building', 'inProgress' ), true ) ) {
					return self::BUILDS_CACHE_DURATION_SHORT; // 5 minutes for active builds.
				}
			}
		}

		return self::BUILDS_CACHE_DURATION; // 30 minutes for stable data.
	}

	/**
	 * App Core Version.
	 *
	 * @return bool
	 */
	public function get_build_app_core_version() {
		$app_core_version = true;

		$builds = $this->get_app_builds( '', '', false, '', 1, 1 );

		// If get app build function return error.
		if ( is_wp_error( $builds ) ) {
			return false;
		}

		if ( ! isset( $builds['data'][0] ) || ! isset( $builds['data'][0]['env'] ) ) {
			return false;
		}

		$build = $builds['data'][0];

		if ( isset( $build['status'] ) && 'completed' === $build['status'] ) {
			$app_core_version = isset( $build['app_core_version'] ) ? $build['app_core_version'] : false;
		}

		return $app_core_version;
	}


	/**
	 * Get the build dir.
	 *
	 * @return mixed|string
	 */
	public function get_build_dir() {
		static $bbapp_build_dir;

		if ( isset( $bbapp_build_dir ) ) {
			return $bbapp_build_dir;
		}

		if ( bbapp()->is_network_activated() ) {
			switch_to_blog( 1 );
		}

		$upload_dir = wp_upload_dir();
		$dir        = $upload_dir['basedir'] . '/bbapp/builds';

		BBAPP_File::create_dir( $dir );
		$dir = str_replace( '/', DIRECTORY_SEPARATOR, $dir );

		if ( bbapp()->is_network_activated() ) {
			restore_current_blog();
		}

		$bbapp_build_dir = $dir;

		return $dir;
	}


	/**
	 * Ajax code for all builds
	 */
	public function bbapp_build_sync() {
		$paged    = ( isset( $_REQUEST['paged'] ) ? sanitize_text_field( wp_unslash( absint( $_REQUEST['paged'] ) ) ) : 1 ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$type     = ( isset( $_REQUEST['type'] ) ? map_deep( wp_unslash( $_REQUEST['type'] ), 'sanitize_text_field' ) : array() ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$env      = ( isset( $_REQUEST['env'] ) ? map_deep( wp_unslash( $_REQUEST['env'] ), 'sanitize_text_field' ) : array() ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$per_page = 12;

		$builds = $this->get_app_builds( $type, $env, false, '', $paged, $per_page );
		ob_start();
		if ( ! is_wp_error( $builds ) && ! empty( $builds ) && ! empty( $builds['data'] ) ) {
			if ( ! empty( $builds['data']['code'] ) && 'invalid_app' === $builds['data']['code'] ) {
				?>
                <p class="bbapp-no-builds"><?php echo wp_kses_post( $builds['data']['message'] ); ?></p>
				<?php
			} else {
				$headers = $builds['headers'];
				global $build_tile;
				foreach ( $builds['data'] as $build ) {
					$build_tile = $build;
					include bbapp()->plugin_dir . 'views/settings/build/build-tile.php';
				}

				if ( ! empty( $headers['total'] ) ) {
					echo '<div class="pagination tablenav-pages">';
					$pagination = paginate_links(
						array(
							'base'      => add_query_arg( 'paged', '%#%' ),
							'format'    => '',
							'total'     => ceil( (int) $headers['total'] / (int) $per_page ),
							'current'   => (int) $paged,
							'prev_text' => __( '&lsaquo;', 'buddyboss-app' ),
							'next_text' => __( '&rsaquo;', 'buddyboss-app' ),
							'mid_size'  => 1,
							'add_args'  => array(),
						)
					);

					if ( $pagination ) {
						$pagination = str_replace( 'page-numbers', 'button page-numbers', $pagination );
					}

					echo wp_kses_post( str_replace( 'page-numbers current', 'page-numbers current disabled', $pagination ) );
					echo '</div>';
				}
			}
		} elseif ( is_wp_error( $builds ) ) {
			?>
			<p class="bbapp-no-builds"><?php echo wp_kses_post( $builds->get_error_message( $builds->get_error_code() ) ); ?></p>
			<?php
		} else {
			$build_link = bbapp_get_super_admin_url( 'admin.php?page=bbapp-build' );
			?>
			<p class="bbapp-no-builds">
				<?php printf( '%1$s <a href="%2$s">%3$s</a> %4$s', esc_html__( 'You haven\'t created any builds yet.', 'buddyboss-app' ), esc_url( $build_link ), esc_html__( 'Request a build', 'buddyboss-app' ), esc_html__( 'to create your first one.', 'buddyboss-app' ) ); ?>
			</p>
			<?php
		}

		$data = ob_get_clean();

		wp_send_json_success( $data );
	}

	/**
	 * Clear build apis transients.
	 */
	public function clear_all_build_transients() {
		bbapp_delete_transients( '_transient_bbapp_all_builds_' );

		// Clear runtime cache as well.
		self::$runtime_builds_cache = array();
	}

	/**
	 * Update build status.
	 *
	 * @return bool|void
	 */
	public function bbapp_update_build_status_now() {
		$build_id = ( isset( $_POST['build_id'] ) ? bbapp_input_clean( wp_unslash( $_POST['build_id'] ) ) : 0 ); // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( empty( $build_id ) ) {
			return;
		}

		// Smart throttling: Prevent rapid successive API calls for same build.
		$throttle_key = 'bbapp_build_status_throttle_' . $build_id;

		if ( false !== get_transient( $throttle_key ) ) {
			// Throttled: Return cached build data instead of API call.
			$cached_builds = $this->get_app_builds( '', '', array( $build_id ), '', 1, 1, false );

			if ( ! is_wp_error( $cached_builds ) && ! empty( $cached_builds['data'] ) ) {
				$build = current( $cached_builds['data'] );

				if ( isset( $build['id'] ) && $build['id'] === $build_id ) {
					Admin\Build\Build::instance()->update_build_status();
				}
			}

			return true;
		}

		// Set throttle for 2 minutes.
		set_transient( $throttle_key, time(), 2 * MINUTE_IN_SECONDS );

		$args['bbapp_id']  = ManageApp::instance()->get_app_id();
		$args['bbapp_key'] = ManageApp::instance()->get_auth_app_key();
		$args['build_ids'] = array( $build_id );

		$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/get-builds' );

		$request = bbapp_remote_get(
			$api_url,
			array(
				'timeout' => 15,
				'body'    => $args,
			)
		);

		$response         = wp_remote_retrieve_body( $request );
		$response         = json_decode( $response, true );
		$response_code    = wp_remote_retrieve_response_code( $request );
		$response_message = wp_remote_retrieve_response_message( $request );

		if ( 'OK' === $response_message && 200 === $response_code && ! empty( $response ) ) {
			$build = current( $response );

			if ( isset( $build['id'] ) && $build['id'] === $build_id ) {
				self::instance()->clear_all_build_transients();
				Admin\Build\Build::instance()->update_build_status();
			}
		}

		return true;
	}

	/**
	 * Limit build api.
	 *
	 * @return array|int|WP_Error
	 */
	public function get_app_builds_limit() {
		$app_info = \BuddyBossApp\ManageApp::instance()->get_app_info( true );

		if ( is_wp_error( $app_info ) ) {
			return $app_info;
		}

		$build_limits = ( isset( $app_info['build_limit'] ) ? $app_info['build_limit'] : array() );

		if ( empty( $build_limits ) || ! isset( $build_limits->remaining_build ) ) {
			return 0;
		}

		return isset( $build_limits->remaining_build ) ? $build_limits->remaining_build : 0;
	}

	/**
	 * Check the installed plugin update.
	 */
	public function check_installed_plugin_update() {
		$plugin_updates = get_site_transient( 'update_plugins' );

		if ( defined( 'WP_CLI' ) && WP_CLI ) {
			$plugin_updates = get_option( '_site_transient_update_plugins' );
		}

		$plugin_updates  = ( ! empty( $plugin_updates ) && isset( $plugin_updates->response ) ) ? $plugin_updates->response : array();
		$required_update = array();

		if ( bbapp()->plugin_version ) {
			$current_version = bbapp()->plugin_version;
			$updated_version = (
			(
				! empty( $plugin_updates ) &&
				array_key_exists( 'buddyboss-app/buddyboss-app.php', $plugin_updates ) &&
				isset( $plugin_updates['buddyboss-app/buddyboss-app.php']->new_version )
			)
				? $plugin_updates['buddyboss-app/buddyboss-app.php']->new_version
				: bbapp()->plugin_version
			);

			if ( version_compare( $current_version, $updated_version, '<' ) ) {
				$required_update[] = array(
					'plugin_name'     => __( 'BuddyBoss App', 'buddyboss-app' ),
					'current_version' => $current_version,
					'latest_version'  => $updated_version,
				);
			}
		}

		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			$current_version = BP_PLATFORM_VERSION;
			$updated_version = (
			(
				! empty( $plugin_updates ) &&
				array_key_exists( 'buddyboss-platform/bp-loader.php', $plugin_updates ) &&
				isset( $plugin_updates['buddyboss-platform/bp-loader.php']->new_version )
			)
				? $plugin_updates['buddyboss-platform/bp-loader.php']->new_version
				: BP_PLATFORM_VERSION
			);

			if ( version_compare( $current_version, $updated_version, '<' ) ) {
				$required_update[] = array(
					'plugin_name'     => __( 'BuddyBoss Platform', 'buddyboss-app' ),
					'current_version' => $current_version,
					'latest_version'  => $updated_version,
				);
			}

			if ( function_exists( 'bb_platform_pro' ) ) {
				$current_version = bb_platform_pro()->version;
				$updated_version = (
				(
					! empty( $plugin_updates ) &&
					array_key_exists( 'buddyboss-platform-pro/buddyboss-platform-pro.php', $plugin_updates ) &&
					isset( $plugin_updates['buddyboss-platform-pro/buddyboss-platform-pro.php']->new_version )
				)
					? $plugin_updates['buddyboss-platform-pro/buddyboss-platform-pro.php']->new_version
					: bb_platform_pro()->version
				);

				if ( version_compare( $current_version, $updated_version, '<' ) ) {
					$required_update[] = array(
						'plugin_name'     => __( 'BuddyBoss Platform Pro', 'buddyboss-app' ),
						'current_version' => $current_version,
						'latest_version'  => $updated_version,
					);
				}
			}
		}

		return $required_update;
	}

	/**
	 * Get builds versions from API.
	 *
	 * @param string  $type           Type of build.
	 * @param string  $env            Type of environment.
	 * @param string  $status         Filter by build status.
	 * @param boolean $override_cache Overwrite cache.
	 *
	 * @return array
	 */
	public function get_app_build_versions( $type = '', $env = 'live', $status = 'completed', $override_cache = false ) {
		// Get connected app details.
		$app = \BuddyBossApp\ManageApp::instance()->get_app();

		if ( ! isset( $app['verified'] ) || ! $app['verified'] ) {
			return;
		}

		$retval = array();
		$args   = array();

		if ( ! empty( $type ) ) {
			$args['type'] = $type;
		}

		if ( ! empty( $env ) ) {
			$args['env'] = $env;
		}

		if ( ! empty( $status ) ) {
			$args['status'] = $status;
		}

		$publish_ios_namespace         = Configure::instance()->option( 'publish.ios.namespace' );
		$publish_ios_dev_namespace     = Configure::instance()->option( 'publish.ios.dev.namespace' );
		$publish_android_namespace     = Configure::instance()->option( 'publish.android.namespace' );
		$publish_android_dev_namespace = Configure::instance()->option( 'publish.android.dev.namespace' );

		if ( empty( $args['type'] ) || ( is_array( $args['type'] ) && in_array( 'ios', $args['type'], true ) ) || 'ios' === $args['type'] ) {
			if ( ( empty( $env ) || 'live' === $env ) && ! empty( $publish_ios_namespace ) ) {
				$args['ios_bundle_ids'][] = $publish_ios_namespace;
			}

			if ( ( empty( $env ) || 'dev' === $env ) && ! empty( $publish_ios_dev_namespace ) ) {
				$args['ios_bundle_ids'][] = $publish_ios_dev_namespace;
			}
		}

		if ( empty( $args['type'] ) || ( is_array( $args['type'] ) && in_array( 'android', $args['type'], true ) ) || 'android' === $args['type'] ) {
			if ( ( empty( $env ) || 'live' === $env ) && ! empty( $publish_android_namespace ) ) {
				$args['android_bundle_ids'][] = $publish_android_namespace;
			}

			if ( ( empty( $env ) || 'dev' === $env ) && ! empty( $publish_android_dev_namespace ) ) {
				$args['android_bundle_ids'][] = $publish_android_dev_namespace;
			}
		}

		$args['bbapp_id']  = ManageApp::instance()->get_app_id();
		$args['bbapp_key'] = ManageApp::instance()->get_auth_app_key();

		$cache_unique_key = get_option( 'bbapp_all_builds_unique_key' ); // this will help forcefully kill all group cache.
		$cache_key        = 'bbapp_all_builds_versions_' . md5( maybe_serialize( $args ) . $cache_unique_key );
		$cached           = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			$retval = $cached;
		} else {
			$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/get-builds-version' );

			$request = bbapp_remote_get(
				$api_url,
				array(
					'timeout' => 15,
					'body'    => $args,
				)
			);

			$response         = wp_remote_retrieve_body( $request );
			$response         = json_decode( $response, true );
			$response_code    = wp_remote_retrieve_response_code( $request );
			$response_message = wp_remote_retrieve_response_message( $request );

			if (
				'OK' === $response_message
				&& 200 === $response_code
			) {
				if ( ! empty( $response ) && ! is_wp_error( $response ) ) {
					$retval = $response;
					set_site_transient( $cache_key, $retval, 10 * 60 ); // cache it for 10 min.
				}
			} elseif ( 400 !== $response_code ) {
				$retval = new WP_Error(
					str_replace( ' ', '_', strtolower( $response_message ) ),
					$response_message,
					array( 'status' => $response_code )
				);
			} else {
				$retval = new WP_Error(
					str_replace( ' ', '_', strtolower( $response['code'] ) ),
					$response['message'],
					array( 'status' => $response_code )
				);
			}
		}

		return $retval;
	}

	/**
	 * Admin notice show when Core App version.
	 */
	public function app_core_version_admin_notice() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$app_core_data = bbapp_get_app_core();
		$notice_id     = 'app_core_version-' . str_replace( '.', '-', $app_core_data['version'] );

		if ( \BuddyBossApp\Admin\Build\Build::instance()->get_stored_notice( 'bbapp-dismissed-notice-' . $notice_id ) ) {
			return;
		}

		$build_app_core_version = $this->get_build_app_core_version();

		if ( false === $build_app_core_version ) {
			return;
		}

		if ( version_compare( $build_app_core_version, $app_core_data['version'], '<' ) ) {
			/* translators: %1$s: Build page URL, %2$s: App core link. */
			$notice = sprintf( __( 'There is an update available for BuddyBoss App. <a href="%1$s">Request a new build</a> to update your app. <a href="%2$s" target="_blank">View Release Notes</a>', 'buddyboss-app' ), bbapp_get_admin_url( 'admin.php?page=bbapp-build&setting=request-build' ), esc_url( $app_core_data['link'] ) );

			$allowed_html          = wp_kses_allowed_html( 'post' );
			$allowed_html['input'] = array(
				'type'  => array(),
				'name'  => array(),
				'value' => array(),
				'id'    => array(),
			);
			echo sprintf( '<div class="notice notice-success is-dismissible bbapp-is-dismissible" data-value"" data-noticeid="%1$s">%2$s%3$s</div>', esc_attr( $notice_id ), wp_kses_post( wpautop( $notice ) ), wp_kses( wp_nonce_field( "bbapp-dismissible-notice-$notice_id", "bbapp-dismissible-nonce-$notice_id", true, false ), $allowed_html ) );
		}
	}

	/**
	 * Get app latest builds from API.
	 *
	 * @param string  $platform_type  Type of build.
	 * @param string  $platform_env   Platform environment.
	 * @param array   $status         Filter by build status.
	 * @param boolean $override_cache Overwrite cache.
	 *
	 * @since 1.5.7
	 *
	 * @return array
	 */
	public function get_app_lateset_builds( $platform_type = '', $platform_env = '', $status = array(), $override_cache = false ) {
		// Get connected app details.
		$app = \BuddyBossApp\ManageApp::instance()->get_app();

		if ( ! isset( $app['verified'] ) || ! $app['verified'] ) {
			return;
		}

		$retval = array();
		$args   = array();

		if ( ! empty( $platform_type ) ) {
			if ( ! is_array( $platform_type ) ) {
				$platform_type = array( $platform_type );
			}
			$args['type'] = $platform_type;
		}

		if ( ! empty( $platform_env ) ) {
			if ( ! is_array( $platform_env ) ) {
				$platform_env = array( $platform_env );
			}

			$args['env'] = $platform_env;
		} else {
			$args['env']  = array( 'live', 'dev' );
			$platform_env = $args['env'];
		}

		if ( ! empty( $status ) ) {
			$args['status'] = $status;
		}

		$args['bbapp_id']  = ManageApp::instance()->get_app_id();
		$args['bbapp_key'] = ManageApp::instance()->get_auth_app_key();
		$cache_unique_key  = get_option( 'bbapp_all_builds_unique_key' ); // this will help forcefully kill all group cache.
		$cache_key         = 'bbapp_all_builds_latest_' . md5( maybe_serialize( $args ) . $cache_unique_key );
		$cached            = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			$retval = $cached;
		} else {
			$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/get-latest-builds' );

			$request = bbapp_remote_get(
				$api_url,
				array(
					'timeout' => 45,
					'body'    => $args,
				)
			);

			$response         = wp_remote_retrieve_body( $request );
			$response         = json_decode( $response, true );
			$response_code    = wp_remote_retrieve_response_code( $request );
			$response_message = wp_remote_retrieve_response_message( $request );

			if (
				'OK' === $response_message
				&& 200 === $response_code
			) {
				$data['data'] = $response;
				$retval       = $data;

				set_site_transient( $cache_key, $retval, 10 * 60 ); // cache it for 10 min.
			} elseif ( 400 !== $response_code ) {
				$retval = new WP_Error(
					str_replace( ' ', '_', strtolower( $response_message ) ),
					$response_message,
					array( 'status' => $response_code )
				);
			} else {
				$retval = new WP_Error(
					str_replace( ' ', '_', strtolower( $response['code'] ) ),
					$response['message'],
					array( 'status' => $response_code )
				);
			}
		}

		return $retval;
	}
}
