<?php
/**
 * Holds App plugin internal API functionality.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

use BuddyBossApp\Admin\Configure;
use BuddyBossApp\ClientQueue;
use MingYuanYun\AppStore\Exceptions\ConfigException;
use Wp_Error;

/**
 * This Class Contain the Code for BuddyBossApp Internal API which Run using admin-ajax.php feature to connect
 * with BuddyBossApp Server.
 * Class BuddyBossAppApi
 *
 * @package BuddyBossApp
 */
class BuddyBossAppApi {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * BuddyBossAppApi constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return BuddyBossAppApi
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Filters/hooks here.
	 */
	public function hooks() {
		add_action( 'admin_init', array( $this, 'admin_init' ) );
	}

	/**
	 * Admin init hook.
	 */
	public function admin_init() {
		// Rest for bbapp server.
		add_action( 'wp_ajax_nopriv_bbapp_api', array( $this, 'response' ) );
		add_action( 'wp_ajax_bbapp_api', array( $this, 'response' ) );
	}

	/**
	 * Response for API.
	 */
	public function response() {
		header( 'Content-Type: application/json' );
		$response = $this->api();
		echo wp_json_encode( $response, JSON_PRETTY_PRINT );
		exit;
	}

	/**
	 * BuddyBossApp API
	 */
	public function api() {
		/**
		 * We don't want to expose any error by any case on this endpoint.
		 */
		$action = ( isset( $_POST['do'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['do'] ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'build-notify' === $action ) {
			return $this->build_notify();
		}

		if ( 'build-upload-notify' === $action ) {
			return $this->build_upload_notify();
		}

		if ( 'publish-notify' === $action ) {
			return $this->publish_notify();
		}

		if ( 'disconnect-site' === $action || 'bbapp-is-trashed-app-id' === $action ) {
			return $this->disconnect_site();
		}

		if ( 'apple-developer-details' === $action ) {
			return $this->apple_developer_account_details();
		}

		if ( 'google-developer-details' === $action ) {
			return $this->android_developer_account_details();
		}

		if ( 'fetch-apple-app-data' === $action ) {
			return $this->fetch_apple_app_data();
		}

		if ( 'fetch-google-app-data' === $action ) {
			return $this->fetch_google_app_data();
		}

		if ( 'bbapp-is-disabled' === $action ) {
			return $this->app_is_disabled();
		}

		// Get the themes and plugins info.
		if ( 'fetch-themes-plugins-info' === $action ) {
			return $this->fetch_themes_plugins_info();
		}

		if ( 'disconnect-development-access' === $action ) {
			return $this->clear_development_access_cache();
		}

		// Is plugin is activated on mutli-site?
		if ( 'bbapp-is-activated-multi-site' === $action ) {
			return $this->app_is_activated_multi_site();
		}

		// Update core branches.
		if ( 'bbapp-udpate-core-branches' === $action ) {
			return $this->bbapp_update_core_branches();
		}

		// License type update.
		if ( 'bb-plan-update' === $action ) {
			return $this->bbapp_update_license_type();
		}

		return $this->return_error( 'no_action', __( 'No action specified.', 'buddyboss-app' ) );
	}

	/**
	 * Return Error Response.
	 *
	 * @param int    $code    Error code.
	 * @param string $message Error message.
	 *
	 * @return null
	 */
	public function return_error( $code, $message ) {
		return wp_send_json_error(
			array(
				'code'    => $code,
				'message' => $message,
			)
		);
	}

	/**
	 * Helps authenticate api using bbapp token.
	 * Returns App instance on success.
	 *
	 * @return array|bool
	 */
	public function authenticate_bbapp_id() {
		$bbapp_app_id = isset( $_POST['bbapp_app_id'] ) ? bbapp_input_clean( wp_unslash( $_POST['bbapp_app_id'] ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$token        = isset( $_POST['bbapp_app_token'] ) ? bbapp_input_clean( wp_unslash( $_POST['bbapp_app_token'] ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$app          = ManageApp::instance()->get_app();

		if ( ! $app ) {
			return false;
		}

		if ( strtolower( $app['bbapp_app_id'] ) === strtolower( $bbapp_app_id ) && sha1( $app['bbapp_app_key'] ) === $token ) {
			return $app;
		}

		return false;
	}

	/**
	 * Build Notify.
	 */
	public function build_upload_notify() {
		// Return App Instance when authentication is valid.
		$app = $this->authenticate_bbapp_id();

		if ( ! $app ) {
			return $this->return_error( 'invalid_bbapp_id', __( 'BuddyBoss App ID Provided is Invalid.', 'buddyboss-app' ) );
		}

		/**
		 * Add the queue job to run on background.
		 */
		$bbapp_queue = Jobs::instance();
		$bbapp_queue->add( 'check_upload_status', array( 'bbapp_app_id' => $app['bbapp_app_id'] ), 9 );
		$bbapp_queue->start(); // start the queue processor.

		return array( 'success' => true );
	}

	/**
	 * Build Notify.
	 */
	public function build_notify() {
		// Return App Instance when authentication is valid.
		$app = $this->authenticate_bbapp_id();

		if ( ! $app ) {
			return $this->return_error( 'invalid_bbapp_id', __( 'BuddyBoss App ID Provided is Invalid.', 'buddyboss-app' ) );
		}

		/**
		 * Add the queue job to run on background.
		 */
		$bbapp_queue = Jobs::instance();
		$bbapp_queue->add( 'check_build_status', array( 'bbapp_app_id' => $app['bbapp_app_id'] ), 9 );
		$bbapp_queue->start(); // start the queue processor.

		return array( 'success' => true );
	}

	/**
	 * Disconnect site hook.
	 */
	public function disconnect_site() {
		// Return App Instance when authentication is valid.
		$app = $this->authenticate_bbapp_id();

		if ( ! $app ) {
			return $this->return_error( 'invalid_bbapp_id', __( 'BuddyBoss App ID Provided is Invalid.', 'buddyboss-app' ) );
		}

		ManageApp::instance()->disconnect_app();

		return array( 'success' => true );
	}

	/**
	 * Function to get Google developer account details.
	 *
	 * @return array|Wp_Error
	 */
	public function android_developer_account_details() {
		$app   = $this->authenticate_bbapp_id();
		$cache = (bool) ( isset( $_POST['cache'] ) ? bbapp_input_clean( wp_unslash( $_POST['cache'] ) ) : false ); //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! $app ) {
			return $this->return_error( 'invalid_bbapp_id', __( 'BuddyBoss App ID Provided is Invalid.', 'buddyboss-app' ) );
		}

		$is_android_connected = \BuddyBossApp\AppStores\Android::instance()->is_google_connected( $cache );

		if ( false === $is_android_connected || is_wp_error( $is_android_connected ) ) {
			return $this->return_error( 'disconnected_android_developer', __( 'This app is not connected to an Android Developer account.', 'buddyboss-app' ) );
		}

		$app_package_id = \BuddyBossApp\AppStores\Android::instance()->get_release_bundle_id();

		if ( empty( $app_package_id ) ) {
			$data['release_app'] = $this->return_error( 'missing_bundle_id', __( 'Bundle ID was not setup.', 'buddyboss-app' ) );
		} else {
			$data['release_app']['bundleId'] = $app_package_id;
		}

		$api_access = \BuddyBossApp\AppStores\Services\Android\AndroidPublisher::instance()->operation_permission();

		if ( false === $api_access ) {
			$data['release_app']['api_access'] = $this->return_error( 'no_api_access', __( 'One of API permission is missing.', 'buddyboss-app' ) );
		} else {
			$data['release_app']['api_access'] = $api_access;
		}

		$release_app_name = Configure::instance()->option( 'publish.android.namespace' );

		if ( ! empty( $release_app_name ) ) {
			$release_app_status               = \BuddyBossApp\AppStores\Android::instance()->validate_package_name( $release_app_name );
			$data['release_app']['published'] = ( 200 === $release_app_status ) ? true : false;
		}

		if ( true === $api_access ) {
			$app_data = \BuddyBossApp\AppStores\Android::instance()->get_required_app_details( $cache );

			if ( empty( $app_data ) || false === $app_data ) {
				$data['release_app']['primary_language'] = $this->return_error( 'missing_primary_language', __( 'Primary language is not setup.', 'buddyboss-app' ) );
			} else {
				$data['release_app']['primary_language'] = $app_data['default_language'];
			}

			$release_app_data = \BuddyBossApp\AppStores\Android::instance()->get_all_production_release_data( $cache );

			if ( empty( $release_app_data ) || false === $app_data ) {
				$data['release_app']['releases'] = $this->return_error( 'missing_app_releases', __( 'There is no production releases for the app', 'buddyboss-app' ) );
			} else {
				$data['release_app']['releases'] = $release_app_data;
			}
		}

		$test_app_package_id = \BuddyBossApp\AppStores\Android::instance()->get_test_bundle_id();

		if ( empty( $app_package_id ) ) {
			$data['test_app'] = $this->return_error( 'missing_test_bundle_id', __( 'Test App Bundle ID was not setup.', 'buddyboss-app' ) );
		} else {
			$data['test_app']['bundleId'] = $test_app_package_id;
		}

		return $data;
	}

	/**
	 * Webhook to return the apple developer details.
	 *
	 * @return Wp_Error
	 * @throws ConfigException IOS store exception message.
	 */
	protected function apple_developer_account_details() {
		$app   = $this->authenticate_bbapp_id();
		$cache = (bool) ( isset( $_POST['cache'] ) ? bbapp_input_clean( wp_unslash( $_POST['cache'] ) ) : false ); //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! $app ) {
			return $this->return_error( 'invalid_bbapp_id', __( 'BuddyBoss App ID Provided is Invalid.', 'buddyboss-app' ) );
		}

		$is_apple_connected = \BuddyBossApp\AppStores\Apple::instance()->is_connected( $cache );

		if ( empty( $is_apple_connected ) || is_wp_error( $is_apple_connected ) ) {
			return $this->return_error( 'disconnected_apple_developer', __( 'This app is not connected to an Apple Developer account.', 'buddyboss-app' ) );
		}

		// Fetch Team information.
		$data           = \BuddyBossApp\AppStores\Apple::instance()->get_apple_team_info( $cache );
		$app_package_id = \BuddyBossApp\AppSettings::instance()->get_setting_value( 'publish.ios.namespace' );

		if ( empty( $app_package_id ) ) {
			$data['release_app'] = $this->return_error( 'missing_bundle_id', __( 'Bundle ID was not setup.', 'buddyboss-app' ) );
		} else {
			$data['release_app']['bundleId'] = $app_package_id;
		}

		$app_id = \BuddyBossApp\AppStores\Apple::instance()->get_apple_release_bundle_store_app_id(  $cache );

		if ( ! $app_id ) {
			$data['release_app'] = __( 'No app found.', 'buddyboss-app' );
		}

		$test_app_bundle_id = \BuddyBossApp\Admin\Configure::instance()->option( 'publish.ios.dev.namespace' );

		if ( ! empty( $test_app_bundle_id ) ) {
			$data['test_app']['bundleId'] = $test_app_bundle_id;
		} else {
			$data['test_app'] = __( 'No app found.', 'buddyboss-app' );
		}

		if ( ! empty( $app_id ) ) {
			// App information.
			$appinfo = \BuddyBossApp\AppStores\Apple::instance()->get_app_info( $app_id, $cache );

			if ( ! empty( $appinfo ) && ! is_wp_error( $appinfo ) ) {
				$data['release_app']['name']          = $appinfo['data']['attributes']['name'];
				$data['release_app']['id']            = $app_id;
				$data['release_app']['bundleId']      = $appinfo['data']['attributes']['bundleId'];
				$data['release_app']['sku']           = $appinfo['data']['attributes']['sku'];
				$data['release_app']['primaryLocale'] = $appinfo['data']['attributes']['primaryLocale'];
			}

			// User account details and permissions.
			$user_info = $this->get_acccount_users_info( $cache );

			if ( ! empty( $user_info ) && ! is_wp_error( $user_info ) ) {
				$data['release_app']['bb_access'] = $user_info;
			} else {
				$data['release_app']['bb_access'] = false;
			}

			// App versions.
			$appversions = \BuddyBossApp\AppStores\Apple::instance()->get_app_store_versions_for_an_app( $app_id, $cache );

			if ( ! empty( $appversions ) || ! is_wp_error( $appversions ) || ! empty( $appversions['data'] ) ) {
				$versions = array();
				$count    = 0;

				foreach ( $appversions['data'] as $version ) {
					$versions[ $count ]['version']           = ( isset( $version['attributes']['versionString'] ) ? $version['attributes']['versionString'] : '' );
					$versions[ $count ]['status']            = ( isset( $version['attributes']['appStoreState'] ) ? \BuddyBossApp\AppStores\Apple::instance()->get_apple_publish_status_label( $version['attributes']['appStoreState'] ) : '' );
					$versions[ $count ]['status_from_apple'] = ( isset( $version['attributes']['appStoreState'] ) ? $version['attributes']['appStoreState'] : '' );
					$versions[ $count ]['date']              = ( isset( $version['attributes']['createdDate'] ) ? $version['attributes']['createdDate'] : '' );
					$count ++;
				}

				$data['release_app']['versions'] = $versions;
			} else {
				$data['release_app']['versions'] = array();
			}
		}

		return $data;
	}

	/**
	 * Get user info and invite pending user info.
	 *
	 * @param bool $cache If need to override cache.
	 *
	 * @return array
	 * @throws ConfigException IOS store exception message.
	 */
	public function get_acccount_users_info( $cache ) {
		$user_data = array();
		// User account details and permissions.
		$user_info         = \BuddyBossApp\AppStores\Apple::instance()->get_acccount_users( $cache );
		$user_invited_info = \BuddyBossApp\AppStores\Apple::instance()->get_acccount_invited_users( $cache );

		if ( ! empty( $user_info ) && ! is_wp_error( $user_info ) && ! empty( $user_info['data'] ) ) {
			$users_info_data = ( ! empty( $user_info['data'] ) ? array_column( $user_info['data'], 'attributes', 'id' ) : array() );

			foreach ( $users_info_data as $id => $user_info_data ) {
				if ( isset( $user_info_data['username'] ) && strpos( $user_info_data['username'], '@buddyboss.com' ) ) {
					$is_bb_access = false;

					if ( ManageApp::instance()->get_appcenter_publisher_email_address( $cache ) === $user_info_data['username'] ) {
						if ( ! empty( $user_info_data['roles'] ) && ( in_array( 'ADMIN', $user_info_data['roles'], true ) || in_array( 'APP_MANAGER', $user_info_data['roles'], true ) ) ) {
							$is_bb_access = true;
						}
					}

					$user_data[ $id ] = array(
						'username'          => $user_info_data['username'],
						'roles'             => $user_info_data['roles'],
						'is_invite_pending' => isset( $user_info_data['expirationDate'] ) ? $user_info_data['expirationDate'] : false,
						'is_bb_access'      => $is_bb_access,
					);
				}
			}
		}

		if ( ! empty( $user_invited_info ) && ! is_wp_error( $user_invited_info ) && ! empty( $user_invited_info['data'] ) ) {
			$users_info_data = ( ! empty( $user_invited_info['data'] ) ? array_column( $user_invited_info['data'], 'attributes', 'id' ) : array() );

			foreach ( $users_info_data as $id => $user_info_data ) {
				if ( isset( $user_info_data['email'] ) && strpos( $user_info_data['email'], '@buddyboss.com' ) ) {
					$is_bb_access = false;

					if ( ManageApp::instance()->get_appcenter_publisher_email_address( $cache ) === $user_info_data['email'] ) {
						if ( ! empty( $user_info_data['roles'] ) && ( in_array( 'ADMIN', $user_info_data['roles'], true ) || in_array( 'APP_MANAGER', $user_info_data['roles'], true ) ) ) {
							$is_bb_access = true;
						}
					}

					$user_data[ $id ] = array(
						'username'          => $user_info_data['email'],
						'roles'             => $user_info_data['roles'],
						'is_invite_pending' => isset( $user_info_data['expirationDate'] ) ? $user_info_data['expirationDate'] : false,
						'is_bb_access'      => $is_bb_access,
					);
				}
			}
		}

		return $user_data;
	}

	/**
	 * API to return the apple app details.
	 * This Return Data to App Center on Request.
	 *
	 * @return Wp_Error
	 * @throws ConfigException IOS store exception message.
	 */
	protected function fetch_apple_app_data() {
		$app = $this->authenticate_bbapp_id();

		if ( ! $app ) {
			return $this->return_error( 'invalid_bbapp_id', __( 'BuddyBoss App ID Provided is Invalid.', 'buddyboss-app' ) );
		}

		$cache  = (bool) ( isset( $_POST['cache'] ) ? bbapp_input_clean( wp_unslash( $_POST['cache'] ) ) : false ); //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$app_id = ( isset( $_POST['apple_app_id'] ) ? bbapp_input_clean( wp_unslash( $_POST['apple_app_id'] ) ) : '' ); //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		return \BuddyBossApp\AppStores\Apple::instance()->fetch_app_data( $app_id, $cache );
	}

	/**
	 *  API to return the Google app details.
	 *  This Return Data to App Center on Request.
	 *
	 * @return array|Wp_Error|false|mixed
	 */
	protected function fetch_google_app_data() {
		$app = $this->authenticate_bbapp_id();

		if ( ! $app ) {
			return $this->return_error( 'invalid_bbapp_id', __( 'BuddyBoss App ID Provided is Invalid.', 'buddyboss-app' ) );
		}

		$cache        = (bool) ( isset( $_POST['cache'] ) ? bbapp_input_clean( wp_unslash( $_POST['cache'] ) ) : false ); //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$release_data = \BuddyBossApp\AppStores\Android::instance()->get_release_data( $cache );

		return $release_data;
	}

	/**
	 * When app center notify anything related to publish.
	 * Here we stores it into background job to process later.
	 *
	 * @return array|Wp_Error
	 */
	public function publish_notify() {
		// Return App Instance when authentication is valid.
		$app = $this->authenticate_bbapp_id();

		if ( ! $app ) {
			return $this->return_error( 'invalid_bbapp_id', __( 'BuddyBoss App ID Provided is Invalid.', 'buddyboss-app' ) );
		}

		/**
		 * Add the queue job to run on background.
		 */
		$bbapp_queue = Jobs::instance();
		$bbapp_queue->add( 'check_publish_status', array( 'bbapp_app_id' => $app['bbapp_app_id'] ), 9 );
		$bbapp_queue->start(); // start the queue processor.

		return array( 'success' => true );
	}

	/**
	 * App is disabled site hook.
	 */
	public function app_is_disabled() {
		// Return App Instance when authentication is valid.
		$app = $this->authenticate_bbapp_id();

		if ( ! $app ) {
			return $this->return_error( 'invalid_bbapp_id', __( 'BuddyBoss App ID Provided is Invalid.', 'buddyboss-app' ) );
		}

		ManageApp::instance()->get_app_info( true );

		// Transients  should be clear.
		$delete_transients = array(
			'_bbapp_app_signed_details',
			bbapp_get_app_info_transient_key(),
			'bbapp_last_build_status_interval_check',
			'bbapp_is_under_maintenance',
			'bbapp_get_app_core',
		);

		foreach ( $delete_transients as $tr ) {
			delete_transient( $tr );
			delete_site_transient( $tr );
		}

		\BuddyBossApp\Build::instance()->clear_all_build_transients();
		\BuddyBossApp\Admin\Publish\Publish::instance()->delete_all_publish_transients();

		return array( 'success' => true );
	}

	/**
	 *  API to return the themes and pluings details.
	 *  This Return Data to App Center on Request.
	 *
	 * @return array|Wp_Error
	 */
	public function fetch_themes_plugins_info() {
		// Return App Instance when authentication is valid.
		$app = $this->authenticate_bbapp_id();

		if ( ! $app ) {
			return $this->return_error( 'invalid_bbapp_id', __( 'BuddyBoss App ID Provided is Invalid.', 'buddyboss-app' ) );
		}

		require_once ABSPATH . 'wp-admin/includes/plugin.php';
		require_once ABSPATH . 'wp-admin/includes/update.php';

		// Get active plugins.
		$active_plugins = (array) get_option( 'active_plugins', array() );

		if ( is_multisite() ) {
			$network_activated_plugins = array_keys( get_site_option( 'active_sitewide_plugins', array() ) );
			$active_plugins            = array_merge( $active_plugins, $network_activated_plugins );
		}

		$active_plugins             = array_unique( $active_plugins );
		$active_themes_plugins_data = array();
		// Get all plugins.
		$plugins = get_plugins();

		// Loop of all the plugins.
		if ( ! empty( $plugins ) ) {
			foreach ( $plugins as $key => $plugin ) {
				// Add plugin data to array response.
				$active_themes_plugins_data['plugins'][] = array(
					'name'    => $plugin['Name'],
					'version' => $plugin['Version'],
					'status'  => ( in_array( $key, $active_plugins, true ) ) ? 1 : 0,
				);
			}
		}

		// Get all the themes.
		$themes = wp_get_themes();

		// Get the active theme.
		$active_theme = wp_get_theme();

		// Loop of all the themes.
		if ( ! empty( $themes ) ) {
			foreach ( $themes as $key => $theme ) {
				// Add theme data to array response.
				$active_themes_plugins_data['themes'][] = array(
					'name'    => $theme->get( 'Name' ),
					'version' => $theme->get( 'Version' ),
					'status'  => ( $theme->get( 'Name' ) === $active_theme->get( 'Name' ) ) ? 1 : 0,
				);
			}
		}

		return $active_themes_plugins_data;
	}

	/**
	 * Clear development access cache.
	 *
	 * @since 1.4.4
	 * @return array Return error or success.
	 */
	public function clear_development_access_cache() {
		// Return App Instance when authentication is valid.
		$app = $this->authenticate_bbapp_id();

		if ( ! $app ) {
			return $this->return_error( 'invalid_bbapp_id', __( 'BuddyBoss App ID Provided is Invalid.', 'buddyboss-app' ) );
		}

		// Transients should be clear.
		$transient_key = bbapp_get_app_info_transient_key();

		delete_transient( $transient_key );
		delete_site_transient( $transient_key );

		return array( 'success' => true );
	}

	/**
	 * Check plugin is activated on multisite.
	 *
	 * @return bool[]
	 * @since 1.6.3
	 */
	public function app_is_activated_multi_site() {
		if ( is_multisite() ) {
			$bbapp_sitewide_plugins = array();
			$bbapp_plugin_file      = 'buddyboss-app/buddyboss-app.php';

			// Get network-activated plugins.
			foreach ( get_site_option( 'active_sitewide_plugins', array() ) as $key => $value ) {
				$bbapp_sitewide_plugins[] = $key;
			}

			$bbapp_plugins = array_merge( $bbapp_sitewide_plugins, get_option( 'active_plugins' ) );

			// Check if BuddyBoss App is activated.
			if ( in_array( $bbapp_plugin_file, $bbapp_plugins, true ) ) {
				return array( 'success' => true );
			}
		}

		return array( 'success' => false );
	}

	/**
	 * Update core branches.
	 *
	 * @return bool[]|Wp_Error
	 * @since 1.6.8
	 */
	public function bbapp_update_core_branches() {
		// Return App Instance when authentication is valid.
		$app = $this->authenticate_bbapp_id();
		if ( ! $app ) {
			return $this->return_error( 'invalid_bbapp_id', __( 'BuddyBoss App ID Provided is Invalid.', 'buddyboss-app' ) );
		}

		$live_ref = ( ! empty( $_POST['live_ref'] ) ? bbapp_input_clean( wp_unslash( $_POST['live_ref'] ) ) : '' ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Missing
		$dev_ref  = ( ! empty( $_POST['dev_ref'] ) ? bbapp_input_clean( wp_unslash( $_POST['dev_ref'] ) ) : '' ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Missing

		$bbapp_settings = Configure::instance()->get_settings();

		if ( ! empty( $live_ref ) ) {
			$bbapp_settings['publish.general.core.branch'] = $live_ref;
		}

		if ( ! empty( $dev_ref ) ) {
			$bbapp_settings['publish.general.core.dev_branch'] = $dev_ref;
		}

		Configure::instance()->save_settings( $bbapp_settings );

		return array( 'success' => true );
	}

	/**
	 * Update license type and trigger in plugin notifications.
	 *
	 * @return true[]
	 * @since 2.2.80
	 */
	public function bbapp_update_license_type() {
		$license_type = ( ! empty( $_POST['license_type'] ) ? bbapp_input_clean( wp_unslash( $_POST['license_type'] ) ) : '' ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Missing
		$trigger_type = ( ! empty( $_POST['type'] ) ? bbapp_input_clean( wp_unslash( $_POST['type'] ) ) : '' ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Missing
		$license_data = ( ! empty( $_POST['license_data'] ) ? bbapp_input_clean( wp_unslash( $_POST['license_data'] ) ) : '' ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Missing

		if ( ! empty( $license_type ) ) {
			$bbapp_license_type = AppSettings::instance()->get_setting_value( 'app_license_type' );

			if ( ! empty( $bbapp_license_type ) && $license_type !== $bbapp_license_type ) {
				AppSettings::instance()->update_setting_value( 'trigger_build_notice', true );
			}

			ManageApp::instance()->update_license_data(
				array(
					'license_type'      => $license_type,
					'license_transient' => $license_type,
					'license_token'     => ( ! empty( $license_data ) ) ? $license_data : '',
				)
			);
		}

		if ( ! empty( $trigger_type ) && 'upgrade' === $trigger_type ) {
			/**
			 * @todo: Trigger the upgrade in plugin notifications.
			 */
		} elseif ( ! empty( $trigger_type ) && 'downgrade' === $trigger_type ) {
			/**
			 * @todo: Trigger the downgrade in plugin notifications.
			 */
		}

		return array( 'success' => true );
	}
}
