<?php
/**
 * Holds browser authentication functionality.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

use BuddyBossApp\Admin\Settings;

/**
 * Browser auth class.
 */
class BrowserAuth {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Number of invalid attempts.
	 *
	 * @var $invalid_attempt
	 */
	public $invalid_attempt;

	/**
	 * Using Singleton, see instance()
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return object
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		add_action( 'init', array( $this, 'add_restriction' ) );
	}

	/**
	 * Browser authentication initiate.
	 */
	public function add_restriction() {
		$settings = $this->get_settings();

		if ( ! empty( $settings['browser_auth_enabled'] ) ) {
			// If use WP_CLI then authentication should not be check in the WP_CLI.
			if ( defined( 'WP_CLI' ) && WP_CLI ) {
				return;
			}

			$asked_password = $this->submit_process( $settings['browser_auth_password'] );
			if ( isset( $_COOKIE['bbapp_auth_psw'] ) ) {
				$asked_password = sanitize_text_field( wp_unslash( $_COOKIE['bbapp_auth_psw'] ) );
			}

			/**
			 *  1. check browser authentication
			 *  2. keep wp-json always free from restriction.
			 *  3. Keep In-app browser page free form restriction
			 *  4. Keep BuddyBoss App embed url free form restriction (modified code)
			 */
			$request_url = ! empty( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
			if (
				! ( md5( $settings['browser_auth_password'] . AUTH_KEY . SECURE_AUTH_KEY ) === $asked_password )
				&& strpos( $request_url, '/wp-json' ) === false
				&& strpos( $request_url, '/wp-admin' ) === false
				&& strpos( $request_url, '/wp-login.php' ) === false
				&& ( function_exists( 'bbapp_is_loaded_from_inapp_browser' ) && ( ! bbapp_is_loaded_from_inapp_browser() ) )
				&& ( empty( $_GET['bbapp_learner_content'] ) && empty( $_GET['bbapp_learner_video'] ) ) // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				&& ( empty( $_GET['mobile-view-content'] ) ) // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				&& ( empty( $_GET['bbapp_installer_screen'] ) ) // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				&& ( empty( $_GET['bbapp_ios_plist'] ) ) // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				&& ( empty( $_GET['register_ios_device'] ) ) // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				&& empty( $_GET['bbapp-vimeo-embed'] ) // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				&& strpos( $request_url, '/bb-media-preview' ) === false
				&& strpos( $request_url, '/bb-document-preview' ) === false
				&& strpos( $request_url, '/bb-document-player' ) === false
				&& strpos( $request_url, '/bb-video-preview' ) === false
				&& strpos( $request_url, '/bb-video-thumb-preview' ) === false
				&& ( is_user_logged_in() && current_user_can( 'manage_options' ) ) === false
			) {
				header( 'HTTP/1.0 401 Unauthorized' );
				$message = $settings['browser_auth_message'];
				$this->get_auth_page( $message );
				exit;
			}
		}
	}

	/**
	 * Auth credentials submit.
	 *
	 * @param string $browser_auth_password Auth password.
	 *
	 * @return false|string
	 */
	private function submit_process( $browser_auth_password ) {
		$auth_password_post = ( ! empty( $_POST['auth_password'] ) ) ? sanitize_text_field( wp_unslash( $_POST['auth_password'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing

		if ( isset( $_POST['auth_submit'] ) && $auth_password_post ) { //phpcs:ignore WordPress.Security.NonceVerification.Missing
			if ( $auth_password_post === $browser_auth_password ) {
				$auth_password = md5( $browser_auth_password . AUTH_KEY . SECURE_AUTH_KEY );

				setcookie( 'bbapp_auth_psw', $auth_password, 0 );

				return $auth_password;
			} else {
				$this->invalid_attempt = true;
			}
		}

		return false;
	}

	/**
	 * Get Page details to show for site restriction because of browser authentication.
	 *
	 * @param string $message Message.
	 */
	private function get_auth_page( $message ) {
		require bbapp()->plugin_dir . 'views/browser-auth/auth.php';
	}

	/**
	 * Returns the Offline Courses Settings.
	 *
	 * @return array
	 */
	public function get_settings() {
		$settings = Settings::instance()->get_settings();
		$values   = array(
			'browser_auth_enabled'  => $this->get_browser_auth_enabled( $settings ),
			'browser_auth_username' => '',
			'browser_auth_password' => $this->get_browser_auth_password( $settings ),
			'browser_auth_message'  => $this->get_browser_auth_message( $settings ),
		);

		return $values;
	}

	/**
	 * Get browser auth enabled.
	 *
	 * @param array $settings Settings.
	 *
	 * @return string
	 */
	public function get_browser_auth_enabled( $settings ) {
		if ( isset( $settings['browser_auth.enabled'] ) && $settings['browser_auth.enabled'] ) {
			return $settings['browser_auth.enabled'];
		}

		return false;
	}

	/**
	 * Get browser auth password.
	 *
	 * @param array $settings Settings.
	 *
	 * @return string
	 */
	public function get_browser_auth_password( $settings ) {
		if ( isset( $settings['browser_auth.password'] ) && $settings['browser_auth.password'] ) {
			return trim( $settings['browser_auth.password'] );
		}

		return '';
	}

	/**
	 * Get browser auth message.
	 *
	 * @param array $settings Settings.
	 *
	 * @return string
	 */
	public function get_browser_auth_message( $settings ) {
		if ( isset( $settings['browser_auth.message'] ) && $settings['browser_auth.message'] ) {
			return trim( $settings['browser_auth.message'] );
		}

		return __( 'You need a password to view this site!', 'buddyboss-app' );
	}
}
