<?php
/**
 * Holds branding main class functionality.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use BuddyBossApp\Helpers\BBAPP_File;

/**
 * Branding class.
 */
class Branding {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	public static $instance;

	/**
	 * Get the instance of class
	 *
	 * @return Branding
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Branding constructor.
	 */
	public function __construct() {

	}

	/**
	 * Returns the branding upload dir.
	 *
	 * @return string
	 */
	public function get_branding_upload_dir() {
		$upload_dir = wp_upload_dir();
		$dir        = $upload_dir['basedir'] . '/bbapp/branding';

		BBAPP_File::create_dir( $dir );

		$dir = $dir . '/' . get_current_blog_id();
		BBAPP_File::create_dir( $dir );

		$dir = str_replace( '/', DIRECTORY_SEPARATOR, $dir );

		return $dir;
	}

	/**
	 * Returns the branding upload url
	 *
	 * @return string
	 */
	public function get_branding_upload_url() {
		$upload_dir = wp_upload_dir();

		$dir = $upload_dir['baseurl'] . '/bbapp/branding';

		$dir = $dir . '/' . get_current_blog_id();

		return $dir;
	}

	/**
	 * Returns the app branding fields.
	 * get_app_upload_fields
	 *
	 * @return array
	 */
	public function get_app_branding_fields() {
		$assets_url           = 'https://buddyboss-app-files.s3.amazonaws.com/';
		$example_launch_png   = $assets_url . 'LaunchScreenSafeZone.png';
		$example_launch_psd   = $assets_url . 'LaunchScreenSafeZone.psd';
		$example_login_png    = $assets_url . 'LoginScreenSafeZone.png';
		$example_login_psd    = $assets_url . 'LoginScreenSafeZone.psd';
		$example_adaptive_png = $assets_url . 'example-adaptive.png';
		$example_adaptive_psd = $assets_url . 'example-adaptive.psd';
		$build_link           = bbapp_get_super_admin_url( 'admin.php?page=bbapp-build&setting=request-build' );

		$return = array(
			'home_screen_notice'            => array(
				'html' => '<div class="desc"><p>' . __( 'The first tab in your Tab Bar will display as the Home Screen. When scrolling up on the Home Screen, the navigation bar will collapse and can optionally display a logo. The image can be square or wide, and can include transparency. For best results upload a logo with maximum dimensions of <strong>250</strong> x <strong>65</strong> pixels.', 'buddyboss-app' ) . '</p></div>',
				'type' => 'html',
			),
			'home_screen_logo'              => array(
				'label'                   => __( 'Home Screen Logo', 'buddyboss-app' ),
				'width'                   => 1200,
				'height'                  => 180,
				'fixedratio'              => false,
				'format'                  => 'png',
				'type'                    => 'upload',
				'class'                   => 'croppie-home_screen_logo',
				'boundary_style_override' => array(
					'width'  => '100%',
					'height' => '15%',
				),
			),
			'home_screen_logo_v1'           => array(
				'label'                   => __( 'Home Screen Logo', 'buddyboss-app' ),
				'width'                   => 250,
				'height'                  => 65,
				'fixedratio'              => false,
				'format'                  => 'png',
				'type'                    => 'upload',
				'class'                   => 'croppie-home_screen_logo_v1',
				'boundary_style_override' => array(
					'width'  => '250px',
					'height' => '65px',
				),
				'viewport' => array(
					'width'  => 250,
					'height' => 65,
					'type'   => 'square'
				)
			),
			'login_notice'                  => array(
				'html' => '<div class="desc"><p>' . __( 'The Login, Forgot Password and related authentication screens can optionally include a background image, and the Login Screen can also include a logo. If no background image is added, they will display the Login Background Color.', 'buddyboss-app' ) . '</p><p>' . sprintf( __( "For best results upload a logo with max dimensions of <strong>1200</strong> x <strong>774</strong> pixels, and a background image sized at <strong>2732</strong> x <strong>2732</strong> pixels. Download a login screen template in <a href='%1\$s' target='_blank'>PNG</a> or <a href='%2\$s' target='_blank'>PSD</a> format to help design your login background, keeping all important graphics within the safety zone.", 'buddyboss-app' ), esc_url( $example_login_png ), esc_url( $example_login_psd ) ) . '</p></div>',
				'type' => 'html',
			),
			'login_logo'                    => array(
				'label'                   => __( 'Login Logo', 'buddyboss-app' ),
				'width'                   => 1200,
				'height'                  => 774,
				'fixedratio'              => false,
				'format'                  => 'png',
				'type'                    => 'upload',
				'class'                   => 'croppie-login_logo',
				'boundary_style_override' => array(
					'width'  => '100%',
					'height' => '64.5%',
				),
			),
			'login_background_img'          => array(
				'label'                   => __( 'Login Background', 'buddyboss-app' ),
				'width'                   => 2732,
				'height'                  => 2732,
				'fixedratio'              => false,
				'format'                  => 'png',
				'type'                    => 'upload',
				'class'                   => 'croppie-login_background_img',
				'boundary_style_override' => array(
					'width'  => '100%',
					'height' => '100%',
				),
			),
			'register_notice'               => array(
				'html' => '<div class="desc"><p>' . __( 'The Register and Activation Code screens can optionally include a background image. If no background image is added, they will display the Register Background Color.', 'buddyboss-app' ) . '</p><p>' . sprintf( __( "For best results upload a background image sized at <strong>%1\$s</strong> x <strong>%2\$s</strong> pixels. Download a register screen template in <a href='%3\$s' target='_blank'>PNG</a> or <a href='%4\$s' target='_blank'>PSD</a> format to help design your register background, keeping all important graphics within the safety zone.", 'buddyboss-app' ), 2732, 2732, $example_launch_png, $example_launch_psd ) . '</p></div>',
				'type' => 'html',
			),
			'register_background_img'       => array(
				'label'                   => __( 'Register Background', 'buddyboss-app' ),
				'width'                   => 2732,
				'height'                  => 2732,
				'fixedratio'              => false,
				'format'                  => 'png',
				'type'                    => 'upload',
				'class'                   => 'croppie-register_background_img',
				'boundary_style_override' => array(
					'width'  => '100%',
					'height' => '100%',
				),
			),
			'splash_screen_build_notice'    => array(
				'html' => '<div class="desc"><p>' . __( 'The Launch Screen appears instantly when your app starts up and is quickly replaced with the app\'s Login Screen. It is a temporary graphic while your app initalizes. For more information see this <a href="https://developer.apple.com/design/human-interface-guidelines/ios/visual-design/launch-screen/" target="_blank">Apple guideline</a>.', 'buddyboss-app' ) . '</p><p>' . sprintf( __( "Download a launch screen template in <a href='%1\$s' target='_blank'>PNG</a> or <a href='%2\$s' target='_blank'>PSD</a> format to help design your Launch Screen, keeping all important graphics within the safety zone. For best results, upload image sized at <strong>2732</strong> x <strong>2732</strong> pixels.", 'buddyboss-app' ), esc_url( $example_launch_png ), esc_url( $example_launch_psd ) ) . '</p></div><div class="bbapp_notice"><p>' . sprintf( "<a href='%1%s'>%2\$s</a>", esc_url( $build_link ), esc_html__( 'Changes will require new iOS and Android builds', 'buddyboss-app' ) ) . '<p></div>',
				'type' => 'html',
			),
			'splash_screen_img'             => array(
				'label'                   => __( 'Launch Screen Background', 'buddyboss-app' ),
				'width'                   => 2732,
				'height'                  => 2732,
				'fixedratio'              => false,
				'format'                  => 'png',
				'type'                    => 'upload',
				'class'                   => 'croppie-splash_screen_img',
				'boundary_style_override' => array(
					'width'  => '100%',
					'height' => '100%',
				),
			),
			'splash_show_spinner'           => array(
				'label'          => __( 'Launch Screen Spinner', 'buddyboss-app' ),
				'label_checkbox' => __( 'Show Spinner', 'buddyboss-app' ),
				'desc'           => sprintf( __( 'Display a spinner at the bottom of your launch screen.', 'buddyboss-app' ), $example_launch_png ),
				'type'           => 'checkbox',
			),
			'splash_spinner_color'          => array(
				'label_checkbox' => __( 'Spinner Color', 'buddyboss-app' ),
				'desc'           => __( 'Color of the spinner shown in launch screen.', 'buddyboss-app' ),
				'type'           => 'colorpicker',
				'default_value'  => '#ffffff',
			),
			'app_icon_bg'                   => array(
				'html' => '<div class="desc"><p>' . __( 'App Icons display on your device\'s home screen, and for iOS will be used in the App Store. For Android, you will upload a separate icon for the Google Play Store while publishing. In modern versions of Android, this app icon will display with a white margin around it, which may not look nice depending on your icon style. Make sure to also add an Android Adaptive Icon to be used in modern versions of Android.', 'buddyboss-app' ) . '</p><p>' . __( 'For best practices when designing your app icons, see this <a href="https://developer.apple.com/design/human-interface-guidelines/ios/icons-and-images/app-icon/" target="_blank">guideline</a>. For best results, upload both images sized at <strong>1024</strong> x <strong>1024</strong> pixels. iOS icon should be opaque and Android icon can include transparency.', 'buddyboss-app' ) . '</p></div><div class="bbapp_notice"><p>' . sprintf( __( "<a href='%1\$s'>%2\$s</a>", 'buddyboss-app' ), esc_url( $build_link ), esc_html__( 'Changes will require new iOS and Android builds.', 'buddyboss-app' ) ) . '<p></div>',
				'type' => 'html',
			),
			'app_icon_ios'                  => array(
				'label'                   => __( 'iOS App Icon', 'buddyboss-app' ),
				'width'                   => 1024,
				'height'                  => 1024,
				'fixedratio'              => false,
				'format'                  => 'png',
				'type'                    => 'upload',
				'super_admin'             => true,
				'class'                   => 'croppie-app_icon_ios',
				'boundary_style_override' => array(
					'width'  => '100%',
					'height' => '100%',
				),
			),
			'app_icon_android'              => array(
				'label'                   => __( 'Android App Icon', 'buddyboss-app' ),
				'width'                   => 1024,
				'height'                  => 1024,
				'fixedratio'              => false,
				'format'                  => 'png',
				'type'                    => 'upload',
				'super_admin'             => true,
				'class'                   => 'croppie-app_icon_android',
				'boundary_style_override' => array(
					'width'  => '100%',
					'height' => '100%',
				),
			),
			'android_adaptive_bg'           => array(
				'html' => '<div class="desc"><p>' . __( 'Android versions from 2017 and newer require adaptive icons for the best experience. These icons can display in a variety of shapes (customizable by the user on their device) and allow for animated parallax effects in some devices.', 'buddyboss-app' ) . '</p><p>' . __( "The icons require an opaque background layer, with a partially transparent foreground layer. Your logo should be in the Foreground later. For more information see this <a href='https://developer.android.com/guide/practices/ui_guidelines/icon_design_adaptive' target='_blank'>Android guideline</a>.", 'buddyboss-app' ) . '</p><p>' . sprintf( __( "Download an example adaptive icon template in <a href='%1\$s' target='_blank'>PNG</a> or <a href='%2\$s' target='_blank'>PSD</a> format to help design your adaptive icon, keeping all important graphics centered within the safety zone.", 'buddyboss-app' ), esc_url( $example_adaptive_png ), esc_url( $example_adaptive_psd ) ) . '</p><p>' . __( 'For best results, upload both images sized at <strong>1024</strong> x <strong>1024</strong> pixels. The inner <strong>683</strong> px will be visible, with a <strong>170</strong> px outer margin reserved for parallax effects (depending on icon shape). Background Icon should be opaque, and Foreground Icon should include transparency.', 'buddyboss-app' ) . '</p></div><div class="bbapp_notice"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require a new Android build.', 'buddyboss-app' ) ) . '</p></div>',
				'type' => 'html',
			),
			'app_icon_android_adaptive_bg'  => array(
				'label'                   => __( 'Background Icon', 'buddyboss-app' ),
				'width'                   => 1024,
				'height'                  => 1024,
				'fixedratio'              => false,
				'format'                  => 'png',
				'type'                    => 'upload',
				'super_admin'             => true,
				'class'                   => 'croppie-app_icon_android_adaptive_bg',
				'boundary_style_override' => array(
					'width'  => '100%',
					'height' => '100%',
				),
			),
			'app_icon_android_adaptive_fg'  => array(
				'label'                   => __( 'Foreground Icon', 'buddyboss-app' ),
				'width'                   => 1024,
				'height'                  => 1024,
				'fixedratio'              => false,
				'format'                  => 'png',
				'type'                    => 'upload',
				'super_admin'             => true,
				'class'                   => 'croppie-app_icon_android_adaptive_fg',
				'boundary_style_override' => array(
					'width'  => '100%',
					'height' => '100%',
				),
			),
			'notification_icon'             => array(
				'html' => '<div class="desc"><p>' . __( 'Notification icons appear to users in different locations, including the status bar, notification center and lock screen. Android requires a dedicated notification icon graphic, while iOS just uses the App Icon. For best results, upload image sized at <strong>96</strong> x <strong>96</strong> pixels. Image can optionally include transparency.', 'buddyboss-app' ) . '</p></div><div class="bbapp_notice"><p>' . sprintf( __( "<a href='%1\$s'>%2\$s</a>", 'buddyboss-app' ), esc_url( $build_link ), esc_html__( 'Changes will require a new Android build.', 'buddyboss-app' ) ) . '<p></div>',
				'type' => 'html',
			),
			'app_icon_android_notification' => array(
				'label'                   => __( 'Android Notification Icon', 'buddyboss-app' ),
				'width'                   => 96,
				'height'                  => 96,
				'fixedratio'              => false,
				'format'                  => 'png',
				'type'                    => 'upload',
				'super_admin'             => true,
				'class'                   => 'croppie-app_icon_android_notification',
				'boundary_style_override' => array(
					'width'  => '100%',
					'height' => '100%',
				),
			),
		);

		return $return;
	}

	/**
	 * Returns the branding options
	 *
	 * @return array|mixed
	 */
	public function get_options() {
		$data = (array) get_option( 'bbapp_branding' );
		$data = isset( $data ) ? $data : array();

		return $data;
	}

	/**
	 * Updates the branding options
	 *
	 * @param array $options Branding options.
	 */
	public function set_options( $options ) {
		update_option( 'bbapp_branding', $options );
	}

	/**
	 * Return the branding details.
	 *
	 * @return array|\WP_Error
	 */
	public function get_branding() {
		$branding_options = $this->get_options(); // user saved values.
		$format           = array();

		// Styles Colors.
		$styling          = Styling::instance();
		$styling_data     = $styling->get_bbapp_styling();
		$format['styles'] = $styling_data;

		// Files.
		$fields    = $this->get_app_branding_fields();
		$files_dir = $this->get_branding_upload_dir();
		$files_url = $this->get_branding_upload_url();

		foreach ( $fields as $field_name => $field ) {
			switch ( $field['type'] ) {
				case 'upload':
					$filename              = $field_name . '.' . $field['format'];
					$file_hash             = isset( $branding_options['uploads_hash'][ $filename ] ) ? $branding_options['uploads_hash'][ $filename ] : '';
					$value                 = ( file_exists( $files_dir . '/' . $filename ) ) ? $files_url . '/' . $filename . '?' . $file_hash : '';
					$format[ $field_name ] = $value;
					break;
				case 'checkbox':
					$format[ $field_name ] = isset( $branding_options[ $field_name ] ) && '1' === $branding_options[ $field_name ] ? true : false;
					break;
				case 'colorpicker':
					$format[ $field_name ] = isset( $branding_options[ $field_name ] ) ? $branding_options[ $field_name ] : '';
					break;
			}
		}

		$format['fonts'] = (object) Typography::instance()->get_fonts_data();

		return $format;
	}

	/**
	 * Returns the branding files.
	 *
	 * @param string $filename  File name.
	 * @param string $file_hash File hash.
	 *
	 * @return string
	 */
	public function get_branding_file( $filename, $file_hash ) {
		$files_dir = $this->get_branding_upload_dir();
		$files_url = $this->get_branding_upload_url();
		$value     = ( file_exists( $files_dir . '/' . $filename ) ) ? $files_url . '/' . $filename . '?' . $file_hash : '';

		return $value;
	}

	/**
	 * Handle output for Branding Endpoint.
	 * Check App Settings Rest Function
	 *
	 * @param string $app_platform Platform name.
	 *
	 * @return \WP_Error
	 */
	public function branding_data( $app_platform ) {
		if ( ! in_array( $app_platform, array( 'android', 'ios' ), true ) ) {
			return new \WP_Error(
				'rest_invalid_app_platform',
				__( 'Invalid Platform', 'buddyboss-app' ),
				array( 'status' => 404 )
			);
		}

		return $this->get_branding();
	}
}
