<?php
/**
 * Abstract class for bookmark.
 *
 * @package BodduBossApp
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Class BB_Bookmark
 */
abstract class BB_Bookmark {

	/**
	 * Class instance.
	 *
	 * @var array
	 */
	private static $instances = array();

	/**
	 * Bookmarks.
	 *
	 * @since 1.7.4
	 *
	 * @var array
	 */
	private $bookmarks = array();

	/**
	 * BB_Bookmark constructor.
	 */
	public function __construct() {
		add_filter( 'bb_bookmark_get_registered_types', array( $this, 'get_registered_types' ), 99, 1 );
		add_filter( 'bb_rest_bookmarks_prepare_links', array( $this, 'rest_prepare_links' ), 99, 3 );
	}

	/**
	 * Class instance called.
	 *
	 * @since 1.7.4
	 * @return mixed
	 */
	public static function instance() {
		$class = get_called_class();
		if ( ! isset( self::$instances[ $class ] ) ) {
			self::$instances[ $class ] = new $class();
			self::$instances[ $class ]->bookmark_setup();
		}

		return self::$instances[ $class ];
	}

	/**
	 * Setup method.
	 *
	 * @since 1.7.4
	 */
	public function bookmark_setup() {
		if ( method_exists( $this, 'setup' ) ) {
			$this->setup();
		}
	}

	/**
	 * Register type.
	 *
	 * @param string $type Bookmark type.
	 * @param array  $args Argument.
	 *
	 * @since 1.7.4
	 */
	public function register_type( $type = false, $args = array() ) {

		$r = bb_parse_args(
			$args,
			array(
				'label'               => array(
					'singular' => '',
					'plural'   => '',
				),
				'items_callback'      => '',
				'rest_links_callback' => '',
			)
		);

		if ( empty( $type ) ) {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Please add type.', 'buddyboss-app' ), '' );
		}

		if ( isset( $r['label'] ) && ! is_array( $r['label'] ) ) {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Please add label', 'buddyboss-app' ), '' );
		}

		if ( isset( $r['items_callback'] ) && empty( $r['items_callback'] ) ) {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Please define condition item callback function.', 'buddyboss-app' ), '' );
		}

		$this->bookmarks[ $type ] = array(
			'label'               => array(
				'singular' => ( ! empty( $r['label']['singular'] ) ? $r['label']['singular'] : $type ),
				'plural'   => ( ! empty( $r['label']['plural'] ) ? $r['label']['plural'] : $type ),
			),
			'items_callback'      => $r['items_callback'],
			'rest_links_callback' => $r['rest_links_callback'],
		);
	}

	/**
	 * Get register types.
	 *
	 * @param array $types Types.
	 *
	 * @since 1.7.4
	 */
	public function get_registered_types( $types = false ) {
		if ( ! empty( $this->bookmarks ) ) {
			foreach ( $this->bookmarks as $type => $bookmark ) {
				$types[ $type ] = $bookmark;
			}
		}

		return $types;
	}

	/**
	 * Save the current bookmark to the database.
	 *
	 * @param array|string $args             {
	 *                                       An array of arguments.
	 *
	 * @type string        $type             The specific bookmark type, used for directory
	 *                                       filtering. 'post', etc.
	 * @type int|bool      $user_id          Optional. The ID of the user associated with the bookmark
	 *                                       item. May be set to false or 0 if the item is not related
	 *                                       to any user. Default: the ID of the currently logged-in user.
	 * @type int           $item_id          Optional. The ID of the associated item.
	 * @type int           $blog_id          Optional. The ID of the associated current blog id for multisite.
	 * @type string        $status           The status associated with
	 *                                       the bookmark item is active or deactivate.
	 * @type string        $date_recorded    Optional. The GMT time, in Y-m-d h:i:s format, when
	 *                                       }
	 *
	 * @since 1.7.4
	 *
	 * @return null|false|int|mixed|WP_Error
	 */
	public static function add( $args ) {
		return bb_add_bookmark( $args );
	}

	/**
	 * Delete the current bookmark.
	 *
	 * @param int $bookmark_id ID of the bookmark to delete.
	 *
	 * @since 1.7.4
	 *
	 * @return bool True on success, false on failure.
	 */
	public static function delete( $bookmark_id ) {
		return bb_delete_bookmark( $bookmark_id );
	}


	/**
	 * Query for bookmarks.
	 *
	 * @param array       $args                   {
	 *                                            An array of optional arguments.
	 *
	 * @type array|string $type                   Optional. Array or comma-separated list of bookmark types.
	 *                                            'post'.
	 * @type int          $item_id                Optional. Get bookmark site wise. Default current site ID.
	 *                                            Default: null.
	 * @type int          $user_id                Optional. If provided, results will be limited to bookmark.
	 *                                            Default: false.
	 * @type int          $per_page               Optional. Number of items to return per page of results.
	 *                                            Default: null (no limit).
	 *                                            }
	 *
	 * @param bool        $force_cache            Bypass the cache if true.
	 *
	 * @since 1.7.4
	 *
	 * @return array {
	 * @type array        $bookmarks              Array of bookmark objects returned by the
	 *                                            paginated query. (IDs only if `fields` is set to `id`.)
	 * @type int          $total                  Total count of all bookmarks matching non-
	 *                                            paginated query params.
	 *                                            }
	 */
	public static function get( $args = array(), $force_cache = false ) {
		return bb_get_bookmarks( $args, $force_cache );
	}

	/**
	 * Filter links prepared for the REST response.
	 *
	 * @param array           $links   The prepared links of the REST response.
	 * @param object          $item    BB_Bookmark object.
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since 1.7.4
	 *
	 * @return mixed
	 */
	public function rest_prepare_links( $links, $item, $request ) {
		$type_data = bb_get_bookmark_types();
		if (
			! empty( $type_data ) &&
			! empty( $type_data[ $item->type ]['rest_links_callback'] ) &&
			is_callable( $type_data[ $item->type ]['rest_links_callback'] )
		) {
			$links = call_user_func( $type_data[ $item->type ]['rest_links_callback'], $links, $item, $request );
		}

		return $links;
	}

	/*********** Abstract methods **********/
	/**
	 * Do all register and initiate from this function when writing extend class.
	 *
	 * @return mixed
	 */
	abstract public function setup();
}
