<?php
/**
 * Bookmark class for Posts.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

use BB_Bookmark;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Class Bookmark_Posts
 */
class Bookmark_Posts extends BB_Bookmark {

	/**
	 * BB_Bookmark constructor.
	 */
	public function __construct() {
		parent::__construct();
		add_filter( 'bb_bookmarks_update_post_type_bookmark_status', array( $this, 'bb_post_type_bookmark_status' ) );
		add_filter( 'bb_bookmark_allow_post_types_filter', array( $this, 'bb_allow_post_types_filter' ) );
		add_filter( 'rest_post_query', array( $this, 'bb_bookmark_filter' ), 10, 2 );
		add_filter( 'rest_post_collection_params', array( $this, 'bb_bookmark_parameters_post_collection' ), 10, 2 );
	}

	/**
	 * Setup method.
	 *
	 * @since 1.7.4
	 * @return mixed|void
	 */
	public function setup() {
		$args = array(
			'label'               => array(
				'singular' => __( 'Post', 'buddyboss-app' ),
				'plural'   => __( 'Posts', 'buddyboss-app' ),
			),
			'items_callback'      => array( $this, 'post_items_callback' ),
			'rest_links_callback' => array( $this, 'post_rest_links_callback' ),
		);
		$this->register_type( 'post', $args );
	}

	/**
	 * Render callback function when get bookmarks.
	 *
	 * @param array $items Array of bookmark list.
	 *
	 * @since 1.7.4
	 *
	 * @return array
	 */
	public function post_items_callback( $items ) {

		if ( ! empty( $items ) ) {

			$blog_ids = array_unique( wp_list_pluck( $items, 'blog_id' ) );
			$blog_id  = ! empty( $blog_ids ) ? current( $blog_ids ) : get_current_blog_id();

			$switch = false;
			// Switch to given blog_id if current blog is not same.
			if ( is_multisite() && get_current_blog_id() !== $blog_id ) {
				switch_to_blog( $blog_id );
				$switch = true;
			}
			foreach ( $items as $item_key => $item ) {
				$bookmark = bb_parse_args(
					$item,
					array(
						'id'      => 0,
						'user_id' => 0,
						'item_id' => 0,
					)
				);

				if (
					empty( $bookmark['id'] ) ||
					empty( $bookmark['item_id'] )
				) {
					continue;
				}

				// Get the post to check the forum is exists or not?
				$bookmark_post = get_post( $bookmark['item_id'] );

				$data = array();
				if ( empty( $bookmark_post ) || is_wp_error( $bookmark_post ) || ( ! empty( $bookmark_post->post_status ) && ! in_array( $bookmark_post->post_status, array( 'publish' ), true ) ) ) {
					$data['link']                = '';
					$data['item_featured_image'] = array(
						'full'  => '',
						'thumb' => '',
					);
					$data['item_title']          = sprintf(
					/* translators: Bookmark label. */
						__( 'Deleted %s', 'buddyboss-app' ),
						__( 'Post', 'buddyboss-app' )
					);
				} else {
					$data['item_title']                   = get_the_title( $bookmark['item_id'] );
					$data['link']                         = get_permalink( $bookmark['item_id'] );
					$data['item_featured_image']['full']  = get_the_post_thumbnail_url( $bookmark['item_id'], 'full' );
					$data['item_featured_image']['thumb'] = get_the_post_thumbnail_url( $bookmark['item_id'], 'thumbnail' );
					$data['author']                       = bbaap_get_user_display_name( $bookmark_post->post_author );
				}

				// Parse the data.
				$data = bb_parse_args(
					$data,
					array(
						'item_title'          => '',
						'item_featured_image' => array(
							'full'  => '',
							'thumb' => '',
						),
						'author'              => '',
						'link'                => '',
					)
				);

				$items[ $item_key ] = (object) array_merge( (array) $item, $data );
			}

			// Restore current blog.
			if ( $switch ) {
				restore_current_blog();
			}
		}

		return $items;
	}

	/**
	 * Filter links prepared for the REST response.
	 *
	 * @param array            $links   The prepared links of the REST response.
	 * @param object           $item    BB_Bookmark object.
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @since 1.7.4
	 *
	 * @return mixed
	 */
	public function post_rest_links_callback( $links, $item, $request ) {
		$links[ $item->type ] = array(
			'href'       => rest_url( 'wp/v2/posts/' . $item->item_id ),
			'embeddable' => true,
		);

		return $links;
	}

	/**
	 * Allow post type on status update clear cache.
	 *
	 * @param array $post_type Post type array.
	 *
	 * @since 1.7.4
	 *
	 * @return mixed
	 */
	public function bb_post_type_bookmark_status( $post_type ) {
		$post_type[] = 'post';

		return $post_type;
	}

	/**
	 * Allow post type on where filter.
	 *
	 * @param array $post_type Post type array.
	 *
	 * @since 1.7.4
	 *
	 * @return mixed
	 */
	public function bb_allow_post_types_filter( $post_type ) {
		$post_type[] = 'post';

		return $post_type;
	}

	/**
	 * Function to pass bb_bookmark filter.
	 *
	 * @param array            $args    Array of arguments for WP_Query.
	 * @param \WP_REST_Request $request The REST API request.
	 *
	 * @since 1.7.4
	 * @return array|mixed
	 */
	public function bb_bookmark_filter( $args, $request ) {
		$bb_bookmark = $request->get_param( 'bb_bookmark' );
		if ( isset( $bb_bookmark ) ) {
			$args['bb_bookmark'] = (bool) $bb_bookmark;
		}

		return $args;
	}

	/**
	 * Function to add extra parameters to post rest endpoint.
	 *
	 * @param array  $query_params query parameters.
	 * @param string $post_type    Post type.
	 *
	 * @since 1.7.4
	 *
	 * @return mixed
	 */
	public function bb_bookmark_parameters_post_collection( $query_params, $post_type ) {
		$query_params['bb_bookmark'] = array(
			'description'       => __( 'Limit response to posts for specific bookmarked.', 'buddyboss-app' ),
			'type'              => 'boolean',
			'enum'              => array( true, false ),
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $query_params;
	}

}
