<?php
/**
 * Class for bookmark controller.
 *
 * @package BodduBossApp
 */

namespace BuddyBossApp;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Class BB_Bookmark
 */
class BB_Bookmarks {

	/**
	 * Class instance.
	 *
	 * @var array
	 */
	private static $instances = array();

	/**
	 * Bookmarks.
	 *
	 * @since 1.7.4
	 *
	 * @var array
	 */
	private $bookmarks = array();

	/**
	 * BB_Bookmark constructor.
	 */
	public function __construct() {
	}

	/**
	 * Class instance called.
	 *
	 * @since 1.7.4
	 * @return mixed
	 */
	public static function instance() {
		$class = get_called_class();
		if ( ! isset( self::$instances[ $class ] ) ) {
			self::$instances[ $class ] = new $class();
			self::$instances[ $class ]->bookmark_setup();
		}

		return self::$instances[ $class ];
	}

	/**
	 * Save the current bookmark to the database.
	 *
	 * @param object $bookmark_obj Bookmark object.
	 *
	 * @since 1.7.4
	 *
	 * @return null|false|int|mixed|WP_Error
	 */
	public static function add( $bookmark_obj ) {
		global $wpdb;

		// Get table name.
		$bookmarks_tbl = self::get_bookmark_tbl();

		$bookmark_obj->type          = apply_filters( 'bb_bookmarks_type_before_save', $bookmark_obj->type, $bookmark_obj->id );
		$bookmark_obj->user_id       = apply_filters( 'bb_bookmarks_user_id_before_save', $bookmark_obj->user_id, $bookmark_obj->id );
		$bookmark_obj->item_id       = apply_filters( 'bb_bookmarks_item_id_before_save', $bookmark_obj->item_id, $bookmark_obj->id );
		$bookmark_obj->blog_id       = apply_filters( 'bb_bookmarks_blog_id_before_save', $bookmark_obj->blog_id, $bookmark_obj->id );
		$bookmark_obj->status        = apply_filters( 'bb_bookmarks_status_before_save', $bookmark_obj->status, $bookmark_obj->id );
		$bookmark_obj->date_recorded = apply_filters( 'bb_bookmarks_date_recorded_before_save', $bookmark_obj->date_recorded, $bookmark_obj->id );

		/**
		 * Fires before the current bookmarks item gets saved.
		 *
		 * Please use this hook to filter the properties above. Each part will be passed in.
		 *
		 * @param BB_Bookmark $bookmark_obj Current instance of the bookmark item being saved. Passed by reference.
		 *
		 * @since 1.7.4
		 */
		do_action_ref_array( 'bb_bookmarks_before_save', array( &$bookmark_obj ) );

		// Bookmark need Type.
		if ( empty( $bookmark_obj->type ) ) {
			if ( isset( $bookmark_obj->error_type ) && 'wp_error' === $bookmark_obj->error_type ) {
				return new WP_Error( 'bb_bookmark_empty_type', __( 'The type is required to create a bookmark.', 'buddyboss-app' ) );
			} else {
				return false;
			}

			// Bookmark need Item ID.
		} elseif ( empty( $bookmark_obj->item_id ) ) {
			if ( isset( $bookmark_obj->error_type ) && 'wp_error' === $bookmark_obj->error_type ) {
				return new WP_Error( 'bb_bookmark_empty_item_id', __( 'The item ID is required to create a bookmark.', 'buddyboss-app' ) );
			} else {
				return false;
			}
		}

		/**
		 * Fires before the current bookmark item gets saved.
		 *
		 * Please use this filter to validate bookmark request. Each part will be passed in.
		 *
		 * @param bool        $is_validate  True when bookmark request correct otherwise false/WP_Error. Default true.
		 * @param BB_Bookmark $bookmark_obj Current instance of the bookmark item being saved.
		 *
		 * @since 1.7.4
		 */
		$is_validate = apply_filters( 'bb_bookmarks_validate_before_save', true, $bookmark_obj );

		if ( ! $is_validate || is_wp_error( $is_validate ) ) {
			return $is_validate;
		}

		if ( ! empty( $bookmark_obj->id ) ) {
			$sql = $wpdb->update( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$bookmarks_tbl,
				array(
					'blog_id'       => $bookmark_obj->blog_id,
					'user_id'       => $bookmark_obj->user_id,
					'type'          => $bookmark_obj->type,
					'item_id'       => $bookmark_obj->item_id,
					'status'        => $bookmark_obj->status,
					'date_recorded' => $bookmark_obj->date_recorded,
				),
				array( 'id' => $bookmark_obj->id ),
				array( '%d', '%d', '%s', '%d', '%d', '%s' ),
				array( '%d' )
			);
		} else {
			$sql = $wpdb->insert( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$bookmarks_tbl,
				array(
					'blog_id'       => $bookmark_obj->blog_id,
					'user_id'       => $bookmark_obj->user_id,
					'type'          => $bookmark_obj->type,
					'item_id'       => $bookmark_obj->item_id,
					'status'        => $bookmark_obj->status,
					'date_recorded' => $bookmark_obj->date_recorded,
				),
				array( '%d', '%d', '%s', '%d', '%d', '%s' )
			);
		}

		if ( false === $sql ) {
			if ( isset( $bookmark_obj->error_type ) && 'wp_error' === $bookmark_obj->error_type ) {
				return new WP_Error( 'bb_bookmark_cannot_create', __( 'There is an error while adding the bookmark.', 'buddyboss-app' ) );
			} else {
				return false;
			}
		}

		if ( empty( $bookmark_obj->id ) ) {
			$bookmark_obj->id = $wpdb->insert_id;
		}

		/**
		 * Fires after the current bookmark item has been saved.
		 *
		 * @param BB_Bookmark $bookmark_obj Current instance of the bookmark item that was saved. Passed by reference.
		 *
		 * @since 1.7.4
		 */
		do_action_ref_array( 'bb_bookmarks_after_save', array( &$bookmark_obj ) );

		return $bookmark_obj->id;
	}

	/**
	 * Delete the current bookmark.
	 *
	 * @param object $bookmark Bookmark object.
	 *
	 * @since 1.7.4
	 *
	 * @return bool True on success, false on failure.
	 */
	public static function delete( $bookmark ) {
		global $wpdb;

		// Get table name.
		$bookmark_tbl = self::get_bookmark_tbl();

		/**
		 * Fires before the deletion of a bookmarks.
		 *
		 * @param BB_Bookmark $this Current instance of the bookmark item being deleted. Passed by reference.
		 * @param int         $id   ID of bookmark.
		 *
		 * @since 1.7.4
		 */
		do_action_ref_array( 'bb_bookmarks_before_delete_bookmark', array( &$bookmark, $bookmark->id ) );

		// Finally, remove the bookmark entry from the DB.
		$delete = $wpdb->delete( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$bookmark_tbl,
			array( 'id' => $bookmark->id ),
			array( '%d' )
		);

		if ( false === $delete ) {
			return false;
		}

		/**
		 * Fires after the deletion of a bookmarks.
		 *
		 * @param BB_Bookmark $this Current instance of the bookmark item being deleted. Passed by reference.
		 * @param int         $id   ID of bookmark.
		 *
		 * @since 1.7.4
		 */
		do_action_ref_array( 'bb_bookmarks_after_delete_bookmark', array( &$bookmark, $bookmark->id ) );

		return true;
	}

	/**
	 * Update the bookmark items status.
	 *
	 * @param string $type    Type bookmark item.
	 * @param int    $item_id The bookmark item ID.
	 * @param int    $status  The bookmark item status, 1 = active, 0 = inactive.
	 * @param int    $blog_id The site ID. Default current site ID.
	 *
	 * @since 1.7.4
	 *
	 * @return bool
	 */
	public static function update_status( $type, $item_id, $status, $blog_id = 0 ) {
		global $wpdb;

		// Get table name.
		$bookmark_tbl = self::get_bookmark_tbl();

		/**
		 * Fires before the update status of a bookmark.
		 *
		 * @param string $type    Type bookmark item.
		 * @param int    $item_id The bookmark item ID.
		 * @param int    $status  The bookmark item status, 1 = active, 0 = inactive.
		 * @param int    $blog_id The site ID.
		 *
		 * @since 1.7.4
		 */
		do_action_ref_array(
			'bb_bookmarks_before_update_bookmark_status',
			array(
				$type,
				$item_id,
				$status,
				$blog_id,
			)
		);

		$where = array(
			'type'    => $type,
			'item_id' => $item_id,
		);

		if ( ! empty( $blog_id ) ) {
			$where['blog_id'] = $blog_id;
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$is_updated = $wpdb->update(
			$bookmark_tbl,
			array(
				'status' => $status,
			),
			$where
		);

		if ( ! is_int( $is_updated ) ) {
			return false;
		}

		/**
		 * Fires after the update status of a bookmarks.
		 *
		 * @param string $type    Type bookmark item.
		 * @param int    $item_id The bookmark item ID.
		 * @param int    $status  The bookmark item status, 1 = active, 0 = inactive.
		 * @param int    $blog_id The site ID.
		 *
		 * @since 1.7.4
		 */
		do_action_ref_array(
			'bb_bookmarks_after_update_bookmark_status',
			array(
				$type,
				$item_id,
				$status,
				$blog_id,
			)
		);

		return true;
	}

	/**
	 * Query for bookmarks.
	 *
	 * @param array       $args                   {
	 *                                            Array of parameters. All items are optional.
	 *
	 * @type array|string $type                   Optional. Array or comma-separated list of bookmark types.
	 *                                            'Post', etc...
	 *                                            Default: null.
	 * @type int          $blog_id                Optional. Get bookmark site wise. Default current site ID.
	 * @type int          $user_id                Optional. If provided, results will be limited to bookmark.
	 *                                            Default: null.
	 * @type int          $item_id                Optional. If provided, results will be limited to bookmark.
	 *                                            Default: null.
	 * @type bool         $status                 Optional. Get all active bookmark if true otherwise return inactive.
	 *                                            Default: true.
	 * @type string       $order_by               Optional. Property to sort by. 'date_recorded', 'item_id', 'user_id', 'id', 'type'
	 *                                            'total_bookmark_count', 'random', 'include'.
	 *                                            Default: 'date_recorded'.
	 * @type string       $order                  Optional. Sort order. 'ASC' or 'DESC'. Default: 'DESC'.
	 * @type int          $per_page               Optional. Number of items to return per page of results.
	 *                                            Default: null (no limit).
	 * @type int          $page                   Optional. Page offset of results to return.
	 * @type array|string $include                Optional. Array or comma-separated list of bookmark IDs.
	 *                                            Results will include the listed bookmarks. Default: false.
	 * @type array|string $exclude                Optional. Array or comma-separated list of bookmark IDs.
	 *                                            Results will exclude the listed bookmarks. Default: false.
	 * @type string       $fields                 Which fields to return. Specify 'id' to fetch a list of IDs.
	 *                                            Default: 'all' (return BB_Bookmark objects).
	 * @type array|string $include_items          Optional. Array or comma-separated list of bookmark item IDs.
	 *                                            Results will include the listed bookmarks. Default: false.
	 * @type array|string $exclude_items          Optional. Array or comma-separated list of bookmark item IDs.
	 *                                            Results will exclude the listed bookmarks. Default: false.
	 * @type bool         $count                  Optional. Fetch total count of all bookmarks matching non-
	 *                                            paginated query params when it false.
	 *                                            Default: true.
	 * @type bool         $cache                  Optional. Fetch the fresh result instead of cache when it true.
	 *                                            Default: false.
	 *                                            }
	 * @since 1.7.4
	 *
	 * @return array {
	 * @type array        $bookmark               Array of bookmark objects returned by the
	 *                                            paginated query. (IDs only if `fields` is set to `id`.)
	 * @type int          $total                  Total count of all bookmarks matching non-
	 *                                            paginated query params.
	 *                                            }
	 */
	public static function get( $args = array() ) {
		global $wpdb;

		$defaults = array(
			'type'          => array(),
			'blog_id'       => get_current_blog_id(),
			'user_id'       => get_current_user_id(),
			'item_id'       => 0,
			'status'        => 1,
			'order_by'      => 'id',
			'order'         => 'DESC',
			'per_page'      => null,
			'page'          => null,
			'include'       => false,
			'exclude'       => false,
			'fields'        => 'all',
			'include_items' => false,
			'exclude_items' => false,
			'count'         => true,
			'cache'         => true,
		);

		$r = bb_parse_args( $args, $defaults, 'bb_bookmarks_bookmark_get' );

		// Sanitize the column name.
		$r['fields'] = self::validate_column( $r['fields'] );

		// Get the database table name.
		$bookmark_tbl = self::get_bookmark_tbl();

		$results = array();
		$sql     = array(
			'select'     => 'SELECT DISTINCT bm.id',
			'from'       => $bookmark_tbl . ' bm',
			'where'      => '',
			'order_by'   => '',
			'pagination' => '',
		);

		$where_conditions = array();

		if ( ! empty( $r['type'] ) ) {
			if ( ! is_array( $r['type'] ) ) {
				$r['type'] = preg_split( '/[\s,]+/', $r['type'] );
			}
			$r['type']                = array_map( 'sanitize_title', $r['type'] );
			$type_in                  = "'" . implode( "','", $r['type'] ) . "'";
			$where_conditions['type'] = "bm.type IN ({$type_in})";
		}

		if ( ! empty( $r['blog_id'] ) ) {
			$where_conditions['blog_id'] = $wpdb->prepare( 'bm.blog_id = %d', $r['blog_id'] );
		}

		if ( ! empty( $r['user_id'] ) ) {
			$where_conditions['user_id'] = $wpdb->prepare( 'bm.user_id = %d', $r['user_id'] );
		}

		if ( ! empty( $r['item_id'] ) ) {
			$where_conditions['item_id'] = $wpdb->prepare( 'bm.item_id = %d', $r['item_id'] );
		}

		if ( null !== $r['status'] ) {
			$where_conditions['status'] = $wpdb->prepare( 'bm.status = %d', (int) $r['status'] );
		}

		if ( ! empty( $r['include'] ) ) {
			$include                     = implode( ',', wp_parse_id_list( $r['include'] ) );
			$where_conditions['include'] = "bm.id IN ({$include})";
		}

		if ( ! empty( $r['exclude'] ) ) {
			$exclude                     = implode( ',', wp_parse_id_list( $r['exclude'] ) );
			$where_conditions['exclude'] = "bm.id NOT IN ({$exclude})";
		}

		if ( ! empty( $r['include_items'] ) ) {
			$include                           = implode( ',', wp_parse_id_list( $r['include_items'] ) );
			$where_conditions['include_items'] = "bm.item_id IN ({$include})";
		}

		if ( ! empty( $r['exclude_items'] ) ) {
			$exclude                           = implode( ',', wp_parse_id_list( $r['exclude_items'] ) );
			$where_conditions['exclude_items'] = "bm.item_id NOT IN ({$exclude})";
		}

		/* Order/orderby ********************************************/
		$order           = strtoupper( trim( $r['order'] ) );
		$order           = 'DESC' === $order ? 'DESC' : 'ASC';
		$order_by        = 'bm.' . $r['order_by'];
		$sql['order_by'] = "ORDER BY {$order_by} {$order}";

		// Random order is a special case.
		if ( 'rand()' === $order_by ) {
			$sql['order_by'] = 'ORDER BY rand()';
		} elseif ( ! empty( $r['include'] ) && 'in' === $order_by ) { // Support order by fields for generally.
			$field_data      = implode( ',', array_map( 'absint', $r['include'] ) );
			$sql['order_by'] = "ORDER BY FIELD(bm.id, {$field_data})";
		}

		if ( ! empty( $r['per_page'] ) && ! empty( $r['page'] ) && - 1 !== $r['per_page'] ) {
			$sql['pagination'] = $wpdb->prepare( 'LIMIT %d, %d', intval( ( $r['page'] - 1 ) * $r['per_page'] ), intval( $r['per_page'] ) );
		}

		/**
		 * Filters the Where SQL statement.
		 *
		 * @param array $r                Array of parsed arguments for the get method.
		 * @param array $where_conditions Where conditions SQL statement.
		 *
		 * @since 1.7.4
		 */
		$where_conditions = apply_filters( 'bb_bookmarks_get_where_conditions', $where_conditions, $r );

		$where = '';
		if ( ! empty( $where_conditions ) ) {
			$sql['where'] = implode( ' AND ', $where_conditions );
			$where        = "WHERE {$sql['where']}";
		}

		/**
		 * Filters the From SQL statement.
		 *
		 * @param array  $r   Array of parsed arguments for the get method.
		 * @param string $sql From SQL statement.
		 *
		 * @since 1.7.4
		 */
		$sql['from'] = apply_filters( 'bb_bookmarks_get_join_sql', $sql['from'], $r );

		$paged_bookmarks_sql = "{$sql['select']} FROM {$sql['from']} {$where} {$sql['order_by']} {$sql['pagination']}";

		/**
		 * Filters the pagination SQL statement.
		 *
		 * @param string $value Concatenated SQL statement.
		 * @param array  $sql   Array of SQL parts before concatenation.
		 * @param array  $r     Array of parsed arguments for the get method.
		 *
		 * @since 1.7.4
		 */
		$paged_bookmarks_sql = apply_filters( 'bb_bookmarks_get_paged_bookmarks_sql', $paged_bookmarks_sql, $sql, $r );

		$cached = bb_core_get_incremented_cache( $paged_bookmarks_sql, 'bb_bookmarks' );
		if ( false === $cached || false === $r['cache'] ) {
			$paged_bookmarks_ids = $wpdb->get_col( $paged_bookmarks_sql ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			bb_core_set_incremented_cache( $paged_bookmarks_sql, 'bb_bookmarks', $paged_bookmarks_ids );
		} else {
			$paged_bookmarks_ids = $cached;
		}

		if ( 'id' === $r['fields'] ) {
			// We only want the IDs.
			$paged_bookmarks = array_map( 'intval', $paged_bookmarks_ids );
		} else {
			$uncached_bookmark_ids = bb_get_non_cached_ids( $paged_bookmarks_ids, 'bb_bookmarks' );
			if ( $uncached_bookmark_ids ) {
				$bookmark_ids_sql      = implode( ',', array_map( 'intval', $uncached_bookmark_ids ) );
				$bookmark_data_objects = $wpdb->get_results( "SELECT bm.* FROM {$bookmark_tbl} bm WHERE bm.id IN ({$bookmark_ids_sql})" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				foreach ( $bookmark_data_objects as $bookmark_data_object ) {
					wp_cache_set( $bookmark_data_object->id, $bookmark_data_object, 'bb_bookmarks' );
				}
			}

			$all_bookmarks = array();
			foreach ( $paged_bookmarks_ids as $paged_bookmark_id ) {
				$all_bookmarks[] = self::get_single_bookmark( $paged_bookmark_id );
			}

			$present_types = array();
			if ( ! empty( $all_bookmarks ) ) {
				foreach ( $all_bookmarks as $additional_bookmark ) {
					$present_types[ $additional_bookmark->type ][] = (array) $additional_bookmark;
				}
			}

			$paged_bookmarks = array();
			if ( ! empty( $present_types ) ) {
				foreach ( $present_types as $type => $items ) {
					$type_data = bb_get_bookmark_types();
					if ( ! empty( $type_data ) && ! empty( $type_data[ $type ]['items_callback'] ) && is_callable( $type_data[ $type ]['items_callback'] ) ) {
						$items_data      = call_user_func( $type_data[ $type ]['items_callback'], $items );
						$paged_bookmarks = array_merge( $paged_bookmarks, $items_data );
					}
				}
			}

			if ( 'all' !== $r['fields'] ) {
				$paged_bookmarks = array_column( $paged_bookmarks, $r['fields'] );
			}
		}

		// Set in response array.
		$results['bookmarks'] = $paged_bookmarks;

		// If count is true then will get total bookmark counts.
		if ( ! empty( $r['count'] ) ) {
			// Find the total number of bookmarks in the results set.
			$total_bookmarks_sql = "SELECT COUNT(DISTINCT bm.id) FROM {$sql['from']} {$where}";

			/**
			 * Filters the SQL used to retrieve total bookmarks results.
			 *
			 * @param string $total_bookmarks_sql Concatenated SQL statement used for retrieving total bookmarks results.
			 * @param array  $sql                 Array of SQL parts for the query.
			 * @param array  $r                   Array of parsed arguments for the get method.
			 *
			 * @since 1.7.4
			 */
			$total_bookmarks_sql = apply_filters( 'bb_bookmarks_get_total_bookmarks_sql', $total_bookmarks_sql, $sql, $r );

			$cached = bb_core_get_incremented_cache( $total_bookmarks_sql, 'bb_bookmarks' );
			if ( false === $cached || false === $r['cache'] ) {
				$total_bookmarks = (int) $wpdb->get_var( $total_bookmarks_sql ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				bb_core_set_incremented_cache( $total_bookmarks_sql, 'bb_bookmarks', array( $total_bookmarks ) );
			} else {
				$total_bookmarks = (int) ( ! empty( $cached ) ? current( $cached ) : 0 );
			}

			// Set in response array.
			$results['total'] = $total_bookmarks;
		}

		return $results;
	}

	/**
	 * Get single bookmark.
	 *
	 * @param int $id Bookmark id.
	 *
	 * @since 1.7.4
	 *
	 * @return int|\stdClass
	 */
	public static function get_single_bookmark( $id ) {
		global $wpdb;

		// Get table name.
		$bookmark_tbl = self::get_bookmark_tbl();

		// Check cache for bookmark data.
		$bookmark = wp_cache_get( $id, 'bb_bookmarks' );

		// Cache missed, so query the DB.
		if ( false === $bookmark ) {
			$bookmark = $wpdb->get_row( $wpdb->prepare( "SELECT bm.* FROM {$bookmark_tbl} bm WHERE bm.id = %d", $id ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

			wp_cache_set( $id, $bookmark, 'bb_bookmarks' );
		}

		// Bail if no bookmark is found.
		if ( empty( $bookmark ) || is_wp_error( $bookmark ) ) {
			return 0;
		}

		/**
		 * Pre validate the bookmark before fetch.
		 *
		 * @param boolean $validate Whether to check the bookmarks is valid or not.
		 * @param object  $bookmark Bookmark object.
		 *
		 * @since 1.7.4
		 */
		$validate = apply_filters( 'bb_bookmarks_pre_validate', true, $bookmark );

		if ( empty( $validate ) ) {
			return 0;
		}

		// Bookmark found so set up the object variables.
		$bookmark_obj                = new \stdClass();
		$bookmark_obj->id            = (int) $bookmark->id;
		$bookmark_obj->blog_id       = (int) $bookmark->blog_id;
		$bookmark_obj->user_id       = (int) $bookmark->user_id;
		$bookmark_obj->type          = $bookmark->type;
		$bookmark_obj->item_id       = (int) $bookmark->item_id;
		$bookmark_obj->status        = (int) $bookmark->status;
		$bookmark_obj->date_recorded = $bookmark->date_recorded;

		$type_data = bb_get_bookmark_types( $bookmark->type );

		if ( ! empty( $type_data ) && ! empty( $type_data['items_callback'] ) && is_callable( $type_data['items_callback'] ) ) {
			$item_data = call_user_func( $type_data['items_callback'], array( $bookmark ) );

			$item_extra = array(
				'item_title'          => '',
				'item_featured_image' => '',
				'author'              => '',
				'link'                => '',
			);

			if ( ! empty( $item_data ) && ! empty( current( $item_data ) ) ) {
				$item_extra = bb_parse_args( (array) current( $item_data ), $item_extra );
			}

			$bookmark_obj->item_title          = $item_extra['item_title'];
			$bookmark_obj->item_featured_image = $item_extra['item_featured_image'];
			$bookmark_obj->author              = $item_extra['author'];
			$bookmark_obj->link                = $item_extra['link'];
			$bookmark_obj->date_recorded       = $item_extra['date_recorded'];
		}

		return $bookmark_obj;
	}

	/**
	 * Get database table name for bookmark.
	 *
	 * @since 1.7.4
	 *
	 * @return string.
	 */
	public static function get_bookmark_tbl() {
		global $wpdb;

		if ( is_multisite() ) {
			switch_to_blog( 1 );
			$bookmark_tbl = $wpdb->prefix . 'bb_bookmarks';
			restore_current_blog();
		} else {
			$bookmark_tbl = $wpdb->prefix . 'bb_bookmarks';
		}

		return $bookmark_tbl;
	}

	/**
	 * Validate the column name.
	 *
	 * @param string $column Column name of database.
	 *
	 * @since 1.7.4
	 *
	 * @return string.
	 */
	public static function validate_column( $column ) {
		$columns = self::get_tbl_columns();

		if ( 'all' === $column ) {
			return $column;
		} elseif ( in_array( $column, $columns, true ) ) {
			return $column;
		}

		return 'all';
	}

	/**
	 * Supported DB columns.
	 *
	 * @since 1.7.4
	 * @return string[]
	 */
	public static function get_tbl_columns() {
		return array(
			'id',
			'blog_id',
			'user_id',
			'type',
			'item_id',
			'status',
			'date_recorded',
		);
	}

	/**
	 * Filter links prepared for the REST response.
	 *
	 * @param array           $links   The prepared links of the REST response.
	 * @param object          $item    BB_Bookmark object.
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since 1.7.4
	 *
	 * @return mixed
	 */
	public function rest_prepare_links( $links, $item, $request ) {
		$type_data = bb_get_bookmark_types();
		if (
			! empty( $type_data ) &&
			! empty( $type_data[ $item->type ]['rest_links_callback'] ) &&
			is_callable( $type_data[ $item->type ]['rest_links_callback'] )
		) {
			$links = call_user_func( $type_data[ $item->type ]['rest_links_callback'], $links, $item, $request );
		}

		return $links;
	}

}
