<?php
/**
 * Used by Actions functions.
 *
 * @package BuddyBossApp\Bookmark
 */

namespace BuddyBossApp\Bookmark;

use BuddyBossApp\BB_Bookmarks;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Menus Actions class.
 */
class Actions {

	/**
	 * Class instance.
	 *
	 * @since 1.7.4
	 * @var object
	 */
	private static $instance;

	/**
	 * Using Singleton, see instance()
	 *
	 * @since 1.7.4
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.7.4
	 * @return Actions
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Add related class and hooks
	 *
	 * @since 1.7.4
	 */
	public function init() {
		add_action( 'bb_create_bookmark', array( $this, 'bb_bookmarks_reset_cache_incrementor' ) );
		add_action( 'bb_create_bookmark', array( $this, 'bb_bookmarks_reset_cache_incrementor' ) );
		add_action( 'bb_bookmarks_before_save', array( $this, 'bb_bookmarks_clear_cache_for_bookmark' ) );
		add_action( 'bb_bookmarks_after_delete_bookmark', array( $this, 'bb_bookmarks_clear_cache_for_bookmark' ) );
		add_action( 'transition_post_status', array( $this, 'bb_update_bookmark_status' ), 999, 3 );
		add_action( 'bb_bookmarks_after_update_bookmark_status', array( $this, 'bb_bookmarks_clear_cache_after_update_status' ), 10, 2 );
		add_filter( 'posts_where', array( $this, 'bb_posts_where' ), 998, 2 );
	}

	/**
	 * Reset incremental cache for add/delete bookmark.
	 *
	 * @since 1.7.4
	 *
	 * @return void
	 */
	public function bb_bookmarks_reset_cache_incrementor() {
		bb_core_reset_incrementor( 'bb_bookmarks' );
	}

	/**
	 * Clear a cached bookmark item when that item is updated.
	 *
	 * @param BB_Bookmarks $bookmark Bookmark object.
	 *
	 * @since 1.7.4
	 */
	public function bb_bookmarks_clear_cache_for_bookmark( $bookmark ) {
		if ( ! empty( $bookmark->id ) ) {
			wp_cache_delete( $bookmark->id, 'bb_bookmarks' );
		}
		if ( ! empty( $bookmark->type ) && ! empty( $bookmark->item_id ) ) {
			wp_cache_delete( $bookmark->type . '_' . $bookmark->item_id, 'bb_bookmarks' );
		}
	}

	/**
	 * Fires when a post is transitioned from one status to another.
	 *
	 * @param string   $new_status New post status.
	 * @param string   $old_status Old post status.
	 * @param \WP_Post $post       Post object.
	 *
	 * @since 1.7.4
	 */
	public function bb_update_bookmark_status( $new_status, $old_status, $post ) {

		$bookmarks_post_type = apply_filters( 'bb_bookmarks_update_post_type_bookmark_status', array() );

		if ( $new_status !== $old_status && ! empty( $post->post_type ) && in_array( $post->post_type, $bookmarks_post_type, true ) ) {

			$blog_id = 0;
			if ( is_multisite() ) {
				$blog_id = get_current_blog_id();
			}

			$bookmark_status = 0;
			if ( ! empty( $new_status ) && in_array( $new_status, array( 'publish' ), true ) ) {
				$bookmark_status = 1;
			}

			foreach ( $bookmarks_post_type as $bookmark_post_type ) {
				if ( $bookmark_post_type === $post->post_type ) {
					$bookmark_type = $bookmark_post_type;
				}
			}

			bb_bookmarks_update_bookmarks_status( $bookmark_type, $post->ID, $bookmark_status, $blog_id );
		}
	}

	/**
	 * Clear cache while updating the status of the bookmarks.
	 *
	 * @param string $type    Type bookmark item.
	 * @param int    $item_id The bookmark item ID.
	 *
	 * @since 1.7.4
	 *
	 * @return void
	 */
	public function bb_bookmarks_clear_cache_after_update_status( $type, $item_id ) {

		if ( empty( $type ) || empty( $item_id ) ) {
			return;
		}

		$bookmark_ids = bb_get_bookmarks(
			array(
				'type'    => $type,
				'item_id' => $item_id,
				'user_id' => false,
				'fields'  => 'id',
				'status'  => null,
			),
			true
		);

		if ( ! empty( $bookmark_ids['bookmarks'] ) ) {
			bb_core_reset_incrementor( 'bb_bookmarks' );

			foreach ( $bookmark_ids['bookmarks'] as $id ) {
				wp_cache_delete( $id, 'bb_bookmarks' );
			}
		}
	}

	/**
	 * Altering the post join logic to show posts based on bookmark status.
	 *
	 * @param string $where    where condition string.
	 * @param object $wp_query WP_Query.
	 *
	 * @since 1.7.4
	 * @return string
	 */
	public function bb_posts_where( $where, $wp_query ) {
		global $wpdb;

		if ( is_admin() ) {
			return $where;
		}

		$post_types = array();

		/**
		 * Filter to use for allowing post type.
		 *
		 * @param array $post_types Post type name in array.
		 *
		 * @since 1.7.4
		 */
		$allow_post_types = apply_filters( 'bb_bookmark_allow_post_types_filter', $post_types );

		$bb_post_type = $wp_query->query_vars['post_type'];
		if ( ! in_array( $bb_post_type, $allow_post_types, true ) ) {
			return $where;
		}

		if ( false === bbapp_is_app_request() ) {
			return $where;
		}

		if ( ! bbapp_is_rest() ) {
			return $where;
		}
		if ( ! isset( $wp_query->query_vars['bb_bookmark'] ) ) {
			return $where;
		}

		// We will allow the posts user has specially based on status.
		$allowed_posts = bb_bookmarks_has_items_sql_query( array( 'type' => $bb_post_type ) );

		if ( empty( $allowed_posts ) ) {
			return $where;
		}

		$operator        = '';
		$bookmark_status = $wp_query->query_vars['bb_bookmark'];
		if ( true === (bool) $bookmark_status ) {
			$operator = 'IN';
		}
		if ( false === (bool) $bookmark_status ) {
			$operator = 'NOT IN';
		}
		if ( ! empty( $operator ) ) {
			$where .= " AND  ({$wpdb->prefix}posts.ID {$operator} ({$allowed_posts}))"; //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}

		return $where;
	}

}
