<?php
/**
 * BuddyBoss App users devices.
 *
 * @package BuddyBossApp\Auth\UserDevice
 */

namespace BuddyBossApp\Auth\UserDevice;

/**
 * Sync the user devices.
 *
 * @class Users
 */
class Users {

	/**
	 * The single instance of the class.
	 *
	 * @var null $instance
	 */
	private static $instance;

	/**
	 * Get the instance of class
	 *
	 * @return Users
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Class construct
	 */
	public function __construct() {
	}

	/**
	 * Load users hooks.
	 */
	public function load() {
		add_filter( 'manage_users_columns', array( $this, 'user_app_installed_columns' ), 10, 1 );
		add_filter( 'manage_users_sortable_columns', array( $this, 'user_app_sortable_installed_columns' ), 10, 1 );
		add_filter( 'manage_users_custom_column', array( $this, 'user_app_installed_columns_data' ), 10, 3 );

		// Filter WP admin users list table to include users of the specified type.
		add_filter( 'users_pre_query', array( $this, 'users_table_filter_by_app_installed' ), 10, 2 );
	}

	/**
	 * Apps Installed.
	 *
	 * @param array $column User list table column.
	 *
	 * @return mixed
	 */
	public function user_app_installed_columns( $column ) {
		$column['app_installed'] = __( 'Apps Installed', 'buddyboss-app' );

		return $column;
	}

	/**
	 * User app sortable installed columns.
	 *
	 * @param array $column Columns.
	 *
	 * @since 1.6.8
	 * @return mixed
	 */
	public function user_app_sortable_installed_columns( $column ) {
		$column['app_installed'] = esc_html__( 'app_installed', 'buddyboss-app' );

		return $column;
	}

	/**
	 * Filter WP Admin users list table to include users those app isntalled.
	 *
	 * @param array|null     $results Return an array of user data to short-circuit WP's user query
	 *                            or null to allow WP to run its normal queries.
	 * @param \WP_User_Query $query This the WP_User_Query instance (passed by reference).
	 *
	 * @return array
	 */
	public function users_table_filter_by_app_installed( $results, $query ) {
		global $pagenow, $wpdb;

		if ( is_admin() && 'users.php' === $pagenow && isset( $_REQUEST['app_installed'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( isset( $query->query_vars['count_total'] ) && $query->query_vars['count_total'] ) {
				$qv =& $query->query_vars;
				$qv = \WP_User_Query::fill_query_vars( $qv );

				if ( is_array( $qv['fields'] ) ) {
					$qv['fields'] = array_unique( $qv['fields'] );

					$query->query_fields = array();
					foreach ( $qv['fields'] as $field ) {
						$field                 = 'ID' === $field ? 'ID' : sanitize_key( $field );
						$query->query_fields[] = "$wpdb->users.$field";
					}
					$query->query_fields = implode( ',', $query->query_fields );
				} elseif ( 'all' === $qv['fields'] ) {
					$query->query_fields = "$wpdb->users.*";
				} else {
					$query->query_fields = "$wpdb->users.ID";
				}
				$query->query_fields = 'SQL_CALC_FOUND_ROWS DISTINCT ' . $query->query_fields;
			}

			$devices_table      = bbapp_get_network_table( 'bbapp_user_devices' );
			$query->query_from .= " LEFT JOIN $devices_table ud ON ( {$wpdb->users}.ID = ud.user_id )";

			$app_installed = (int) sanitize_text_field( wp_unslash( $_REQUEST['app_installed'] ) ); //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( $app_installed ) {
				$query->query_where .= " AND ud.device_token != ''";
			} else {
				$query->query_where .= " AND ( ud.device_token IS NULL or ud.device_token = '' )";
			}
		}

		// Added shorting for app installed.
		$order_by = ! empty( $_REQUEST['orderby'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['orderby'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		if ( 'app_installed' === $order_by ) {
			$order                = ! empty( $_REQUEST['order'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['order'] ) ) : 'ASC'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$devices_table        = bbapp_get_network_table( 'bbapp_user_devices' );
			$query->query_fields .= ", (SELECT ud.device_token FROM {$devices_table} AS ud WHERE {$wpdb->users}.ID = ud.user_id ORDER BY ud.device_token DESC LIMIT 1) AS app_installed";
			$query->query_orderby = "ORDER BY app_installed {$order}";
		}

		return $results;
	}

	/**
	 * App install display.
	 *
	 * @param int    $value       Column value.
	 * @param string $column_name Column name.
	 * @param int    $user_id     Coumn user id.
	 *
	 * @return int
	 */
	public function user_app_installed_columns_data( $value, $column_name, $user_id ) {
		switch ( $column_name ) {
			case 'app_installed':
				return $this->app_installed_html( $user_id );
			default:
		}

		return $value;
	}

	/**
	 * App install html.
	 *
	 * @param int $user_id User id.
	 *
	 * @return string
	 */
	public function app_installed_html( $user_id ) {
		$app_installed = $this->user_has_app_installed( $user_id );
		$html          = '';

		if ( $app_installed ) {
			$html .= '<div class="app-users app-installed"><a href="' . add_query_arg(
				array(
					'page'    => 'bbapp-user-devices',
					'user_id' => $user_id,
				),
				admin_url( 'users.php' )
			) . '">';
			/* translators: %s: Number of device(s).  */
			$html .= '<span>' . sprintf( _n( '%s Device', '%s Devices', $app_installed, 'buddyboss-app' ), $app_installed ) . '</span>';
			$html .= '</a></div>';
		} else {
			$html .= esc_html__( 'None', 'buddyboss-app' );
		}

		return $html;
	}

	/**
	 * Get user has app install or not.
	 *
	 * @param int $user_id User id.
	 *
	 * @return bool
	 */
	public function user_has_app_installed( $user_id ) {
		global $wpdb;

		$devices_table    = bbapp_get_network_table( 'bbapp_user_devices' );
		$is_app_installed = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(user_id) FROM `$devices_table` WHERE user_id = %d", $user_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		return $is_app_installed;
	}
}
