<?php
/**
 * User device information.
 *
 * @package BuddyBossApp\Auth\UserDevice
 */

namespace BuddyBossApp\Auth\UserDevice;

use DateTime;
use Exception;

if ( ! class_exists( '\WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Render list of user devices
 *
 * Class UserDevicesList
 */
class UserDevicesList extends \WP_List_Table {

	/**
	 * User id.
	 *
	 * @var int $user_id
	 */
	public $user_id;

	/** Class constructor */
	public function __construct() {

		parent::__construct(
			array(
				'singular' => __( 'User Device', 'buddyboss-app' ), // singular name of the listed records.
				'plural'   => __( 'User Devices', 'buddyboss-app' ), // plural name of the listed records.
				'ajax'     => false, // does this table support ajax?
				'screen'   => 'user_devices', // load push data in custom screen.
			)
		);

		add_action( 'admin_footer', array( &$this, 'admin_header' ) );
		add_action( 'bbapp_admin_update_user_device', array( &$this, 'user_devices_actions' ), 10, 4 );
		add_action( 'after_bbapp_user_devices_list', array( $this, 'render_remove_all_devices_btn' ), 10, 2 );

	}

	/**
	 * Style list table
	 */
	public function admin_header() {
		$page = ( isset( $_GET['page'] ) ) ? esc_attr( bbapp_input_clean( wp_unslash( $_GET['page'] ) ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		if ( 'bbapp-user-devices' !== $page ) {
			return;
		}
	}

	/**
	 * Trigger logout action from backend side.
	 *
	 * @param string $doaction    Action hook.
	 * @param int    $user_id     User id.
	 * @param array  $request     Request data.
	 * @param string $redirect_to Redirect URL.
	 */
	public function user_devices_actions( $doaction = '', $user_id = 0, $request = array(), $redirect_to = '' ) {

		// Logout from single device.
		if ( 'delete_device' === $doaction ) {

			check_admin_referer( 'delete_device' );

			$redirect_to = remove_query_arg( '_wpnonce', $redirect_to );
			if ( self::delete_user_device( $request['device_id'] ) ) {
				$redirect_to = add_query_arg( 'updated', 'device', $redirect_to );
			} else {
				$redirect_to = add_query_arg( 'error', 'device', $redirect_to );
			}

			wp_safe_redirect( $redirect_to );
			exit;
		}
		// Logout from all devices.
		if ( 'delete_all_device' === $doaction ) {

			check_admin_referer( 'delete_all_device' );

			$redirect_to = remove_query_arg( '_wpnonce', $redirect_to );
			if ( self::delete_user_devices( $request['user_id'] ) ) {
				$redirect_to = add_query_arg( 'updated', 'devices', $redirect_to );
			} else {
				$redirect_to = add_query_arg( 'error', 'devices', $redirect_to );
			}

			wp_safe_redirect( $redirect_to );
			exit;
		}
	}

	/**
	 *  Associative array of columns
	 *
	 * @return array
	 */
	public function get_columns() {
		$columns = array(
			'device'         => __( 'Device Type', 'buddyboss-app' ),
			'data_activated' => __( 'Date Activated', 'buddyboss-app' ),
			'device_token'   => __( 'Device Token', 'buddyboss-app' ),
			'token_date'     => __( 'Token Date', 'buddyboss-app' ),
			'last_active'    => __( 'Last Active', 'buddyboss-app' ),
			'action'         => __( 'Action', 'buddyboss-app' ),
		);

		// Admin can see only device token for other users.
		if ( ! current_user_can( 'administrator' ) ) {
			unset( $columns['device_token'] );
		}

		return $columns;
	}

	/**
	 * Render a column when no column specific method exist.
	 *
	 * @param object $item        Table column item.
	 * @param string $column_name Table column name.
	 *
	 * @return mixed
	 * @throws Exception Date and time exception.
	 */
	public function column_default( $item, $column_name ) {
		/* translators: %1$s is the date and %2$ is the time */
		$date_time_format = sprintf( _x( '%1$s %2$s', 'date_at_time', 'buddyboss-app' ), get_option( 'date_format' ), get_option( 'time_format' ) );

		switch ( $column_name ) {
			case 'device':
				if ( empty( $item->device_model ) ) {
					if ( 'ios' === $item->platform ) {
						echo esc_html__( 'iOS', 'buddyboss-app' );
					} else {
						echo esc_html__( 'Android', 'buddyboss-app' );
					}
				} else {
					echo esc_html( $item->device_model );
				}
				break;
			case 'data_activated':
				echo '0000-00-00 00:00:00' === $item->date_registered ? esc_html__( 'N/A', 'buddyboss-app' ) : esc_html( get_date_from_gmt( $item->date_registered, $date_time_format ) );
				break;
			case 'device_token':
				$device_token_len    = strlen( $item->device_token );
				$device_token_masked = substr( $item->device_token, 0, abs( $device_token_len / 3 ) ) . str_repeat( '*', abs( $device_token_len / 3 ) ) . substr( $item->device_token, abs( $device_token_len / 3 ) * 2 );

				echo esc_html( $device_token_masked );
				break;
			case 'token_date':
				$device_data       = ! empty( $item->data ) ? maybe_unserialize( $item->data ) : array();
				$device_token_date = ( ! empty( $device_data['device_token_date'] ) ) ? $device_data['device_token_date'] : '0000-00-00 00:00:00';

				echo '0000-00-00 00:00:00' === $device_token_date ? esc_html__( 'N/A', 'buddyboss-app' ) : esc_html( get_date_from_gmt( $device_token_date, $date_time_format ) );
				break;
			case 'last_active':
				$date_time_format = get_option( 'date_format' );
				$last_active_date = new DateTime( get_date_from_gmt( $item->date_active ) );
				$today_date       = new DateTime( get_date_from_gmt( gmdate( 'Y-m-d H:i:s' ) ) );
				$interval         = $last_active_date->diff( $today_date );
				$last_active_text = get_date_from_gmt( $item->date_active, $date_time_format );

				if ( 0 === $interval->y && 0 === $interval->m && 0 === $interval->d && 23 >= $interval->h && 59 >= $interval->i && 59 >= $interval->s ) {
					$last_active_text = esc_html__( 'Last 24 hours', 'buddyboss-app' );
				}

				if ( 0 === $interval->y && 0 === $interval->m && 0 === $interval->d && 0 === $interval->h && 59 >= $interval->i && 59 >= $interval->s ) {
					$last_active_text = esc_html__( 'Last hour', 'buddyboss-app' );
				}

				echo esc_html( $last_active_text );
				break;
			case 'action':
				$query_args = array(
					'device_id' => $item->id,
					'user_id'   => $item->user_id,
					'action'    => 'delete_device',
				);

				$referer_url = $this->get_referer_url( $query_args );
				$delete_link = wp_nonce_url( $referer_url, 'delete_device' );
				echo sprintf( '<strong><a href="%s" class="bbapp-confirm">%s</a></strong>', esc_url( $delete_link ), esc_html__( 'Log out', 'buddyboss-app' ) );
				break;

		}
	}

	/**
	 * Handles data query and filter, sorting, and pagination.
	 */
	public function prepare_items() {

		$this->_column_headers = $this->get_column_info();

		$per_page     = $this->get_items_per_page( 'per_page', 10 );
		$current_page = $this->get_pagenum();

		$this->items = $this->get_user_devices( $per_page, $current_page );

		$total_items = $this->get_users_devices_count();

		$this->set_pagination_args(
			array(
				'total_items' => $total_items, // WE have to calculate the total number of items.
				'per_page'    => $per_page, // WE have to determine how many items to show on a page.
			)
		);
	}

	/** Text displayed when no user app data is available */
	public function no_items() {
		esc_html_e( 'No devices found.', 'buddyboss-app' );
	}

	/**
	 * Get the user ID.
	 *
	 * Look for $_GET['user_id']. If anything else, force the user ID to the
	 * current user's ID so they aren't left without a user to edit.
	 *
	 * @return int
	 */
	private function get_user_id() {
		if ( ! empty( $this->user_id ) ) {
			return $this->user_id;
		}

		$this->user_id = (int) get_current_user_id();

		// We'll need a user ID when not on self profile.
		if ( ! empty( $_GET['user_id'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$this->user_id = (int) $_GET['user_id']; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
		}

		return $this->user_id;
	}

	/**
	 * Get Devices
	 *
	 * @param int $per_page    user device per pgae.
	 * @param int $page_number Current page number.
	 *
	 * @return null|array|object
	 */
	public function get_user_devices( $per_page = 10, $page_number = 1 ) {
		global $wpdb;

		$table_name = bbapp_get_network_table( 'bbapp_user_devices' );
		$sql        = $wpdb->prepare( "SELECT * FROM {$table_name} WHERE user_id = %d", $this->get_user_id() ); //phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! empty( $_REQUEST['orderby'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$orderby = 'date_updated';

			$sql .= ' ORDER BY ' . esc_sql( $orderby );
			$sql .= ! empty( $_REQUEST['order'] ) ? ' ' . esc_sql( bbapp_input_clean( wp_unslash( $_REQUEST['order'] ) ) ) : ' ASC'; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		} else {
			$sql .= ' ORDER BY date_updated';
			$sql .= ! empty( $_REQUEST['order'] ) ? ' ' . esc_sql( bbapp_input_clean( wp_unslash( $_REQUEST['order'] ) ) ) : ' DESC'; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		}

		$sql   .= " LIMIT $per_page";
		$sql   .= ' OFFSET ' . ( $page_number - 1 ) * $per_page;
		$result = $wpdb->get_results( $sql ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

		return $result;
	}

	/**
	 * Get Devices count
	 *
	 * @return null|int
	 */
	public function get_users_devices_count() {
		global $wpdb;

		$table_name = bbapp_get_network_table( 'bbapp_user_devices' );
		return $wpdb->get_var( $wpdb->prepare( "SELECT count(*) FROM {$table_name} WHERE user_id = %d", $this->get_user_id() ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Return the user device.
	 *
	 * @param int $_id User device id.
	 *
	 * @return array|bool|null|object|void
	 */
	public function get_user_device( $_id ) {
		global $wpdb;
		static $cache;

		// Return the cache.
		if ( is_array( $cache ) && isset( $cache[ $_id ] ) ) {
			return $cache[ $_id ];
		}

		$table_name = bbapp_get_network_table( 'bbapp_user_devices' );
		$get        = $wpdb->get_row( $wpdb->prepare( "SELECT *FROM {$table_name} WHERE id=%d", $_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		if ( empty( $get ) ) {
			return false;
		}
		$cache[ $_id ] = $get;

		return $get;
	}

	/**
	 * Delete a user app device record.
	 *
	 * @param int $id user device ID.
	 *
	 * @return bool|false|int
	 */
	public static function delete_user_device( $id ) {
		global $wpdb;

		$user_device = ( new self() )->get_user_device( $id );

		if ( empty( $user_device ) ) {
			return false;
		}

		/**
		 * Check permissions.
		 * Only admin user or the owner of device can delete.
		 */
		if ( ! current_user_can( 'manage_options' ) ) {
			return false;
		}

		$_bbapp_jwt_jti = get_user_meta( $user_device->user_id, '_bbapp_jwt_jti', true );
		if ( ! empty( $_bbapp_jwt_jti ) ) {
			foreach ( $_bbapp_jwt_jti as $jwt => $data ) {
				if ( hash( 'sha256', $jwt ) === $user_device->auth_token_hash ) {
					unset( $_bbapp_jwt_jti[ $jwt ] );
				}
			}
			update_user_meta( $user_device->user_id, '_bbapp_jwt_jti', $_bbapp_jwt_jti );
		}

		$table_name = bbapp_get_network_table( 'bbapp_user_devices' );

		return $wpdb->delete( "{$table_name}", array( 'id' => $id ), array( '%d' ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
	}

	/**
	 * Delete a user app device records.
	 *
	 * @param int $user_id user ID.
	 *
	 * @return bool|false|int
	 */
	public static function delete_user_devices( $user_id ) {

		/**
		 * Check permissions.
		 * Only admin user or the owner of devices can delete.
		 */
		if ( ! current_user_can( 'manage_options' ) ) {
			return false;
		}

		global $wpdb;

		delete_user_meta( $user_id, '_bbapp_jwt_jti' );
		$table_name = bbapp_get_network_table( 'bbapp_user_devices' );

		return $wpdb->delete( "{$table_name}", array( 'user_id' => $user_id ), array( '%d' ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
	}

	/**
	 * Logout user from all devices
	 *
	 * @param object $user         User data.
	 * @param object $user_devices User device items.
	 */
	public function render_remove_all_devices_btn( $user, $user_devices ) {

		if ( empty( $user_devices ) || empty( $user_devices->items ) ) {
			return;
		}

		if ( $user instanceof \WP_User ) {
			$user_id = $user->ID;
		} else {
			$user_id = (int) get_current_user_id();
			// We'll need a user ID when not on self profile.
			if ( ! empty( $_GET['user_id'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$user_id = (int) $_GET['user_id']; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			}
		}
		$query_args = array(
			'user_id' => $user_id,
			'action'  => 'delete_all_device',
		);

		$community_url = $this->get_referer_url( $query_args );
		$delete_link   = wp_nonce_url( $community_url, 'delete_all_device' );
		echo sprintf( '<a href="%s" class="page-title-action bbapp-lgout-all-devices bbapp-confirm">%s</a>', esc_url( $delete_link ), esc_html__( 'Log Out Everywhere', 'buddyboss-app' ) );
	}

	/**
	 * Get referer url.
	 *
	 * @param array $query_args Query args.
	 *
	 * @since 1.6.9
	 *
	 * @return string
	 */
	public function get_referer_url( $query_args ) {
		if ( ! empty( $_REQUEST['wp_http_referer'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$wp_http_referer               = bbapp_input_clean( wp_unslash( $_REQUEST['wp_http_referer'] ) ); //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$wp_http_referer               = remove_query_arg( array( 'action', 'updated' ), $wp_http_referer );
			$wp_http_referer               = wp_validate_redirect( esc_url_raw( $wp_http_referer ) );
			$query_args['wp_http_referer'] = urlencode_deep( $wp_http_referer );
		}

		$referer_url = add_query_arg( $query_args, add_query_arg( array( 'page' => 'bbapp-user-devices' ), bbapp_get_admin_url( 'users.php' ) ) );

		return $referer_url;
	}
}
