<?php
/**
 * Holds user device related functionality.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp\Auth;

use BuddyBossApp\Auth\UserDevice\Users;
use mysqli_result;
use WP_Error;
use WP_REST_Request;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * User device class.
 */
class UserDevice {
	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * User device table name.
	 *
	 * @var string $table_name
	 */
	protected $table_name = 'bbapp_user_devices';

	/**
	 * UserDevice constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get class instance.
	 *
	 * @since 1.7.7
	 * @return UserDevice
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Filters/Hooks here.
	 *
	 * @since 1.7.7
	 * @return void
	 */
	private function hooks() {
		$this->table_name = bbapp_get_network_table( $this->table_name );

		Users::instance();

		// Helper hook responsible for mark unread counts to 0.
		add_action( 'bbapp_auth_delete_jwt_token_request', array( $this, 'delete_device_for_user' ), 9 );
	}

	/**
	 * Delete user device tokens from database.
	 *
	 * @param WP_REST_Request $request Request data.
	 *
	 * @return bool
	 */
	public function delete_device_for_user( $request ) {
		global $bbapp_var, $wpdb;

		$access_token = trim( $bbapp_var['access_token'] );
		$uuid         = $request->get_param( 'uuid' );
		$secret_key   = $request->get_param( 'secret_key' );

		if ( ! empty( $uuid ) && ! empty( $secret_key ) ) {
			if ( ! is_wp_error( $this->delete_app_device_by_uuid( $uuid, $secret_key ) ) ) {
				return true;
			}
		}

		if ( empty( $access_token ) ) {
			return false;
		}

		$auth_jti          = bbapp_get_jwt_jti( $access_token );
		$access_token_hash = hash( 'sha256', $auth_jti );

		return $wpdb->delete( $this->table_name, array( 'auth_token_hash' => $access_token_hash ), array( '%s' ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	}

	/**
	 * Add new device.
	 *
	 * @param string $secret       Secret key.
	 * @param string $uuid         Device UUID.
	 * @param string $device_token Device token.
	 * @param string $device_id    Device id.
	 * @param string $device_model Device model.
	 *
	 * @since 1.7.7
	 *
	 * @return array|WP_Error
	 */
	public function add_new_device( $secret, $uuid = '', $device_token = '', $device_id = '', $device_model = '' ) {
		global $wpdb;

		$user_id = get_current_user_id();

		if ( ! empty( $device_id ) && ! empty( $user_id ) ) {
			$device = $wpdb->get_results( $wpdb->prepare( "SELECT id FROM $this->table_name WHERE user_id=%d AND device_id=%s", $user_id, $device_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

			if ( $device ) {
				$user_devices  = wp_list_pluck( $device, 'id' );
				$seperated_ids = "'" . implode( "','", $user_devices ) . "'";

				$wpdb->query( "DELETE FROM {$this->table_name} WHERE id IN ($seperated_ids)" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		$insert_data['uuid']            = ( ! empty( $uuid ) ) ? $uuid : \BuddyBossApp\Library\Composer::instance()->uuid_instance()->uuid4()->toString();
		$insert_data['date_active']     = current_time( 'mysql', 1 );
		$insert_data['date_registered'] = current_time( 'mysql', 1 );
		$insert_data['date_updated']    = current_time( 'mysql', 1 );

		if ( ! empty( $device_token ) ) {
			$device_token_hash                       = hash( 'sha256', $device_token );
			$insert_data['device_token']             = $device_token;
			$insert_data['device_token_hash']        = $device_token_hash;
			$insert_device_data['device_token_date'] = current_time( 'mysql', 1 );
		}

		$insert_device_data['secret_key'] = $secret;
		$insert_data['data']              = maybe_serialize( $insert_device_data );

		if ( ! empty( $device_id ) ) {
			$insert_data['device_id'] = $device_id;
		}

		if ( ! empty( $device_model ) ) {
			$insert_data['device_model'] = $device_model;
		}

		$insert_data = $this->prepare_args( $insert_data );
		$insert      = $wpdb->insert( $this->table_name, $insert_data ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery

		if ( $insert ) {
			$response['uuid']         = ( ! empty( $insert_data['uuid'] ) ) ? $insert_data['uuid'] : '';
			$response['secret_key']   = ( ! empty( $insert_device_data['secret_key'] ) ) ? $insert_device_data['secret_key'] : '';
			$response['device_token'] = $device_token;
		} else {
			$response = new WP_Error(
				'bbapp_add_app_device',
				__( 'There is an error while adding the device', 'buddyboss-app' ),
				array(
					'status' => 404,
				)
			);
		}

		return $response;
	}

	/**
	 * Prepare device extra argument.
	 *
	 * @param array $args Argument of device.
	 *
	 * @since 1.7.7
	 *
	 * @return array
	 */
	public function prepare_args( $args = array() ) {
		global $wpdb, $bbapp_var;

		if ( is_user_logged_in() ) {
			$args['user_id'] = get_current_user_id();

			if ( method_exists( '\BuddyBossApp\Auth\Common', 'get_access_token' ) ) {
				$bbapp_auth_common = Common::instance();
				$auth_token        = $bbapp_auth_common->get_access_token();

				/**
				 * Get the auth jti value.
				 * jti a user unique key. on v2 same jti is used for refresh token & access token.
				 * For reference check authentication plugin JWT generation.
				 */
				$auth_jti                = bbapp_get_jwt_jti( $auth_token );
				$auth_token_hash         = hash( 'sha256', $auth_jti );
				$args['auth_token_hash'] = $auth_token_hash;

				/**
				 * Delete any older entry which is not associated with current auth_token or user_id
				 */
				$wpdb->query( $wpdb->prepare( "DELETE FROM {$this->table_name} WHERE  device_token_hash = %s AND (user_id != %d OR auth_token_hash != %s)", $args['device_token_hash'], $args['user_id'], $auth_token_hash ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		if ( ! empty( $bbapp_var ) ) {
			$args['platform'] = $bbapp_var['app_platform'];
			$args['app_ver']  = $bbapp_var['app_code_version'];
		}

		$push_instance = bbapp_get_app_push_instance();

		if ( $push_instance ) {
			$args['push_type'] = $push_instance->push_type();
		}

		return $args;
	}

	/**
	 * Mark as empty if device token exists.
	 *
	 * @param string $device_token Device token.
	 * @param string $uuid         Uuid.
	 *
	 * @since 1.7.7
	 *
	 * @return bool|int|mysqli_result|resource|null
	 */
	public function make_device_token_empty( $device_token, $uuid ) {
		global $wpdb;

		if ( empty( $device_token ) ) {
			return false;
		}

		if ( empty( $uuid ) ) {
			return false;
		}

		$device_items = $wpdb->get_results( $wpdb->prepare( "SELECT * from {$this->table_name} WHERE device_token=%s AND ( uuid != %s OR uuid IS NULL )", $device_token, $uuid ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! empty( $device_items ) ) {
			$device_ids = wp_list_pluck( $device_items, 'id' );

			if ( ! empty( $device_ids ) ) {
				$in_placeholders = implode( ', ', array_fill( 0, count( $device_ids ), '%d' ) );
				$where_clause    = $wpdb->prepare( "id IN ({$in_placeholders})", $device_ids ); //phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

				return $wpdb->query( $wpdb->prepare( "UPDATE {$this->table_name} SET device_token='' WHERE {$where_clause}" ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
		}

		return false;
	}

	/**
	 * Delete app device by UUID.
	 *
	 * @param string $uuid       Device UUID.
	 * @param string $secret_key App device secret key.
	 *
	 * @since 1.7.7
	 *
	 * @return array|WP_Error
	 */
	public function delete_app_device_by_uuid( $uuid, $secret_key ) {
		global $wpdb;

		$get_device_by_uuid = $wpdb->get_row( $wpdb->prepare( "SELECT data from $this->table_name WHERE uuid=%s", $uuid ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$device_data        = ( ! empty( $get_device_by_uuid->data ) ) ? maybe_unserialize( $get_device_by_uuid->data ) : array();

		if ( empty( $device_data ) ) {
			return new WP_Error( 'bbapp_delete_app_device', __( 'No device found with provided UUID.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		if ( empty( $device_data['secret_key'] ) ) {
			return new WP_Error( 'bbapp_delete_app_device', __( 'No secret key found with provided UUID.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		if ( $device_data['secret_key'] !== $secret_key ) {
			return new WP_Error( 'bbapp_delete_app_device', __( 'Secret key doesn\'t match with the provided secret key.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		$delete = $wpdb->delete( $this->table_name, array( 'uuid' => $uuid ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

		if ( $delete ) {
			return array(
				'uuid'    => $uuid,
				'message' => __( 'Device deleted successfully.', 'buddyboss-app' ),
			);
		}

		return new WP_Error( 'bbapp_delete_app_device', __( 'There was a problem to delete app device.', 'buddyboss-app' ), array( 'status' => 404 ) );
	}

	/**
	 * Get Uuid by devices.
	 *
	 * @param string $device_uuid Device uuid.
	 *
	 * @since 1.7.7
	 * @return array
	 */
	public function get_device_ids_by_uuid( $device_uuid ) {
		global $wpdb;

		if ( empty( $device_uuid ) ) {
			return array();
		}

		$device_items = $wpdb->get_results( $wpdb->prepare( "SELECT * from {$this->table_name} WHERE uuid=%s", $device_uuid ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! empty( $device_items ) ) {
			$device_ids = wp_list_pluck( $device_items, 'id' );

			if ( ! empty( $device_ids ) ) {
				return $device_ids;
			}
		}

		return array();
	}
}
