<?php
/**
 * Holds mobile app switch user related functionality.
 *
 * @package BuddyBossApp\Auth
 */

namespace BuddyBossApp\Auth;

/**
 * SwitchUser class.
 */
class SwitchUser {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get class instance
	 *
	 * @return SwitchUser
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * SwitchUser constructor.
	 */
	public function __construct() {
	}

	/**
	 * Load method.
	 *
	 * @since 1.8.20
	 */
	public function load() {
		add_action( 'init', array( $this, 'app_header' ) );
		add_filter( 'rest_post_dispatch_header_cache', array( $this, 'rest_post_dispatch_header_cache' ), 10, 2 );
	}

	/**
	 * Add required headers.
	 */
	public function app_header() {

		// if current user is switched then give info in header.
		$original_id     = $this->get_original_user();
		$current_user_id = get_current_user_id();
		if ( ! empty( $original_id ) ) {
			header( "bbapp-original-user:$original_id" );
			header( "bbapp-switched-user:$current_user_id" );
		}

	}

	/**
	 * Switch and restore user callback function.
	 *
	 * @param bool $switch_back remove meta if user switch back.
	 *
	 * @param bool $user_id user id which we want to switch.
	 *
	 * @return bool|\WP_User
	 */
	public function switch_to_user( $switch_back = false, $user_id = false ) {

		// Extract JTI information from current access token.
		$bbapp_auth_common = Common::instance();
		$auth_token        = $bbapp_auth_common->get_access_token();
		$auth_jti          = bbapp_get_jwt_jti( $auth_token );

		if ( empty( $auth_jti ) ) {
			return false;
		}

		if ( true === $switch_back ) { // restore the original user.

			$original_user_id = $this->get_original_user();

			if ( empty( $original_user_id ) ) {
				return false;
			}

			$switch_data = get_user_meta( $original_user_id, '_bbapp_jwt_switch_user', true );
			$switch_data = ( ! is_array( $switch_data ) ) ? array() : $switch_data;

			// Since while switching back current user ID is probably switched userID.
			unset( $switch_data[ $auth_jti ] ); // Remove this jti information from user switch info.
			update_user_meta( $original_user_id, '_bbapp_jwt_switch_user', $switch_data );

			return $original_user_id;

		} else { // switch the user.

			$user = get_userdata( $user_id );
			if ( ! $user ) {
				return false;
			}

			$user_id = $user->ID;

			// After restore get_original_user will be blank then use current user id.
			$original_user_id = $this->get_original_user();
			if ( empty( $original_user_id ) ) {
				$original_user_id = get_current_user_id();
			}

			/**
			 * Permission Check.
			 */
			if ( ! $this->user_can_do_switch( $original_user_id ) ) {
				return false;
			}

			$switch_data = get_user_meta( $original_user_id, '_bbapp_jwt_switch_user', true );
			$switch_data = ( ! is_array( $switch_data ) ) ? array() : $switch_data;

			$switch_data[ $auth_jti ] = $user_id;
			update_user_meta( $original_user_id, '_bbapp_jwt_switch_user', $switch_data );

			// Override switched user id.
			header( "bbapp-switched-user:$user_id" );

			return $user_id;
		}

	}

	/**
	 * Check if user can do switch or not.
	 *
	 * @param int $user_id user id to check user can able to switch permission.
	 *
	 * @return bool
	 */
	public function user_can_do_switch( $user_id ) {
		if ( is_multisite() ) {
			return is_super_admin( $user_id );
		}

		return user_can( $user_id, 'edit_users' );
	}

	/**
	 * Perform a user switch and give the switched user ID to be used.
	 *
	 * @param int $loggedin_user_id logged in user id to switch user.
	 *
	 * @return bool
	 */
	public function do_switch_user( $loggedin_user_id ) {

		// Check loggedin user can able to switch.
		if ( ! $this->user_can_do_switch( $loggedin_user_id ) ) {
			return false;
		}

		// Extract JTI information from current access token.
		$bbapp_auth_common = Common::instance();
		$auth_token        = $bbapp_auth_common->get_access_token();
		$auth_jti          = bbapp_get_jwt_jti( $auth_token );
		if ( empty( $auth_jti ) ) {
			return false;
		}

		$switch_data = get_user_meta( $loggedin_user_id, '_bbapp_jwt_switch_user', true );

		$switched_user_id = false;

		if ( isset( $switch_data[ $auth_jti ] ) && is_numeric( $switch_data[ $auth_jti ] ) ) {
			$this->set_original_user( $loggedin_user_id );
			$switched_user    = $switch_data[ $auth_jti ];
			$switched_user_id = $switched_user;
		}

		return $switched_user_id;

	}

	/**
	 * Set the original user id.
	 *
	 * @param int $original_user_id original user id to set global.
	 */
	public function set_original_user( $original_user_id ) {
		global $bbapp_switch_user;
		if ( ! is_array( $bbapp_switch_user ) ) {
			$bbapp_switch_user = array();
		}
		$bbapp_switch_user['original_user_id'] = $original_user_id;
	}

	/**
	 * Get the original user id.
	 *
	 * @return mixed
	 */
	public function get_original_user() {
		global $bbapp_switch_user;
		if ( isset( $bbapp_switch_user['original_user_id'] ) ) {
			return $bbapp_switch_user['original_user_id'];
		}

		return false;
	}

	/**
	 * Remove switched user header from cache.
	 *
	 * @param array $header           cache header.
	 * @param array $cached_endpoints registered endpoint cache.
	 *
	 * @since  1.4.9
	 *
	 * @return mixed
	 */
	public function rest_post_dispatch_header_cache( $header, $cached_endpoints ) {
		$original_user_id = $this->get_original_user();
		if ( empty( $original_user_id ) ) {
			unset( $header['bbapp-original-user'] );
			unset( $header['bbapp-switched-user'] );
		}

		return $header;
	}

}
