<?php
/**
 * Holds JWT authentication related functionality.
 *
 * @package BuddyBossApp\Auth
 */

namespace BuddyBossApp\Auth;

use BuddyBossApp\ClientCommon;

/**
 * This class handles most part for JWT.
 * We have not made use of I18n because the library is not loaded when we hook determine current user.
 */
class Jwt {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * JWT data.
	 *
	 * @var array $jwt_iss
	 */
	private $jwt_iss = array();

	/**
	 * Jwt constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get class instance
	 *
	 * @return Jwt
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Hooks/filters here.
	 */
	public function hooks() {
		add_filter( 'determine_current_user', array( $this, 'determine_current_user' ), 99 );
		add_filter( 'show_user_profile', array( $this, 'show_in_profile' ), 99 );
		add_filter( 'edit_user_profile', array( $this, 'show_in_profile' ), 99 );
		add_action( 'wp_ajax_app_destroy_sessions', array( $this, 'destroy_mobile_session_action' ) );
		add_action( 'set_logged_in_cookie', array( $this, 'set_logged_in_cookie' ), 10, 1 );
		$this->plugins_loaded();
	}

	/**
	 * Set some global which depends on bbapp other classes.
	 */
	public function plugins_loaded() {
		$this->jwt_iss['rest'] = get_bloginfo( 'url' ) . ':rest-api';
		$this->remove_cookie_auth_on_jwt();

		if ( bbapp()->is_network_activated() ) {
			$this->jwt_iss['rest'] = network_home_url() . ':rest-api';
		}
	}

	/**
	 * We have to forcefully remove cookie auth to avoid multiple conflicts when access token is provided.
	 * Known Issues.
	 * 1. Auto Logout.
	 * 2. Cache Issues.
	 */
	public function remove_cookie_auth_on_jwt() {
		$access_token  = \BuddyBossApp\Auth\Common::instance()->get_access_token();
		$custom_cookie = \BuddyBossApp\Auth\Common::instance()->get_custom_cookie();

		// if it's rest & access_token is provided thats means client is trying to authenticate using access token.
		// We will forcefully disable cookie based auth in this case.
		$request_url = ! empty( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( $_SERVER['REQUEST_URI'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		if ( strpos( $request_url, 'wp-json/' ) !== false && ! empty( $access_token ) ) {
			/**
			 * These filters were added at
			 * wp-includes/default-filters.php @ 391
			 */
			$_COOKIE            = array(); // Clearing the cookies to make rest stateless.
			$GLOBALS['_COOKIE'] = array(); // Clearing the cookies to make rest stateless.
			remove_filter( 'determine_current_user', 'wp_validate_auth_cookie' );
			remove_filter( 'determine_current_user', 'wp_validate_logged_in_cookie', 20 );
			remove_filter( 'rest_authentication_errors', 'rest_cookie_check_errors', 100 );
		}

		/**
		 * Adding Custom cookies in cookies list.
		 */
		if ( ( bbapp_is_loaded_from_inapp_browser( false ) || isset( $_GET['mobile-view-content'] ) ) && ! empty( $access_token ) && ! empty( $custom_cookie ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$cookie_part = explode( '=', $custom_cookie );

			if ( ! empty( $cookie_part[0] ) ) {
				$_COOKIE[ trim( $cookie_part[0] ) ] = $this->utf8_rawurldecode( $cookie_part[1] );
			}
		}
	}

	/**
	 * Rest cookie to convert in correct format.
	 *
	 * @param string $raw_url_encoded Encoded raw URL.
	 *
	 * @return false|string
	 */
	protected function utf8_rawurldecode( $raw_url_encoded ) {
		$enc = rawurldecode( $raw_url_encoded );

		if ( utf8_encode( utf8_decode( $enc ) ) === $enc ) {
			return rawurldecode( $raw_url_encoded );
		} else {
			return utf8_encode( rawurldecode( $raw_url_encoded ) );
		}
	}

	/**
	 * Will try to authentication the user in between of WP Rest API
	 *
	 * @param int $user_id User id.
	 *
	 * @return mixed
	 */
	public function determine_current_user( $user_id ) {
		$is_switched_user = ClientCommon::instance()->capture_header( 'switchuser' );

		/**
		 * If user is already set than we will not do anything. else things will become recursive.
		 */
		if ( ! empty( $user_id ) && false === $is_switched_user ) {
			return $user_id;
		}

		/**
		 * If on current URL we are validating the token then don't do anything.
		 */
		$request_url = ! empty( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( $_SERVER['REQUEST_URI'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		if ( strpos( $request_url, 'auth/jwt' ) > 0 ) {
			return $user_id;
		}

		$verify = $this->verify_token( false );

		// don't allow authentication using refresh token.
		if ( is_wp_error( $verify ) || ! isset( $verify['user_id'] ) || ( isset( $verify['refresh_token'] ) && true === (bool) $verify['refresh_token'] ) ) {
			return $user_id;
		} else {
			/**
			 * User Switch
			 */
			remove_filter( 'determine_current_user', array( $this, 'determine_current_user' ), 99 ); // remove filter to avoid loops pm switch user.

			$switched_user_id = SwitchUser::instance()->do_switch_user( $verify['user_id'] );

			add_filter( 'determine_current_user', array( $this, 'determine_current_user' ), 99 ); // add filter back filter to avoid loops pm switch user.

			if ( $switched_user_id ) {
				$verify['user_id'] = $switched_user_id;
			}

			/**
			 * End User Switch.
			 */

			// if we found token is fine.
			return $verify['user_id'];
		}
	}

	/**
	 * This function will Verify the token received to the WordPress.
	 *
	 * @param bool $pre_token if required to verify the given token from parameter.
	 *
	 * @return array|\WP_Error
	 */
	public function verify_token( $pre_token = false ) {
		if ( ! $pre_token ) {
			// find token on header.
			$header    = \BuddyBossApp\Auth\Common::instance()->getallheaders();
			$jwt_token = false;

			foreach ( $header as $k => $v ) {
				if ( 'accesstoken' === strtolower( $k ) ) {
					$jwt_token = $v;
					break;
				}
			}

			if ( ! $jwt_token ) {
				return new \WP_Error(
					'invalid_jwt_auth_token',
					'Valid or no jwt auth token found.'
				);
			}
		} else {
			$jwt_token = $pre_token;
		}

		$jwt_secret = $this->get_jwt_secret();

		if ( ! $jwt_secret || empty( $jwt_secret ) ) {
			$_SESSION['bbapp_access_token'] = false;

			return new \WP_Error(
				'not_configured_jwt',
				'Jwt is not currently configured correctly.'
			);
		}

		// Lets Verify The Token.
		try {
			// just be sure thing get working if class is not available..

			$token = \BuddyBossApp\Library\Composer::instance()->firebase_jwt_instance()->firebase_jwt_decode( $jwt_token, $jwt_secret, array( 'HS256' ) );
			if ( ! $this->verify_token_iss( $token ) ) {
				$_SESSION['bbapp_access_token'] = false;

				return new \WP_Error(
					'invalid_auth_token',
					'Jwt token is invalid.'
				);
			}

			$user = $this->get_user_from_token( $token );

			if ( isset( $user->ID ) ) {
				// Check if user is suspended.
				if ( ! empty( $user->ID ) && function_exists( 'bp_is_active' ) && bp_is_active( 'moderation' ) && bp_moderation_is_user_suspended( $user->ID ) ) {
					return new \WP_Error( 'bbapp_rest_user_suspended', __( 'User is suspended. Please contact site administrator.', 'buddyboss-app' ) );
				}
			}

			if ( ! $user ) {
				$_SESSION['bbapp_access_token'] = false;

				return new \WP_Error(
					'invalid_user_jwt_token',
					'Invalid user in token. this error can show in situation if user is deleted from WordPress after generating of this token.'
				);
			}

			// verify if token is valid.
			$data = get_user_meta( $user->ID, '_bbapp_jwt_jti', true );
			$data = ( ! is_array( $data ) ) ? array() : $data;

			if ( ! isset( $data[ $token->jti ] ) ) {
				$_SESSION['bbapp_access_token'] = false;

				return new \WP_Error(
					'invalid_user_jwt_token',
					'Invalid user in token. this error can show in situation if user is deleted from WordPress after generating of this token.'
				);
			}

			$is_refresh_token = false;

			if ( isset( $token->data->refreshtoken ) && true === $token->data->refreshtoken ) {
				$is_refresh_token = true;
			}

			$request_url = ! empty( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( $_SERVER['REQUEST_URI'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
			// We will set cookie & auto authenticate when access token is provided on web version.
			// Setting cookie also allow us to get ride of cache system.
			// So if cache is integrated properly with WordPress system our system will work also.
			if (
				strpos( $request_url, 'wp-json/' ) === false && // We don't allow cookies for rest endpoint request.
				strpos( $request_url, '&cert-nonce=' ) === false// We need remove cookies for certification for app.
			) {

				remove_filter( 'determine_current_user', array( $this, 'determine_current_user' ), 99 ); // Remove filter to avoid loops verifying token.

				wp_set_auth_cookie( $user->ID, false );

				add_filter( 'determine_current_user', array( $this, 'determine_current_user' ), 99 ); // Add filter back filter to avoid loops verifying token.

				/**
				 * We are loading this function because we want to set the cookie if any page mode is available.
				 * Reason for this function to load exactly in this place is that we want to load it after
				 * wp_set_auth_cookie. if the auth cookie is being sent.
				 * Reason is in some hosting with cache wp_set_auth_cookie doesn't work if it's being triggered after
				 * setting some other cookie.
				 *
				 * You can check below in else condition we are calling this function also when wp_set_auth_cookie is
				 * not calling.
				 */
				bbapp_is_loaded_from_inapp_browser( true );

				// Reload the page cookie to take effect properly.
				// This code is needed to properly authenticate user if website is using deep cache system.
				add_action(
					'init',
					function () {
						// Remove our determine_current_user logic to know if a user is successfully logged in with cookie-based authentication. if not redirect the user to allow the browser to set the authentication cookies.
						remove_filter( 'determine_current_user', array( $this, 'determine_current_user' ), 99 );

						if ( ! is_user_logged_in() ) {
							$http_host   = ! empty( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( $_SERVER['HTTP_HOST'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
							$request_url = ! empty( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( $_SERVER['REQUEST_URI'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
							$actual_link = ( is_ssl() ? 'https://' : 'http://' ) . $http_host . $request_url;
							// Adding query param to identify app redirect.
							$actual_link = add_query_arg( 'rtime', time(), $actual_link );
							wp_safe_redirect( $actual_link );
							exit;
						}

						add_filter( 'determine_current_user', array( $this, 'determine_current_user' ), 99 );
					}
				);
			} else {
				/**
				 * Set the cookie for pagemode without any worry.
				 * ? Reference check code under if above.
				 */
				bbapp_is_loaded_from_inapp_browser( true );
			}

			// Get user data.
			$user = get_userdata( $user->ID );

			// Return if User is Spammer.
			if ( 1 === absint( $user->user_status ) ) {
				return new \WP_Error(
					'user_spam',
					'User reported as a Spam.'
				);
			}

			// Looks everything is good.
			return array(
				'refresh_token' => $is_refresh_token,
				'user'          => $user,
				'user_id'       => $user->ID,
				'token'         => $token,
			);
		} catch ( \Exception $e ) {
			$_SESSION['bbapp_access_token'] = false;

			return new \WP_Error(
				'invalid_auth_token',
				'Jwt token is invalid.'
			);
		}
	}

	/**
	 * Will function will return jwt secret key which is used to sign the token
	 *
	 * @return string
	 */
	private function get_jwt_secret() {
		if ( bbapp()->is_network_activated() ) {
			$secret = get_network_option( 1, 'bbapp_jwt_secret' );

			if ( empty( $secret ) ) {
				$secret = wp_generate_password( 43, true, false );

				update_network_option( 1, 'bbapp_jwt_secret', $secret );
			}
		} else {
			$secret = get_option( 'bbapp_jwt_secret' );

			if ( empty( $secret ) ) {
				$secret = wp_generate_password( 43, true, false );
				update_option( 'bbapp_jwt_secret', $secret );
			}
		}

		return $secret;
	}

	/**
	 * Verify the iss part of jwt token
	 *
	 * @param object $token Decoded jwt token.
	 *
	 * @return boolean
	 */
	public function verify_token_iss( $token ) {
		$iss = $token->iss;

		if ( $iss !== $this->jwt_iss['rest'] ) {
			return false;
		} else {
			return true;
		}
	}

	/**
	 * Will return user from the token provided
	 *
	 * @param object $token Provided token much be decoded.
	 *
	 * @return false|\WP_User
	 */
	public function get_user_from_token( $token ) {
		$user_id = $token->data->user->id;

		if ( ! $user_id || empty( $user_id ) ) {
			return false;
		}

		$user = get_userdata( $user_id );

		if ( ! $user || is_wp_error( $user ) ) {
			return false;
		} else {
			return $user;
		}
	}

	/**
	 * Generates JWt token for user securely.
	 *
	 * @param string $username               User name.
	 * @param string $password               Password.
	 * @param array  $args                   Extra arguments.
	 * @param bool   $generate_refresh_token If need to generate refreshed token.
	 * @param bool   $pre_user               If user is already authenticated.
	 *
	 * @return array|bool|\WP_Error
	 */
	public function generate_user_token( $username, $password, $args, $generate_refresh_token = false, $pre_user = false ) {
		// Make sure no auth cookies are set.
		add_filter( 'send_auth_cookies', array( $this, 'prevent_sending_cookie' ) );
		// Avoid unwanted triggers to execute. # memberium had issues on these hooks.
		remove_all_actions( 'wp_login_failed' ); // Remove all action so it won't create any issue.

		if ( ! $pre_user ) {
			$user = wp_authenticate( $username, $password );
		} else {
			$user = $pre_user;
		}

		if ( is_wp_error( $user ) ) {
			if ( isset( $user->errors['invalid_username'][0] ) ) {
				return new \WP_Error( 'invalid_username', wp_strip_all_tags( $user->errors['invalid_username'][0] ), array( 'status' => 500 ) );
			} elseif ( isset( $user->errors['incorrect_password'][0] ) ) {
				return new \WP_Error( 'incorrect_password', wp_strip_all_tags( $user->errors['incorrect_password'][0] ), array( 'status' => 500 ) );
			} else {
				// NOTE : Below must return false so parent wrapper will send 'jwt_token_error'.
				return false;
			}
		}

		// Set global current-user.
		wp_set_current_user( $user->ID );

		$data = $this->generate_jwt_base( $user, $generate_refresh_token, false, $args );

		/**
		 * Fire after user authentication.
		 *
		 * @type string   $user_login User login.
		 * @type \WP_User $user       User data.
		 * @type array    $data       JWT data.
		 */
		do_action( 'bbapp_auth_wp_login', $user->user_login, $user, $data );

		/**
		 * Fires to catch wp_login hooks same as web login.
		 *
		 * @type string   $user_login User login.
		 * @type \WP_User $user       User data.
		 */
		do_action( 'wp_login', $user->user_login, $user );

		return $data;
	}


	/**
	 * Generate JWT base token.
	 *
	 * @param \WP_User $user                   User data.
	 * @param bool     $generate_refresh_token If need to generate new user token.
	 * @param bool     $pre_refresh_token      If need refreshed token.
	 * @param array    $args                   Arguemtns.
	 *
	 * @return array|bool
	 */
	public function generate_jwt_base( $user, $generate_refresh_token = false, $pre_refresh_token = false, $args = array() ) {
		$default_args = array(
			'expire_at_days' => 7, // when the access token will expire.
		);
		$args         = wp_parse_args( $args, $default_args );
		$jwt_secret   = $this->get_jwt_secret();

		if ( ! $jwt_secret || empty( $jwt_secret ) ) {
			return false;
		}

		if ( ! $user ) {
			return false;
		}

		$current_time = time(); // this will be probaly going to be used as issue time.

		/**
		 * Fires before auth expire time.
		 *
		 * @type string $current_time Current time stamp.
		 * @type string $current_time Current time stamp.
		 */
		$not_before = apply_filters( 'bbapp_jwt_auth_notbefore', $current_time, $current_time );

		$expire_at = $current_time + ( DAY_IN_SECONDS * $args['expire_at_days'] ); // Make it expire in one week.

		/**
		 * Fires after auth expire time.
		 *
		 * @type string $expire_at Expire time.
		 * @type string $expire_at Expire time.
		 */
		$expire_at = apply_filters( 'bbapp_jwt_auth_expireon', $expire_at, $expire_at );

		if ( $generate_refresh_token || $pre_refresh_token ) {
			if ( ! $pre_refresh_token ) {
				$refresh_token = $this->generate_refresh_token( $user );

				if ( ! $refresh_token ) {
					return false;
				}

				$refresh_token = $refresh_token['token'];
			} else {
				$refresh_token = $pre_refresh_token;
			}

			if ( ! $refresh_token ) {
				return false;
			}

			$jti = $this->get_token_jti( $refresh_token );
		} else {
			$jti = $this->generate_jti_for_user( $user->data->ID, __( 'App (Legacy)', 'buddyboss-app' ), $expire_at );
		}

		$access_token     = array(
			'iss'  => $this->jwt_iss['rest'],
			'iat'  => $current_time,
			'nbf'  => $not_before,
			'exp'  => $expire_at,
			'jti'  => $jti,
			'data' => array(
				'user' => array(
					'id' => $user->data->ID,
				),
			),
		);
		$jwt_access_token = \BuddyBossApp\Library\Composer::instance()->firebase_jwt_instance()->firebase_jwt_encode( $access_token, $jwt_secret );
		$data             = array(
			'user_email'        => $user->data->user_email,
			'user_nicename'     => $user->data->user_nicename,
			'user_display_name' => $user->data->display_name,
			'user_id'           => $user->data->ID,
			'token'             => $jwt_access_token,
		);

		if ( isset( $refresh_token ) ) {
			$data['refresh_token'] = $refresh_token;
		}

		return $data;
	}

	/**
	 * Get refreshed user token.
	 *
	 * @param \WP_User $user User data.
	 *
	 * @return array|bool
	 *
	 * Generates the Refresh Token.
	 */
	public function generate_refresh_token( $user ) {
		$jwt_secret = $this->get_jwt_secret();

		if ( ! $jwt_secret || empty( $jwt_secret ) ) {
			return false;
		}

		if ( ! $user ) {
			return false;
		}

		$current_time = time();
		$not_before   = apply_filters( 'bbapp_jwt_auth_refresh_token_notbefore', $current_time, $current_time );
		$expire_at    = $current_time + ( MONTH_IN_SECONDS * 6 ); // Make it expire in 6 month.
		$expire_at    = apply_filters( 'bbapp_jwt_auth_refresh_token_expireon', $expire_at, $expire_at );
		$jti          = $this->generate_jti_for_user( $user->data->ID, __( 'App', 'buddyboss-app' ), $expire_at );
		$token        = array(
			'iss'  => $this->jwt_iss['rest'],
			'iat'  => $current_time,
			'nbf'  => $not_before,
			'exp'  => $expire_at,
			'jti'  => $jti,
			'data' => array(
				'user'         => array(
					'id' => $user->data->ID,
				),
				'refreshtoken' => true,
			),
		);
		$jwt_token    = \BuddyBossApp\Library\Composer::instance()->firebase_jwt_instance()->firebase_jwt_encode( $token, $jwt_secret );

		return array(
			'jti'    => $jti,
			'token'  => $jwt_token,
			'expire' => $expire_at,
		);
	}

	/**
	 * This will generate new jti for user and save it into DB so later user can revoke it.
	 *
	 * @param int    $user_id     User id.
	 * @param string $client_info Client information.
	 * @param string $expire      Expire time.
	 *
	 * @return string
	 */
	public function generate_jti_for_user( $user_id, $client_info, $expire ) {
		$data        = get_user_meta( $user_id, '_bbapp_jwt_jti', true );
		$data        = ( ! is_array( $data ) ) ? array() : $data;
		$switch_data = get_user_meta( $user_id, '_bbapp_jwt_switch_user', true );
		$switch_data = ( ! is_array( $switch_data ) ) ? array() : $switch_data;
		$jti         = wp_generate_password( 43, true, false );
		$date        = gmdate( 'Y-m-d H:i:s' );

		if ( ! empty( $_SERVER['HTTP_CLIENT_IP'] ) ) {
			$ip = sanitize_text_field( $_SERVER['HTTP_CLIENT_IP'] ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		} elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			$ip = sanitize_text_field( $_SERVER['HTTP_X_FORWARDED_FOR'] ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		} else {
			$ip = ! empty( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( $_SERVER['REMOTE_ADDR'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		}

		$data[ $jti ]            = array(
			'client' => $client_info,
			'time'   => strtotime( $date ),
			'ip'     => $ip,
			'expire' => $expire,
		);
		$switch_data[ $user_id ] = $jti;

		update_user_meta( $user_id, '_bbapp_jwt_jti', $data );
		update_user_meta( $user_id, '_bbapp_jwt_switch_user', $switch_data );

		/**
		 * Fires after new JWT token entry.
		 *
		 * @param int $user_id User id.
		 * @param string $jti JTI hash.
		 * @param string $data JTI data.
		 */
		do_action( 'bbapp_jwt_new_jti_entry', $user_id, $jti, $data[ $jti ] );

		return $jti;
	}

	/**
	 * Return jti from token.
	 *
	 * @param string $token User token.
	 *
	 * @return mixed
	 */
	public function get_token_jti( $token ) {
		$jwt_secret = $this->get_jwt_secret();

		try {

			$token = \BuddyBossApp\Library\Composer::instance()->firebase_jwt_instance()->firebase_jwt_decode( $token, $jwt_secret, array( 'HS256' ) );
			if ( isset( $token->jti ) ) {
				return $token->jti;
			} else {
				return false;
			}
		} catch ( \Exception $e ) {
			return false;
		}
	}

	/**
	 * Prevent sending cookie to keep stateless.
	 *
	 * @param string $val Value.
	 *
	 * @return bool
	 */
	public function prevent_sending_cookie( $val ) {
		return false;
	}

	/**
	 * Revoke token and make it useless for authentication.
	 *
	 * @param bool $refresh_token Get refresh token.
	 *
	 * @return array|bool|\WP_Error
	 */
	public function revoke_jwt_token( $refresh_token = false ) {
		$token = $this->verify_token( $refresh_token );

		if ( is_wp_error( $token ) || ! $token ) {
			return $token;
		}

		if ( isset( $token['token']->jti ) ) {
			$data = get_user_meta( $token['user_id'], '_bbapp_jwt_jti', true );
			$data = ( ! is_array( $data ) ) ? array() : $data;

			unset( $data[ $token['token']->jti ] );

			update_user_meta( $token['user_id'], '_bbapp_jwt_jti', $data );

			return true;
		}

		return new \WP_Error(
			'invalid_jwt_auth_token',
			'invalid or no jwt auth token found.'
		);
	}

	/**
	 * Will show revoke button on user profile
	 *
	 * @param \WP_User $profile_user User data.
	 *
	 * @return bool
	 */
	public function show_in_profile( $profile_user ) {
		if ( ! current_user_can( 'edit_user', $profile_user->ID ) ) {
			return false;
		}

		?>
		<table class="form-table">
			<tr class="user-sessions-wrap hide-if-no-js">
				<th><?php esc_html_e( 'App Sessions', 'buddyboss-app' ); ?></th>
				<td aria-live="assertive">
					<div class="app-destroy-sessions">
						<button type="button" class="button button-secondary">
							<?php esc_html_e( 'Log Out Everywhere', 'buddyboss-app' ); ?>
						</button>
					</div>
					<p class="description">
						<?php esc_html_e( 'You will logout from every app.', 'buddyboss-app' ); ?>
                    </p>
                </td>
            </tr>
        </table>
        <script>
            jQuery( document ).on( 'click', '.app-destroy-sessions > button', function( e ) {
                $ = jQuery;
                const $this = $( this );

				wp.ajax.post ( 'app_destroy_sessions', {
					nonce: $ ( '#_wpnonce' ).val (),
					user_id: $ ( '#user_id' ).val (),
				} ).done ( function( response ) {
					$this.prop ( 'disabled', true );
					$this.siblings ( '.notice' ).remove ();
					$this.before ( '<div class="notice notice-success inline"><p>' + response.message + '</p></div>' );
				} ).fail ( function( response ) {
					$this.siblings ( '.notice' ).remove ();
					$this.before ( '<div class="notice notice-error inline"><p>' + response.message + '</p></div>' );
				} );

                e.preventDefault();
            } );
        </script>
		<?php
	}

	/**
	 * This function will distroy all session for mobile apps
	 *
	 * @return void
	 */
	public function destroy_mobile_session_action() {
		$user_id_post = ( ! empty( $_POST['user_id'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['user_id'] ) ) : 0; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$user         = get_userdata( (int) $user_id_post );
		$nonce_post   = ( ! empty( $_POST['nonce'] ) ) ? wp_unslash( $_POST['nonce'] ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( $user ) {
			if ( ! current_user_can( 'edit_user', $user->ID ) ) {
				$user = false;
			} elseif ( ! wp_verify_nonce( $nonce_post, 'update-user_' . $user->ID ) ) {
				$user = false;
			}
		}

		if ( ! $user ) {
			wp_send_json_error(
				array(
					'message' => __( 'Could not log out user sessions. Please try again.', 'buddyboss-app' ),
				)
			);
		}

		update_user_meta( $user->ID, '_bbapp_jwt_jti', array() );

		$message = __( 'You are now logged out everywhere.', 'buddyboss-app' );

		wp_send_json_success( array( 'message' => $message ) );
	}

	/**
	 * Set logged in cookie for mobile page mode.
	 *
	 * @since 1.6.1
	 *
	 * @param string $logged_in_cookie Logged in cookie.
	 *
	 * @return void
	 */
	public function set_logged_in_cookie( $logged_in_cookie ) {
		if ( bbapp_is_loaded_from_inapp_browser() || ! empty( $_GET['mobile-view-content'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$_COOKIE[ LOGGED_IN_COOKIE ] = $logged_in_cookie;
		}
	}
}
