<?php
/**
 * Holds auth related common functions.
 *
 * @package BuddyBossApp\Auth
 */

namespace BuddyBossApp\Auth;

use WP_Error;
use WP_User;

/**
 * Contains all common function.
 */
class Common {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Common constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get class instance
	 *
	 * @return Common
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Hooks/filters here.
	 */
	public function hooks() {
	}

	/**
	 * Get password.
	 *
	 * @param string $user_login User login.
	 *
	 * @return bool|WP_Error
	 */
	public static function api_retrieve_password( $user_login ) {
		$errors    = new WP_Error();
		$user_data = false;

		if ( empty( $user_login ) ) {
			$errors->add( 'empty_username', __( 'ERROR: Enter a username or email address.', 'buddyboss-app' ), array( 'status' => 400 ) );
		} elseif ( strpos( $user_login, '@' ) ) {
			$user_data = get_user_by( 'email', trim( $user_login ) );

			if ( empty( $user_data ) ) {
				$errors->add( 'invalid_email', __( 'ERROR: There is no user registered with that email address.', 'buddyboss-app' ), array( 'status' => 400 ) );
			}
		} else {
			$login     = trim( $user_login );
			$user_data = get_user_by( 'login', $login );
		}

		/**
		 * Fires before errors are returned from a password reset request.
		 *
		 * @param WP_Error      $errors    A \WP_Error object containing any errors generated
		 *                                 by using invalid credentials.
		 * @param WP_User|false $user_data WP_User object if found, false if the user does not exist.
		 *
		 * @since 2.1.0
		 * @since 4.4.0 Added the `$errors` parameter.
		 * @sice 2.2.50 Added $user_data parameter.
		 */
		do_action( 'lostpassword_post', $errors, $user_data );

		if ( $errors->get_error_code() ) {
			return $errors;
		}

		if ( ! $user_data ) {
			$errors->add( 'invalidcombo', __( 'ERROR: Invalid username or email.', 'buddyboss-app' ), array( 'status' => 400 ) );

			return $errors;
		}

		// Redefining user_login ensures we return the right case in the email.
		$user_login = $user_data->user_login;
		$user_email = $user_data->user_email;
		$key        = get_password_reset_key( $user_data );

		if ( is_wp_error( $key ) ) {
			return $key;
		}

		$message  = __( 'Someone has requested a password reset for the following account:', 'buddyboss-app' ) . "\r\n\r\n";
		$message .= network_home_url( '/' ) . "\r\n\r\n";
		$message .= sprintf( '%1$s %2$s', esc_html__( 'Username:', 'buddyboss-app' ), $user_login ) . "\r\n\r\n";
		$message .= __( 'If this was a mistake, kindly ignore this email, and no further action is required.', 'buddyboss-app' ) . "\r\n\r\n";
		$message .= __( 'To reset your password, visit the following address:', 'buddyboss-app' ) . "\r\n\r\n";
		$message .= '<' . network_site_url( "wp-login.php?action=rp&key=$key&login=" . rawurlencode( $user_login ), 'login' ) . ">\r\n";

		if ( is_multisite() ) {
			$blogname = $GLOBALS['current_site']->site_name;

			/**
			 * The blogname option is escaped with esc_html on the way into the database
			 * in sanitize_option we want to reverse this for the plain text arena of emails.
			 */
		} else {
			$blogname = wp_specialchars_decode( get_option( 'blogname' ), ENT_QUOTES );
		}

		$title = sprintf( '[%1$s] %2$s', $blogname, esc_html__( 'Password Reset', 'buddyboss-app' ) );

		/**
		 * Password title.
		 *
		 * @param string $title Title.
		 * @param string $user_login user login.
		 * @param object $user_data user data.
		 */
		$title = apply_filters( 'retrieve_password_title', $title, $user_login, $user_data );

		/**
		 * Password message.
		 *
		 * @param string $message Password message.
		 * @param string $key Password key.
		 * @param string $user_login User login.
		 * @param object $user_data User data.
		 */
		$message = apply_filters( 'retrieve_password_message', $message, $key, $user_login, $user_data );

		if ( $message && ! wp_mail( $user_email, wp_specialchars_decode( $title ), $message ) ) {
			return new WP_Error(
				'couldnt_send_mail',
				__( 'The email could not be sent. Possible reason: your host may have disabled the mail() function.', 'buddyboss-app' ),
				array(
					'status' =>
						500,
				)
			);
		}

		return true;
	}

	/**
	 * Get access token.
	 *
	 * @return bool|mixed
	 */
	public function get_access_token() {
		$header    = $this->getallheaders();
		$jwt_token = false;

		foreach ( $header as $k => $v ) {
			if ( 'accesstoken' === strtolower( $k ) ) {
				$jwt_token = $v;
				break;
			}
		}

		return $jwt_token;
	}

	/**
	 * Add Custom Cookie for extra cookie adding.
	 *
	 * @return bool|mixed
	 */
	public function get_custom_cookie() {
		$header        = $this->getallheaders();
		$custom_cookie = false;

		foreach ( $header as $k => $v ) {
			if ( 'bbapp-cookie' === strtolower( $k ) ) {
				$custom_cookie = $v;
				break;
			}
		}

		return $custom_cookie;
	}

	/**
	 * Get all headers.
	 *
	 * @return array|false|string
	 */
	public function getallheaders() {
		if ( function_exists( 'getallheaders' ) ) {
			return getallheaders();
		}

		if ( ! is_array( $_SERVER ) ) {
			return array();
		}

		$headers = array();
		foreach ( $_SERVER as $name => $value ) {
			if ( 'HTTP_' === substr( $name, 0, 5 ) ) {
				$headers[ str_replace( ' ', '-', ucwords( strtolower( str_replace( '_', ' ', substr( $name, 5 ) ) ) ) ) ] = $value;
			}
		}

		return $headers;
	}

	/**
	 * Get app type.
	 *
	 * @return string
	 *
	 * @since 2.2.80
	 */
	public function get_app_type() {
		// Get the header here and block data list to exclude.
		$headers  = Common::instance()->getallheaders();
		$app_type = '';

		foreach ( $headers as $k => $v ) {
			if ( 'apptype' === strtolower( $k ) ) {
				$app_type = $v;
				break;
			}
		}

		return $app_type;
	}
}
