<?php
/**
 * Holds all the auth class load.
 *
 * @package BuddyBossApp\Auth
 */

namespace BuddyBossApp\Auth;

if ( ! function_exists( 'add_action' ) ) {
	die( "Sorry, you can't access this directly - Security established" );
}

/**
 * Class Auth
 */
class Auth {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Auth constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get class instance
	 *
	 * @return Auth
	 */
	public static function instance() {

		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
			self::$instance->initiate();
		}

		return self::$instance;
	}

	/**
	 * Loads when the class is loaded first time.
	 */
	public function initiate() {

		// Register all hooks.
		$this->hooks();
		// Load Auth API.
		\BuddyBossApp\Api\Auth\Main::instance();
		// Load Common function.
		Common::instance();
		Jwt::instance();
		Account::instance();
		UserDevice::instance();
	}

	/**
	 * Register all hooks
	 *
	 * @return void
	 * @since 1.0,0
	 */
	public function hooks() {
	}

	/**
	 * Generate random strong string
	 *
	 * @param int $length Length number.
	 *
	 * @return string
	 */
	public function generate_random_string( $length = 20 ) {
		$characters        = '0123456789abcdefghijklmnopqrstuvwxyz-';
		$characters_length = strlen( $characters );
		$random_string     = '';
		for ( $i = 0; $i < $length; $i ++ ) {
			$random_string .= $characters[ wp_rand( 0, $characters_length - 1 ) ];
		}

		return $random_string;
	}

	/**
	 * Execute when the plugin is activated.
	 */
	public function on_activation() {

		/**
		 * We should't update this key without any reason.
		 * If the key not exists while activating plugin let's auto generate it.
		 */
		$key = get_option( 'jwt_auth_secret_key' );

		if ( empty( $key ) ) {
			update_option( 'jwt_auth_secret_key', $this->generate_random_string() );
		}

	}

	/**
	 * Check if the Registration is available on current site.
	 *
	 * @return bool
	 */
	public function is_registration_available() {
		if ( function_exists( 'bp_enable_site_registration' ) ) {
			return bp_enable_site_registration();
		}

		return get_option( 'users_can_register' );
	}

	/**
	 * Return if Registration is enabled or not.
	 *
	 * @return bool
	 */
	public function is_app_registration_enabled() {
		// If BuddyBoss Platform Registration is Disabled not registration available here to.
		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			if ( ! $this->is_registration_available() ) {
				return false;
			}
		}
		$settings = \BuddyBossApp\Admin\Settings::instance()->get_global_settings();

		return isset( $settings['app_auth.enable_signup'] ) ? $settings['app_auth.enable_signup'] : false;
	}

	/**
	 * Return selected registration forms.
	 *
	 * @return string
	 */
	public function is_app_registration_form() {
		$default_registration = 'wordpress_registration';
		// If BuddyBoss Platform enable then get registration form from app side.
		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			$settings             = \BuddyBossApp\Admin\Settings::instance()->get_global_settings();
			$default_registration = isset( $settings['app_auth.signup_form'] ) ? $settings['app_auth.signup_form'] : 'buddyboss_registration';
		}
		return $default_registration;
	}
}
