<?php
/**
 * Holds mobile app account related functionality.
 *
 * @package BuddyBossApp\Auth
 */

namespace BuddyBossApp\Auth;

use BuddyBossApp\Admin\Configure;

/**
 * App account class.
 */
class Account {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	public static $instance;

	/**
	 * Account constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get class instance
	 *
	 * @return Account
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * BuddyPress/WordPress Registration/activation Integration.
	 */
	public function hooks() {
		add_action( 'init', array( $this, 'app_launch_code_template' ), 99, 4 );
		// Add BuddyBoss App user activation data for new user signup.
		add_filter( 'bp_after_bp_core_signups_add_args_parse_args', array( $this, 'bbapp_activation_code_add' ) );
		add_filter( 'signup_user_meta', array( $this, 'wp_core_bbapp_activation_code_add' ) );
		// Update BuddyPress activation email with ab code.
		add_action( 'bp_send_email', array( $this, 'bbapp_bp_send_email' ), 99, 4 );
		add_filter( 'wpmu_signup_user_notification_email', array( $this, 'wp_bbapp_activation_email' ), 99, 3 );
		// Triggers when user activation process happens.
		add_action( 'bp_core_activated_user', array( $this, 'bp_core_activated_user' ), 99, 3 );
		add_action( 'wpmu_activate_user', array( $this, 'wp_core_activated_user' ), 99, 3 );
		add_action( 'bp_suspend_user_suspended', array( $this, 'ap_suspend_user_suspended' ), 99, 1 );
		add_action( 'make_spam_user', array( $this, 'bbapp_make_spam_user' ), 10, 1 );
	}

	/**
	 * Remove the Access token when user suspended or make spam user from backend.
	 *
	 * @param int $user_id User id.
	 */
	public function ap_suspend_user_suspended( $user_id ) {
		if ( bp_moderation_is_user_suspended( $user_id ) ) {
			$this->bbapp_make_spam_user( $user_id );
		}
	}

	/**
	 * Remove the Access token when user suspended or make spam user from backend.
	 *
	 * @param int $user_id User id.
	 *
	 * @since 1.5.9
	 */
	public function bbapp_make_spam_user( $user_id ) {
		delete_user_meta( $user_id, '_bbapp_jwt_jti' );
		delete_user_meta( $user_id, '_bbapp_jwt_switch_user' );

		if ( function_exists( 'bbapp_notifications' ) ) {
			// Delete devices token.
			bbapp_notifications()->delete_user_devices_by_user_id( $user_id );
		}
	}

	/**
	 * Get the auth settings.
	 *
	 * @return array|mixed
	 */
	public function get_settings() {
		$settings = \BuddyBossApp\Admin\Settings::instance()->get_global_settings();

		if ( ! is_array( $settings ) ) {
			$settings = array();
		}

		return $settings;
	}

	/**
	 * Sets the auth settings.
	 *
	 * @param array $settings settings array.
	 */
	public function set_settings( $settings ) {
		\BuddyBossApp\Admin\Settings::instance()->save_global_settings( $settings );
	}

	/**
	 * Activation Code added when user signup via buddyPress
	 *
	 * @param array $args Activation code arguments.
	 *
	 * @return mixed
	 */
	public function bbapp_activation_code_add( $args ) {
		$args['meta'] = $this->wp_core_bbapp_activation_code_add( $args['meta'] );

		return $args;
	}

	/**
	 * Activation Code added when user signup via WordPress
	 *
	 * @param array $args Activation code arguments.
	 *
	 * @return mixed
	 */
	public function wp_core_bbapp_activation_code_add( $args ) {
		global $bbapp_var;

		$args['bbapp_code'] = $this->generate_activation_code();

		if ( isset( $bbapp_var ) && isset( $bbapp_var['app_id'] ) ) {
			$bbapp_app_id   = $bbapp_var['app_id'];
			$args['app_id'] = $bbapp_app_id;
		}

		return $args;
	}

	/**
	 * Update user date when activation is triggered by BuddyPress core.
	 *
	 * @param int    $user_id User id.
	 * @param string $key     Key.
	 * @param array  $user    User data.
	 */
	public function bp_core_activated_user( $user_id, $key, $user ) {
		if ( is_array( $user ) ) {
			$this->wp_core_activated_user( $user_id, $key, $user['meta'] );
		}
	}

	/**
	 * Update user date when activation is triggered by WordPress core.
	 *
	 * @param int    $user_id User id.
	 * @param string $key     Key.
	 * @param array  $meta    Array of metadata.
	 */
	public function wp_core_activated_user( $user_id, $key, $meta ) {
		global $wpdb;

		$meta = maybe_unserialize( $meta );

		if ( ! empty( $meta['app_id'] ) ) {
			update_user_meta( $user_id, '_bbapp_register_via_app_id', $meta['app_id'] );
		}

		$data = array(
			'ID' => $user_id,
		);

		if ( isset( $meta['first_name'] ) ) {
			$data['first_name'] = $meta['first_name'];
		}

		if ( isset( $meta['last_name'] ) ) {
			$data['last_name'] = $meta['last_name'];
		}

		if ( isset( $meta['display_name'] ) ) {
			$data['display_name'] = $meta['display_name'];
		}

		if ( isset( $meta['description'] ) ) {
			$data['description'] = $meta['description'];
		}

		wp_update_user( $data );

		// Replace the password automatically generated by WordPress by the one the user chose.
		if ( is_multisite() && isset( $meta['user_pass'] ) ) {
			$wpdb->update( $wpdb->users, array( 'user_pass' => $meta['user_pass'] ), array( 'ID' => $user_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		}
	}

	/**
	 * App launch code template.
	 */
	public function app_launch_code_template() {
		$app_launch_code_get = isset( $_GET['app-launch-code'] ) ? bbapp_input_clean( wp_unslash( $_GET['app-launch-code'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( $app_launch_code_get ) {
			$app_launch_code = json_decode( base64_decode( $app_launch_code_get ), true ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
			$action          = $app_launch_code['action'];
			$email           = $app_launch_code['email'];
			$code            = $app_launch_code['code'];
			$site_id         = $app_launch_code['site_id'];

			switch ( $action ) {
				case 'activation':
					if ( is_email( $email ) && is_numeric( $code ) ) {
						// Validate.
						$signup = $this->get_signup_user( $email );
						if ( ! empty( $signup ) && ! is_wp_error( $signup ) ) {
							$scheme_param = "//activation/{$email}:{$code}:{$site_id}";
							$this->bbapp_launch_app_script( $scheme_param );
						}
					}
					break;
				default:
					wp_die( esc_html__( 'Given app URL is incorrect. Please try again.', 'buddyboss-app' ) );
			}
		}
	}

	/**
	 * Creates a app launch button based on device it's being rendered.
	 *
	 * @param string $scheme_param Schema parameter.
	 */
	public function bbapp_launch_app_script( $scheme_param ) {
		$app_package_id = false;
		// Get the BundleID or Application ID.
		$mobile_detect             = \BuddyBossApp\Library\Composer::instance()->mobile_detect_instance()->Mobile_Detect();
		$publish_ios_namespace     = Configure::instance()->option( 'publish.ios.namespace' );
		$publish_android_namespace = Configure::instance()->option( 'publish.android.namespace' );

		if ( ! empty( $publish_ios_namespace ) && $mobile_detect->isiOS() ) {
			$app_package_id = $publish_ios_namespace;
		}

		if ( ! empty( $publish_android_namespace ) && $mobile_detect->isAndroidOS() ) {
			$app_package_id = $publish_android_namespace;
		}

		if ( ! empty( $app_package_id ) ) {
			// Convert namespace to schema.
			$app_schema = str_replace( '.', '', $app_package_id );
			$url        = "{$app_schema}:{$scheme_param}";
			include bbapp()->plugin_dir . 'views/open-on-app.php';
			exit;
		}
	}

	/**
	 * Get the user by there username and email.
	 * This function treats username as first priority. then after ward look for email if require.
	 *
	 * @param string $username_email Username or Email.
	 *
	 * @return \WP_User|bool
	 */
	public function get_user( $username_email ) {
		$user = get_user_by( 'login', $username_email );

		if ( $user && ! is_wp_error( $user ) ) {
			return $user;
		}

		// Let's give second try with email.
		if ( is_email( $username_email ) ) {
			$user = get_user_by( 'email', $username_email );
		}

		if ( $user && ! is_wp_error( $user ) ) {
			return $user;
		}

		return false;
	}

	/**
	 * Handle wp activation email to overrides it with bbapp one. (For multisite backend user create.)
	 *
	 * @param string $body       Email body.
	 * @param string $user_login User login.
	 * @param string $user_email User email.
	 *
	 * @return string
	 */
	public function wp_bbapp_activation_email( $body, $user_login, $user_email ) {
		if ( true !== $this->is_rest() ) {
			return $body;
		}

		// Prepare Body content with App changes.
		$bbapp_body = $this->get_email_activation_body();
		$key_code   = $this->get_activation_code( $user_email );

		// Replaces.
		$replace_keys = array(
			'KEY_CODE'      => $key_code,
			'KEY_CODE_LINK' => $this->get_mobile_code_link( 'activation', $user_email, $key_code ),
		);

		foreach ( $replace_keys as $replace_key => $replace_value ) {
			$replace_key = '{{' . $replace_key . '}}';
			$bbapp_body  = str_replace( $replace_key, $replace_value, $bbapp_body );
		}

		$bbapp_body = wp_strip_all_tags( str_replace( '%', '%%', $bbapp_body ) );

		return $body . "\n\n" . $bbapp_body;
	}

	/**
	 * Handle bp email to overrides it with bbapp one.
	 *
	 * @param string $email      User email.
	 * @param string $email_type Email type.
	 * @param string $to         To user email.
	 * @param array  $args       Arguments.
	 *
	 * @return bool|void
	 */
	public function bbapp_bp_send_email( $email, $email_type, $to, $args ) {

		if ( 'core-user-registration' === $email_type ) {
			if ( true === $this->is_rest() ) {
				$this->bbapp_activation_email( $email, $args );
			}
		}
	}

	/**
	 * Tell if current request is coming from rest api or not.
	 *
	 * @return bool
	 */
	public function is_rest() {
		return ( strpos( sanitize_text_field( $_SERVER['REQUEST_URI'] ), '/wp-json' ) !== false ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
	}

	/**
	 * Handle bp email activation email to overrides it with bbapp one.
	 *
	 * @param string $email User meial.
	 * @param array  $args  Arguments.
	 *
	 * @return bool|void
	 */
	public function bbapp_activation_email( $email, $args ) {
		$content_plaintext = $email->get_content_plaintext();
		$content_html      = $email->get_content_html();

		/**
		 * Auto Add Plain Email Content
		 * Add the custom bbapp content if user didn't added it before.
		 */
		if ( strpos( 'app_email_contains', $content_plaintext ) === false ) {
			$content_plaintext = $content_plaintext . '\n\n
			{{{bbapp_email_contents}}}';
			$email->set_content_plaintext( $content_plaintext );
		}

		/**
		 * Auto Add Plain HTML Content
		 * Add the custom bbapp content if user didn't added it before.
		 */
		if ( strpos( 'app_email_contains', $content_html ) === false ) {
			$content_html = $content_html . '<hr>
			{{{bbapp_email_contents}}}';
			$email->set_content_html( $content_html );
		}

		// Prepare Body content with App changes.
		$body       = $this->get_email_activation_body();
		$user_email = $args['tokens']['user.email'];

		$key_code = $this->get_activation_code( $user_email );

		// Replaces.
		$replace_keys = array(
			'KEY_CODE'      => $key_code,
			'KEY_CODE_LINK' => sprintf( '<a href="%1$s">%1$s</a>', $this->get_mobile_code_link( 'activation', $user_email, $key_code ) ),
		);

		foreach ( $replace_keys as $replace_key => $replace_value ) {
			$replace_key = '{{' . $replace_key . '}}';
			$body        = str_replace( $replace_key, $replace_value, $body );
		}

		$body                           = nl2br( $body );
		$tokens                         = $email->get_tokens();
		$tokens['bbapp_email_contents'] = $body;

		$email->set_tokens( $tokens );
	}

	/**
	 * Get signup mobile link.
	 *
	 * @param string $action     Action.
	 * @param string $user_email User email.
	 * @param int    $code       Activation code.
	 *
	 * @return string
	 */
	public function get_mobile_code_link( $action, $user_email, $code ) {
		$key_value = base64_encode( // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
			wp_json_encode(
				array(
					'action'  => $action,
					'email'   => $user_email,
					'code'    => $code,
					'site_id' => get_current_blog_id(),
				)
			)
		);

		return site_url() . "?app-launch-code={$key_value}";
	}

	/**
	 * Generates activation code for user.
	 *
	 * @return string
	 */
	public function generate_activation_code() {
		return wp_rand( 100000, 999999 );
	}

	/**
	 * Get user activation code.
	 *
	 * @param string $user_email User email.
	 *
	 * @return bool
	 */
	public function get_activation_code( $user_email ) {
		$row = $this->get_signup_user( $user_email );

		if ( ! empty( $row ) ) {
			$meta = maybe_unserialize( $row->meta );

			return $meta['bbapp_code'];
		}

		return false;
	}

	/**
	 * Get signup user.
	 *
	 * @param string $user_email User email.
	 *
	 * @return array|bool|object|\stdClass
	 */
	public function get_signup_user( $user_email ) {
		global $wpdb;

		$signups_table = false;

		if ( class_exists( 'BP_Signup' ) ) {
			$signups_table = buddypress()->members->table_name_signups;
		} elseif ( is_multisite() ) {
			$signups_table = $wpdb->signups;
		}

		if ( ! $signups_table ) {
			return $signups_table;
		}

		$row = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$signups_table} WHERE user_email=%s ORDER BY signup_id desc limit 1", $user_email ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		return $row;
	}

	/**
	 * Return's account activation email body.
	 *
	 * @return string
	 */
	public function get_email_activation_body() {
		$settings = $this->get_settings();

		if ( isset( $settings['app_auth.email_activation_body'] ) ) {
			return $settings['app_auth.email_activation_body'];
		}

		return '';
	}

	/**
	 * Activate the account by given activation code and email.
	 *
	 * @param string $user_email      User email.
	 * @param int    $activation_code User activation code.
	 *
	 * @return \WP_Error|(int) [user id]
	 */
	public function activate_account( $user_email, $activation_code ) {
		// Validate.
		$signup = $this->get_signup_user( $user_email );

		if ( empty( $signup ) ) {
			return new \WP_Error( 'user_not_found', __( 'Given user information not found.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		if ( $signup->active ) {
			return new \WP_Error( 'nothing_to_verify', __( 'Given user is already verified.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		$user_activation_code = $this->get_activation_code( $user_email );

		// Validate the code.
		if ( (int) $user_activation_code !== (int) $activation_code ) {
			return new \WP_Error( 'activation_code_not_matched', __( "Given activation code doesn't match.", 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		$do_activate = false;

		// If BP activate use it's method for activating user.
		if ( class_exists( 'BP_Signup' ) ) {
			$do_activate = $this->bp_core_activate_signup( $user_email );
		} elseif ( is_multisite() ) {
			$do_activate = $this->wpmu_core_activate_signup( $user_email );
		}

		$do_activate = apply_filters(
			'bbapp_auth_do_activate_flag',
			$do_activate,
			array(
				'user_email'      => $user_email,
				'activation_code' => $activation_code,
			)
		);

		// Do activate if do_activate is true.
		if ( $do_activate ) {
			$user = get_user_by( 'email', $user_email );

			return $user->ID;
		} else {
			return new \WP_Error( 'error_activating', __( 'Error while activating user account.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}
	}

	/**
	 * Activate user by there email in BuddyPress.
	 *
	 * @param string $user_email User email.
	 *
	 * @return bool
	 */
	public function bp_core_activate_signup( $user_email ) {
		$row            = $this->get_signup_user( $user_email );
		$activation_key = false;

		if ( ! empty( $row ) ) {
			$activation_key = $row->activation_key;
		}

		if ( $activation_key ) {
			$activate = bp_core_activate_signup( $activation_key );
			if ( ! empty( $activate ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 *  Activate user by there email in WordPress.
	 *
	 * @param string $user_email User email.
	 *
	 * @return bool
	 */
	public function wpmu_core_activate_signup( $user_email ) {
		$row            = $this->get_signup_user( $user_email );
		$activation_key = false;

		if ( ! empty( $row ) ) {
			$activation_key = $row->activation_key;
		}

		if ( $activation_key ) {
			$user = wpmu_activate_signup( $activation_key );

			if ( ! is_wp_error( $user ) ) {
				// Add user into main blog.
				return true;
			}
		}

		return false;
	}
}
