<?php
/**
 * Holds Apple store connection functionality.
 *
 * @package BuddyBossApp\AppStores
 */

namespace BuddyBossApp\AppStores;

use BuddyBossApp\Admin\Configure;
use BuddyBossApp\Admin\InAppPurchases\Helpers;
use BuddyBossApp\Helpers\BBAPP_File;
use BuddyBossApp\Tools\Logger;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Permissions;
use MingYuanYun\AppStore\Exceptions\ConfigException;

/**
 * IOS connection class.
 */
class Apple {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * IOS connection error messages.
	 *
	 * @var bool $apple_connect_error
	 */
	public $apple_connect_error = false;

	/**
	 * IOS connect message.
	 *
	 * @var string $apple_connect_msg
	 */
	public $apple_connect_msg = '';

	/**
	 * Get the instance of the class.
	 *
	 * @return Apple
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Build constructor.
	 */
	public function __construct() {
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		add_action( 'admin_init', array( $this, 'automatic_generate_provisioning_profiles' ), 99 );
		add_action( 'admin_init', array( $this, 'automatic_generate_certs' ), 99 );
		add_action( 'wp_ajax_refresh_bundle_ids', array( $this, 'refresh_bundle_ids' ) );
		add_action( 'wp_ajax_bbapp_regenerate_provisional_profile', array( $this, 'regenerate_provisional_profile' ) );
	}

	/**
	 * Return the Issuer ID.
	 */
	public function get_issuer_id() {
		return Configure::instance()->option( 'publish.ios.issuer_id' );
	}

	/**
	 * Return the Get key ID.
	 */
	public function get_key_id() {
		return Configure::instance()->option( 'publish.ios.key_id' );
	}

	/**
	 * Return the Get key ID.
	 */
	public function get_private_key() {
		return Configure::instance()->option( 'publish.ios.private_key' );
	}

	/**
	 * Return the Get key ID.
	 */
	public function get_ios_store_app_id() {
		$ios_store_app_id = ( self::instance()->is_connected() ) ? self::instance()->get_apple_release_bundle_store_app_id() : false;

		return empty( $ios_store_app_id ) ? Configure::instance()->option( 'publish.ios.store_app_id' ) : $ios_store_app_id;
	}

	/**
	 * Return the App Specific Password.
	 */
	public function get_ios_app_specific_password() {
		return Configure::instance()->option( 'publish.ios.app_specific_password' );
	}

	/**
	 * Return the App Specific Password User.
	 * It's a user whom app specific password is given.
	 */
	public function get_ios_app_specific_password_account_id() {
		return Configure::instance()->option( 'publish.ios.app_specific_password_account_id' );
	}

	/**
	 * Return's the release cert ID
	 */
	private function get_release_cert_id() {
		return trim( Configure::instance()->option( 'publish.ios.signing_certificate_id' ) );
	}

	/**
	 * Return's the dev bundle ID.
	 */
	private function get_dev_bundle_id() {
		return Configure::instance()->option( 'publish.ios.dev.namespace' );
	}

	/**
	 * Return's the dev bundle ID.
	 */
	public function get_release_bundle_id() {
		return Configure::instance()->option( 'publish.ios.namespace' );
	}

	/**
	 * Return's the signing certificate password
	 */
	public function get_signing_certificate_password() {
		$signing_certificate_password = Configure::instance()->option( 'publish.ios.signing_certificate_password' );

		return trim( $signing_certificate_password );
	}

	/**
	 * Return's the dev signing certificate password
	 */
	public function get_dev_signing_certificate_password() {
		$dev_signing_certificate_password = Configure::instance()->option( 'publish.ios.dev.signing_certificate_password' );

		return trim( $dev_signing_certificate_password );
	}

	/**
	 * Function to ger IOS client.
	 *
	 * @return \BuddyBossAppScoping\Google_Client|\MingYuanYun\AppStore\Client|\WP_Error
	 * @throws ConfigException Apple configuration exception message.
	 */
	private function get_client() {
		$issuer_id   = $this->get_issuer_id();
		$key_id      = $this->get_key_id();
		$private_key = $this->get_private_key();

		if ( empty( $issuer_id ) || empty( $key_id ) || empty( $private_key ) ) {
			return new \WP_Error(
				'error_connecting_account',
				__( 'Error while connecting to your Apple Developer account. Please make sure you have configured correct apple api keys.', 'buddyboss-app' )
			);
		}

		$private_key = bbapp_get_upload_full_path( $private_key );

		if ( ! file_exists( $private_key ) ) {
			return new \WP_Error(
				'error_connecting_account',
				__( 'Error while connecting to your Apple Developer account. Please make sure you have configured correct apple api keys.', 'buddyboss-app' )
			);
		}

		$client = \BuddyBossApp\Library\Composer::instance()->apple_instance()->ItunesConnectClient( $issuer_id, $key_id, $private_key );

		if ( empty( $client ) || is_wp_error( $client ) ) {
			return new \WP_Error(
				'error_connecting_account',
				__( 'Error while connecting to your Apple Developer account. Please make sure you have configured correct apple api keys.', 'buddyboss-app' )
			);
		}

		return $client;
	}

	/**
	 * Check if the AppStore is Connected.
	 *
	 * @param bool $override_cache Override cache.
	 *
	 * @return bool|\WP_Error
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function is_connected( $override_cache = false ) {
		static $clear_cache = null;

		$is_connect  = false;
		$issuer_id   = $this->get_issuer_id();
		$key_id      = $this->get_key_id();
		$private_key = $this->get_private_key();
		$cache_key   = 'bbapp_apple_is_connected_cache_' . md5( $issuer_id . $key_id . $private_key );
		$cached      = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			$apps       = $cached;
			$is_connect = true;

			if ( empty( $apps ) || isset( $apps['errors'] ) ) {
				$is_connect                = false;
				$this->apple_connect_error = true;
				$this->apple_connect_msg   = isset( $apps['errors'][0]['detail'] ) ? $apps['errors'][0]['detail'] : '';
			}
		} else {
			$apps = '';

			if ( $this->verify_ios_connect_fields() ) {
				$client = $this->get_client();

				if ( empty( $client ) || is_wp_error( $client ) ) {
					$this->apple_connect_error = true;
					$is_connect                = false;
				} else {
					try {
						$i_connect_in_apps = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectApps( $client );
						$apps              = $i_connect_in_apps->query();
					} catch ( \Throwable $e ) {
						$this->apple_connect_error = true;
					}

					if ( empty( $apps ) || isset( $apps['errors'] ) ) {
						$this->apple_connect_error = true;
						$this->apple_connect_msg   = isset( $apps['errors'][0]['detail'] ) ? $apps['errors'][0]['detail'] : '';
					} else {
						$is_connect = true;
					}
				}
			}

			set_site_transient( $cache_key, $apps, 24 * HOUR_IN_SECONDS );
		}

		return $is_connect;
	}

	/**
	 * Verify IOS fields.
	 *
	 * @return bool
	 */
	public function verify_ios_connect_fields() {
		$issuer_id   = $this->get_issuer_id();
		$key_id      = $this->get_key_id();
		$private_key = $this->get_private_key();

		if ( empty( $issuer_id ) || empty( $key_id ) || empty( $private_key ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Automatic creates certs.
	 *
	 * @param bool $override_cache Override cache.
	 *
	 * @return false|void
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function automatic_generate_certs( $override_cache = false ) {
		if ( ! is_admin() ) {
			return false;
		}

		$triggered = bbapp_get_network_option( '_bbapp_automatic_cert_triggered' );

		if ( ! empty( $triggered ) ) {
			return false;
		}

		$this->update_cert( true ); // dev.
		$this->update_cert( false ); // release.

		bbapp_set_network_option( '_bbapp_automatic_cert_triggered', '1' );
	}

	/**
	 * Regenerate provisional profile.
	 */
	public function regenerate_provisional_profile() {
		if ( Permissions::instance()->can_manage_app() ) {
			$nonce_post = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';

			if ( wp_verify_nonce( $nonce_post, 'bbapp_create_new_profile' ) ) {
				$this->update_provisioning_profile( true );
				$this->update_provisioning_profile( false );
				wp_send_json_success( __( 'Request processed, Please wait while provisional profile generates.', 'buddyboss-app' ) );
			} else {
				wp_send_json_error( __( 'Unable to verify security nonce, Please refresh the page and try again.', 'buddyboss-app' ) );
			}
		}
	}

	/**
	 * Responsible for automatic generating provisioning profiles for release.
	 * Dev provisioning profiles will be generated when a device will be added.
	 *
	 * @return bool|void
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function automatic_generate_provisioning_profiles() {
		if ( ! is_admin() ) {
			return false;
		}

		$triggered = bbapp_get_network_option( '_bbapp_automatic_provisioning_profiles_triggered' );

		if ( ! empty( $triggered ) ) {
			return false;
		}

		$settings                  = ManageApp::instance()->get_app_settings( true );
		$value_release             = ( isset( $settings['publish.ios.provisioning_profile'] ) ) ? $settings['publish.ios.provisioning_profile'] : false;
		$value_dev                 = ( isset( $settings['publish.ios.dev.provisioning_profile'] ) ) ? $settings['publish.ios.dev.provisioning_profile'] : false;
		$singing_cert_release      = ( isset( $settings['publish.ios.signing_certificate'] ) ) ? $settings['publish.ios.signing_certificate'] : false;
		$singing_cert_dev          = ( isset( $settings['publish.ios.dev.signing_certificate'] ) ) ? $settings['publish.ios.dev.signing_certificate'] : false;
		$last_singing_cert_release = bbapp_get_network_option( '_bbapp_release_signing_certificate_last_provisioning_profiles' );
		$last_singing_cert_dev     = bbapp_get_network_option( '_bbapp_dev_signing_certificate_last_provisioning_profiles' );

		// Trigger release.
		if ( empty( $value_release ) || $last_singing_cert_release !== $singing_cert_release ) {
			$this->update_provisioning_profile( false ); // don't worry. this func won't create if field is set to manual.
			bbapp_set_network_option( '_bbapp_release_signing_certificate_last_provisioning_profiles', $singing_cert_release );
		}

		// Trigger dev.
		if ( empty( $value_dev ) || $last_singing_cert_dev !== $singing_cert_dev ) {
			$this->update_provisioning_profile( true ); // don't worry. this func won't create if field is set to manual.
			bbapp_set_network_option( '_bbapp_dev_signing_certificate_last_provisioning_profiles', $singing_cert_dev );
		}

		bbapp_set_network_option( '_bbapp_automatic_provisioning_profiles_triggered', '1' );
	}

	/**
	 * Return's the development app bundle id.
	 * Don't use it on recurring events! there is no cache in it.
	 *
	 * @param bool $dev Development or not.
	 *
	 * @return bool|\WP_Error
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function get_store_bundle_id( $dev = true ) {
		if ( $dev ) {
			$bundle_id = $this->get_dev_bundle_id();
		} else {
			$bundle_id = $this->get_release_bundle_id();
		}

		if ( empty( $bundle_id ) ) {
			return $bundle_id;
		}

		$client = $this->get_client();

		if ( is_wp_error( $client ) ) {
			return $client;
		}

		$store_bundle_id = false;

		// find the correct bundle which matches the test app configuration one.
		$connect_bundle_id = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectBundleId( $client );
		$bundles           = $connect_bundle_id->all(
			array(
				'limit'  => 200,
				'filter' => array( 'identifier' => $bundle_id ),
			)
		);

		if ( $bundles['errors'] ) {
			return new \WP_Error(
				'error_connecting_api',
				__( 'There was an error creating provisioning profiles in your Apple Developer account. Please make sure you have configured your Apple API Key correctly.',
					'buddyboss-app' )
			);
		}

		foreach ( $bundles['data'] as $bundle ) {
			if ( 'bundleIds' === $bundle['type'] ) {
				if ( isset( $bundle['attributes'] ) && isset( $bundle['attributes']['identifier'] ) && $bundle['attributes']['identifier'] === $bundle_id ) {
					$store_bundle_id = $bundle['id'];
				}
			}
		}

		if ( empty( $store_bundle_id ) ) {
			if ( $dev ) {
				return new \WP_Error(
					'bundle_id_not_found',
					__( 'The Bundle ID you have entered for your Test App was not found in your Apple Developer account.', 'buddyboss-app' )
				);
			} else {
				return new \WP_Error(
					'bundle_id_not_found',
					__( 'The Bundle ID you have entered for your Release App was not found in your Apple Developer account.', 'buddyboss-app' )
				);
			}
		}

		return $store_bundle_id;
	}

	/**
	 * Return's the app certs.
	 * Don't use it on recurring events! there is no cache in it.
	 *
	 * @param bool $dev Development or not.
	 *
	 * @return \WP_Error|array
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function get_certs( $dev = true ) {
		$client = $this->get_client();

		if ( is_wp_error( $client ) ) {
			return $client;
		}

		// Get the Certificates.
		$apple_cert = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectCert( $client );
		$certs      = $apple_cert->all();

		if ( $certs['errors'] ) {
			return new \WP_Error(
				'error_getting_cert',
				__( 'There was an error retrieving signing certificates from your Apple Developer account.', 'buddyboss-app' )
			);
		}

		$cert_ids   = array();
		$cert_types = array(
			'IOS_DISTRIBUTION',
			'DISTRIBUTION',
		);

		if ( $dev ) {
			$cert_types = array(
				'DEVELOPMENT',
				'IOS_DEVELOPMENT',
			);
		}

		foreach ( $certs['data'] as $cert ) {
			if ( isset( $cert['attributes'] ) && isset( $cert['attributes']['certificateType'] ) && in_array( $cert['attributes']['certificateType'], $cert_types, true ) ) {
				$cert_ids[] = $cert['id'];
			}
		}

		return $cert_ids;
	}

	/**
	 * Return the list of devices dev app has
	 * Don't use it on recurring events! there is no cache in it.
	 */
	public function get_development_devices() {
		$client = $this->get_client();

		if ( is_wp_error( $client ) ) {
			return $client;
		}

		$devices     = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectDevices( $client );
		$get_devices = $devices->all( array( 'limit' => 200 ) );

		if ( isset( $get_devices['errors'] ) ) {
			return new \WP_Error(
				'error_connecting_api',
				__( 'There was an error creating provisioning profiles in your Apple Developer account. Please make sure you have configured your Apple API Key correctly.',
					'buddyboss-app' )
			);
		}

		$devices_id = array();

		if ( ! empty( $get_devices['data'] ) && ! is_array( $get_devices['data'] ) ) {
			return $devices_id;
		}

		foreach ( $get_devices['data'] as $device ) {
			if ( isset( $device['attributes'] ) && isset( $device['attributes']['platform'] ) && 'IOS' === $device['attributes']['platform'] && 'DISABLED' !== $device['attributes']['status'] ) {
				$devices_id[ $device['id'] ] = array(
					'id'   => $device['id'],
					'udid' => $device['attributes']['udid'],
					'name' => $device['attributes']['name'],
				);
			}
		}

		return $devices_id;
	}

	/**
	 * Return the local devices which is synced from store.
	 * Use this function to fetch on recurring events.
	 *
	 * @param bool $override_cache Override cache.
	 *
	 * @return bool|array
	 */
	public function get_local_devices( $override_cache = false ) {
		$cache_key = 'bbapp_apple_get_local_devices';
		$cached    = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			return $cached;
		} else {
			$devices = $this->get_development_devices();

			if ( is_wp_error( $devices ) ) {
				set_site_transient( $cache_key, array(), 30 * 60 ); // Cache it for less time for retry early.

				return array();
			}

			set_site_transient( $cache_key, $devices, DAY_IN_SECONDS * 7 ); // cache it for 7 weeks.

			return $devices;
		}
	}

	/**
	 * Register device to get app download.
	 *
	 * @param string $name Device name.
	 * @param string $udid Device UDID.
	 *
	 * @return \Google_Client|\MingYuanYun\AppStore\Client|bool|\WP_Error
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function register_development_device( $name, $udid ) {
		$name   = wp_strip_all_tags( $name );
		$client = $this->get_client();

		if ( is_wp_error( $client ) ) {
			return $client;
		}

		$devices  = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectDevices( $client );
		$register = $devices->register( $name, 'IOS', $udid );

		if ( $register['errors'] ) {
			// if it's already exists error.
			if ( isset( $register['errors'][0] ) && isset( $register['errors'][0]['detail'] ) && '409' === (string) $register['errors'][0]['status'] ) {
				return true;
			}

			return new \WP_Error(
				'error_registering_udid',
				__( 'There was an error registering a UDID in your Apple Developer account.', 'buddyboss-app' )
			);
		}

		// Refresh latest.
		$this->get_local_devices( true );

		return true;
	}

	/**
	 * Automatic updates the profile.
	 *
	 * @param bool $dev Development or not.
	 *
	 * @return bool|\WP_Error
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function update_provisioning_profile( $dev = true ) {
		$settings       = ManageApp::instance()->get_app_settings( true );
		$configure_type = ( isset( $settings['publish.ios.provisioning_profile_automatic'] ) ) ? $settings['publish.ios.provisioning_profile_automatic'] : false;
		$env_type       = ( $dev ) ? 'dev' : 'release';

		if ( true !== (bool) $configure_type ) {
			bbapp_set_network_option( "_bbapp_automatic_{$env_type}_provisioning_profiles_error_msg", '' );
			Logger::instance()->add( 'info_log', 'skipped automatic provisioning profile creation as the field is set to manual.' );

			return false;
		}

		$create_profile = $this->create_provisioning_profile( $dev );

		bbapp_set_network_option( "bbapp_{$env_type}_release_provisioning_profiles_date", time() );

		if ( is_wp_error( $create_profile ) ) {
			bbapp_set_network_option( "_bbapp_automatic_{$env_type}_provisioning_profiles_error_msg", $create_profile->get_error_message() );
			Logger::instance()->add( 'info_log', $create_profile->get_error_message() );

			return $create_profile;
		}

		bbapp_set_network_option( "_bbapp_automatic_{$env_type}_provisioning_profiles_error_msg", '' );

		$upload_dir = wp_upload_dir();
		$dir        = 'bbapp/uploads';
		$dir        = "{$upload_dir['basedir']}/{$dir}";

		if ( $dev ) {
			$old_file = ( $settings['publish.ios.dev.provisioning_profile'] );
		} else {
			$old_file = ( $settings['publish.ios.provisioning_profile'] );
		}

		if ( ! empty( $old_file ) && file_exists( $upload_dir['basedir'] . '/' . $old_file ) ) {
			wp_delete_file( $upload_dir['basedir'] . '/' . $old_file );
		}

		$unique_hash = bbapp_generate_password( 20, false ); // unguessable..
		$new_file    = $dir . "/{$create_profile["id"]}-{$unique_hash}.mobileprovision";

		BBAPP_File::write_file( $new_file, base64_decode( $create_profile['content'] ) ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode

		if ( file_exists( $new_file ) ) {
			$new_file = str_replace( $upload_dir['basedir'], '', $new_file );

			if ( $dev ) {
				$settings['publish.ios.dev.provisioning_profile'] = $new_file;
			} else {
				$settings['publish.ios.provisioning_profile'] = $new_file;
			}

			ManageApp::instance()->update_app_settings( $settings );

			return true;
		}

		return false;
	}

	/**
	 * Update Cert.
	 * This has all the function which decides if cert needs to create or where to save it.
	 *
	 * @param bool $dev Development or not.
	 *
	 * @return bool
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function update_cert( $dev = true ) {
		$settings       = ManageApp::instance()->get_app_settings( true );
		$env_type       = ( $dev ) ? 'dev' : 'release';
		$configure_type = ( isset( $settings['publish.ios.signing_certificates_automatic'] ) ) ? $settings['publish.ios.signing_certificates_automatic'] : false;

		if ( true !== (bool) $configure_type ) {
			bbapp_set_network_option( "_bbapp_automatic_{$env_type}_cert_error_msg", '' );
			Logger::instance()->add( 'info_log', 'skipped automatic cert creation as the field is set to manual.' );

			return false;
		}

		if ( ! $dev ) {
			$cert = ( isset( $settings['publish.ios.signing_certificate'] ) ) ? $settings['publish.ios.signing_certificate'] : false;
		} else {
			$cert = ( isset( $settings['publish.ios.dev.signing_certificate'] ) ) ? $settings['publish.ios.dev.signing_certificate'] : false;
		}

		if ( empty( $cert ) ) {
			$create_cert = $this->create_cert( $dev );

			if ( ! $create_cert || is_wp_error( $create_cert ) ) {
				if ( is_wp_error( $create_cert ) ) {
					bbapp_set_network_option( "_bbapp_automatic_{$env_type}_cert_error_msg", $create_cert->get_error_message() );
					Logger::instance()->add( 'info_log', 'Auto Cert Creating Error - ' . $create_cert->get_error_message() );
				} else {
					bbapp_set_network_option( "_bbapp_automatic_{$env_type}_cert_error_msg", __( 'Unknown error while creating cert.', 'buddyboss-app' ) );
				}

				return false;
			}

			// Looks like it's created.
			$upload_dir  = wp_upload_dir();
			$dir         = 'bbapp/uploads';
			$dir         = "{$upload_dir['basedir']}/{$dir}";
			$unique_hash = bbapp_generate_password( 20, false ); // unguessable..
			$new_file    = $dir . "/{$create_cert["cert_id"]}-{$unique_hash}.p12";

			BBAPP_File::write_file( $new_file, base64_decode( $create_cert['p12_content'] ) ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode

			if ( file_exists( $new_file ) ) {
				$new_file = str_replace( $upload_dir['basedir'], '', $new_file );

				if ( $dev ) {
					$settings['publish.ios.dev.signing_certificate']          = $new_file;
					$settings['publish.ios.dev.signing_certificate_id']       = $create_cert['cert_id'];
					$settings['publish.ios.dev.signing_certificate_password'] = $create_cert['p12_password'];
				} else {
					$settings['publish.ios.signing_certificate']          = $new_file;
					$settings['publish.ios.signing_certificate_id']       = $create_cert['cert_id'];
					$settings['publish.ios.signing_certificate_password'] = $create_cert['p12_password'];
				}

				ManageApp::instance()->update_app_settings( $settings );

				bbapp_set_network_option( "_bbapp_automatic_{$env_type}_cert_error_msg", '' );

				// Regenerate the Profiles.
				$this->update_provisioning_profile( $dev );

				return true;
			} else {
				bbapp_set_network_option( "_bbapp_automatic_{$env_type}_cert_error_msg",
					__( 'Error while creating certificate file, please check the file permissions.', 'buddyboss-app' ) );
			}

			return false;
		}

		return false;
	}

	/**
	 * Create the certificate.
	 *
	 * @param bool $dev       Development or not.
	 * @param bool $force_new when provided true it will skip all stages and try to create new.
	 *
	 * This function shouldn't be called directly on any page load.
	 * Recurring run of this function should be avoided.
	 *
	 * @return bool|\Google_Client|\MingYuanYun\AppStore\Client|\WP_Error
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function create_cert( $dev = true, $force_new = false ) {
		$client = $this->get_client();

		if ( is_wp_error( $client ) ) {
			return $client;
		}

		$type_slug = ( $dev ) ? 'dev' : 'release';

		if ( ! $force_new ) {
			$generated_cert = bbapp_get_network_option( '_bbapp_generated_cert_' . $type_slug );
		} else {
			$generated_cert = false;
		}

		// validate if cert exists if it was already generated in past.
		if ( ! empty( $generated_cert ) ) {
			$apple_cert = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectCert( $client );
			$certs      = $apple_cert->all();

			if ( isset( $certs['errors'] ) ) {
				return new \WP_Error(
					'error_getting_cert',
					__( 'There was an error retrieving signing certificates from your Apple Developer account.', 'buddyboss-app' )
				);
			}

			$found = false;

			foreach ( $certs['data'] as $cert ) {
				if ( $cert['id'] === $generated_cert['cert_id'] ) {
					$found = true;
					break;
				}
			}

			if ( ! $found ) { // if no cert found empty the previous data.
				$generated_cert = array();
				bbapp_set_network_option( '_bbapp_generated_cert_' . $type_slug, '' );
			}
		}

		if ( empty( $generated_cert ) ) {
			// Get Signing Cert.
			$get_csr = $this->get_csr();

			if ( is_wp_error( $get_csr ) ) {
				return $get_csr;
			}

			// Create Certificate Now.
			$apple_cert   = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectCert( $client );
			$csr_file     = $get_csr['file']['path'];
			$csr_key_file = $get_csr['file']['private_key_file_path'];
			$type         = 'DISTRIBUTION';

			if ( $dev ) {
				$type = 'DEVELOPMENT';
			}

			$create = $apple_cert->create( $type, $csr_file );

			if ( isset( $create['data'] ) && isset( $create['data']['id'] ) && isset( $create['data']['attributes']['certificateContent'] ) ) {
				$generated_cert = array(
					'cert_id'                 => $create['data']['id'],
					'cert_content'            => $create['data']['attributes']['certificateContent'],
					'expirationDate'          => $create['data']['attributes']['expirationDate'],
					'csr_content'             => BBAPP_File::read_file( $csr_file ),
					'csr_private_key_content' => BBAPP_File::read_file( $csr_key_file ),
				);

				bbapp_set_network_option( '_bbapp_generated_cert_' . $type_slug, $generated_cert );
			} elseif ( isset( $create['errors'] ) ) {
				$error     = $create['errors'][0];
				$error_msg = "{$error["title"]}. {$error["detail"]}.";

				return new \WP_Error(
					'error_creating_cert',
					$error_msg
				);
			}
		}

		// if p12 is not create try to create it.
		if ( ! isset( $generated_cert['p12_content'] ) ) {
			$p12_password = bbapp_generate_password( 8, false );
			$p12_password = trim( strtolower( $p12_password ) ); // keep it easy to type.

			// create p12 file.
			$cert_content_decode = base64_decode( $generated_cert['cert_content'] ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
			$p12                 = $this->generate_p12( $generated_cert['csr_content'], $cert_content_decode, $generated_cert['csr_private_key_content'], $p12_password );

			if ( is_wp_error( $p12 ) ) {
				return $p12;
			} else {
				$generated_cert['p12_content']  = base64_encode( $p12 ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
				$generated_cert['p12_password'] = $p12_password;
				bbapp_set_network_option( '_bbapp_generated_cert_' . $type_slug, $generated_cert );
			}
		}

		return $generated_cert;
	}

	/**
	 * Create new development profile & updates it.
	 *
	 * @param bool $dev Development or not.
	 *
	 * @return bool|\WP_Error|array
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function create_provisioning_profile( $dev = true ) {
		$client = $this->get_client();

		if ( is_wp_error( $client ) ) {
			return $client;
		}

		$store_bundle_id = $this->get_store_bundle_id( $dev );

		if ( is_wp_error( $store_bundle_id ) ) {
			return $store_bundle_id;
		}

		if ( $dev ) {
			$cert_ids = $this->get_certs( $dev );

			if ( is_wp_error( $cert_ids ) ) {
				return $cert_ids;
			}
		} else {
			$cert_id = $this->get_release_cert_id();

			if ( empty( $cert_id ) ) {
				return new \WP_Error(
					'no_release_cert_id',
					__( 'Please provide the signing certificate id to automatically create provisioning profile for release app.', 'buddyboss-app' )
				);
			}

			$cert_ids = array( $cert_id );
		}

		$device_ids = array();

		if ( $dev ) {
			$devices = $this->get_development_devices();

			if ( is_wp_error( $devices ) ) {
				return $devices;
			}

			foreach ( $devices as $k => $v ) {
				$device_ids[] = $k;
			}

			if ( empty( $device_ids ) ) {
				return new \WP_Error(
					'no_devices_found',
					__( 'Please register at least one iOS device to start generating provisioning profiles.', 'buddyboss-app' )
				);
			}
		}

		$i_connect_profile = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectProfile( $client );

		if ( $dev ) {
			$bundle_id = $this->get_dev_bundle_id();
		} else {
			$bundle_id = $this->get_release_bundle_id();
		}

		if ( $dev ) {
			$profile_name = "{$bundle_id}(dev) created at " . gmdate( 'd-m-y H:i:s' );
		} else {
			$profile_name = "{$bundle_id}(release) created at " . gmdate( 'd-m-y H:i:s' );
		}

		$profile_type = 'IOS_APP_STORE';

		if ( $dev ) {
			$profile_type = 'IOS_APP_DEVELOPMENT';
		}

		$create = $i_connect_profile->create( $profile_name, $store_bundle_id, $profile_type, $device_ids, $cert_ids );

		if ( $create['errors'] ) {
			$error = __( 'There was an unknown error creating a provisioning profile in your Apple Developer account.', 'buddyboss-app' );
			if ( isset( $create['errors'][0]['title'] ) ) {
				$error = "{$create["errors"][0]["title"]}. {$create["errors"][0]["detail"]}.";
			}

			return new \WP_Error(
				'error_creating_profile',
				$error
			);
		}

		return array(
			'id'      => $create['data']['id'],
			'content' => $create['data']['attributes']['profileContent'],
		);
	}

	/**
	 * Get error message while apple connect.
	 *
	 * @return string|void
	 */
	public function get_error_message() {
		if ( true === $this->apple_connect_error ) {
			if ( ! empty( $this->apple_connect_msg ) ) {
				return $this->apple_connect_msg;
			}

			return __( 'We were unable to connect to your Apple Developer account using the API Key provided. Please provide a valid API Key with App Manager access or higher.',
				'buddyboss-app' );
		}

		return false;
	}

	/**
	 * Generate & return CSR file.
	 * If the cert is already generated we don't create new.
	 */
	public function get_csr() {
		$blogname  = html_entity_decode( get_option( 'blogname' ) );
		$dev_name  = ! empty( $blogname ) && strlen( $blogname ) > 64 ? substr( $blogname, 0, 63 ) : $blogname;
		$dev_email = get_option( 'admin_email' );

		if ( empty( $dev_name ) || ! is_email( $dev_email ) ) {
			return new \WP_Error(
				'csr_no_email_or_name_found',
				__( 'Please set your Site Title and Administrator Email Address in your WordPress Settings to start generating signing certificates.', 'buddyboss-app' )
			);
		}

		$bbapp_generated_csr = bbapp_get_network_option( '_bbapp_generated_csr' );

		if ( ! empty( $bbapp_generated_csr ) && isset( $bbapp_generated_csr['file'] ) && file_exists( $bbapp_generated_csr['file']['private_key_file_path'] ) && file_exists( $bbapp_generated_csr['file']['path'] ) ) {
			return $bbapp_generated_csr;
		}

		$csr_generator_url         = 'https://bbapp-cert-services.buddyboss.com/api/csr';
		$private_key_generator_url = 'https://bbapp-cert-services.buddyboss.com/api/private-key';

		$do_request = bbapp_remote_post(
			$csr_generator_url,
			array(
				'method'  => 'POST',
				'timeout' => 45,
				'body'    => array(
					'common_name' => $dev_name,
					'email'       => $dev_email,
				),
			)
		);

		if ( is_wp_error( $do_request ) || empty( $do_request ) ) {
			return $do_request;
		}

		$data    = array(
			'name'  => $dev_name,
			'email' => $dev_email,
			'file'  => '',
		);
		$success = false;
		$body    = wp_remote_retrieve_body( $do_request );
		$success = false;

		if ( ! empty( $body ) && 200 === wp_remote_retrieve_response_code( $do_request ) ) {
			$file_name     = wp_remote_retrieve_header( $do_request, 'content-disposition' );
			$file_name     = str_replace( 'inline; filename=', '', $file_name );
			$upload        = wp_upload_dir();
			$file_path     = $upload['basedir'] . '/bbapp/csr/';
			$file_url      = $upload['baseurl'] . '/bbapp/csr/';
			$hash_name_arr = explode( '.', $file_name );
			$file_uploaded = BBAPP_File::file_handler( $file_path . $file_name, $body );

			if ( ! empty( $file_uploaded ) || ! is_wp_error( $file_uploaded ) ) {
				$do_request_private_key = bbapp_remote_get( $private_key_generator_url . '?csr_hash=' . $hash_name_arr[0] );
				$private_key_body       = wp_remote_retrieve_body( $do_request_private_key );
				$private_key_file_path  = '';
				$private_key_file_url   = '';

				if ( ! empty( $private_key_body ) && 200 === wp_remote_retrieve_response_code( $do_request_private_key ) ) {
					$private_key_file_name     = wp_remote_retrieve_header( $do_request_private_key, 'content-disposition' );
					$private_key_file_name     = str_replace( 'inline; filename=', '', $private_key_file_name );
					$private_key_file_uploaded = BBAPP_File::file_handler( $file_path . $private_key_file_name, $private_key_body );

					if ( ! empty( $private_key_file_uploaded ) || ! is_wp_error( $private_key_file_uploaded ) ) {
						$private_key_file_path = $file_path . $private_key_file_name;
						$private_key_file_url  = $file_url . $private_key_file_name;
					}
				}

				$data['file']        = array(
					'path'                  => $file_path . $file_name,
					'url'                   => $file_url . $file_name,
					'file_name'             => $file_name,
					'file_hash'             => ( ! empty( $hash_name_arr[0] ) ) ? $hash_name_arr[0] : '',
					'private_key_file_path' => $private_key_file_path,
					'private_key_file_url'  => $private_key_file_url,
				);
				$bbapp_generated_csr = $data;

				bbapp_set_network_option( '_bbapp_generated_csr', $data );
				$success = true;
			}
		}

		if ( $success ) {
			return $bbapp_generated_csr;
		} else {
			return new \WP_Error(
				'unknown_error_generating_csr',
				__( 'This were an unknown error while generating a CSR file for your signing certificates. Please try again later.', 'buddyboss-app' )
			);
		}
	}

	/**
	 * Function to generate .p12 file.
	 *
	 * @param string $csr_content  CSR certificate content.
	 * @param string $cer_content  Certificate content.
	 * @param string $key_content  Certificate key.
	 * @param string $p12_password P12 file password.
	 *
	 * @return bool|string|\WP_Error
	 */
	public function generate_p12( $csr_content, $cer_content, $key_content, $p12_password ) {
		$upload            = wp_upload_dir();
		$p12_generator_url = 'https://bbapp-cert-services.buddyboss.com/api/p12?csr_hash=' . md5( $csr_content );
		$path              = $upload['basedir'] . '/bbapp/csr/';
		$cer_file          = bbapp_generate_password( 15, false ) . '.cer';
		$key_file          = bbapp_generate_password( 15, false ) . '.key';
		$cer_upload        = BBAPP_File::file_handler( $path . $cer_file, $cer_content );
		$key_upload        = BBAPP_File::file_handler( $path . $key_file, $key_content );

		if ( ! $cer_upload || is_wp_error( $cer_upload ) || ! $key_upload || is_wp_error( $key_upload ) ) {
			return new \WP_Error(
				'error_creating_p12',
				__( 'Error while creating p12 file please try again.', 'buddyboss-app' )
			);
		}

		$do_request = $this->send_p12_request(
			$p12_generator_url,
			array(
				'cer_file'        => new \CURLFILE( $path . $cer_file ),
				'key_file'        => new \CURLFILE( $path . $key_file ),
				'export_password' => $p12_password,
			)
		);

		// remove temp files.
		unlink( $path . $cer_file );
		unlink( $path . $key_file );

		if ( ! is_wp_error( $do_request ) ) {
			return $do_request; // return the p12 content.
		} else {
			return new \WP_Error(
				'error_creating_p12',
				__( 'Error while creating p12 file please try again.', 'buddyboss-app' )
			);
		}
	}

	/**
	 * Retrieve the p12 file from the API.
	 *
	 * @param string $api_url     API url.
	 * @param array  $post_fields Fields that will be sending along API request.
	 *
	 * @note In this function we have used CURL instead of WP get OR set request because multiple files are not being read by WP methods.
	 * @return bool|string|\WP_Error
	 */
	public function send_p12_request( $api_url, $post_fields ) {
		$result = false;
		// phpcs:disable
		$curl = curl_init();

		curl_setopt_array(
			$curl,
			array(
				CURLOPT_URL            => $api_url,
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_ENCODING       => '',
				CURLOPT_MAXREDIRS      => 10,
				CURLOPT_TIMEOUT        => 45,
				CURLOPT_FOLLOWLOCATION => true,
				CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
				CURLOPT_CUSTOMREQUEST  => 'POST',
				CURLOPT_POSTFIELDS     => $post_fields,
				CURLOPT_HTTPHEADER     => array(),
			)
		);

		$response = curl_exec( $curl );

		if ( ! curl_errno( $curl ) ) {
			switch ( $http_code = curl_getinfo( $curl, CURLINFO_HTTP_CODE ) ) {
				case 200:
					$result = $response;
					break;
				default:
					$result = new \WP_Error( $http_code, curl_error( $curl ) );
			}
		}

		curl_close( $curl );

		// phpcs:enable
		return $result;
	}

	/**
	 * Return's the app bundles.
	 *
	 * @param bool $override_cache Override cache.
	 *
	 * @return array
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function get_registered_bundles( $override_cache = false ) {
		$issuer_id    = $this->get_issuer_id();
		$key_id       = $this->get_key_id();
		$private_key  = $this->get_private_key();
		$is_connected = $this->is_connected();

		if ( empty( $is_connected ) || is_wp_error( $is_connected ) ) {
			return array();
		}

		$client = $this->get_client();

		if ( is_wp_error( $client ) ) {
			return array();
		}

		$store_bundle_ids = array();

		$cache_key = 'bbapp_apple_bundle_ids_cache_' . md5( $issuer_id . $key_id . $private_key );
		$cached    = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			return $cached;
		} else {
			// find the correct bundle which matches the test app configuration one.
			$connect_bundle_id = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectBundleId( $client );
			$bundles           = $connect_bundle_id->all();

			// If there is an error or no data found.
			if ( isset( $bundles['errors'] ) || empty( $bundles ) || is_wp_error( $bundles ) ) {
				return array();
			}

			$store_bundle_ids = $this->get_store_bundle_ids( $bundles, $store_bundle_ids, $connect_bundle_id );
			set_site_transient( $cache_key, $store_bundle_ids, 1 * HOUR_IN_SECONDS );

			return $store_bundle_ids;
		}
	}

	/**
	 * Get the app bundles.
	 *
	 * @param object $connect_bundle_id Connect Bundle Id.
	 * @param string $next_page_url     Next Page URL.
	 *
	 * @since 2.0.90
	 * @return array
	 */
	public function next_page_bundles_ids( $connect_bundle_id, $next_page_url ) {
		$store_bundle_ids = array();
		$query_params     = wp_parse_url( $next_page_url, PHP_URL_QUERY );
		// Parse the query string into an array
		parse_str( $query_params, $params );
		$app_bundles = $connect_bundle_id->all( $params );

		// If there is an error or no data found.
		if ( isset( $app_bundles['errors'] ) || empty( $app_bundles ) || is_wp_error( $app_bundles ) ) {
			$store_bundle_ids = array();
		}

		if ( isset( $app_bundles['data'] ) ) {
			$store_bundle_ids = $this->get_store_bundle_ids( $app_bundles, $store_bundle_ids, $connect_bundle_id );
		}

		return $store_bundle_ids;
	}

	/**
	 * Get store bundle ids.
	 *
	 * @param array  $app_bundles       App Bundles.
	 * @param array  $store_bundle_ids  Store Bundle Ids.
	 * @param object $connect_bundle_id Connect Bundle Id.
	 *
	 * @since 2.0.90
	 * @return array
	 */
	public function get_store_bundle_ids( $app_bundles, $store_bundle_ids, $connect_bundle_id ) {
		if ( ! empty( $app_bundles['data'] ) ) {
			foreach ( $app_bundles['data'] as $app_bundle ) {
				if ( 'bundleIds' === $app_bundle['type'] ) {
					if ( isset( $app_bundle['attributes'] ) && ! empty( $app_bundle['attributes']['identifier'] ) ) {
						$store_bundle_ids[] = $app_bundle['attributes']['identifier'];
					}
				}
			}
		}

		if ( isset( $app_bundles['links']['next'] ) ) {
			$bundles_ids      = $this->next_page_bundles_ids( $connect_bundle_id, $app_bundles['links']['next'] );
			$store_bundle_ids = array_merge( $store_bundle_ids, $bundles_ids );
		}

		return $store_bundle_ids;
	}

	/**
	 * Sync store product with local data.
	 */
	public function sync_store_product() {
		$issuer_id            = $this->get_issuer_id();
		$key_identifier       = $this->get_key_id();
		$secret_key_file_name = $this->get_private_key();
		$ios_app_id           = $this->get_ios_store_app_id();
		$secret_key_file      = bbapp_get_upload_full_path( $secret_key_file_name );
		$products             = array();

		if ( empty( $secret_key_file ) || ! file_exists( $secret_key_file ) || empty( $issuer_id ) || empty( $key_identifier ) || empty( $ios_app_id ) ) {
			return $products;
		}

		try {
			$client  = $this->get_client();
			$iap_api = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectInAppPurchases( $ios_app_id, $client );

			// Fetch all non-subscription in-app purchases (inAppPurchasesV2) with pagination
			$inAppPurchaseProducts = $this->fetch_all_pages( $iap_api, 'queryInAppPurchasesV2', [] );

			// Fetch all subscription groups and their subscriptions with pagination
			$subscriptionGroups = $this->fetch_all_pages( $iap_api, 'listSubscriptionGroups', [] );

			if ( ! empty( $subscriptionGroups ) ) {
				foreach ( $subscriptionGroups as $group ) {
					$subscriptions = $this->fetch_all_pages( $iap_api, 'listSubscriptionsInGroup', [], $group['id'] );

					if ( ! empty( $subscriptions ) ) {
						foreach ( $subscriptions as $subscription ) {
							$products[] = array(
								'id'                 => $subscription['attributes']['productId'],
								'name'               => $subscription['attributes']['name'],
								'type'               => Helpers::platform_store_product_typeslug( 'subscription' ),
								'status'             => strtolower( $subscription['attributes']['state'] ),
								'familySharable'     => $subscription['attributes']['familySharable'],
								'subscriptionPeriod' => $subscription['attributes']['subscriptionPeriod'],
								'reviewNote'         => $subscription['attributes']['reviewNote'],
								'groupLevel'         => $subscription['attributes']['groupLevel'],
							);
						}
					}
				}
			}

			// Merge inAppPurchasesV2 with subscription products
			if ( ! empty( $inAppPurchaseProducts ) ) {
				foreach ( $inAppPurchaseProducts as $item ) {
					$products[] = array(
						'id'             => $item['attributes']['productId'],
						'name'           => $item['attributes']['name'],
						'type'           => Helpers::platform_store_product_typeslug( $item['attributes']['inAppPurchaseType'] ),
						'status'         => strtolower( $item['attributes']['state'] ),
						'familySharable' => $item['attributes']['familySharable'],
						'reviewNote'     => $item['attributes']['reviewNote'],
					);
				}
			}

		} catch ( \Throwable $e ) {
			Logger::instance()->add( 'info_log', 'BuddyBossApp\AppStores\Apple -> sync_store_product:::' . $e->getMessage() );
		}

		return $products;
	}

	/**
	 * Fetch all pages for a given API method
	 *
	 * @param object      $api_instance
	 * @param string      $method
	 * @param array       $args
	 * @param string|null $extraId
	 *
	 * @since 2.2.90
	 * @return array
	 */
	private function fetch_all_pages( $api_instance, $method, array $args = [], $extraId = null ) {
		$all_results  = [];
		$has_next     = true;
		$current_args = $args;

		while ( $has_next ) {
			$response = is_null( $extraId ) ? $api_instance->$method( $current_args ) : $api_instance->$method( $extraId, $current_args );

			if ( ! empty( $response['data'] ) ) {
				$all_results = array_merge( $all_results, $response['data'] );
			}

			// Check for next page
			$has_next = isset( $response['links']['next'] );
			if ( $has_next ) {
				// Set next page cursor
				$current_args['cursor'] = $this->get_cursor_from_link( $response['links']['next'] );
			}
		}

		return $all_results;
	}

	/**
	 * Extracts cursor parameter from pagination link.
	 *
	 * @param string $next_link
	 *
	 * @since 2.2.90
	 * @return string|null
	 */
	private function get_cursor_from_link( $next_link ) {
		$url_parts = parse_url( $next_link );
		parse_str( $url_parts['query'] ?? '', $query_params );

		return $query_params['cursor'] ?? null;
	}

	/**
	 * Refresh Bundle IDs from Apple account.
	 */
	public function refresh_bundle_ids() {
		$this->get_registered_bundles( true );
		wp_send_json_success();
	}

	/**
	 * Get Apple client app data.
	 *
	 * @param bool $override_cache Override Cache.
	 *
	 * @return array|mixed
	 * @throws ConfigException
	 */
	public function get_client_app_data( $override_cache = false ) {
		$cache_key = 'bbapp_apple_client_app_id';
		$cached    = get_site_transient( $cache_key );
		$app       = array();

		if ( false !== $cached && ! $override_cache ) {
			$app = $cached;
		} else {
			if ( $this->is_connected( $override_cache ) ) {
				$client = $this->get_client();

				try {
					$i_connect_in_apps = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectApps( $client );
					$app_args          = array(
						'limit'            => 200,
						'filter[bundleId]' => $this->get_release_bundle_id(),
					);
					$app               = $i_connect_in_apps->query( $app_args );
				} catch ( \Throwable $e ) {
					return array();
				}

				if ( ! empty( $app ) && isset( $app['data'] ) && ! is_wp_error( $app ) ) {
					set_site_transient( $cache_key, $app, 24 * HOUR_IN_SECONDS );
				}
			}
		}

		if ( ! empty( $app ) && isset( $app['data'] ) && ! is_wp_error( $app ) ) {
			return $app['data'];
		} else {
			return array();
		}
	}

	/**
	 * Get Apple client app Bundle Id.
	 *
	 * @param false $override_cache Override Cache.
	 *
	 * @return array|mixed
	 */
	public function get_client_apps_bundle_id( $override_cache = false ) {
		$cache_key = 'bbapp_apple_client_apps_bundle_id';
		$cached    = get_site_transient( $cache_key );
		$bundle_id = array();
		if ( false !== $cached && empty( $override_cache ) ) {
			$bundle_id = $cached;
		} else {
			$client_app = $this->get_client_app_data( $override_cache );
			if ( ! empty( $client_app ) ) {
				$app_bundle_data = array_column( $client_app, 'attributes', 'id' );
				$bundle_id       = ( ! empty( $app_bundle_data ) ? array_filter( array_combine( array_keys( $app_bundle_data ),
					array_column( $app_bundle_data, 'bundleId' ) ) ) : array() );
				set_site_transient( $cache_key, $bundle_id, 24 * HOUR_IN_SECONDS );
			}
		}

		return $bundle_id;
	}

	/**
	 * Check if the bundle id is client app bundle id.
	 *
	 * @param string|int $bundle_id      Bundle Id.
	 * @param bool       $override_cache Override Cache.
	 *
	 * @since 2.0.90
	 * @return bool
	 */
	public function has_client_app_bundle_id( $bundle_id, $override_cache = false ) {
		$client_apps_bundle_ids = $this->get_client_apps_bundle_id( $override_cache );

		return in_array( $bundle_id, $client_apps_bundle_ids );
	}

	/**
	 * Return the apple release bundle ID store app ID.
	 *
	 * @param bool $override_cache Override cache.
	 *
	 * @return false|int
	 */
	public function get_apple_release_bundle_store_app_id( $override_cache = false ) {
		$app               = $this->get_client_apps_bundle_id( $override_cache );
		$release_bundle_id = $this->get_release_bundle_id();

		return array_search( $release_bundle_id, $app, true );
	}

	/**
	 * Get app information.
	 *
	 * @param int   $app_id         APP ID.
	 * @param false $override_cache Override Cache.
	 *
	 * @return array|mixed
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function get_app_info( $app_id, $override_cache = false ) {
		$cache_key = 'bbapp_apple_app_info_' . $app_id;
		$cached    = get_site_transient( $cache_key );
		$appinfo   = array();

		if ( false !== $cached && empty( $override_cache ) ) {
			$appinfo = $cached;
		} else {
			if ( $this->is_connected( $override_cache ) ) {
				$client = $this->get_client();

				try {
					$i_connect_in_apps = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectApps( $client );
					$appinfo           = $i_connect_in_apps->get_app( $app_id );
				} catch ( \Throwable $e ) {
					return array();
				}

				if ( ! empty( $appinfo ) && ! is_wp_error( $appinfo ) ) {
					set_site_transient( $cache_key, $appinfo, 24 * HOUR_IN_SECONDS );
				}
			}
		}

		return $appinfo;
	}

	/**
	 * Return the root user of connected apple account.
	 *
	 * @param bool $override_cache Override cache.
	 *
	 * @return bool|string
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function get_root_apple_account( $override_cache = false ) {
		$users  = $this->get_acccount_users( $override_cache, array( 'ACCOUNT_HOLDER' ) );
		$return = false;

		if ( ! isset( $users['data'] ) ) {
			return $return;
		}

		foreach ( $users['data'] as $user ) {
			$return = $user['attributes']['username'];
		}

		return $return;
	}


	/**
	 * Get Account Users information.
	 *
	 * @param false $override_cache Override Cache.
	 * @param array $filter_roles   Array of filtered roles.
	 *
	 * @return array|mixed
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function get_acccount_users( $override_cache = false, $filter_roles = array() ) {
		$cache_key = 'bbapp_apple_client_users';
		$cached    = get_site_transient( $cache_key );
		$users     = array();

		if ( false !== $cached && empty( $override_cache ) ) {
			$users = $cached;
		} else {
			if ( $this->is_connected( $override_cache ) ) {
				$client = $this->get_client();

				try {
					$i_connect_in_apps = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectApps( $client );
					$users             = $i_connect_in_apps->get_request( '/users' );
				} catch ( \Throwable $e ) {
					return array();
				}

				if ( ! empty( $users ) && ! is_wp_error( $users ) && empty( $users['errors'] ) ) {
					set_site_transient( $cache_key, $users, 24 * HOUR_IN_SECONDS );
				}
			}
		}

		if ( ! empty( $filter_roles ) ) {
			if ( ! empty( $users['data'] ) ) {
				foreach ( $users['data'] as $k => $user ) {
					$found = false;

					if ( isset( $user['attributes'] ) || isset( $user['attributes']['roles'] ) ) {
						foreach ( $user['attributes']['roles'] as $role ) {
							if ( in_array( $role, $filter_roles, true ) ) {
								$found = true;
								continue;
							}
						}
					}

					if ( ! $found ) {
						unset( $users['data'][ $k ] );
					}
				}
			}
		}

		return $users;
	}

	/**
	 * Get Account Invited Users information.
	 *
	 * @param false $override_cache Override Cache.
	 *
	 * @return array|mixed
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function get_acccount_invited_users( $override_cache = false ) {
		$cache_key = 'bbapp_apple_client_invited_users';
		$cached    = get_site_transient( $cache_key );
		$users     = array();

		if ( false !== $cached && empty( $override_cache ) ) {
			$users = $cached;
		} else {
			if ( $this->is_connected( $override_cache ) ) {
				$client = $this->get_client();

				try {
					$i_connect_in_apps = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectApps( $client );
					$users             = $i_connect_in_apps->get_request( '/userInvitations' );
				} catch ( \Throwable $e ) {
					return array();
				}

				if ( ! empty( $users ) && ! is_wp_error( $users ) && empty( $users['errors'] ) ) {
					set_site_transient( $cache_key, $users, 24 * HOUR_IN_SECONDS );
				}
			}
		}

		return $users;
	}

	/**
	 * Get app versions information.
	 *
	 * @param int   $app_id         APP ID.
	 * @param false $override_cache Override Cache.
	 *
	 * @return array|mixed
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function get_app_store_versions_for_an_app( $app_id, $override_cache = false ) {
		$cache_key  = 'bbapp_apple_app_store_versions_for_an_app_' . $app_id;
		$cached     = get_site_transient( $cache_key );
		$appvesions = array();

		if ( false !== $cached && empty( $override_cache ) ) {
			$appvesions = $cached;
		} else {
			if ( $this->is_connected( $override_cache ) ) {
				$client = $this->get_client();

				try {
					$i_connect_in_apps = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectApps( $client );
					$appvesions        = $i_connect_in_apps->get_request( "/apps/{$app_id}/appStoreVersions", array( 'filter' => array( 'platform' => 'IOS' ) ) );
				} catch ( \Throwable $e ) {
					return array();
				}

				if ( ! empty( $appvesions ) && ! is_wp_error( $appvesions ) ) {
					set_site_transient( $cache_key, $appvesions, 24 * HOUR_IN_SECONDS );
				}
			}
		}

		return $appvesions;
	}

	/**
	 * Get Apple team info.
	 *
	 * @param false $override_cache Override Cache.
	 *
	 * @return mixed|string[]
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function get_apple_team_info( $override_cache = false ) {
		$cache_key = 'bbapp_apple_team_info';
		$cached    = get_site_transient( $cache_key );
		$app_info  = array(
			'key_id'         => '',
			'account_holder' => '',
		);

		if ( false !== $cached && empty( $override_cache ) ) {
			$app_info = $cached;
		} else {
			$app_info['key_id'] = \BuddyBossApp\AppSettings::instance()->get_setting_value( 'publish.ios.key_id' );
			$user_info          = $this->get_acccount_users( $override_cache );
			$user_name          = '';

			if ( ! empty( $user_info ) && ! is_wp_error( $user_info ) && ! empty( $user_info['data'] ) ) {
				$users = ( ! empty( $user_info['data'] ) ? array_column( $user_info['data'], 'attributes', 'id' ) : array() );

				if ( ! empty( $users ) ) {
					foreach ( $users as $user ) {
						if ( ! empty( $user['roles'] ) && in_array( 'ACCOUNT_HOLDER', $user['roles'], true ) && empty( $user_name ) ) {
							$user_name = $user['firstName'] . ' ' . $user['lastName'];
							continue;
						}
					}
				}
			}

			$app_info['account_holder'] = $user_name;

			set_site_transient( $cache_key, $app_info, 24 * HOUR_IN_SECONDS );
		}

		return $app_info;
	}

	/**
	 * Validate App.
	 *
	 * @param false $override_cache Override Cache.
	 *
	 * @return array|mixed
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function get_required_app_details( $override_cache = false ) {
		$app_id    = $this->get_apple_release_bundle_store_app_id( $override_cache );
		$cache_key = 'bbapp_apple_app_details_' . $app_id;
		$cached    = get_site_transient( $cache_key );

		if ( false !== $cached && empty( $override_cache ) ) {
			return $cached;
		} else {
			$app_data = array(
				'name'             => '',
				'subtitle'         => '',
				'primaryCategory'  => '',
				'LicenseAgreement' => '',
				'appPrices'        => false,
				'privacyPolicyUrl' => '',
				'appStoreState'    => '',
			);

			if ( $this->is_connected( $override_cache ) ) {
				$client            = $this->get_client();
				$i_connect_in_apps = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectApps( $client );

				$param = array(
					'include' => 'appInfos,endUserLicenseAgreement,betaAppLocalizations,appStoreVersions',
					'fields'  => array(
						'appInfos'             => 'appInfoLocalizations,appStoreState',
						'betaAppLocalizations' => 'privacyPolicyUrl',
						'appStoreVersions'     => 'ageRatingDeclaration',
					),
				);

				$appinfos   = $i_connect_in_apps->get_request( "/apps/{$app_id}", $param );
				$app_prices = $i_connect_in_apps->get_request(
					"/apps/{$app_id}/appPriceSchedule",
					array(
						'include' => 'automaticPrices,baseTerritory,manualPrices',
						'fields'  => array(
							'appPriceSchedules' => 'automaticPrices,baseTerritory,manualPrices',
							'appPrices'         => 'appPricePoint,endDate,manual,startDate,territory',
						),
					)
				);

				$price_data = ( ! empty( $app_prices ) && ! is_wp_error( $app_prices ) && ! empty( $app_prices['included'] ) ) ? $app_prices['included'] : array();

				if ( ! empty( $price_data ) ) {
					$price_data = array_column( $price_data, 'type', 'id' );
				}
				$appinfo_id      = ( ! empty( $appinfos ) && ! is_wp_error( $appinfos ) && isset( $appinfos['data']['relationships']['appInfos']['data'][0]['id'] ) ? $appinfos['data']['relationships']['appInfos']['data'][0]['id'] : '' );
				$include_data    = ( ! empty( $appinfos ) && ! is_wp_error( $appinfos ) && isset( $appinfos['included'] ) ) ? $appinfos['included'] : array();
				$applocalization = $i_connect_in_apps->get_request( "/appInfos/{$appinfo_id}/appInfoLocalizations" );

				if ( ! empty( $applocalization ) && ! is_wp_error( $applocalization ) && isset( $applocalization['data'][0]['attributes'] ) ) {
					$app_data = array_merge( $app_data, $applocalization['data'][0]['attributes'] );
				}

				$primary_category = $i_connect_in_apps->get_request( "/appInfos/{$appinfo_id}/primaryCategory" );
				if ( ! empty( $primary_category ) && ! is_wp_error( $primary_category ) && isset( $primary_category['data']['id'] ) ) {
					$app_data['primaryCategory'] = $primary_category['data']['id'];
				}

				$age_id = '';

				if ( ! empty( $include_data ) ) {
					foreach ( $include_data as $data ) {
						if ( ! empty( $price_data ) && in_array( 'appPrices', $price_data, true ) ) {
							$app_data['appPrices'] = true;
						}

						if ( 'appInfos' === $data['type'] && isset( $data['attributes']['appStoreState'] ) ) {
							$app_data['appStoreState'] = $data['attributes']['appStoreState'];
						}

						if ( 'appStoreVersions' === $data['type'] && isset( $data['id'] ) && empty( $age_id ) ) {
							$age_id = $data['id'];
						}
					}
				}

				$app_data['contentRightsDeclaration'] = ( ! empty( $appinfos ) && ! is_wp_error( $appinfos ) && isset( $appinfos['data']['attributes']['contentRightsDeclaration'] ) ? $appinfos['data']['attributes']['contentRightsDeclaration'] : '' );
				$age_rating                           = $i_connect_in_apps->get_request( "/appStoreVersions/{$age_id}/ageRatingDeclaration" );
				$app_data['ageRatingDeclarations']    = ( ! empty( $age_rating ) && ! is_wp_error( $age_rating ) && isset( $age_rating['data']['attributes'] ) ? $age_rating['data']['attributes'] : '' );
			}

			set_site_transient( $cache_key, $app_data, 24 * HOUR_IN_SECONDS );
		}

		return $app_data;
	}

	/**
	 * Validate App.
	 *
	 * @param false $override_cache Override Cache.
	 *
	 * @return array
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function validate_app( $override_cache = false ) {
		$app_details = $this->get_required_app_details( $override_cache );
		$app_id      = $this->get_apple_release_bundle_store_app_id( $override_cache );
		$errors      = array();
		$builds      = \BuddyBossApp\Build::instance()->get_app_builds( 'ios', 'live', false, 'completed', 1, 10, $override_cache );

		if ( ! is_wp_error( $builds ) && empty( $builds['data'] ) ) {
			$errors[] = array(
				'headline'    => __( 'No iOS Release App Builds', 'buddyboss-app' ),
				'description' => __( 'Please generate an iOS release app build to publish.', 'buddyboss-app' ),
				'link'        => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-build&setting=request-build' ) ),
			);
		}

		if ( ! isset( $app_details['name'] ) || empty( $app_details['name'] ) ) {
			$errors[] = array(
				'headline'    => __( 'App Name Missing', 'buddyboss-app' ),
				'description' => __( 'Please enter a name for your app in at least one localization.', 'buddyboss-app' ),
				'link'        => esc_url( 'https://appstoreconnect.apple.com/apps/' . $app_id . '/appstore/info' ),
			);
		}

		if ( ! isset( $app_details['subtitle'] ) || empty( $app_details['subtitle'] ) ) {
			$errors[] = array(
				'headline'    => __( 'Subtitle Missing', 'buddyboss-app' ),
				'description' => __( 'Please enter a name for your app in at least one localization.', 'buddyboss-app' ),
				'link'        => esc_url( 'https://appstoreconnect.apple.com/apps/' . $app_id . '/appstore/info' ),
			);
		}

		if ( ! isset( $app_details['primaryCategory'] ) || empty( $app_details['primaryCategory'] ) ) {
			$errors[] = array(
				'headline'    => __( 'Primary Category Missing', 'buddyboss-app' ),
				'description' => __( 'Please select a primary category for your app.', 'buddyboss-app' ),
				'link'        => esc_url( 'https://appstoreconnect.apple.com/apps/' . $app_id . '/appstore/info' ),
			);
		}

		if ( ! isset( $app_details['appPrices'] ) || empty( $app_details['appPrices'] ) ) {
			$errors[] = array(
				'headline'    => __( 'No Price Schedule Set', 'buddyboss-app' ),
				'description' => __( 'Please enter pricing and availability information for your app.', 'buddyboss-app' ),
				'link'        => esc_url( 'https://appstoreconnect.apple.com/apps/' . $app_id . '/appstore/pricing' ),
			);
		}

		if ( ! isset( $app_details['privacyPolicyUrl'] ) || empty( $app_details['privacyPolicyUrl'] ) ) {
			$errors[] = array(
				'headline'    => __( 'Privacy Policy URL Missing', 'buddyboss-app' ),
				'description' => __( 'Please enter the URL to your privacy policy.', 'buddyboss-app' ),
				'link'        => esc_url( 'https://appstoreconnect.apple.com/apps/' . $app_id . '/appstore/privacy' ),
			);
		}

		if ( ! isset( $app_details['contentRightsDeclaration'] ) || empty( $app_details['contentRightsDeclaration'] ) ) {
			$errors[] = array(
				'headline'    => __( 'No Content Rights Set', 'buddyboss-app' ),
				'description' => __( 'Please set your app\'s contents rights information.', 'buddyboss-app' ),
				'link'        => esc_url( 'https://appstoreconnect.apple.com/apps/' . $app_id . '/appstore/info' ),
			);
		}

		$ages = array();

		if ( ! empty( $app_details['ageRatingDeclarations'] ) ) {
			$rating_key = $this->get_ageinfo_labels();
			foreach ( $rating_key as $key => $value ) {
				$ages[ $value ] = ( isset( $app_details['ageRatingDeclarations'][ $key ] ) ? $app_details['ageRatingDeclarations'][ $key ] : '' );
			}
		}

		if ( empty( $ages ) || empty( array_filter( $ages ) ) ) {
			$errors[] = array(
				'headline'    => __( 'No Age Rating Set', 'buddyboss-app' ),
				'description' => __( 'Please set an age rating for your app.', 'buddyboss-app' ),
				'link'        => esc_url( 'https://appstoreconnect.apple.com/apps/' . $app_id . '/appstore/info' ),
			);
		}

		$approved_status_app = array(
			'READY_FOR_SALE',
			'PENDING_APPLE_RELEASE',
			'PENDING_DEVELOPER_RELEASE',
			'PREORDER_READY_FOR_SALE',
		);

		if ( isset( $app_details['appStoreState'] ) && ! empty( $app_details['appStoreState'] ) && in_array( $app_details['appStoreState'], $approved_status_app, true ) ) {
			$errors[] = array(
				'headline'    => __( 'App Already Approved', 'buddyboss-app' ),
				'description' => __( 'Your app has already been approved by Apple. To publish an update, please create and prepare a new version.', 'buddyboss-app' ),
				'link'        => esc_url( 'https://appstoreconnect.apple.com/apps/' . $app_id . '/appstore' ),
			);
		}

		$app_not_read_for_review = array(
			'DEVELOPER_REMOVED_FROM_SALE',
			'DEVELOPER_REJECTED',
			'REMOVED_FROM_SALE',
			'REPLACED_WITH_NEW_VERSION',
		);

		if ( isset( $app_details['appStoreState'] ) && ! empty( $app_details['appStoreState'] ) && in_array( $app_details['appStoreState'], $app_not_read_for_review, true ) ) {
			$errors[] = array(
				'headline'    => __( 'App Not Ready for Review', 'buddyboss-app' ),
				'description' => __( 'Your app doesn\'t have a new version ready for review. Please create and prepare a new version.', 'buddyboss-app' ),
				'link'        => esc_url( 'https://appstoreconnect.apple.com/apps/' . $app_id . '/appstore' ),
			);
		}

		$app_in_review = array(
			'IN_REVIEW',
			'INVALID_BINARY',
			'METADATA_REJECTED',
			'REJECTED',
			'PROCESSING_FOR_APP_STORE',
			'WAITING_FOR_EXPORT_COMPLIANCE',
			'WAITING_FOR_REVIEW',
		);

		if ( isset( $app_details['appStoreState'] ) && ! empty( $app_details['appStoreState'] ) && in_array( $app_details['appStoreState'], $app_in_review, true ) ) {
			$errors[] = array(
				'headline'    => __( 'App Release In Progress', 'buddyboss-app' ),
				'description' => __( 'You have a release in progress. Please cancel this release or wait until it has been approved by Apple.', 'buddyboss-app' ),
				'link'        => esc_url( 'https://appstoreconnect.apple.com/apps/' . $app_id . '/appstore' ),
			);
		}

		if ( isset( $app_details['appStoreState'] ) && ! empty( $app_details['appStoreState'] ) && 'PENDING_CONTRACT' === $app_details['appStoreState'] ) {
			$errors[] = array(
				'headline'    => __( 'Agreements Not Signed', 'buddyboss-app' ),
				'description' => __( 'You have agreements that need to be signed inside your Apple Developer account.', 'buddyboss-app' ),
				'link'        => esc_url( 'https://appstoreconnect.apple.com/agreements' ),
			);
		}

		$user_added      = $this->is_bb_app_user_added( true );
		$user_invited    = $this->is_bb_app_user_invited( true );
		$account_created = false;

		if ( empty( $user_added ) && empty( $user_invited ) ) {
			$account_created = $this->create_acccount_users( true );
		}

		if ( empty( $user_added ) && empty( $user_invited ) && empty( $account_created ) ) {
			ob_start();
			?>
            <div class="error-bb-user">
                <p><?php echo esc_html__( 'To continue, please invite the BuddyBoss Publishing Team to manage your app in App Store Connect.', 'buddyboss-app' ); ?></p>
                <ul>
                    <li><?php echo sprintf( '%1$s <a href="https://appstoreconnect.apple.com/access/users" target="_blank">%2$s</a> %3$s',
							esc_html__( 'Go to', 'buddyboss-app' ),
							esc_html__( 'Users and Access', 'buddyboss-app' ),
							esc_html__( 'in App Store Connect', 'buddyboss-app' ) ); ?></li>
                    <li><?php echo esc_html__( 'In the People tab, click on the + symbol', 'buddyboss-app' ); ?></li>
                    <li><?php echo esc_html__( 'In the First and Last Name fields, enter BuddyBoss Team', 'buddyboss-app' ); ?></li>
                    <li>
						<?php
						echo esc_html__( 'In the Email field, enter', 'buddyboss-app' ) . ' ' . esc_html( ManageApp::instance()->get_appcenter_publisher_email_address() );
						?>
                    </li>
                    <li><?php echo esc_html__( 'Under Roles, select App Manager', 'buddyboss-app' ); ?></li>
                    <li><?php echo esc_html__( 'Under App, select your app', 'buddyboss-app' ); ?></li>
                    <li><?php echo esc_html__( 'Click Invite', 'buddyboss-app' ); ?></li>
                </ul>
            </div>
			<?php
			$desc     = ob_get_clean();
			$errors[] = array(
				'headline'    => __( 'Invite the BuddyBoss Publishing Team', 'buddyboss-app' ),
				'description' => $desc,
				'link'        => esc_url( 'https://appstoreconnect.apple.com/access/users' ),
			);
		}

		return $errors;
	}

	/**
	 * Fetch App data.
	 *
	 * @param int   $app_id         App ID.
	 * @param false $override_cache Override Cache.
	 *
	 * @return array|mixed
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function fetch_app_data( $app_id = 0, $override_cache = false ) {
		if ( empty( $app_id ) ) {
			$app_id = $this->get_apple_release_bundle_store_app_id( $override_cache );
		}

		$app_data = array();

		if ( ! empty( $app_id ) ) {
			$cache_key = 'bbapp_apple_app_fetched_data_' . $app_id;
			$cached    = get_site_transient( $cache_key );

			if ( false !== $cached && empty( $override_cache ) ) {
				return $cached;
			} else {
				$app_data['apple_app_id'] = $app_id;
				if ( $this->is_connected( $override_cache ) ) {
					$client            = $this->get_client();

					$i_connect_in_apps = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectApps( $client );
					$app_versions      = $i_connect_in_apps->get_request(
						"/apps/{$app_id}/appStoreVersions",
						array(
							'filter' => array( 'platform' => 'IOS' ),
						)
					);

					if ( ! empty( $app_versions ) && ! is_wp_error( $app_versions ) ) {
						$prepare_app            = ( ! empty( $app_versions['data'] ) ? current( $app_versions['data'] ) : array() );
						$prepare_version_id     = isset( $prepare_app['id'] ) ? $prepare_app['id'] : '';
						$localization_id        = '';
						$primary_local          = '';
						$app_data['version_id'] = $prepare_version_id;

						if ( ! empty( $prepare_version_id ) ) {
							$app_data = array_merge( $app_data, $prepare_app['attributes'] );

							$localizations = $i_connect_in_apps->get_request(
								"/appStoreVersions/{$prepare_version_id}",
								array(
									'fields'  => array( 'appStoreVersions' => 'appStoreVersionLocalizations' ),
									'include' => 'app,appStoreVersionLocalizations,appStoreReviewDetail,appStoreVersionPhasedRelease,appStoreVersionSubmission,routingAppCoverage',
								)
							);

							if ( ! empty( $localizations ) && ! is_wp_error( $localizations ) && ! empty( $localizations['included'] ) ) {
								foreach ( $localizations['included'] as $localization ) {
									if ( isset( $localization['type'] ) && 'apps' === $localization['type'] && empty( $primary_local ) ) {
										$primary_local = $localization['attributes']['primaryLocale'];
									}

									if ( isset( $localization['type'] ) && 'apps' === $localization['type'] && ! empty( $localization['attributes']['subscriptionStatusUrl'] ) && ! empty( $localization['attributes']['subscriptionStatusUrlVersion'] ) ) {
										$app_data['server_notification']['production']['url']     = $localization['attributes']['subscriptionStatusUrl'];
										$app_data['server_notification']['production']['version'] = $localization['attributes']['subscriptionStatusUrlVersion'];
									}

									if ( isset( $localization['type'] ) && 'apps' === $localization['type'] && ! empty( $localization['attributes']['subscriptionStatusUrlForSandbox'] ) && ! empty( $localization['attributes']['subscriptionStatusUrlVersionForSandbox'] ) ) {
										$app_data['server_notification']['sandbox']['url']     = $localization['attributes']['subscriptionStatusUrlForSandbox'];
										$app_data['server_notification']['sandbox']['version'] = $localization['attributes']['subscriptionStatusUrlVersionForSandbox'];
									}

									if ( isset( $localization['type'] ) && 'appStoreVersionLocalizations' === $localization['type'] && empty( $localization_id ) && ! empty( $primary_local ) && isset( $localization['attributes']['locale'] ) && $localization['attributes']['locale'] === $primary_local ) {
										$localization_id = $localization['id'];
										$app_data        = array_merge( $app_data, $localization['attributes'] );
									}

									if ( isset( $localization['type'] ) && 'appStoreReviewDetails' === $localization['type'] && isset( $localization['attributes'] ) ) {
										$app_store_review_details_id = $localization['id'];
										$app_data                    = array_merge( $app_data, $localization['attributes'] );

										if ( ! empty( $app_store_review_details_id ) ) {
											$attachments = $i_connect_in_apps->get_request( "/appStoreReviewDetails/{$app_store_review_details_id}/appStoreReviewAttachments" );
											if ( ! empty( $attachments ) && ! is_wp_error( $attachments ) && ! empty( $attachments['data'] ) ) {
												$attachment = current( $attachments['data'] )['attributes'];
												if ( ! empty( $attachment ) ) {
													$app_data = array_merge( $app_data, array( 'attachment' => $attachment ) );
												}
											}
										}
									}
								}
							}

							$screenshots = $i_connect_in_apps->get_request(
								"/appStoreVersionLocalizations/{$localization_id}/appScreenshotSets",
								array( 'include' => 'appScreenshots' )
							);

							$screenshot_type = array();

							if ( ! empty( $screenshots ) && ! is_wp_error( $screenshots ) && ! empty( $screenshots['data'] ) ) {
								foreach ( $screenshots['data'] as $screenshot ) {
									if ( isset( $screenshot['attributes']['screenshotDisplayType'] ) ) {
										$screenshot_type[ $screenshot['attributes']['screenshotDisplayType'] ] = array();

										if ( isset( $screenshot['relationships']['appScreenshots']['data'] ) ) {
											foreach ( $screenshot['relationships']['appScreenshots']['data'] as $screen ) {
												$screenshot_type[ $screenshot['attributes']['screenshotDisplayType'] ][ $screen['id'] ] = '';
											}
										}
									}
								}

								if ( ! empty( $screenshots['included'] ) ) {
									$included_screenshot = array_column( $screenshots['included'], 'attributes', 'id' );
									if ( ! empty( $included_screenshot ) && ! empty( $screenshot_type ) ) {
										foreach ( $screenshot_type as $key => $screenshot_ids ) {
											if ( ! empty( $screenshot_ids ) ) {
												foreach ( $screenshot_ids as $screenshot_id => $value ) {
													if ( array_key_exists( $screenshot_id, $included_screenshot ) ) {
														$screenshot_type[ $key ][ $screenshot_id ] = $included_screenshot[ $screenshot_id ];
													}
												}
											}
										}

										$app_data = array_merge( $app_data, array( 'screenshots' => $screenshot_type ) );
									}
								}
							}
						}

						if ( is_array( $app_versions['data'] ) && ! empty( $app_versions['data'] ) && count( $app_versions['data'] ) < 2 && isset( $app_data['whatsNew'] ) ) {
							unset( $app_data['whatsNew'] );
						}
					}

					set_site_transient( $cache_key, $app_data, 24 * HOUR_IN_SECONDS );
				}
			}
		}

		return $app_data;
	}


	/**
	 * Display App data.
	 *
	 * @param false  $override_cache Override Cache.
	 * @param string $app_id         App id.
	 *
	 * @return array
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function display_app_data( $override_cache = false, $app_id = '' ) {
		$display_data = array();

		$app_data = $this->fetch_app_data( $app_id, $override_cache );

		if ( ! empty( $app_data ) ) {
			$display_data['app_title'] = array(
				'label'    => __( 'iOS App', 'buddyboss-app' ),
				'type'     => '',
				'value'    => ( ! empty( $app_data['versionString'] ) ? $app_data['versionString'] : '' ),
				'validate' => true,
				'error'    => '',
			);

			$display_data['app_status'] = array(
				'label'    => '',
				'type'     => '',
				'value'    => ( ! empty( $app_data['appStoreState'] ) ? $this->get_apple_publish_status_label( $app_data['appStoreState'] ) : '' ),
				'validate' => true,
				'error'    => '',
			);

			$display_data['app_screenshots'] = array(
				'label'    => __( 'App Screenshots', 'buddyboss-app' ),
				'type'     => 'headline',
				'value'    => '',
				'validate' => true,
				'error'    => '',
			);
			// Get app screenshot list from function
			$app_screenshot_keys = $this->app_screenshot_list();

			// Check for required screenshots conflict between app_iphone_67 and app_iphone_65
			if ( ! empty( $app_data['screenshots'] ) ) {
				$has_iphone_67 = ! empty( $app_data['screenshots'][ strtoupper( 'app_iphone_67' ) ] );
				$has_iphone_65 = ! empty( $app_data['screenshots'][ strtoupper( 'app_iphone_65' ) ] );

				// Adjust `required` logic dynamically
				if ( $has_iphone_67 ) {
					$app_screenshot_keys['app_iphone_65']['required'] = false;
				} elseif ( $has_iphone_65 ) {
					$app_screenshot_keys['app_iphone_67']['required'] = false;
				}
			}

			if ( ! empty( $app_data['screenshots'] ) ) {
				foreach ( $app_data['screenshots'] as $key => $screenshot ) {
					$label = '';

					if ( array_key_exists( strtolower( $key ), $app_screenshot_keys ) ) {
						$label = $app_screenshot_keys[ strtolower( $key ) ]['label'];
					}

					if ( ! empty( $label ) ) {
						$screenshots = array();
						if ( ! empty( $screenshot ) ) {
							foreach ( $screenshot as $k => $screen ) {
								if ( isset( $screen['assetDeliveryState']['state'] ) && 'COMPLETE' === $screen['assetDeliveryState']['state'] ) {
									$image     = $screen['imageAsset']['templateUrl'];
									$height    = $screen['imageAsset']['height'];
									$width     = $screen['imageAsset']['width'];
									$filename  = $screen['fileName'];
									$filename  = array_filter( explode( '.', $filename ) );
									$extension = end( $filename );

									$image         = str_replace( '{w}', $width, $image );
									$image         = str_replace( '{h}', $height, $image );
									$image         = str_replace( '{f}', $extension, $image );
									$screenshots[] = $image;
								}
							}
						}

						if ( array_key_exists( strtolower( $key ), $app_screenshot_keys ) ) {
							$display_data[ strtolower( $key ) ] = array(
								'label'    => $label,
								'type'     => $app_screenshot_keys[ strtolower( $key ) ]['type'],
								'required' => $app_screenshot_keys[ strtolower( $key ) ]['required'],
								'value'    => $screenshots,
								'validate' => ! empty( $screenshots ),
								'error'    => sprintf( '%1$s %2$s.', esc_html__( 'Please provide at least one screenshot for', 'buddyboss-app' ), $label ),
							);
							unset( $app_screenshot_keys[ strtolower( $key ) ] );
						}
					}
				}
			}

			if ( ! empty( $app_screenshot_keys ) ) {
				foreach ( $app_screenshot_keys as $key => $data ) {
					// Skip non-required screenshots
					if ( ! $data['required'] ) {
						continue;
					}

					$label                = $data['label'];
					$display_data[ $key ] = array(
						'label'    => $label,
						'type'     => $data['type'],
						'required' => $data['required'],
						'value'    => array(),
						'validate' => false,
						'error'    => sprintf( '%1$s %2$s.', esc_html__( 'Please provide at least one screenshot for', 'buddyboss-app' ), $label ),
					);
				}
			}

			$display_data['version_information'] = array(
				'label'    => __( 'Version Information', 'buddyboss-app' ),
				'type'     => 'headline',
				'value'    => '',
				'validate' => true,
				'error'    => '',
			);

			$display_data['promotional_text'] = array(
				'label'    => __( 'Promotional Text', 'buddyboss-app' ),
				'type'     => '',
				'value'    => ( ! empty( $app_data['promotionalText'] ) ? $app_data['promotionalText'] : '' ),
				'validate' => ! empty( $app_data['promotionalText'] ),
				'error'    => __( 'Please enter promotional text for your app.', 'buddyboss-app' ),
			);

			$display_data['description'] = array(
				'label'    => __( 'Description', 'buddyboss-app' ),
				'type'     => '',
				'value'    => ( ! empty( $app_data['description'] ) ? nl2br( $app_data['description'] ) : '' ),
				'validate' => ! empty( $app_data['description'] ),
				'error'    => __( 'Please enter a description for your app.', 'buddyboss-app' ),
			);

			$display_data['keywords'] = array(
				'label'    => __( 'Keywords', 'buddyboss-app' ),
				'type'     => '',
				'value'    => ( ! empty( $app_data['keywords'] ) ? $app_data['keywords'] : '' ),
				'validate' => ! empty( $app_data['keywords'] ),
				'error'    => __( 'Please enter keywords for your app.', 'buddyboss-app' ),
			);

			$display_data['support_url'] = array(
				'label'    => __( 'Support URL', 'buddyboss-app' ),
				'type'     => '',
				'value'    => ( ! empty( $app_data['supportUrl'] ) ? $app_data['supportUrl'] : '' ),
				'validate' => ! empty( $app_data['supportUrl'] ),
				'error'    => __( 'Please enter a support URL for your app.', 'buddyboss-app' ),
			);

			$display_data['marketing_url'] = array(
				'label'    => __( 'Marketing URL', 'buddyboss-app' ),
				'type'     => '',
				'value'    => ( ! empty( $app_data['marketingUrl'] ) ? $app_data['marketingUrl'] : '' ),
				'validate' => ! empty( $app_data['marketingUrl'] ),
				'error'    => __( 'Please enter a marketing URL for your app.', 'buddyboss-app' ),
			);

			if ( isset( $app_data['whatsNew'] ) ) {
				$display_data['whats_new'] = array(
					'label'    => __( 'What\'s New In This Version', 'buddyboss-app' ),
					'type'     => '',
					'value'    => ( ! empty( $app_data['whatsNew'] ) ? $app_data['whatsNew'] : '' ),
					'validate' => ! empty( $app_data['whatsNew'] ),
					'error'    => __( 'Please enter what\'s new in this version.', 'buddyboss-app' ),
				);
			}

			$display_data['general_app_info'] = array(
				'label'    => __( 'General App Information', 'buddyboss-app' ),
				'type'     => 'headline',
				'value'    => '',
				'validate' => true,
				'error'    => '',
			);

			$display_data['copyright'] = array(
				'label'    => __( 'Copyright', 'buddyboss-app' ),
				'type'     => '',
				'value'    => ( ! empty( $app_data['copyright'] ) ? $app_data['copyright'] : '' ),
				'validate' => ! empty( $app_data['copyright'] ),
				'error'    => __( 'Please enter copyright information for your app.', 'buddyboss-app' ),
			);

			$display_data['version'] = array(
				'label'    => __( 'Version', 'buddyboss-app' ),
				'type'     => '',
				'value'    => ( ! empty( $app_data['versionString'] ) ? $app_data['versionString'] : '' ),
				'validate' => ! empty( $app_data['versionString'] ),
				'error'    => __( 'Please enter Version information for your app.', 'buddyboss-app' ),
			);

			$display_data['app_review_information'] = array(
				'label'    => __( 'App Review Information', 'buddyboss-app' ),
				'type'     => 'headline',
				'value'    => '',
				'validate' => true,
				'error'    => '',
			);

			if ( ! empty( $app_data['demoAccountRequired'] ) ) {
				$display_data['account_signin_required'] = array(
					'label'    => __( 'Sign-In Required', 'buddyboss-app' ),
					'type'     => '',
					'value'    => ( ! empty( $app_data['demoAccountRequired'] ) ? __( 'Yes', 'buddyboss-app' ) : __( 'No', 'buddyboss-app' ) ),
					'validate' => ! empty( $app_data['demoAccountRequired'] ),
					'error'    => __( 'Please specify if your app requires a signing in to access the content.', 'buddyboss-app' ),
				);

				$display_data['account_signin_information'] = array(
					'label'    => __( 'Sign-In Information', 'buddyboss-app' ),
					'type'     => '',
					'value'    => ( ! empty( $app_data['demoAccountName'] ) ? $app_data['demoAccountName'] : '' ) . '<br/>' . ( ! empty( $app_data['demoAccountPassword'] ) ? $app_data['demoAccountPassword'] : '' ),
					'validate' => ! empty( $app_data['demoAccountName'] ) && ! empty( $app_data['demoAccountPassword'] ),
					'error'    => __( 'Please enter sign-in information for your app.', 'buddyboss-app' ),
				);
			}

			$display_data['contact_information'] = array(
				'label'    => __( 'Contact Information', 'buddyboss-app' ),
				'type'     => '',
				'value'    => ( ! empty( $app_data['contactFirstName'] ) ? $app_data['contactFirstName'] . '<br/>' : '' ) . ( ! empty( $app_data['contactLastName'] ) ? $app_data['contactLastName'] . '<br/>' : '' ) . ( ! empty( $app_data['contactPhone'] ) ? $app_data['contactPhone'] . '<br/>' : '' ) . ( ! empty( $app_data['contactEmail'] ) ? $app_data['contactEmail'] : '' ),
				'validate' => ! empty( $app_data['contactFirstName'] ) && ! empty( $app_data['contactLastName'] ) && ! empty( $app_data['contactPhone'] ) && ! empty( $app_data['contactEmail'] ),
				'error'    => __( 'Please enter contact information for Apple to contact you with.', 'buddyboss-app' ),
			);

			$display_data['notes'] = array(
				'label'    => __( 'Notes', 'buddyboss-app' ),
				'type'     => '',
				'value'    => ( ! empty( $app_data['notes'] ) ? $app_data['notes'] : '' ),
				'validate' => ! empty( $app_data['notes'] ),
				'error'    => __( 'Please enter notes to help the Apple review team understand and navigate your app.', 'buddyboss-app' ),
			);

			if ( ! empty( $app_data['attachment'] ) && isset( $app_data['attachment']['assetDeliveryState']['state'] ) && 'COMPLETE' === $app_data['attachment']['assetDeliveryState']['state'] ) {
				$display_data['attachments'] = array(
					'label'    => __( 'Attachments', 'buddyboss-app' ),
					'type'     => '',
					'value'    => ( ( ! empty( $app_data['attachment'] ) && isset( $app_data['attachment']['assetDeliveryState']['state'] ) && 'COMPLETE' === $app_data['attachment']['assetDeliveryState']['state'] ) ? $app_data['attachment']['fileName'] : '' ),
					'validate' => ! empty( $app_data['attachment'] ) && isset( $app_data['attachment']['assetDeliveryState']['state'] ) && 'COMPLETE' === $app_data['attachment']['assetDeliveryState']['state'] && ! empty( $app_data['attachment']['fileName'] ),
					'error'    => __( 'Please enter notes to help the Apple review team understand and navigate your app.', 'buddyboss-app' ),
				);
			}

			$display_data['version_release'] = array(
				'label'    => __( 'Version Release', 'buddyboss-app' ),
				'type'     => 'headline',
				'value'    => '',
				'validate' => true,
				'error'    => '',
			);

			$display_data['release_schedule'] = array(
				'label'    => __( 'Release Schedule', 'buddyboss-app' ),
				'type'     => '',
				'value'    => ( ! empty( $app_data['releaseType'] ) ? $this->get_release_type_label( $app_data['releaseType'] ) : '' ) . ( ! empty( $app_data['earliestReleaseDate'] ) && 'SCHEDULED' === $app_data['releaseType'] ? ' ' . gmdate( 'F j, Y g:i A',
							strtotime( $app_data['earliestReleaseDate'] ) ) . ' (GMT)' : '' ),
				'validate' => true,
				'error'    => '',
			);
		}

		return $display_data;
	}

	/**
	 * Get Apple Publish status.
	 *
	 * @param string $key Key.
	 *
	 * @return mixed
	 */
	public function get_apple_publish_status_label( $key ) {
		$status_code = array(
			'PREPARE_FOR_SUBMISSION'        => __( 'Prepare for Submission', 'buddyboss-app' ),
			'WAITING_FOR_REVIEW'            => __( 'Waiting for Review', 'buddyboss-app' ),
			'IN_REVIEW'                     => __( 'In Review', 'buddyboss-app' ),
			'PENDING_AGREEMENT'             => __( 'Pending Agreement', 'buddyboss-app' ),
			'WAITING_FOR_EXPORT_COMPLIANCE' => __( 'Waiting for Export Compliance', 'buddyboss-app' ),
			'PENDING_DEVELOPER_RELEASE'     => __( 'Pending Developer Release', 'buddyboss-app' ),
			'PROCESSING_FOR_APP_STORE'      => __( 'Processing for App Store', 'buddyboss-app' ),
			'PENDING_APPLE_RELEASE'         => __( 'Pending Apple Release', 'buddyboss-app' ),
			'PENDING_APP_TRANSFER'          => __( 'Pending App Transfer', 'buddyboss-app' ),
			'PROCESSING_APP_TRANSFER'       => __( 'Processing App Transfer', 'buddyboss-app' ),
			'READY_FOR_SALE'                => __( 'Ready for Sale', 'buddyboss-app' ),
			'PRE_ORDER_READY_FOR_SALE'      => __( 'Pre-Order Ready for Sale', 'buddyboss-app' ),
			'PREORDER_READY_FOR_SALE'       => __( 'Pre-Order Ready for Sale', 'buddyboss-app' ),
			'REJECTED'                      => __( 'Rejected', 'buddyboss-app' ),
			'METADATA_REJECTED'             => __( 'Metadata Rejected', 'buddyboss-app' ),
			'REMOVED_FROM_SALE'             => __( 'Removed from Sale', 'buddyboss-app' ),
			'DEVELOPER_REJECTED'            => __( 'Developer Rejected', 'buddyboss-app' ),
			'DEVELOPER_REMOVED_FROM_SALE'   => __( 'Developer Removed from Sale', 'buddyboss-app' ),
			'REPLACED_WITH_NEW_VERSION'     => __( 'Replaced with New Version', 'buddyboss-app' ),
			'INVALID_BINARY'                => __( 'Invalid Binary', 'buddyboss-app' ),
		);

		return ( isset( $status_code[ $key ] ) ? $status_code[ $key ] : $key );
	}

	/**
	 * Get Apple Release label.
	 *
	 * @param string $key Key.
	 *
	 * @return mixed
	 */
	public function get_release_type_label( $key ) {
		$status_code = array(
			'MANUAL'         => __( 'Manually release this version', 'buddyboss-app' ),
			'AFTER_APPROVAL' => __( 'Automatically release this version', 'buddyboss-app' ),
			'SCHEDULED'      => __( 'Automatically release this version after App Review, no earlier than', 'buddyboss-app' ),
		);

		return ( isset( $status_code[ $key ] ) ? $status_code[ $key ] : $key );
	}

	/**
	 * Get Apple Release label.
	 *
	 * @param string $key Key.
	 *
	 * @return mixed
	 */
	public function get_ageinfo_labels( $key = '' ) {
		$status_code = array(
			'violenceCartoonOrFantasy'                    => __( 'Cartoon or Fantasy Violence', 'buddyboss-app' ),
			'violenceRealistic'                           => __( 'Realistic Violence', 'buddyboss-app' ),
			'violenceRealisticProlongedGraphicOrSadistic' => __( 'Prolonged Graphic or Sadistic Realistic Violence', 'buddyboss-app' ),
			'profanityOrCrudeHumor'                       => __( 'Profanity or Crude Humor', 'buddyboss-app' ),
			'matureOrSuggestiveThemes'                    => __( 'Mature/Suggestive Themes', 'buddyboss-app' ),
			'horrorOrFearThemes'                          => __( 'Horror/Fear Themes', 'buddyboss-app' ),
			'medicalOrTreatmentInformation'               => __( 'Medical/Treatment Information', 'buddyboss-app' ),
			'alcoholTobaccoOrDrugUseOrReferences'         => __( 'Alcohol, Tobacco, or Drug Use or References', 'buddyboss-app' ),
			'gamblingSimulated'                           => __( 'Simulated Gambling', 'buddyboss-app' ),
			'sexualContentOrNudity'                       => __( 'Sexual Content or Nudity', 'buddyboss-app' ),
			'sexualContentGraphicAndNudity'               => __( 'Graphic Sexual Content and Nudity', 'buddyboss-app' ),
			'contests'                                    => __( 'Contests', 'buddyboss-app' ),
			'unrestrictedWebAccess'                       => __( 'Unrestricted Web Access', 'buddyboss-app' ),
			'gambling'                                    => __( 'Gambling', 'buddyboss-app' ),
			'kidsAgeBand'                                 => __( 'Made for Kids', 'buddyboss-app' ),
		);

		return ( empty( $key ) ? $status_code : ( isset( $status_code[ $key ] ) ? $status_code[ $key ] : $key ) );
	}

	/**
	 * Check the apphelp user added into the apple account or not.
	 *
	 * @param false $override_cache Override Cache.
	 *
	 * @return bool
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function is_bb_app_user_added( $override_cache = false ) {
		$users_data                      = $this->get_acccount_users( $override_cache );
		$appcenter_support_email_address = ManageApp::instance()->get_appcenter_publisher_email_address( $override_cache );
		$user_exist                      = false;

		if ( ! empty( $users_data ) && ! is_wp_error( $users_data ) && ! empty( $users_data['data'] ) ) {
			$users = ( ! empty( $users_data['data'] ) ? array_column( $users_data['data'], 'attributes', 'id' ) : array() );
			$users = ( ! empty( $users ) ? array_filter( array_combine( array_keys( $users ), array_column( $users, 'username' ) ) ) : array() );

			if ( empty( $users ) || ! in_array( $appcenter_support_email_address, $users, true ) ) {
				$user_exist = false;
			} else {
				$user_id   = array_search( $appcenter_support_email_address, $users, true );
				$index     = array_search( $user_id, array_column( $users_data['data'], 'id' ), true );
				$user_info = ( isset( $users_data['data'][ $index ] ) && isset( $users_data['data'][ $index ]['attributes']['roles'] ) ) ? $users_data['data'][ $index ]['attributes']['roles'] : array();

				if ( ! empty( $user_info ) && ( in_array( 'ADMIN', $user_info, true ) || in_array( 'APP_MANAGER', $user_info, true ) ) ) {
					$user_exist = true;
				} else {
					$user_exist = false;
				}
			}
		} else {
			$user_exist = false;
		}

		return $user_exist;
	}

	/**
	 * Check the app-help user added into the apple account or not.
	 *
	 * @param false $override_cache Override Cache.
	 *
	 * @return bool
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function is_bb_app_user_invited( $override_cache = false ) {
		$users_data                      = $this->get_acccount_invited_users( $override_cache );
		$appcenter_support_email_address = ManageApp::instance()->get_appcenter_publisher_email_address( $override_cache );
		$user_invited                    = false;

		if ( ! empty( $users_data ) && ! is_wp_error( $users_data ) && ! empty( $users_data['data'] ) ) {
			$users = ( ! empty( $users_data['data'] ) ? array_column( $users_data['data'], 'attributes', 'id' ) : array() );
			$users = ( ! empty( $users ) ? array_filter( array_combine( array_keys( $users ), array_column( $users, 'email' ) ) ) : array() );

			if ( empty( $users ) || ! in_array( $appcenter_support_email_address, $users, true ) ) {
				$user_invited = false;
			} else {
				$user_id   = array_search( $appcenter_support_email_address, $users, true );
				$index     = array_search( $user_id, array_column( $users_data['data'], 'id' ), true );
				$user_info = ( isset( $users_data['data'][ $index ] ) && isset( $users_data['data'][ $index ]['attributes']['roles'] ) ) ? $users_data['data'][ $index ]['attributes']['roles'] : array();

				if ( ! empty( $user_info ) && ( in_array( 'ADMIN', $user_info, true ) || in_array( 'APP_MANAGER', $user_info, true ) ) ) {
					$user_invited = true;
				} else {
					$user_invited = false;
				}
			}
		} else {
			$user_invited = false;
		}

		return $user_invited;
	}

	/**
	 * Get Account Users information.
	 *
	 * @param false $override_cache Override Cache.
	 *
	 * @return bool
	 * @throws ConfigException Apple configuration exception message.
	 */
	public function create_acccount_users( $override_cache = false ) {
		$users                           = $this->get_acccount_users( $override_cache );
		$user_exist                      = false;
		$appcenter_support_email_address = ManageApp::instance()->get_appcenter_publisher_email_address( $override_cache );

		if ( ! empty( $users ) && ! is_wp_error( $users ) && ! empty( $users['data'] ) ) {
			$users = ( ! empty( $users['data'] ) ? array_column( $users['data'], 'attributes', 'id' ) : array() );
			$users = ( ! empty( $users ) ? array_filter( array_combine( array_keys( $users ), array_column( $users, 'username' ) ) ) : array() );

			if ( ! empty( $users ) || ! in_array( $appcenter_support_email_address, $users, true ) ) {
				if ( $this->is_connected( $override_cache ) ) {
					$client = $this->get_client();

					try {
						$i_connect_user = \BuddyBossApp\Library\Composer::instance()->apple_instance()->IConnectUsers( $client );
						$users          = $i_connect_user->create( 'App', 'Help', $appcenter_support_email_address );
					} catch ( \Throwable $e ) {
						return false;
					}

					if ( ! empty( $users ) && ! is_wp_error( $users ) && ! empty( $users['data'] ) ) {
						$user_exist = true;
					} else {
						$user_exist = false;
					}
				}
			} else {
				$user_exist = true;
			}
		}

		return $user_exist;
	}

	/**
	 * Get App screenshot list.
	 *
	 * @since 2.2.60
	 * @return array
	 */
	public function app_screenshot_list() {
		$app_screenshot_keys = array(
			'app_iphone_67'         => array(
				'label'       => __( 'iPhone 6.9" Display', 'buddyboss-app' ),
				'type'        => 'iphone',
				'required'    => true,
				'display_key' => 'display_6_7',
			),
			'app_iphone_65'         => array(
				'label'       => __( 'iPhone 6.5" Display', 'buddyboss-app' ),
				'type'        => 'iphone',
				'required'    => true,
				'display_key' => 'display_6_5',
			),
			'app_iphone_61'         => array(
				'label'       => __( 'iPhone 6.3" Display', 'buddyboss-app' ),
				'type'        => 'iphone',
				'required'    => false,
				'display_key' => 'display_6_1',
			),
			'app_iphone_58'         => array(
				'label'       => __( 'iPhone 5.8" Display', 'buddyboss-app' ),
				'type'        => 'iphone',
				'required'    => false,
				'display_key' => 'display_5_8',
			),
			'app_iphone_55'         => array(
				'label'       => __( 'iPhone 5.5" Display', 'buddyboss-app' ),
				'type'        => 'iphone',
				'required'    => false,
				'display_key' => 'display_5_5',
			),
			'app_iphone_47'         => array(
				'label'       => __( 'iPhone 4.7" Display', 'buddyboss-app' ),
				'type'        => 'iphone',
				'required'    => false,
				'display_key' => 'display_4_7',
			),
			'app_iphone_40'         => array(
				'label'       => __( 'iPhone 4" Display', 'buddyboss-app' ),
				'type'        => 'iphone',
				'required'    => false,
				'display_key' => 'display_4_0',
			),
			'app_iphone_35'         => array(
				'label'       => __( 'iPhone 3.5" Display', 'buddyboss-app' ),
				'type'        => 'iphone',
				'required'    => false,
				'display_key' => 'display_3_5',
			),
			'app_ipad_pro_3gen_129' => array(
				'label'       => __( 'iPad Pro 13" Display', 'buddyboss-app' ),
				'type'        => 'ipad',
				'required'    => true,
				'display_key' => 'display_ipad_13'
			),
			'app_ipad_pro_3gen_11'         => array(
				'label'       => __( 'iPad Pro 11" Display', 'buddyboss-app' ),
				'type'        => 'ipad',
				'required'    => false,
				'display_key' => 'display_1_1',
			),
			'app_ipad_pro_129'         => array(
				'label'       => __( 'iPad Pro 12.9" Display', 'buddyboss-app' ),
				'type'        => 'ipad',
				'required'    => false,
				'display_key' => 'display_12_9',
			),
			'app_ipad_105'         => array(
				'label'       => __( 'iPad 10.5" Display', 'buddyboss-app' ),
				'type'        => 'ipad',
				'required'    => false,
				'display_key' => 'display_10_5',
			),
			'app_ipad_97'         => array(
				'label'       => __( 'iPad 9.7" Display', 'buddyboss-app' ),
				'type'        => 'ipad',
				'required'    => false,
				'display_key' => 'display_9_7',
			),
		);

		return $app_screenshot_keys;
	}
}