<?php
/**
 * Holds Google store connection functionality.
 *
 * @package BuddyBossApp\AppStores
 */

namespace BuddyBossApp\AppStores;

use BuddyBossApp\Admin\InAppPurchases\Helpers;
use BuddyBossApp\Admin\Configure;
use BuddyBossApp\AppStores\Services\Android\AndroidPublisher;
use BuddyBossApp\Tools\Logger;

/**
 * Google store connection class.
 */
class Android {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Google connection variable.
	 *
	 * @var bool $client
	 */
	private static $client = false;

	/**
	 * Google app store connection error messages.
	 *
	 * @var array $error_messages
	 */
	private static $error_messages;

	/**
	 * Google store connection error messages.
	 *
	 * @var string $google_client_error_messages
	 */
	private static $google_client_error_messages = '';

	/**
	 * Get the instance of the class.
	 *
	 * @return Android
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Android constructor.
	 */
	public function __construct() {
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
	}

	/**
	 * Function to connect with android store.
	 */
	public function get_client() {
		$file_name = $this->get_private_key();

		if ( empty( $file_name ) ) {
			return new \WP_Error(
				'error_connecting_account',
				__( 'Error while connecting to your account, make sure you have configured correct android api service json.', 'buddyboss-app' )
			);
		}

		$google_service_account_json = bbapp_get_upload_full_path( $file_name );

		if ( ! file_exists( $google_service_account_json ) ) {
			return new \WP_Error(
				'error_connecting_account',
				__( 'Error while connecting to your account, make sure you have configured correct android api service json.', 'buddyboss-app' )
			);
		}

		if ( ! self::$client ) {
			try {
				$google_client = \BuddyBossApp\Library\Composer::instance()->google_instance()->Google_Client();
				$google_client->setScopes( 'https://www.googleapis.com/auth/androidpublisher' );
				$google_client->setAuthConfig( $google_service_account_json );
				$google_client->fetchAccessTokenWithAssertion();
				self::$client = $google_client;
			} catch ( \Throwable $e ) {
				self::$google_client_error_messages = $e->getMessage();
				Logger::instance()->add( 'info_log', 'BuddyBossApp\AppStores\Android -> get_client :::' . $e->getMessage() );
				return new \WP_Error( 'error_connecting_account', $e->getCode() );
			}
		}

		return self::$client;
	}

	/**
	 * Function to check google connection.
	 *
	 * @param false $override_cache Over-ride cache.
	 *
	 * @return bool|mixed
	 */
	public function is_google_connected( $override_cache = false ) {
		$file_name = $this->get_private_key();
		$cache_key = 'bbapp_android_is_google_client_connected_cache_' . md5( $file_name );
		$cached    = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			$is_token = (int) $cached;
		} else {
			$google_acc_connection = $this->get_client();
			$token                 = ( ! is_wp_error( $google_acc_connection ) ) ? $google_acc_connection->getAccessToken() : '';
			$is_token              = ( ! empty( $token['access_token'] ) ) ? 1 : 0;
			set_site_transient( $cache_key, $is_token, 24 * HOUR_IN_SECONDS );
		}

		return $is_token;
	}

	/**
	 * Function to get google connection error.
	 *
	 * @return string
	 */
	public function get_google_connection_error() {
		return self::$google_client_error_messages;
	}

	/**
	 * Check if the GoogleStore is Connected.
	 *
	 * @param bool $override_cache Over-ride cache or not.
	 *
	 * @return bool|\WP_Error
	 */
	public function is_connected( $override_cache = false ) {
		$file_name = $this->get_private_key();
		$cache_key = 'bbapp_android_is_connected_cache_' . md5( $file_name );
		$cached    = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			$connected = $cached;
		} else {
			// do api call here.
			$client = $this->get_client();

			if ( ! $client || is_wp_error( $client ) ) {
				return false;
			}

			$application_id = $this->get_namespace();
			$connected      = 'not_connected';

			try {
				$google_client = \BuddyBossApp\Library\Composer::instance()->google_instance()->Google_Client();
				$service       = \BuddyBossApp\Library\Composer::instance()->google_instance()->Google_Service_AndroidPublisher( $client );
				$google_client->setApplicationName( $application_id );
				$products_list_response = $service->monetization_subscriptions->listMonetizationSubscriptions( $application_id );
				$response               = $products_list_response->getSubscriptions();
				$connected              = 'connected';
			} catch ( \Exception $e ) {
				self::$error_messages = $e->getErrors();

				return new \WP_Error( 'error_connecting_account', $e->getMessage() );
			}

			set_site_transient( $cache_key, $connected, 24 * HOUR_IN_SECONDS );
		}

		if ( 'connected' === $connected ) {
			return true;
		}

		return false;
	}

	/**
	 * Function to check if package name is already take on google play store.
	 *
	 * @param string $package_name package name to check.
	 *
	 * @return int|string
	 */
	public function validate_package_name( $package_name ) {
		$request_uri = bbapp_remote_get( 'https://play.google.com/store/apps/details?id=' . $package_name );

		return wp_remote_retrieve_response_code( $request_uri );
	}

	/**
	 * Get connection error messages.
	 *
	 * @return mixed
	 */
	public function get_connection_error_messages() {
		return self::$error_messages;
	}

	/**
	 * Return the Namespace.
	 *
	 * @return false
	 */
	public function get_namespace() {
		return Configure::instance()->option( 'publish.android.namespace' );
	}

	/**
	 * Return the Get key ID.
	 *
	 * @return false
	 */
	public function get_private_key() {
		return Configure::instance()->option( 'publish.android.account_key' );
	}

	/**
	 * Sync store product with local data.
	 */
	public function sync_store_product() {
		$google_service_account_name = __( 'Play Publisher', 'buddyboss-app' );
		$android_app_id              = $this->get_namespace();
		$file_name                   = $this->get_private_key();
		$google_service_account_json = bbapp_get_upload_full_path( $file_name );
		$products                    = array();

		if ( empty( $google_service_account_json ) || ! file_exists( $google_service_account_json ) || empty( $android_app_id ) ) {
			return $products;
		}

		try {
			$google_client = \BuddyBossApp\Library\Composer::instance()->google_instance()->Google_Client();
			$google_client->setApplicationName( $google_service_account_name );
			$google_client->setScopes( 'https://www.googleapis.com/auth/androidpublisher' );
			$google_client->setAuthConfig( $google_service_account_json );

			// Use our custom Google Play API wrapper for PHP 7.4 compatibility.
			$google_play_api = \BuddyBossApp\Library\Composer::instance()->google_play_api_instance();
			$google_play_api->init( $google_client, $android_app_id );

			// Fetch all products (in-app products and subscriptions).
			$all_products = $google_play_api->get_all_products();
			if ( ! empty( $all_products ) ) {
				foreach ( $all_products as $product ) {
					$products[] = array(
						'id'     => $product['id'],
						'name'   => $product['name'],
						'type'   => Helpers::platform_store_product_typeslug( $product['type'] ),
						'status' => $product['status'],
					);
				}
			}
		} catch ( \Throwable $e ) {
			Logger::instance()->add( 'info_log', 'BuddyBossApp\AppStores\Android -> sync_store_product:::' . $e->getMessage() );
		}

		return $products;
	}

	/**
	 * Validate App.
	 *
	 * @param false $override_cache Override Cache.
	 */
	public function validate_app( $override_cache = false ) {
		$release_details = $this->get_release_data( $override_cache );
		$errors          = array();
		$builds          = \BuddyBossApp\Build::instance()->get_app_builds( 'android', 'live', false, 'completed', 1, 10, $override_cache );

		if ( ! is_wp_error( $builds ) && empty( $builds['data'] ) ) {
			$errors[] = array(
				'headline'    => __( 'No Android Release App Builds', 'buddyboss-app' ),
				'description' => __( 'Please generate an Android release app build to publish.', 'buddyboss-app' ),
				'link'        => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-build&setting=request-build' ) ),
			);
		}

		$release_status = array(
			'status_unspecified',
			'halted',
		);

		if ( empty( $release_details ) || false === $release_details || in_array( $release_details['release_status'], $release_status, true ) ) {
			$errors[] = array(
				'headline'    => __( 'No Release Found', 'buddyboss-app' ),
				'description' => __( 'Your app doesn\'t have a new release ready for review. Please create and prepare a new release.', 'buddyboss-app' ),
				'link'        => esc_url( 'https://play.google.com/console/developers' ),
			);
		}

		if ( ( ! empty( $release_details ) || false === $release_details ) && 'completed' === $release_details['release_status'] ) {
			$errors[] = array(
				'headline'    => __( 'App Already Approved', 'buddyboss-app' ),
				'description' => __( 'our app has already been approved by Google. To publish an update, please create and prepare a new release.', 'buddyboss-app' ),
				'link'        => esc_url( 'https://play.google.com/console/developers' ),
			);
		}

		if ( ( ! empty( $release_details ) || false === $release_details ) && 'in_progress' === $release_details['release_status'] ) {
			$errors[] = array(
				'headline'    => __( 'App Release In Progress', 'buddyboss-app' ),
				'description' => __( 'Your app already has a release in progress. Please cancel the open release or wait for it to complete to start a new publish request.', 'buddyboss-app' ),
				'link'        => esc_url( 'https://play.google.com/console/developers' ),
			);
		}

		return $errors;
	}

	/**
	 * Function to get Release app details.
	 *
	 * @param boolean $override_cache whether override cache or not.
	 *
	 * @return array|false
	 */
	public function get_required_app_details( $override_cache ) {
		$cache_key = 'bbapp_google_app_details';
		$cached    = get_site_transient( $cache_key );

		if ( false !== $cached && empty( $override_cache ) ) {
			return $cached;
		} else {
			$connection = $this->is_connected( $override_cache );

			if ( ! $connection || is_wp_error( $connection ) ) {
				return false;
			}

			$app_details = AndroidPublisher::instance()->initialize();
			$app_data    = array();

			if ( ! empty( $app_details ) ) {
				$app_data['email']            = $app_details->contactEmail; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				$app_data['website']          = $app_details->contactWebsite; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				$app_data['phone']            = $app_details->contactPhone; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				$app_data['default_language'] = $app_details->defaultLanguage; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
			}

			set_site_transient( $cache_key, $app_data, 24 * HOUR_IN_SECONDS );
		}

		return $app_data;
	}

	/**
	 * Function to get latest production release data.
	 *
	 * @param bool $override_cache Whether bypass the cache or not.
	 *
	 * @return array|false|mixed
	 */
	public function get_release_data( $override_cache = false ) {
		$cache_key = 'bbapp_google_release_data';
		$cached    = get_site_transient( $cache_key );

		if ( false !== $cached && empty( $override_cache ) ) {
			return $cached;
		} else {
			$connection = $this->is_connected( $override_cache );

			if ( ! $connection || is_wp_error( $connection ) ) {
				return false;
			}

			$release_tracks             = AndroidPublisher::instance()->initialize( 'EditsTracks' );
			$app_data                   = array();
			$app_data['has_release']    = false;
			$app_data['release_status'] = '';
			$app_data['release_notes']  = array();

			if ( ! empty( $release_tracks->tracks ) ) {
				foreach ( $release_tracks->tracks as $release_track ) {
					if ( ! empty( $release_track->track ) && 'production' === $release_track->track ) {
						$app_data['name']           = $release_track->releases[0]->name;
						$app_data['has_release']    = true;
						$app_data['release_status'] = $release_track->releases[0]->status;
						$count                      = 1;

						if ( ! empty( $release_track->releases[0]->releaseNotes ) ) {
							foreach ( $release_track->releases[0]->releaseNotes as $release_notes ) {
								$app_data['release_notes'][ $count ]['language'] = $release_notes->language;
								$app_data['release_notes'][ $count ]['notes']    = $release_notes->text;
								$count ++;
							}
						}
						break;
					}
				}
			}

			set_site_transient( $cache_key, $app_data, 24 * HOUR_IN_SECONDS );
		}

		return $app_data;
	}

	/**
	 * Function to get all production release data.
	 *
	 * @param bool $override_cache Whether to bypass the cache.
	 *
	 * @return array|false|mixed
	 */
	public function get_all_production_release_data( $override_cache = false ) {
		$cache_key = 'bbapp_google_all_production_release_data';
		$cached    = get_site_transient( $cache_key );

		if ( false !== $cached && empty( $override_cache ) ) {
			return $cached;
		} else {
			$connection = $this->is_connected( $override_cache );

			if ( ! $connection || is_wp_error( $connection ) ) {
				return false;
			}

			$release_tracks = AndroidPublisher::instance()->initialize( 'EditsTracks' );
			$app_data       = array();
			$count          = 1;

			if ( ! empty( $release_tracks->tracks ) ) {
				foreach ( $release_tracks->tracks as $release_track ) {
					if ( ! empty( $release_track->track ) && 'production' === $release_track->track ) {
						if ( ! empty( $release_track->releases ) ) {
							foreach ( $release_track->releases as $release_item ) {
								$inr_count                    = 1;
								$app_data[ $count ]['name']   = $release_item->name;
								$app_data[ $count ]['status'] = $release_item->status;

								if ( ! empty( $release_item->releaseNotes ) ) { // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
									foreach ( $release_item->releaseNotes as $release_notes ) { // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
										$app_data[ $count ]['release_notes'][ $inr_count ]['language'] = $release_notes->language;
										$app_data[ $count ]['release_notes'][ $inr_count ]['notes']    = $release_notes->text;
										$inr_count ++;
									}
								}
								$count ++;
							}
						}
					}
				}
			}

			set_site_transient( $cache_key, $app_data, 24 * HOUR_IN_SECONDS );
		}

		return $app_data;
	}

	/**
	 * Returns the release bundle ID.
	 *
	 * @return false
	 */
	public function get_release_bundle_id() {
		return Configure::instance()->option( 'publish.android.namespace' );
	}

	/**
	 * Returns the test bundle ID.
	 *
	 * @return false
	 */
	public function get_test_bundle_id() {
		return Configure::instance()->option( 'publish.android.dev.namespace' );
	}

	/**
	 * Function to get the App language.
	 *
	 * @param string $lang_code Language code.
	 *
	 * @return string
	 */
	public function get_language( $lang_code = '' ) {

		$languages = array(
			'af'      => 'Afrikaans',
			'am'      => 'Amharic',
			'bg'      => 'Bulgarian',
			'ca'      => 'Catalan',
			'zh-HK'   => 'Chinese (Hong Kong)',
			'zh-CN'   => 'Chinese (PRC)',
			'zh-TW'   => 'Chinese (Taiwan)',
			'hr'      => 'Croatian',
			'cs'      => 'Czech',
			'da'      => 'Danish',
			'nl'      => 'Dutch',
			'en-GB'   => 'English (UK)',
			'en-US'   => 'English (US)',
			'et'      => 'Estonian',
			'fil'     => 'Filipino',
			'fi'      => 'Finnish',
			'fr-CA'   => 'French (Canada)',
			'fr-FR'   => 'French (France)',
			'de'      => 'German',
			'el'      => 'Greek',
			'he'      => 'Hebrew',
			'hi'      => 'Hindi',
			'hu'      => 'Hungarian',
			'is'      => 'Icelandic',
			'id / in' => 'Indonesian',
			'it'      => 'Italian',
			'ja'      => 'Japanese',
			'ko'      => 'Korean',
			'lv'      => 'Latvian',
			'lt'      => 'Lithuanian',
			'ms'      => 'Malay',
			'no'      => 'Norwegian',
			'pl'      => 'Polish',
			'pt-BR'   => 'Portuguese (Brazil)',
			'pt-PT'   => 'Portuguese (Portugal)',
			'ro'      => 'Romanian',
			'ru'      => 'Russian',
			'sr'      => 'Serbian',
			'sk'      => 'Slovak',
			'sl'      => 'Slovenian',
			'es-419'  => 'Spanish (Latin America)',
			'es-ES'   => 'Spanish (Spain)',
			'sw'      => 'Swahili',
			'sv'      => 'Swedish',
			'th'      => 'Thai',
			'tr'      => 'Turkish',
			'uk'      => 'Ukrainian',
			'vi'      => 'Vietnamese',
			'zu'      => 'Zulu',
		);

		return ( ! empty( $lang_code ) && isset( $languages[ $lang_code ] ) ? $languages[ $lang_code ] : $lang_code );
	}
}
