<?php
/**
 * Holds app settings rest functinality.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

use BuddyBossApp\Admin\Configure;
use BuddyBossApp\App\App;
use BuddyBossApp\AppStores\Apple;
use BuddyBossApp\Auth\Auth;
use BuddyBossApp\InAppPurchases\Controller;
use WP_REST_Request;
use WP_REST_Response;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * App settings rest class.
 */
class AppSettings {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	public static $instance;

	/**
	 * Get the instance of class
	 *
	 * @return AppSettings
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * AppSettings constructor.
	 */
	public function __construct() {
	}

	/**
	 * Returns the settings value.
	 *
	 * @param string $key     Key name.
	 * @param bool   $network Fetch the network setting or current blog.
	 *
	 * @return null
	 */
	public function get_setting_value( $key, $network = false ) {
		$bbapp_settings = ManageApp::instance()->get_app_settings( $network );
		$settings       = ( isset( $bbapp_settings ) ) ? $bbapp_settings : array();

		if ( isset( $settings[ $key ] ) ) {
			return $settings[ $key ];
		}

		return '';
	}

	/**
	 * Update the settings value.
	 *
	 * @param string $key Key name.
	 * @param string $value Value.
	 * @param bool   $network Fetch the network setting or current blog.
	 *
	 * @return bool|mixed
	 * @since 2.2.80
	 */
	public function update_setting_value( $key, $value, $network = false ) {
		$bbapp_settings = ManageApp::instance()->get_app_settings( $network );
		$settings       = ( isset( $bbapp_settings ) ) ? $bbapp_settings : array();

		$settings[ $key ] = $value;

		if ( true === $network && bbapp()->is_network_activated() ) {
			return update_blog_option( get_current_network_id(), 'bbapp_settings', $settings );
		} else {
			return update_option( 'bbapp_settings', $settings );
		}
	}

	/**
	 * Delete the settings value.
	 *
	 * @param string $key Key name.
	 * @param bool   $network Fetch the network setting or current blog.
	 *
	 * @return bool|mixed
	 * @since 2.3.00
	 */
	public function delete_setting_value( $key, $network = false ) {
		$bbapp_settings = ManageApp::instance()->get_app_settings( $network );
		$settings       = ( isset( $bbapp_settings ) ) ? $bbapp_settings : array();

		if ( isset( $settings[ $key ] ) ) {
			unset( $settings[ $key ] );
		}

		if ( true === $network && bbapp()->is_network_activated() ) {
			return update_blog_option( get_current_network_id(), 'bbapp_settings', $settings );
		} else {
			return update_option( 'bbapp_settings', $settings );
		}
	}

	/**
	 * Get rest response.
	 *
	 * @param WP_REST_Request $request Wp rest request.
	 *
	 * @return \WP_Error
	 * @apiPrivate
	 * @api            {GET} /wp-json/buddyboss-app/core/v1/settings/:app_id App core settings
	 * @apiName        GetCoreSettings
	 * @apiGroup       CoreSettings
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Handle output for Settings Endpoint. You can pass a 'language_code' header to receive translations for a specific language.
	 * You can also use the 'appLang' header which is the preferred method for specifying the language.
	 * @apiUse         apidocForGetCoreSettingsV1
	 */
	public function rest_response( $request ) {
		// First check if app_id is in the URL path parameters
		$app_id = $request->get_param( 'app_id' );
		
		// If not in URL, check headers (for backward compatibility)
		if ( empty( $app_id ) ) {
				$app_id = $request->get_header( 'appid' );

				// If found in headers, set it as a parameter for consistency
			if ( ! empty( $app_id ) ) {
				$request->set_param( 'app_id', $app_id );
			}
		}
		

		global $bbapp_var;

		$response           = array();
		$response['config'] = array();

		/**
		 * Demo Account.
		 * This is used if we want to prefill the login form on app with given account.
		 */
		$demo_acc = array(
			'is'       => false,
			'username' => '',
			'password' => '',
		);

		/**
		 * Filters app demo account
		 *
		 * @param array $demo_acc Demo account credentials.
		 */
		$demo_acc = apply_filters( 'bbapp_settings_app_demo_account', $demo_acc );

		$response['demo_credentials'] = $demo_acc;

		/**
		 * App Default Language
		 */
		$app_default_language = get_locale();

		/**
		 * Filters app default language.
		 *
		 * @param string $app_default_language App default language.
		 */
		$app_default_language = apply_filters( 'bbapp_settings_app_default_lang', $app_default_language );

		$response['language'] = $app_default_language;

		/**
		 * App default language.
		 *
		 * @todo: remove this in future
		 */
		$response['config']['language'] = $app_default_language;

		/**
		 * App Default Language
		 */
		$app_default_language_rtl = is_rtl();

		/**
		 * Filters Settings app default language.
		 *
		 * @param bool $app_default_language_rtl If default language is RTL.
		 */
		$app_default_language_rtl = apply_filters( 'bbapp_settings_app_default_language_rtl', $app_default_language_rtl );

		$response['rtl_enabled'] = (bool) $app_default_language_rtl;

		/**
		 * Bug Email
		 */
		$app_report_bug = $this->get_setting_value( 'bugs_email.enabled' );
		$bugs_email     = $this->get_setting_value( 'bugs_email.email' );

		if ( empty( $bugs_email ) ) {
			$bugs_email = get_option( 'admin_email' );
		}

		$response['bugs_email'] = array(
			'enabled' => (bool) $app_report_bug,
			'email'   => $bugs_email,
		);

		if ( $this->get_setting_value( 'app_sharing.enabled' ) ) {
			$response['app_sharing'] = array(
				'link'  => home_url( 'dl' ),
				'label' => esc_html__( 'Share App', 'buddyboss-app' ),
			);
		}

		/**
		 * Support Email
		 */
		$support_email_enabled = $this->get_setting_value( 'support_email.enabled' );
		$support_email         = $this->get_setting_value( 'support_email.email' );

		if ( empty( $support_email ) ) {
			$support_email = get_option( 'admin_email' );
		}

		$response['support_email'] = array(
			'enabled' => (bool) $support_email_enabled,
			'email'   => $support_email,
		);

		/**
		 * Minimum version support.
		 */
		$ios_min_version_code                         = Configure::instance()->option( 'publish.general.ios.min_app' );
		$android_min_version_code                     = Configure::instance()->option( 'publish.general.android.min_app' );
		$response['min_supported_version']['ios']     = ! empty( $ios_min_version_code ) ? (int) $ios_min_version_code : 0;
		$response['min_supported_version']['android'] = ! empty( $android_min_version_code ) ? (int) $android_min_version_code : 0;

		/**
		 * Auth & Signup Settings.
		 */
		$response['app_auth']['enable_signup']                     = (bool) Auth::instance()->is_app_registration_enabled();
		$response['app_auth']['enable_inapp_account_verification'] = (bool) Auth::instance()->is_registration_available(); // BB registration enabled, Activation will enabled.
		$response['app_auth']['signup_form']                       = Auth::instance()->is_app_registration_form(); // Registration form type.

		if ( ! defined( 'BP_PLATFORM_VERSION' ) ) {

			/**
			 * Delete Account
			 */
			$admin_email = $this->get_setting_value( 'delete_account.admin_email' );

			if ( empty( $admin_email ) ) {
				$admin_email = get_option( 'admin_email' );
			}

			$response['app_auth']['can_request_delete_account'] = array(
				'enabled' => true,
				'email'   => $admin_email,
			);
		}

		/**
		 * App Menu
		 */
		$response['app_menu']['ios']['show_labels']     = (bool) $this->get_setting_value( 'app_menu.appmenu_labels' );
		$response['app_menu']['android']['show_labels'] = (bool) $this->get_setting_value( 'app_menu.appmenu_labels' );

		/**
		 * Tab Bar Visibility
		 */
		$tab_bar_visibility                                        = $this->get_setting_value( 'app_menu.tab_bar_visibility' );
		$response['app_menu']['ios']['show_tabbar_on_all_screens'] = false;
		$response['app_menu']['android']['show_tabbar_on_all_screens'] = false;

		if ( ! empty( $tab_bar_visibility ) && 'show_on_all_screen' === $tab_bar_visibility ) {
			$response['app_menu']['ios']['show_tabbar_on_all_screens']     = true;
			$response['app_menu']['android']['show_tabbar_on_all_screens'] = true;
		}

		/**
		 * Headerbar settings.
		 */
		$tabbar_id    = \BuddyBossApp\Menu::instance()->bbapp_get_user_nav_menu_id();
		$headerbar_id = \BuddyBossApp\Menus\MenuManager::instance()->get_headerbar_menu_from_tabbar_id( $tabbar_id );

		// Title position on Scroll.
		$header_bar_position                               = \BuddyBossApp\Menus\MenuManager::instance()->get_title_position( $headerbar_id );
		$response['app_menu']['ios']['title_position']     = 'left';
		$response['app_menu']['android']['title_position'] = 'left';

		if ( ! empty( $header_bar_position ) ) {
			$response['app_menu']['ios']['title_position']     = $header_bar_position;
			$response['app_menu']['android']['title_position'] = $header_bar_position;
		}

		// More Menu Icon.
		$more_menu_position                                    = \BuddyBossApp\Menus\MenuManager::instance()->get_more_menu_position( $headerbar_id );
		$response['app_menu']['ios']['more_menu_position']     = 'tabbar';
		$response['app_menu']['android']['more_menu_position'] = 'tabbar';

		if ( ! empty( $more_menu_position ) ) {
			$response['app_menu']['ios']['more_menu_position']     = $more_menu_position;
			$response['app_menu']['android']['more_menu_position'] = $more_menu_position;
		}

		// Post input button.
		$post_input_field                                    = \BuddyBossApp\Menus\MenuManager::instance()->get_post_input_field_status( $headerbar_id );
		$response['app_menu']['ios']['post_input_field']     = false;
		$response['app_menu']['android']['post_input_field'] = false;

		if ( $post_input_field ) {
			$response['app_menu']['ios']['post_input_field']     = true;
			$response['app_menu']['android']['post_input_field'] = true;
		}

		// Get ios store app id.
		$ios_store_app_id = Apple::instance()->get_ios_store_app_id();

		/**
		 * Generals
		 * This setting is ovverided by app rating settings. can be removed in future completely.
		 *
		 * @todo: to be remove in future.
		 */
		$response['general']['ios']['store_app_id'] = $ios_store_app_id;

		/**
		 * App Rating Settings
		 */
		$response['app_rating']                        = array();
		$response['app_rating']['ios']['store_app_id'] = $ios_store_app_id;
		$response['app_rating']['ios']['enabled']      = ! empty( $response['app_rating']['ios']['store_app_id'] ) ? (bool) $this->get_setting_value( 'app_rating.ios.enabled' ) : false;
		$response['app_rating']['android']['enabled']  = ! empty( Configure::instance()->option( 'publish.android.namespace' ) ) ? (bool) $this->get_setting_value( 'app_rating.android.enabled' ) : false;
		$app_platform                                  = isset( $bbapp_var['app_platform'] ) ? $bbapp_var['app_platform'] : '';

		if ( ! in_array( $app_platform, array( 'android', 'ios' ), true ) ) {
			return new \WP_Error( 'rest_invalid_app_platform', __( 'Invalid App Platform', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		$branding      = Branding::instance();
		$branding_data = $branding->branding_data( $app_platform );

		if ( is_wp_error( $branding_data ) ) {
			$branding_data = null;
		}

		// Unset appHeaderTitle size from API.
		if ( isset( $branding_data['styles']['typography']['appHeaderTitle']['size'] ) ) {
			unset( $branding_data['styles']['typography']['appHeaderTitle']['size'] );
		}

		$response['branding'] = $branding_data;

		// Add information about the requested language
		$response['app_languages_config'] = \BuddyBossApp\AppLanguages::instance()->get_app_languages_config();

		/**
		 * Features
		 */
		$features             = App::instance()->get_app_features_availability();
		$response['features'] = array();

		foreach ( $features as $feature_key => $feature ) {
			$response['features'][ $feature_key ] = array(
				'is_enabled_android' => $feature['is_enabled_android'],
				'is_enabled_ios'     => $feature['is_enabled_ios'],
			);
		}

		/**
		 * Multi-Site Network
		 */
		$response['multisite_network'] = array(
			'enabled'      => is_multisite(),
			'network_mode' => bbapp()->is_network_activated(),
		);

		/**
		 * Vimeo setting
		 */
		$vimeo_auth_code   = $this->get_setting_value( 'vimeo.auth_code' );
		$response['vimeo'] = array(
			'auth_code' => $vimeo_auth_code,
		);

		/**
		 * Compatibility Setting
		 */
		$open_link_in                              = $this->get_setting_value( 'compatibility.open_link_in' );
		$response['compatibility']['open_link_in'] = ! empty( $open_link_in ) ? $open_link_in : 'in_app_browser';

		/**
		 * IAP
		 */
		if ( bbapp_is_active( 'iap' ) ) {
			$get_iap_rest_settings = Controller::instance()->get_rest_settings();
			$response['iap']       = array(
				'has_access'                      => $get_iap_rest_settings['has_access'],
				'lock_app'                        => $get_iap_rest_settings['lock_app'],
				'purchase_before_register'        => $get_iap_rest_settings['purchase_before_register'],
				'register_show_products'          => $get_iap_rest_settings['register_show_products'],
				'registration_products_available' => ( ! empty( $request->get_header( 'appplatform' ) ) && bbapp_is_active( 'iap' ) && self::instance()->get_setting_value( 'iap.purchase_before_register' ) && method_exists( 'BuddyBossApp\Admin\InAppPurchases\Helpers', 'bbapp_device_platform_products_available' ) && true === \BuddyBossApp\Admin\InAppPurchases\Helpers::bbapp_device_platform_products_available( $request->get_header( 'appplatform' ) ) ) ? true : false,
				'terms_page'                      => array(
					'id'    => $get_iap_rest_settings['terms_page_id'],
					'title' => ! empty( $get_iap_rest_settings['terms_page_id'] ) ? get_the_title( $get_iap_rest_settings['terms_page_id'] ) : '',
					'url'   => ! empty( $get_iap_rest_settings['terms_page_id'] ) ? get_permalink( $get_iap_rest_settings['terms_page_id'] ) : '',
				),
				'privacy_page'                    => array(
					'id'    => $get_iap_rest_settings['policy_page_id'],
					'title' => ! empty( $get_iap_rest_settings['policy_page_id'] ) ? get_the_title( $get_iap_rest_settings['policy_page_id'] ) : '',
					'url'   => ! empty( $get_iap_rest_settings['policy_page_id'] ) ? get_permalink( $get_iap_rest_settings['policy_page_id'] ) : '',
				),
				'review_version'                  => array(
					'ios' => $get_iap_rest_settings['review_version_ios'],
				),
				'subscribe_message'               => $get_iap_rest_settings['subscribe_message'],
			);
		}

		/**
		 * Connected Sites.
		 */
		$get_secondary_sites           = ManageApp::instance()->get_secondary_sites_url();
		$get_primary_site              = ManageApp::instance()->get_primary_site_url();
		$response['connected_sites']   = array();
		$response['connected_sites'][] = array(
			'label' => str_replace( array( 'http://', 'https://' ), '', $get_primary_site ),
			'url'   => $get_primary_site,
			'type'  => 'primary',
		);

		foreach ( $get_secondary_sites as $site ) {
			$response['connected_sites'][] = array(
				'label' => str_replace( array( 'http://', 'https://' ), '', $site ),
				'url'   => $site,
				'type'  => 'secondary',
			);
		}

		/**
		 * Bundle ids
		 */
		$ios_release_bundle_id     = $this->get_setting_value( 'publish.ios.namespace' );
		$ios_test_bundle_id        = $this->get_setting_value( 'publish.ios.dev.namespace' );
		$android_release_bundle_id = $this->get_setting_value( 'publish.android.namespace' );
		$android_test_bundle_id    = $this->get_setting_value( 'publish.android.dev.namespace' );

		$response['bundle_ids']['ios']     = array(
			'release' => ! empty( $ios_release_bundle_id ) ? $ios_release_bundle_id : '',
			'test'    => ! empty( $ios_test_bundle_id ) ? $ios_test_bundle_id : '',
		);
		$response['bundle_ids']['android'] = array(
			'release' => ! empty( $android_release_bundle_id ) ? $android_release_bundle_id : '',
			'test'    => ! empty( $android_test_bundle_id ) ? $android_test_bundle_id : '',
		);

		// App license type.
		$response['license_type'] = $this->get_setting_value( 'app_license_type' );

		// App license data token.
		$response['license_data_token'] = $this->get_setting_value( 'license_data_token' );

		// Feature and full license links.
		$upgrade_url              = 'https://www.buddyboss.com/app-upgrade';
		$response['all_features'] = $upgrade_url;
		$response['full_license'] = $upgrade_url;

		/**
		 * Filters settings rest response.
		 *
		 * @param WP_REST_Response $response WP rest response.
		 * @param WP_REST_Request  $request  WP rest request.
		 */
		$response = apply_filters( 'bbapp_settings_rest_response', $response, $request );

		// change cast for empty.
		if ( empty( $response ) ) {
			$response = new \stdClass();
		}

		$response = rest_ensure_response( $response );

		$response->add_links( $this->prepare_links() );

		return rest_ensure_response( $response );
	}

	/**
	 * Prepare links for the request.
	 *
	 * @return array Links for the given post.
	 */
	protected function prepare_links() {
		// Entity meta.
		$links = array(
			'bb_settings' => array(
				'href'       => rest_url( '/buddyboss/v1/settings' ),
				'embeddable' => true,
			),
		);

		return $links;
	}
}
