<?php
/**
 * Holds app languages functionality.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use BuddyBossApp\Helpers\BBAPP_File;

/**
 * App languages class.
 */
class AppLanguages {

	/**
	 * Class instance.
	 *
	 * @var AppLanguages $instance
	 */
	public static $instance;

	/**
	 * Per page.
	 *
	 * @var int $per_page
	 */
	public $per_page = 10;

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.0.0
	 *
	 * @return AppLanguages
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * AppLanguages constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
	}

	/**
	 * Filters/hooks here.
	 *
	 * @since 1.0.0
	 */
	public function load() {
		add_action( 'bbapp_on_plugin_activate', array( $this, 'on_plugin_activate' ) );
		add_action( 'upgrader_process_complete', array( $this, 'on_plugin_update' ), 10, 2 );
	}

	/**
	 * Return Languages of App.
	 *
	 * @since 1.0.0
	 *
	 * @return array|mixed
	 */
	public function get_languages() {
		$data = get_option( 'bbapp_languages' );

		// Merge main site language when empty on network enabled.
		if ( bbapp()->is_network_activated() && ! is_main_site() && empty( $data ) ) {
			$current_blog_id = get_current_blog_id();
			$main_site_id    = get_main_site_id();
			switch_to_blog( $main_site_id );
			$main_site_data = get_option( 'bbapp_languages' );
			switch_to_blog( $current_blog_id );

			if ( ! empty( $main_site_data ) ) {
				update_option( 'bbapp_languages', $main_site_data );
			}
		}

		$data = isset( $data ) ? $data : array();

		return $data;
	}

	/**
	 * Transform the language for rest API output.
	 *
	 * @param array $args                 Additional arguments:
	 *                                    - 'include_empty': Whether to include empty translations. Default false.
	 *                                    - 'include_default_matches': Whether to include translations matching defaults. Default false.
	 *                                    - 'include_deleted': Whether to include deleted translations. Default false.
	 *                                    - 'only_build_strings': Whether to only include build strings. Default false.
	 *
	 * @since 2.4.10
	 *
	 * @return array|\WP_Error Array of language translations or WP_Error on failure.
	 */
	public function get_languages_strings_by_language_code( $args = array() ) {
		$translations = $this->get_translations( $args );

		$language_translations = array();
		if ( ! empty( $translations ) ) {
			foreach ( $translations as $translation ) {
				$string_value = ( ! empty( $translation->updated_string ) )
					? $translation->updated_string
					: $translation->default_string;

				if ( ! empty( $string_value ) ) {
					// Convert database language code (e.g., 'vn') to API language code (e.g., 'vi') for consistency.
					$lang_code = $this->mapped_language_code( strtolower( $translation->language_code ) );
					$language_translations[ $lang_code ][ $translation->string_handle ] = $string_value;
				}
			}
		}

		return $language_translations;
	}

	/**
	 * Get the currently selected app language from the dropdown
	 *
	 * @since 2.4.10
	 *
	 * @return string The language code of the currently selected app language
	 */
	public function get_app_language() {
		// Get the selected language from option.
		$selected_language = get_option( 'bbapp_selected_language' );

		// Get all supported languages.
		$supported_languages = $this->get_app_supported_languages();

		// Get saved active languages.
		$languages_codes = get_option( 'bbapp_active_languages', array() );
		if ( ! is_array( $languages_codes ) ) {
			$languages_codes = array();
		}

		// If a language is selected and it's in both supported and active languages.
		if ( ! empty( $selected_language ) &&
			isset( $supported_languages[ $selected_language ] ) &&
			in_array( $selected_language, $languages_codes, true ) ) {
			return $selected_language;
		}

		// Get WordPress selected language.
		$wp_locale = bbapp_wp_locale_to_app_locale();

		// If WordPress locale is supported and active, use it.
		if ( isset( $supported_languages[ $wp_locale ] ) && in_array( $wp_locale, $languages_codes, true ) ) {
			return $wp_locale;
		}

		// Default to English.
		return 'en';
	}

	/**
	 * Get the language dir.
	 *
	 * @param bool $delete_dir If need to delete directory.
	 *
	 * @since 1.4.7
	 *
	 * @return array|mixed|string|string[]
	 */
	public function get_language_dir( $delete_dir = false ) {
		static $bbapp_language_dir;

		if ( isset( $bbapp_language_dir ) ) {
			return $bbapp_language_dir;
		}

		if ( bbapp()->is_network_activated() ) {
			switch_to_blog( 1 );
		}

		$upload_dir = wp_upload_dir();
		$dir        = $upload_dir['basedir'] . '/bbapp/language';

		if ( true === $delete_dir ) {
			BBAPP_File::delete_dir( $dir );
		}
		BBAPP_File::create_dir( $dir );
		$dir = str_replace( '/', DIRECTORY_SEPARATOR, $dir );

		if ( bbapp()->is_network_activated() ) {
			restore_current_blog();
		}

		$bbapp_language_dir = $dir;

		return $dir;
	}

	/**
	 * Returns the language url.
	 *
	 * @since 1.4.7
	 *
	 * @return string
	 */
	public function get_language_url() {
		global $bbapp_language_url;

		if ( isset( $bbapp_language_url ) ) {
			return $bbapp_language_url;
		}

		if ( bbapp()->is_network_activated() ) {
			$curr_blog_id = get_current_blog_id();
			switch_to_blog( 1 );
		}

		$upload_dir = wp_upload_dir();
		$url        = $upload_dir['baseurl'] . '/bbapp/language';

		if ( bbapp()->is_network_activated() ) {
			switch_to_blog( $curr_blog_id );
		}

		$bbapp_language_url = $url;

		return $url;
	}

	/**
	 * Create a language file.
	 *
	 * @since 1.4.7
	 *
	 * @return array Language translations data with file paths and language information.
	 */
	public function get_languages_translations() {
		$language_dir = $this->get_language_dir( true );
		$language_url = $this->get_language_url();

		$active_languages_stored  = $this->get_active_languages();
		$build_selected_languages = maybe_unserialize( AppSettings::instance()->get_setting_value( 'build_selected_languages' ) );
		$filtered_languages       = array();

		// Only filter if we have valid build selected languages and they are an array.
		if ( ! empty( $build_selected_languages ) && is_array( $build_selected_languages ) ) {
			foreach ( $build_selected_languages as $language_code ) {
				if ( isset( $active_languages_stored[ $language_code ] ) ) {
					$filtered_languages[ $language_code ] = $active_languages_stored[ $language_code ];
				}
			}
		} else {
			// If no build selected languages, use all active languages from the stored option.
			$filtered_languages = $active_languages_stored;
		}

		$active_language_codes = array_keys( $filtered_languages );
		$args                  = array(
			'only_build_strings' => true,
			'include_deleted'    => true,
			'only_updated'       => true,
			'include_custom'     => true,
			'language_code'      => $active_language_codes,
		);

		$build_translate_lang = $this->get_languages_strings_by_language_code( $args );

		// If no build strings found, ensure we have at least an empty English entry.
		if ( empty( $build_translate_lang ) ) {
			$build_translate_lang = array( 'en' => array() );
		}

		// Create the JSON file with the build translations.
		$translated_language = wp_json_encode( $build_translate_lang );
		$file_name           = 'languages-' . uniqid() . '.json';
		BBAPP_File::write_file( trailingslashit( $language_dir ) . $file_name, $translated_language );

		// Get WordPress default language.
		$wp_locale = bbapp_wp_locale_to_app_locale();

		// Format active languages as required.
		$formatted_languages = array();

		foreach ( $filtered_languages as $lang_code => $lang_details ) {
			$formatted_languages[ $lang_code ] = array(
				'name'       => $lang_details['name'],
				'code'       => $lang_code,
				'is_default' => ( $lang_code === $wp_locale ),
			);
		}

		// Ensure English is always included.
		if ( ! isset( $formatted_languages['en'] ) && ( 'en' === $wp_locale ) ) {
			$formatted_languages['en'] = array(
				'name'       => 'English',
				'code'       => 'en',
				'is_default' => ( 'en' === $wp_locale ),
			);
		}

		// Create the response array with file paths and language info.
		return array(
			'dir'                 => trailingslashit( $language_dir ) . $file_name,
			'url'                 => trailingslashit( $language_url ) . $file_name,
			'wp_default_language' => $wp_locale,
			'supported_languages' => $formatted_languages,
		);
	}

	/**
	 * Get the path to the main language file
	 *
	 * @since 2.4.10
	 *
	 * @return string Path to the language file
	 */
	public function get_language_file_path() {
		return bbapp()->plugin_dir . 'include/App/Format/master-lang.json';
	}

	/**
	 * Get the path to the PTC language file
	 *
	 * @since 2.4.10
	 *
	 * @return string Path to the PTC language file
	 */
	public function get_ptc_language_file_path() {
		return bbapp()->plugin_dir . 'include/App/Format/ptc-langs.json';
	}

	/**
	 * When BuddyBoss App plugin is activated
	 *
	 * @since 2.4.10
	 */
	public function on_activation() {
		global $wpdb;

		$charset_collate = $wpdb->get_charset_collate();

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$sql = "CREATE TABLE {$wpdb->prefix}bbapp_string_translations (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			site_id bigint(20) DEFAULT NULL,
			string_handle varchar(255) DEFAULT NULL,
			default_string longtext DEFAULT NULL,
			is_build_string BOOLEAN DEFAULT 0,
			language_code varchar(255) DEFAULT NULL,
			updated_string longtext DEFAULT NULL,
			is_custom_string BOOLEAN DEFAULT 0,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			deleted_at datetime DEFAULT NULL,
			PRIMARY KEY (id),
			KEY site_id (site_id),
			KEY string_handle (string_handle),
			KEY created_at (created_at),
			KEY updated_at (updated_at),
			KEY deleted_at (deleted_at)
		) {$charset_collate}";

		dbDelta( $sql );
	}

	/**
	 * Get app supported languages
	 *
	 * This function has been updated to use the new bbapp_get_languages_codes and bbapp_get_languages_locales
	 * functions instead of having a hardcoded list. This provides better maintainability and consistency
	 * across the codebase.
	 *
	 * @since 2.4.10
	 *
	 * @return array Array of supported languages with language code as key and language name as value
	 */
	public function get_app_supported_languages() {
		// Get language codes and locales.
		$language_codes   = $this->bbapp_get_languages_codes();
		$language_locales = $this->bbapp_get_languages_locales();

		// Build the array with locale codes as keys and language names as values.
		$supported_languages = array();

		foreach ( $language_codes as $language_name => $language_code ) {
			if ( isset( $language_locales[ $language_code ] ) ) {
				$supported_languages[ $language_code ] = $language_name;
			}
		}

		$ptc_language_file_path = $this->get_ptc_language_file_path();
		$ptc_languages          = json_decode( BBAPP_File::read_file( $ptc_language_file_path ), true );

		// Only include languages that are in the ptc-langs.json file.
		$filtered_languages = array();

		// Always include English.
		if ( isset( $supported_languages['en'] ) ) {
			$filtered_languages['en'] = $supported_languages['en'];
		}

		foreach ( $ptc_languages as $ptc_lang_code ) {
			$ptc_lang_code_lower = strtolower( $ptc_lang_code );

			// First try with mapped language code.
			$mapped_code = $this->mapped_language_code( $ptc_lang_code_lower );
			if ( $mapped_code !== $ptc_lang_code_lower && isset( $supported_languages[ $mapped_code ] ) ) {
				$filtered_languages[ $mapped_code ] = $supported_languages[ $mapped_code ];
				continue;
			}

			// Then try regular matching.
			foreach ( $supported_languages as $lang_code => $lang_name ) {
				if ( strtolower( $lang_code ) === $ptc_lang_code_lower ) {
					$filtered_languages[ $lang_code ] = $lang_name;
					break;
				}
			}
		}

		asort( $filtered_languages );

		return $filtered_languages;
	}

	/**
	 * Get active languages
	 *
	 * @since 2.4.10
	 *
	 * @return array Array of active languages with language code as key and language details as value
	 */
	public function get_active_languages() {
		$active_languages = array();

		// Get WordPress selected language.
		$wp_locale = bbapp_wp_locale_to_app_locale();

		// Get all supported languages.
		$supported_languages = $this->get_app_supported_languages();

		// Get saved active languages.
		$languages_codes = get_option( 'bbapp_active_languages', array() );
		if ( ! is_array( $languages_codes ) ) {
			$languages_codes = array();
		}

		// Filter out unsupported languages.
		$languages_codes = array_filter(
			$languages_codes,
			function ( $code ) use ( $supported_languages ) {
				return isset( $supported_languages[ $code ] );
			}
		);

		// If WordPress locale is supported, ensure it's in the list.
		if ( isset( $supported_languages[ $wp_locale ] ) && ! in_array( $wp_locale, $languages_codes, true ) ) {
			array_unshift( $languages_codes, $wp_locale );
		}

		// Only add 'en' if WordPress locale is not supported.
		if ( ! empty( $languages_codes ) && ! in_array( 'en', $languages_codes, true ) && ! isset( $supported_languages[ $wp_locale ] ) ) {
			array_unshift( $languages_codes, 'en' );
		}

		if ( empty( $languages_codes ) ) {
			$languages_codes = array( 'en' );
		}

		// Process each language code.
		foreach ( $languages_codes as $language_code ) {
			$active_languages[ $language_code ] = array(
				'name'          => $supported_languages[ $language_code ],
				'is_wp_default' => ( $language_code === $wp_locale ),
			);
		}

		// Check if any language is set as WordPress default.
		$has_wp_default = false;
		foreach ( $active_languages as $language ) {
			if ( $language['is_wp_default'] ) {
				$has_wp_default = true;
				break;
			}
		}

		// If no language is set as WordPress default, set English as default.
		if ( ! $has_wp_default && isset( $active_languages['en'] ) ) {
			$active_languages['en']['is_wp_default'] = true;
		}

		return $active_languages;
	}

	/**
	 * Get app languages config
	 *
	 * @since 2.4.10
	 *
	 * @return array Array of app languages config
	 */
	public function get_app_languages_config() {
		$languages_config = array();

		// Get language code from header using the common function.
		$language_code     = bbapp_get_app_language_header();
		$languages_strings = $this->get_languages_strings_by_language_code(
			array(
				'language_code'   => $language_code,
				'only_updated'    => true,
				'include_custom'  => true,
				'include_deleted' => true,
			)
		);
		$app_languages     = $this->get_active_languages();

		if ( ! empty( $language_code ) ) {
			$languages_config['requested_language']    = $language_code;
			$languages_config['is_supported_language'] = isset( $app_languages[ $language_code ] );
			$languages_config['fallback_to_english']   = ! isset( $app_languages[ $language_code ] );
			$languages_config['app_active_languages']  = $app_languages;
			$languages_config['languages_strings']     = $languages_strings;
		}

		return $languages_config;
	}

	/**
	 * Get languages codes from language name
	 *
	 * @since 2.4.10
	 *
	 * @return array
	 */
	public function bbapp_get_languages_codes() {
		static $result = null;

		if ( null === $result ) {
			$result = array(
				'Arabic'                => 'ar',
				'Bulgarian'             => 'bg',
				'Chinese (Simplified)'  => 'zh-hans',
				'Czech'                 => 'cs',
				'Danish'                => 'da',
				'Dutch'                 => 'nl',
				'English'               => 'en',
				'Estonian'              => 'et',
				'Finnish'               => 'fi',
				'French'                => 'fr',
				'German'                => 'de',
				'Greek'                 => 'el',
				'Hebrew'                => 'he',
				'Hungarian'             => 'hu',
				'Indonesian'            => 'id',
				'Italian'               => 'it',
				'Japanese'              => 'ja',
				'Korean'                => 'ko',
				'Latvian'               => 'lv',
				'Lithuanian'            => 'lt',
				'Norwegian Bokmål'      => 'no',
				'Polish'                => 'pl',
				'Portuguese, Brazil'    => 'pt-br',
				'Portuguese, Portugal'  => 'pt',
				'Romanian'              => 'ro',
				'Russian'               => 'ru',
				'Slovak'                => 'sk',
				'Slovenian'             => 'sl',
				'Spanish'               => 'es',
				'Swedish'               => 'sv',
				'Thai'                  => 'th',
				'Turkish'               => 'tr',
				'Ukrainian'             => 'uk',
				'Vietnamese'            => 'vi',
			);
		}

		return $result;
	}

	/**
	 * Get languages locales from language code
	 *
	 * @since 2.4.10
	 *
	 * @return array
	 */
	public function bbapp_get_languages_locales() {
		static $result = null;

		if ( ! $result ) {
			$result = array(
				'ar'      => 'ar',
				'bg'      => 'bg_BG',
				'cs'      => 'cs_CZ',
				'da'      => 'da_DK',
				'de'      => 'de_DE',
				'el'      => 'el',
				'en'      => 'en_US',
				'es'      => 'es_ES',
				'et'      => 'et',
				'fi'      => 'fi',
				'fr'      => 'fr_FR',
				'he'      => 'he_IL',
				'hu'      => 'hu_HU',
				'id'      => 'id_ID',
				'it'      => 'it_IT',
				'ja'      => 'ja',
				'ko'      => 'ko_KR',
				'lt'      => 'lt_LT',
				'lv'      => 'lv_LV',
				'nb'      => 'nb_NO',
				'no'      => 'nb_NO',
				'nl'      => 'nl_NL',
				'pl'      => 'pl_PL',
				'pt-br'   => 'pt_BR',
				'pt'      => 'pt_PT',
				'ro'      => 'ro_RO',
				'ru'      => 'ru_RU',
				'sk'      => 'sk_SK',
				'sl'      => 'sl_SI',
				'sv'      => 'sv_SE',
				'th'      => 'th',
				'tr'      => 'tr_TR',
				'uk'      => 'uk',
				'vi'      => 'vi',
				'zh-hans' => 'zh_CN',
			);
		}

		return $result;
	}

	/**
	 * Get language name from language code.
	 *
	 * @param string $language_code Language code.
	 *
	 * @since 2.4.10
	 *
	 * @return  string Language name or the original language code if not found.
	 */
	public function get_language_name( $language_code ) {
		$supported_languages = $this->get_app_supported_languages();

		if ( isset( $supported_languages[ $language_code ] ) ) {
			return $supported_languages[ $language_code ];
		}

		return $language_code;
	}

	/**
	 * Map language code
	 *
	 * @param string $language_code Language code to map.
	 *
	 * @since 2.4.10
	 *
	 * @return string Mapped language code
	 */
	public function mapped_language_code( $language_code ) {
		$mapped_language_code = array(
			'nb' => 'no',
			'vn' => 'vi',
		);
		if ( isset( $mapped_language_code[ $language_code ] ) ) {
			return $mapped_language_code[ $language_code ];
		}

		return $language_code;
	}

	/**
	 * Get PTC supported languages from JSON file (cached for performance)
	 *
	 * @since 2.4.10
	 * @return array Array of PTC supported language codes
	 */
	public function get_ptc_supported_languages() {
		static $ptc_languages = null;

		if ( is_null( $ptc_languages ) ) {
			$ptc_file_path = plugin_dir_path( __FILE__ ) . 'App/Format/ptc-langs.json';

			if ( file_exists( $ptc_file_path ) ) {
				$ptc_content   = file_get_contents( $ptc_file_path ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
				$ptc_languages = json_decode( $ptc_content, true );

				if ( ! is_array( $ptc_languages ) ) {
					$ptc_languages = array();
				}
			} else {
				$ptc_languages = array();
			}

			// Always ensure English is included.
			if ( ! in_array( 'en', $ptc_languages, true ) ) {
				$ptc_languages[] = 'en';
			}
		}

		return array_map( 'strtolower', $ptc_languages );
	}

	/**
	 * Get a single translation from the database by ID or a combination of filters
	 *
	 * @param int|array $id_or_args Translation ID or array of arguments to filter by.
	 *
	 * @since 2.4.10
	 *
	 * @return object|null Translation object or null if not found
	 */
	public function get_translation( $id_or_args ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'bbapp_string_translations';

		if ( is_numeric( $id_or_args ) ) {
			// Get by ID.
			return $wpdb->get_row( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
				$wpdb->prepare(
					"SELECT * FROM {$table_name} WHERE id = %d", // phpcs:ignore
					intval( $id_or_args )
				)
			);
		} elseif ( is_array( $id_or_args ) ) {
			// Map language code for consistency if present in the arguments.
			if ( isset( $id_or_args['language_code'] ) ) {
				// Convert API language code to database format for querying if it's an array.
				$id_or_args['language_code'] = $this->get_db_language_code( $id_or_args['language_code'] );
			}

			// Build query from args.
			$query      = "SELECT * FROM {$table_name} WHERE 1=1";
			$query_args = array();

			// Add filters for specific fields.
			if ( isset( $id_or_args['language_code'] ) ) {
				$query       .= ' AND language_code = %s';
				$query_args[] = $id_or_args['language_code'];
			}

			if ( isset( $id_or_args['string_handle'] ) ) {
				$query       .= ' AND string_handle = %s';
				$query_args[] = $id_or_args['string_handle'];
			}

			if ( isset( $id_or_args['site_id'] ) ) {
				$query       .= ' AND site_id = %d';
				$query_args[] = intval( $id_or_args['site_id'] );
			}

			// Include/exclude deleted.
			if ( isset( $id_or_args['include_deleted'] ) && ! $id_or_args['include_deleted'] ) {
				$query .= ' AND deleted_at IS NULL';
			} elseif ( isset( $id_or_args['only_deleted'] ) && $id_or_args['only_deleted'] ) {
				$query .= ' AND deleted_at IS NOT NULL';
			}

			// Add LIMIT 1 to ensure we only get one record.
			$query .= ' LIMIT 1';

			if ( count( $query_args ) > 0 ) {
				return $wpdb->get_row( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
					$wpdb->prepare(
						$query, // phpcs:ignore
						$query_args
					)
				);
			} else {
				return $wpdb->get_row( $query ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.NotPrepared
			}
		}

		return null;
	}

	/**
	 * Get multiple translations from the database with filtering options
	 *
	 * @param array $args                 {
	 *                                    Optional. Arguments to filter and retrieve translations.
	 *
	 * @type string|array $language_code        Specific language code(s) to retrieve. Default empty (all languages).
	 * @type string|array $string_handle        Specific string handle(s) to retrieve. Default empty (all handles).
	 * @type string       $search               Search term to filter results by. Default empty.
	 * @type array        $search_columns       Database columns to apply search against. Default: ['string_handle', 'default_string', 'updated_string'].
	 * @type bool         $include_deleted      Whether to include soft-deleted translations in results. Default false.
	 * @type bool         $only_deleted         Whether to retrieve only soft-deleted translations. Default false.
	 * @type bool         $only_build_strings   Whether to retrieve only build-time strings. Default false.
	 * @type bool         $only_empty           Whether to retrieve only translations with empty values. Default false.
	 * @type bool         $only_default_matches Whether to retrieve only translations that match their default value. Default false.
	 * @type bool         $include_custom       Whether to include custom-added strings in results. Default false.
	 * @type bool         $exclude_custom       Whether to exclude custom-added strings from results. Default false.
	 * @type bool         $only_updated         Whether to retrieve only strings with custom translations. Default false.
	 * @type string       $order_by             Database column to sort results by. Accepts: 'id', 'string_handle', 'language_code', 'created_at', 'updated_at'. Default
	 *                                          'string_handle'.
	 * @type string       $order                Sort direction. Accepts 'ASC' or 'DESC'. Default 'ASC'.
	 * @type int          $limit                Maximum number of translations to retrieve. Default 0 (no limit).
	 * @type int          $offset               Number of translations to skip. Default 0.
	 * @type bool         $count                Whether to return only the count of matching translations. Default false.
	 *                                          }
	 * @since 2.4.10
	 *
	 * @return array|int Array of translation objects or count if $args['count'] is true.
	 */
	public function get_translations( $args = array() ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'bbapp_string_translations';

		$default_args = array(
			'language_code'        => '',
			'string_handle'        => '',
			'search'               => '',
			'search_columns'       => array( 'string_handle', 'default_string', 'updated_string' ),
			'include_deleted'      => false,
			'only_deleted'         => false,
			'only_build_strings'   => false,
			'only_empty'           => false,
			'only_default_matches' => false,
			'include_custom'       => false,
			'exclude_custom'       => false,
			'only_updated'         => false,
			'order_by'             => 'string_handle',
			'order'                => 'ASC',
			'limit'                => 0,
			'offset'               => 0,
			'count'                => false,
		);

		$args = wp_parse_args( $args, $default_args );

		// Map language codes to ensure consistency for database queries and API responses.
		if ( ! empty( $args['language_code'] ) ) {
			if ( is_array( $args['language_code'] ) ) {
				foreach ( $args['language_code'] as $key => $code ) {
					// Convert API language codes to database format for querying.
					$args['language_code'][ $key ] = $this->get_db_language_code( $code );
				}
			} else {
				// Convert API language code to database format for querying.
				$args['language_code'] = $this->get_db_language_code( $args['language_code'] );
			}
		}

		// Start building the query.
		$query      = 'SELECT ' . ( $args['count'] ? 'COUNT(*)' : '*' ) . " FROM {$table_name} WHERE 1=1";
		$query_args = array();

		// Add filters for language code.
		if ( ! empty( $args['language_code'] ) ) {
			$query .= $this->build_in_clause( $args['language_code'], 'language_code', $query_args );
		}

		// Add filters for string handle.
		if ( ! empty( $args['string_handle'] ) ) {
			$query .= $this->build_in_clause( $args['string_handle'], 'string_handle', $query_args );
		}

		// Handle deletion status filters.
		if ( ! $args['include_deleted'] && ! $args['only_deleted'] ) {
			$query .= ' AND deleted_at IS NULL';
		} elseif ( $args['only_deleted'] ) {
			$query .= ' AND deleted_at IS NOT NULL';
		}

		// Handle build strings filter.
		if ( $args['only_build_strings'] ) {
			$query .= ' AND is_build_string = 1';
		}

		// Handle empty translations filter.
		if ( $args['only_empty'] ) {
			$query .= " AND (updated_string IS NULL OR updated_string = '')";
		}

		// Handle default matches filter.
		if ( $args['only_default_matches'] ) {
			$query .= ' AND (updated_string = default_string)';
		}

		// Handle updated and custom string filters.
		$query .= $this->build_string_type_conditions(
			$args['only_updated'],
			$args['include_custom'],
			$args['exclude_custom']
		);

		// Handle search functionality.
		if ( ! empty( $args['search'] ) && ! empty( $args['search_columns'] ) ) {
			$query .= $this->build_search_conditions( $args['search'], $args['search_columns'], $query_args );
		}

		// Don't add ORDER BY for count queries.
		if ( ! $args['count'] ) {
			// Build order clause.
			$query .= $this->build_order_clause( $args['order_by'], $args['order'] );

			// Add limit and offset if provided.
			if ( $args['limit'] > 0 ) {
				$query       .= ' LIMIT %d';
				$query_args[] = intval( $args['limit'] );

				if ( $args['offset'] > 0 ) {
					$query       .= ' OFFSET %d';
					$query_args[] = intval( $args['offset'] );
				}
			}
		}

		// Prepare the query if we have arguments.
		$prepared_query = count( $query_args ) > 0 ? $wpdb->prepare(
			$query, // phpcs:ignore
			$query_args
		) : $query; // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.NotPrepared

		// Return count or results.
		return $args['count'] ? (int) $wpdb->get_var( $prepared_query ) : $wpdb->get_results( $prepared_query ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.NotPrepared
	}

	/**
	 * Build SQL IN clause for array values
	 *
	 * @param string|array $values     The value(s) to include in the IN clause.
	 * @param string       $column     The column name to match against.
	 * @param array        $query_args The array of query arguments to append values to.
	 *
	 * @since 2.4.10
	 *
	 * @return string The SQL condition string.
	 */
	private function build_in_clause( $values, $column, &$query_args ) {
		if ( is_array( $values ) ) {
			$placeholders = implode( ', ', array_fill( 0, count( $values ), '%s' ) );
			$query_args   = array_merge( $query_args, $values );

			return " AND {$column} IN ({$placeholders})";
		} else {
			$query_args[] = $values;

			return " AND {$column} = %s";
		}
	}

	/**
	 * Build SQL conditions for string type filtering
	 *
	 * @param bool $only_updated   Whether to only include updated strings.
	 * @param bool $include_custom Whether to include custom strings.
	 * @param bool $exclude_custom Whether to exclude custom strings.
	 *
	 * @since 2.4.10
	 *
	 * @return string The SQL condition string.
	 */
	private function build_string_type_conditions( $only_updated, $include_custom, $exclude_custom ) {
		if ( $only_updated && $include_custom ) {
			// Include both updated regular strings AND custom strings.
			return " AND ((updated_string IS NOT NULL AND updated_string != '' AND updated_string != default_string) OR is_custom_string = 1)";
		} elseif ( $only_updated ) {
			// Only regular updated strings.
			return " AND (updated_string IS NOT NULL AND updated_string != '' AND updated_string != default_string)";
		} elseif ( $include_custom ) {
			// Only custom strings.
			return ' AND is_custom_string = 1';
		} elseif ( $exclude_custom ) {
			// Exclude custom strings.
			return ' AND is_custom_string = 0';
		}

		return '';
	}

	/**
	 * Build SQL search conditions
	 *
	 * @param string $search         The search term.
	 * @param array  $search_columns Columns to search within.
	 * @param array  $query_args     The array of query arguments to append values to.
	 *
	 * @since 2.4.10
	 *
	 * @return string The SQL condition string.
	 */
	private function build_search_conditions( $search, $search_columns, &$query_args ) {
		global $wpdb;

		$search_terms = array();
		foreach ( $search_columns as $column ) {
			$search_terms[] = $column . ' LIKE %s';
			$query_args[]   = '%' . $wpdb->esc_like( $search ) . '%';
		}

		return ! empty( $search_terms ) ? ' AND (' . implode( ' OR ', $search_terms ) . ')' : '';
	}

	/**
	 * Build SQL ORDER BY clause
	 *
	 * @param string $order_by Column to order by.
	 * @param string $order    Order direction (ASC or DESC).
	 *
	 * @since 2.4.10
	 *
	 * @return string The SQL ORDER BY clause.
	 */
	private function build_order_clause( $order_by, $order ) {
		$allowed_order_columns = array( 'id', 'string_handle', 'language_code', 'created_at', 'updated_at' );
		$sanitized_order_by    = in_array( $order_by, $allowed_order_columns, true ) ? $order_by : 'string_handle';
		$sanitized_order       = 'DESC' === $order ? 'DESC' : 'ASC';

		return " ORDER BY {$sanitized_order_by} {$sanitized_order}";
	}

	/**
	 * Insert a new translation into the database
	 *
	 * @param array $data Translation data to insert.
	 *
	 * @since 2.4.10
	 *
	 * @return int|false The ID of the inserted translation or false on failure
	 */
	public function insert_translation( $data ) {
		global $wpdb;
		$table_name   = $wpdb->prefix . 'bbapp_string_translations';
		$current_time = current_time( 'mysql' );

		// Required fields.
		if ( empty( $data['language_code'] ) || empty( $data['string_handle'] ) ) {
			return false;
		}

		// Check if this translation already exists.
		$exists = $this->get_translation(
			array(
				'language_code'   => $data['language_code'],
				'string_handle'   => $data['string_handle'],
				'include_deleted' => true,
			)
		);

		if ( $exists ) {
			// If it was previously deleted, update it instead.
			if ( null !== $exists->deleted_at ) {
				$update_data = array(
					'default_string'   => isset( $data['default_string'] ) ? $data['default_string'] : $exists->default_string,
					'updated_string'   => isset( $data['updated_string'] ) ? $data['updated_string'] : $exists->updated_string,
					'is_build_string'  => isset( $data['is_build_string'] ) ? intval( $data['is_build_string'] ) : $exists->is_build_string,
					'is_custom_string' => isset( $data['is_custom_string'] ) ? intval( $data['is_custom_string'] ) : $exists->is_custom_string,
					'updated_at'       => $current_time,
					'deleted_at'       => null,
				);

				$result = $wpdb->update( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
					$table_name,
					$update_data,
					array( 'id' => $exists->id )
				);

				return $result ? $exists->id : false;
			}

			// Otherwise, it's a duplicate - return false or update.
			return false;
		}

		// Prepare data for insertion.
		$insert_data = array(
			'site_id'          => isset( $data['site_id'] ) ? intval( $data['site_id'] ) : get_current_blog_id(),
			'string_handle'    => $data['string_handle'],
			'language_code'    => $data['language_code'],
			'default_string'   => isset( $data['default_string'] ) ? $data['default_string'] : '',
			'updated_string'   => isset( $data['updated_string'] ) ? $data['updated_string'] : '',
			'is_build_string'  => isset( $data['is_build_string'] ) ? intval( $data['is_build_string'] ) : 0,
			'is_custom_string' => isset( $data['is_custom_string'] ) ? intval( $data['is_custom_string'] ) : 0,
			'created_at'       => $current_time,
			'updated_at'       => isset( $data['updated_string'] ) ? $current_time : null,
			'deleted_at'       => null,
		);

		$result = $wpdb->insert( $table_name, $insert_data ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery

		return $result ? $wpdb->insert_id : false;
	}

	/**
	 * Update an existing translation in the database
	 *
	 * @param int   $id   Translation ID.
	 * @param array $data Translation data to update.
	 *
	 * @since 2.4.10
	 *
	 * @return bool Whether the update was successful
	 */
	public function update_translation( $id, $data ) {
		global $wpdb;
		$table_name   = $wpdb->prefix . 'bbapp_string_translations';
		$current_time = current_time( 'mysql' );

		// Get the existing translation.
		$existing = $this->get_translation( $id );
		if ( ! $existing ) {
			return false;
		}

		// Prepare update data.
		$update_data = array();

		if ( isset( $data['default_string'] ) ) {
			$update_data['default_string'] = $data['default_string'];
		}

		if ( isset( $data['updated_string'] ) ) {
			$update_data['updated_string'] = $data['updated_string'];
			$update_data['updated_at']     = $current_time;
		}

		if ( isset( $data['is_build_string'] ) ) {
			$update_data['is_build_string'] = intval( $data['is_build_string'] );
		}

		if ( isset( $data['is_custom_string'] ) ) {
			$update_data['is_custom_string'] = intval( $data['is_custom_string'] );
		}

		if ( isset( $data['deleted_at'] ) ) {
			$update_data['deleted_at'] = $data['deleted_at'] ? $current_time : null;
		}

		// If no data to update, return true (no changes needed).
		if ( empty( $update_data ) ) {
			return true;
		}

		$result = $wpdb->update( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
			$table_name,
			$update_data,
			array( 'id' => $id )
		);

		return false !== $result;
	}

	/**
	 * Delete a translation from the database (soft delete by default)
	 *
	 * @param int  $id           Translation ID.
	 * @param bool $force_delete Whether to permanently delete the record.
	 *
	 * @since 2.4.10
	 *
	 * @return bool Whether the deletion was successful
	 */
	public function delete_translation( $id, $force_delete = false ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'bbapp_string_translations';

		// Get the existing translation.
		$existing = $this->get_translation( $id );
		if ( ! $existing ) {
			return false;
		}

		if ( $force_delete ) {
			// Permanently delete the record.
			$result = $wpdb->delete( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
				$table_name,
				array( 'id' => $id )
			);
		} else {
			// Soft delete - mark as deleted.
			$current_time = current_time( 'mysql' );
			$result       = $wpdb->update( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
				$table_name,
				array( 'deleted_at' => $current_time ),
				array( 'id' => $id )
			);
		}

		return false !== $result;
	}

	/**
	 * Batch delete multiple translations at once
	 *
	 * @param array $ids          Array of translation IDs to delete.
	 * @param bool  $force_delete Whether to permanently delete the records.
	 *
	 * @since 2.4.10
	 *
	 * @return array Results array with counts of success and failure
	 */
	public function batch_delete_translations( $ids, $force_delete = false ) {
		global $wpdb;

		$results = array(
			'success' => 0,
			'failure' => 0,
		);

		// Begin transaction.
		$wpdb->query( 'START TRANSACTION' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery

		try {
			foreach ( $ids as $id ) {
				$result = $this->delete_translation( $id, $force_delete );

				if ( $result ) {
					++$results['success'];
				} else {
					++$results['failure'];
				}
			}

			// Commit transaction.
			$wpdb->query( 'COMMIT' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
		} catch ( \Exception $e ) {
			// Rollback on error.
			$wpdb->query( 'ROLLBACK' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery

			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'BuddyBoss App: Error in batch_delete_translations: ' . $e->getMessage() ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			}

			$results['failure'] = count( $ids );
			$results['success'] = 0;
		}

		return $results;
	}

	/**
	 * On plugin activate import languages from master-lang.json file.
	 *
	 * @since 2.4.10
	 *
	 * @return void
	 */
	public function on_plugin_activate() {
		$json_file_path = $this->get_language_file_path();

		// Import languages.
		$import_result = $this->import_languages();

		// If import was successful, update the stored modification time.
		if ( $import_result && file_exists( $json_file_path ) ) {
			update_option( 'bbapp_languages_file_mod_time', filemtime( $json_file_path ) );
		}
	}

	/**
	 * On plugin update import languages from master-lang.json file.
	 *
	 * @param object $upgrader_object WP_Upgrader instance.
	 * @param array  $options         Array of bulk item update data.
	 *
	 * @since 2.4.10
	 *
	 * @return void
	 */
	public function on_plugin_update( $upgrader_object, $options ) {
		// Check if our plugin was updated.
		if ( 'update' === $options['action'] && 'plugin' === $options['type'] ) {
			if ( isset( $options['plugins'] ) ) {
				foreach ( $options['plugins'] as $plugin ) {
					if ( plugin_basename( bbapp()->plugin_dir . 'buddyboss-app.php' ) === $plugin ) {
						// Force re-import by deleting the stored modification time.
						delete_option( 'bbapp_languages_file_mod_time' );

						// This will trigger the import on the next admin page load
						// But for immediate effect, we'll import now.
						$this->import_languages();
					}
				}
			}
		}
	}

	/**
	 * Import languages with change tracking
	 *
	 * @since 2.4.10
	 *
	 * @return bool True on success, false on failure
	 */
	public function import_languages() {
		// Read the JSON file.
		$json      = BBAPP_File::read_file( $this->get_language_file_path() );
		$json_data = json_decode( $json, true );

		if ( empty( $json_data ) || ! is_array( $json_data ) ) {
			error_log( 'Failed to parse language JSON file or empty data' ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log

			return false;
		}

		// Restructure the data into language-based format.
		$new_data = array();
		foreach ( $json_data as $string_item ) {
			if ( ! isset( $string_item['handle'] ) || ! isset( $string_item['default'] ) || ! isset( $string_item['translation'] ) ) {
				continue;
			}

			// Add default English translation if not specified.
			if ( ! isset( $string_item['translation']['en'] ) ) {
				$string_item['translation']['en'] = $string_item['default'];
			}

			// Organize by language.
			foreach ( $string_item['translation'] as $lang_code => $translation ) {
				if ( ! isset( $new_data[ $lang_code ] ) ) {
					$new_data[ $lang_code ] = array();
				}
				$new_data[ $lang_code ][ $string_item['handle'] ] = $translation;
			}
		}

		global $wpdb;
		$table_name   = $wpdb->prefix . 'bbapp_string_translations';
		$current_time = current_time( 'mysql' );
		$site_id      = get_current_blog_id();

		// Get all existing translations.
		$existing_translations = $this->get_existing_translations();

		// Track which strings we've processed.
		$processed_strings = array();

		// Begin transaction.
		$wpdb->query( 'START TRANSACTION' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery

		try {
			// Process each language and string.
			foreach ( $new_data as $language_code => $language_data ) {
				foreach ( $language_data as $string_handle => $translated_string ) {
					$key                 = $language_code . '_' . $string_handle;
					$processed_strings[] = $key;

					// Determine if this is a build string based on the prefix.
					$is_build_string = 0 === ( strpos( $string_handle, 'bbAppBuildTime.' ) ) ? 1 : 0;

					// Check if string exists.
					if ( isset( $existing_translations[ $key ] ) ) {
						$existing_translation = $existing_translations[ $key ];

						// Check if this string was previously marked as deleted.
						if ( ! empty( $existing_translation['deleted_at'] ) ) {
							// String has been restored in the JSON file, so un-delete it.
							$wpdb->update( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
								$table_name,
								array(
									'default_string'   => $translated_string,
									'deleted_at'       => null, // Un-delete the string.
									'is_build_string'  => $is_build_string, // Update the build string flag.
									'is_custom_string' => 0, // Update the custom string flag.
								),
								array(
									'id' => $existing_translation['id'],
								)
							);
						} elseif ( $existing_translation['default_string'] !== $translated_string ) { // Only update if default string has changed and there's no custom translation.
							$update_data = array(
								'default_string'   => $translated_string,
								'is_build_string'  => $is_build_string, // Update the build string flag.
								'is_custom_string' => 0, // Update the custom string flag.
							);

							$wpdb->update( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
								$table_name,
								$update_data,
								array(
									'id' => $existing_translation['id'],
								)
							);
						}
					} else { // Check if this string was previously deleted.
						$deleted_string = $wpdb->get_row( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
							$wpdb->prepare(
								"SELECT id FROM {$table_name} 
								WHERE language_code = %s 
								AND string_handle = %s 
								AND deleted_at IS NOT NULL",
								$language_code,
								$string_handle
							)
						);

						if ( $deleted_string ) {
							// String was previously deleted, restore it.
							$wpdb->update( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
								$table_name,
								array(
									'default_string'   => $translated_string,
									'is_build_string'  => $is_build_string, // Update the build string flag.
									'is_custom_string' => 0, // Update the custom string flag.
									'deleted_at'       => null, // Un-delete the string.
								),
								array(
									'id' => $deleted_string->id,
								)
							);
						} else {
							// Insert new string.
							$wpdb->insert( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
								$table_name,
								array(
									'site_id'          => $site_id,
									'language_code'    => $language_code,
									'string_handle'    => $string_handle,
									'default_string'   => $translated_string,
									'updated_string'   => '',
									'is_build_string'  => $is_build_string, // Set based on string handle prefix.
									'is_custom_string' => 0, // Set to 0 as it's not a custom string.
									'created_at'       => $current_time,
									'updated_at'       => null, // Initially null as no manual updates.
									'deleted_at'       => null,
								)
							);
						}
					}
				}
			}

			// Mark deleted strings.
			foreach ( $existing_translations as $key => $translation ) {
				if ( ! in_array( $key, $processed_strings, true ) && empty( $translation['deleted_at'] ) ) {
					$wpdb->update( // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery
						$table_name,
						array(
							'deleted_at' => $current_time,
						),
						array(
							'id' => $translation['id'],
						)
					);
				}
			}

			// Commit transaction.
			$wpdb->query( 'COMMIT' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery

			return true;
		} catch ( \Exception $e ) {
			// Rollback on error.
			$wpdb->query( 'ROLLBACK' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			error_log( 'Error importing languages: ' . $e->getMessage() ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log

			return false;
		}
	}

	/**
	 * Get existing translations from database
	 *
	 * @since 2.4.10
	 *
	 * @return array Array of translations indexed by language_code_string_handle
	 */
	private function get_existing_translations() {
		// Get all translations with no filters.
		$results = $this->get_translations(
			array(
				'include_deleted' => true,
				'exclude_custom'  => true,
				'limit'           => 0, // No limit.
			)
		);

		$translations = array();
		foreach ( $results as $row ) {
			$key                  = $row->language_code . '_' . $row->string_handle;
			$translations[ $key ] = (array) $row;
		}

		return $translations;
	}

	/**
	 * Map language code from API to database
	 * This is the reverse of mapped_language_code function
	 * Used when we need to query the database with language codes from API
	 *
	 * @param string $language_code Language code to convert back to DB format.
	 *
	 * @since 2.4.10
	 *
	 * @return string Original database language code
	 */
	public function get_db_language_code( $language_code ) {
		$reverse_mapped_language_code = array(
			'vi' => 'vn', // Convert 'vi' back to 'vn' for database queries.
			'no' => 'nb', // Convert 'no' back to 'nb' for database queries.
		);
		if ( isset( $reverse_mapped_language_code[ $language_code ] ) ) {
			return $reverse_mapped_language_code[ $language_code ];
		}

		return $language_code;
	}
}
