<?php
/**
 * Holds features class functionality.
 *
 * @package BuddyBossApp\App
 */

namespace BuddyBossApp\App;

use BuddyBossApp\Admin\Settings;
use BuddyBossApp\AppSettings;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Menus\Types\AppSettingsMenus;
use BuddyBossApp\Menus\Types\BBMenus;
use BuddyBossApp\Menus\Types\CoreMenus;
use BuddyBossApp\Menus\Types\CustomLinkMenus;
use BuddyBossApp\Menus\Types\CustomMenus;
use BuddyBossApp\Menus\Types\PostTypeMenus;
use BuddyBossApp\Menus\Types\SectionsMenus;

/**
 * Features class.
 */
class Features {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Features constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Use singleton.
	 *
	 * @return mixed
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		add_filter( 'bbapp_feature_availability_buddyboss_platform', array( $this, 'buddyboss_platform_feature_availability' ), 1, 1 );
		add_filter( 'bbapp_feature_availability_private_app', array( $this, 'private_app_feature_availability' ), 1, 1 );
		add_filter( 'bbapp_feature_availability_offline', array( $this, 'offline_feature_availability' ), 1, 1 );
		add_filter( 'bbapp_feature_availability_learndash.offline', array( $this, 'offline_feature_availability' ), 1, 1 );
		add_filter( 'bbapp_feature_availability_learndash.author_visible', array( $this, 'author_visible_feature_availability' ), 1, 1 );
		add_filter( 'bbapp_feature_availability_learndash.date_visible', array( $this, 'date_visible_feature_availability' ), 1, 1 );
		add_filter( 'bbapp_feature_availability_learndash.reader_app_compatibility', array( $this, 'reader_app_compatibility_feature_availability' ), 1, 1 );
		add_filter( 'bbapp_feature_availability_push_notifications', array( $this, 'push_notification_feature_availability' ), 1, 1 );
		add_filter( 'bbapp_feature_availability_developer_access', array( $this, 'developer_access_feature_availability' ), 1, 1 );
		add_filter( 'bbapp_feature_availability_post_sharing', array( $this, 'post_sharing_feature_availability' ), 1, 1 );
		add_filter( 'bbapp_feature_availability_related_post', array( $this, 'related_post_feature_availability' ), 1, 1 );
		add_filter( 'bbapp_feature_availability_tutor_lms_app', array( $this, 'tutor_lms_app_feature_availability' ), 1, 1 );
		add_filter( 'bbapp_feature_availability_tutor_lms_pro_app', array( $this, 'tutor_lms_pro_app_feature_availability' ), 1, 1 );

		add_filter(
			'bbapp_feature_availability_share_app_link',
			array(
				$this,
				'bbapp_share_app_link_feature_availability',
			),
			1,
			1
		);
		add_action( 'init', array( $this, 'local_features' ) );
		add_action( 'init', array( $this, 'load_core_menus' ) ); // Custom post type load in init.
	}

	/**
	 * Core Menus.
	 */
	public function load_core_menus() {
		CoreMenus::instance();
		PostTypeMenus::instance();
		BBMenus::instance();
		AppSettingsMenus::instance();
		CustomLinkMenus::instance();
		CustomMenus::instance();
		SectionsMenus::instance();
	}

	/**
	 * Add location features related to social features.
	 */
	public function local_features() {
		// BuddyBoss Platform.
		App::instance()->add_local_feature(
			'buddyboss_platform',
			array(
				'is_enabled_android' => false,
				'is_enabled_ios'     => false,
			)
		);

		// Private app support.
		App::instance()->add_local_feature(
			'private_app',
			array(
				'is_enabled_android' => true,
				'is_enabled_ios'     => true,
			)
		);

		// Offline support.
		App::instance()->add_local_feature(
			'offline',
			array(
				'is_enabled_android' => false,
				'is_enabled_ios'     => false,
			)
		);

		App::instance()->add_local_feature(
			'learndash.offline',
			array(
				'is_enabled_android' => false,
				'is_enabled_ios'     => false,
			)
		);

		App::instance()->add_local_feature(
			'learndash.author_visible',
			array(
				'is_enabled_android' => false,
				'is_enabled_ios'     => false,
			)
		);

		App::instance()->add_local_feature(
			'learndash.date_visible',
			array(
				'is_enabled_android' => false,
				'is_enabled_ios'     => false,
			)
		);

		App::instance()->add_local_feature(
			'learndash.reader_app_compatibility',
			array(
				'is_enabled_android' => false,
				'is_enabled_ios'     => false,
			)
		);

		// Push Notification.
		App::instance()->add_local_feature(
			'push_notifications',
			array(
				'is_enabled_android' => true,
				'is_enabled_ios'     => true,
			)
		);

		// Network Mode.
		App::instance()->add_local_feature(
			'multisite_network',
			array(
				'is_enabled_android' => bbapp()->is_network_activated(),
				'is_enabled_ios'     => bbapp()->is_network_activated(),
			)
		);

		// Developer Access.
		App::instance()->add_local_feature(
			'developer_access',
			array(
				'is_enabled_android' => false,
				'is_enabled_ios'     => false,
			)
		);

		// Post sharing.
		App::instance()->add_local_feature(
			'post_sharing',
			array(
				'is_enabled_android' => false,
				'is_enabled_ios'     => false,
			)
		);

		// Related Post.
		App::instance()->add_local_feature(
			'related_post',
			array(
				'is_enabled_android' => false,
				'is_enabled_ios'     => false,
			)
		);

		// Share App link.
		App::instance()->add_local_feature(
			'share_app_link',
			array(
				'is_enabled_android' => false,
				'is_enabled_ios'     => false,
			)
		);

		// Tutor LMS.
		App::instance()->add_local_feature(
			'tutor_lms_app',
			array(
				'is_enabled_android' => false,
				'is_enabled_ios'     => false,
			)
		);

		// Tutor LMS Pro.
		App::instance()->add_local_feature(
			'tutor_lms_pro_app',
			array(
				'is_enabled_android' => false,
				'is_enabled_ios'     => false,
			)
		);
	}

	/**
	 * Changes the feature status based on WordPress requirements.
	 *
	 * @param array $feature App features.
	 *
	 * @return mixed
	 */
	public function buddyboss_platform_feature_availability( $feature ) {
		$enabled                       = defined( 'BP_PLATFORM_VERSION' );
		$feature['is_enabled_ios']     = $enabled;
		$feature['is_enabled_android'] = $enabled;

		return $feature;
	}

	/**
	 * Changes the feature status based on WordPress requirements.
	 *
	 * @param array $feature App features.
	 *
	 * @return mixed
	 */
	public function private_app_feature_availability( $feature ) {
		$settings                      = Settings::instance()->get_settings();
		$enabled                       = ( isset( $settings['private_app.enabled'] ) && $settings['private_app.enabled'] );
		$feature['is_enabled_ios']     = $enabled;
		$feature['is_enabled_android'] = $enabled;

		return $feature;
	}

	/**
	 * Whether the Offline Courses functionality is available or not.
	 *
	 * @param array $feature App features.
	 *
	 * @return mixed
	 */
	public function offline_feature_availability( $feature ) {
		$settings = $this->get_settings();
		$enabled  = false;

		if ( isset( $settings['learndash_course_downloading'] ) && '1' === (string) $settings['learndash_course_downloading'] ) {
			$enabled = true;
		}

		$feature['is_enabled_ios']     = (bool) $enabled;
		$feature['is_enabled_android'] = (bool) $enabled;

		return $feature;
	}

	/**
	 * Show author
	 *
	 * @param array $feature App features.
	 *
	 * @return mixed
	 */
	public function author_visible_feature_availability( $feature ) {
		$settings = $this->get_settings();
		$enabled  = false;

		if ( isset( $settings['learndash_author_visible'] ) && '1' === (string) $settings['learndash_author_visible'] ) {
			$enabled = true;
		}

		$feature['is_enabled_ios']     = (bool) $enabled;
		$feature['is_enabled_android'] = (bool) $enabled;

		return $feature;
	}

	/**
	 * Show Date.
	 *
	 * @param array $feature App features.
	 *
	 * @return mixed
	 */
	public function date_visible_feature_availability( $feature ) {
		$settings = $this->get_settings();
		$enabled  = false;

		if ( isset( $settings['learndash_date_visible'] ) && '1' === (string) $settings['learndash_date_visible'] ) {
			$enabled = true;
		}

		$feature['is_enabled_ios']     = (bool) $enabled;
		$feature['is_enabled_android'] = (bool) $enabled;

		return $feature;
	}

	/**
	 * Show Reader App Compatibility.
	 *
	 * @param array $feature App features.
	 *
	 * @return mixed
	 */
	public function reader_app_compatibility_feature_availability( $feature ) {
		$settings = $this->get_settings();
		$enabled  = false;

		if ( isset( $settings['learndash_reader_app_compatibility'] ) && '1' === (string) $settings['learndash_reader_app_compatibility'] ) {
			$enabled = true;
		}

		$feature['is_enabled_ios']     = (bool) $enabled;
		$feature['is_enabled_android'] = (bool) $enabled;

		return $feature;
	}

	/**
	 * Push Notification feature availability.
	 *
	 * @param array $feature App features.
	 *
	 * @return mixed
	 */
	public function push_notification_feature_availability( $feature ) {
		$enabled                       = bbapp_is_active( 'push_notification' ); // it's always enabled when current plugin is enabled.
		$feature['is_enabled_ios']     = $enabled;
		$feature['is_enabled_android'] = $enabled;

		return $feature;
	}

	/**
	 * Developer Access feature availability.
	 *
	 * @param array $feature App features.
	 *
	 * @return mixed
	 */
	public function developer_access_feature_availability( $feature ) {
		$app_info         = ManageApp::instance()->get_app_info();
		$developer_access = false;

		if ( ! is_wp_error( $app_info ) || ! $app_info ) {
			$developer_access = ( isset( $app_info['has_developer_access'] ) ? $app_info['has_developer_access'] : false );
			$developer_access = (bool) $developer_access;
		}

		$feature['is_enabled_ios']     = $developer_access;
		$feature['is_enabled_android'] = $developer_access;

		return $feature;
	}

	/**
	 * Post-Sharing feature availability.
	 *
	 * @param array $feature App features.
	 *
	 * @return mixed
	 */
	public function post_sharing_feature_availability( $feature ) {
		$settings = $this->get_settings();
		$enabled  = false;

		if ( ! empty( $settings['post_share_enable'] ) ) {
			$enabled = true;
		}

		$feature['is_enabled_ios']     = (bool) $enabled;
		$feature['is_enabled_android'] = (bool) $enabled;

		return $feature;
	}

	/**
	 * Related Post feature availability.
	 *
	 * @param array $feature App features.
	 *
	 * @return mixed
	 */
	public function related_post_feature_availability( $feature ) {
		$settings = $this->get_settings();
		$enabled  = false;

		if ( ! empty( $settings['related_post_enable'] ) ) {
			$enabled = true;
		}

		$feature['is_enabled_ios']     = (bool) $enabled;
		$feature['is_enabled_android'] = (bool) $enabled;

		return $feature;
	}

	/**
	 * Share App Link feature availability.
	 *
	 * @param array $feature App features.
	 *
	 * @return array
	 */
	public function bbapp_share_app_link_feature_availability( $feature ) {
		if ( AppSettings::instance()->get_setting_value( 'app_sharing.enabled' ) ) {
			$feature['is_enabled_ios']     = bbapp_is_ios_share_app_link_enabled();
			$feature['is_enabled_android'] = bbapp_is_android_share_app_link_enabled();
		}

		return $feature;
	}

	/**
	 * Tutor LMS feature availability.
	 *
	 * @param array $feature App features.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	public function tutor_lms_app_feature_availability( $feature ) {
		$enabled                       = bbapp_is_tutor_lms_plugins_active();
		$feature['is_enabled_ios']     = $enabled;
		$feature['is_enabled_android'] = $enabled;

		return $feature;
	}

	/**
	 * Tutor LMS Pro feature availability.
	 *
	 * @param array $feature App features.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	public function tutor_lms_pro_app_feature_availability( $feature ) {
		$enabled                       = bbapp_is_tutor_lms_plugins_active( true );
		$feature['is_enabled_ios']     = $enabled;
		$feature['is_enabled_android'] = $enabled;

		return $feature;
	}

	/**
	 * Returns the Offline Courses Settings.
	 *
	 * @return array
	 */
	public function get_settings() {
		return ManageApp::instance()->get_app_settings();
	}
}
