<?php
/**
 * REST API: WP_REST_Users_Controller class
 *
 * @since      4.7.0
 * @subpackage REST_API
 * @package    WordPress
 */

namespace BuddyBossApp\Api\WpCore\V2;

use WP_REST_Request;
use WP_REST_Server;
use WP_REST_Users_Controller;

/**
 * Core class used to manage users via the REST API.
 *
 * @since 4.7.0
 *
 * @see   WP_REST_Controller
 */
class BuddyBossApp_REST_Users_Controller extends WP_REST_Users_Controller {

	/**
	 * Allow batch.
	 *
	 * @var true[] $allow_batch
	 */
	protected $allow_batch = array( 'v1' => true );

	/**
	 * Registers the routes for users.
	 *
	 * @since 4.7.0
	 *
	 * @see   register_rest_route()
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				'args'        => array(
					'id' => array(
						'description' => __( 'Unique identifier for the user.' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'allow_batch' => $this->allow_batch,
			),
			true
		);

		// Register the platform user field.
		register_rest_field(
			'user',
			'platform_user',
			array(
				'get_callback' => array( $this, 'get_platform_user' ),
				'schema'       => null,
			)
		);
	}

	/**
	 * Get platform user.
	 *
	 * @param array           $user       User object.
	 * @param string          $field_name Field name.
	 * @param WP_REST_Request $request    Request object.
	 *
	 * @since 2.1.20
	 * @return array
	 */
	public function get_platform_user( $user, $field_name, $request ) {
		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			/**
			 * Get Platform user data.
			 */
			$request_curl = new \WP_REST_Request( 'GET', '/buddyboss/v1/members/' . $user['id'] );
			$query_params = $request_curl->get_params();

			if ( ! empty( $request['_embed'] ) ) {
				$query_params['_embed'] = $request['_embed'];
			}

			$request_curl->set_query_params( $query_params );
			$server   = rest_get_server();
			$response = $server->dispatch( $request_curl );

			return $response->get_data();
		}

		return array();
	}
}
