<?php
/**
 * Holds related post rest functionality.
 *
 * @package BuddyBossApp\Api\WpCore\V2
 */

namespace BuddyBossApp\Api\WpCore\V2;

use WP_Error;
use WP_Query;
use WP_REST_Posts_Controller;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Class BuddyBossApp_REST_Related_Post_Controller
 */
class BuddyBossApp_REST_Related_Post_Controller extends WP_REST_Posts_Controller {

	/**
	 * Registers the routes for attachments.
	 *
	 * @since 2.0.30
	 *
	 * @see   register_rest_route()
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<post_id>[\d]+)/related-posts',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_related_posts_collection_params(),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);
	}

	/**
	 * Retrieve Courses.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since 2.0.30
	 */
	public function get_items( $request ) {
		$registered = $this->get_related_posts_collection_params();
		$post_id    = $request->get_param( 'post_id' );

		/**
		 * Filter the request.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.0.30
		 */
		$request = apply_filters( 'bbapp_related_posts_request', $request );

		$parameter_mappings = array(
			'posts_per_page' => $registered['posts_per_page']['default'],
			'paged'          => $registered['paged']['default'],
		);

		$args = wp_parse_args( $request, $parameter_mappings );
		$cats = wp_get_post_categories( $post_id, array( 'exclude' => array( 1 ) ) ); // Exclude uncategorized category.

		if ( ! empty( $cats ) ) {
			$args['category__in'] = $cats;
		}

		/**
		 * Filter the query arguments for the request.
		 *
		 * @param array           $args    Key value array of query var to query value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.0.30
		 */
		$args = apply_filters( 'bbapp_related_posts_args', $args, $request );

		// This fields shouldn't be overwritten.
		$args['post_type']         = $this->post_type;
		$args['post_status']       = 'publish';
		$args['post__not_in']      = array( $post_id );
		$count_selected_categories = 0;
		$combined_posts            = array();
		$remaining_posts           = 5;
		$combine_post_ids          = array( $post_id );

		if ( ! empty( $cats ) ) {
			// Query posts from selected categories.
			$query_selected_categories = new WP_Query( $args );

			// Get the count of posts retrieved from selected categories.
			$count_selected_categories = $query_selected_categories->post_count;
			$combined_posts            = $query_selected_categories->posts;
			// Calculate the remaining number of posts needed.
			$remaining_posts  = $remaining_posts - $count_selected_categories;
			$combine_post_ids = array_unique( array_merge( $combine_post_ids, wp_list_pluck( $combined_posts, 'ID' ) ) );
		}

		// Check if more posts are needed to reach the total count.
		if ( $count_selected_categories < 5 ) {
			$sticky_post_ids = get_option( 'sticky_posts' );

			if ( ! empty( $sticky_post_ids ) ) {
				$remained_sticky_posts = array();

				foreach ( $sticky_post_ids as $sticky_post_id ) {
					if ( ! in_array( $sticky_post_id, $combine_post_ids, true ) ) {
						$remained_sticky_posts[] = $sticky_post_id;
					}
				}

				if ( ! empty( $remained_sticky_posts ) ) {
					// Query sticky posts.
					$args_sticky_posts = array(
						'post_type'           => $this->post_type,
						'posts_per_page'      => $remaining_posts,
						'post__in'            => $remained_sticky_posts,
						'ignore_sticky_posts' => 1,
						'orderby'             => 'date',
						'order'               => 'DESC',
					);

					$query_sticky_posts = new WP_Query( $args_sticky_posts );

					// Combine selected category posts and sticky posts.
					$combined_posts   = array_merge( $combined_posts, $query_sticky_posts->posts );
					$combine_post_ids = array_unique( array_unique( array_merge( array( $post_id ), wp_list_pluck( $combined_posts, 'ID' ) ) ) ); // Exclude current sticky post and already selected posts from previous query.

					// Check if more posts are needed.
					$remaining_posts -= count( $query_sticky_posts->posts );
				}
			}
		}

		if ( $remaining_posts > 0 ) {
			// Query additional latest posts.
			$args_latest_posts = array(
				'post_type'      => $this->post_type,
				'posts_per_page' => $remaining_posts,
				'post__not_in'   => $combine_post_ids,
				'orderby'        => 'date',
				'order'          => 'DESC',
			);

			$query_latest_posts = new WP_Query( $args_latest_posts );

			// Combine all posts.
			$combined_posts = array_merge( $combined_posts, $query_latest_posts->posts );
		}

		/**
		 * Fires list of Courses is fetched via Query.
		 *
		 * @param array            $courses Fetched courses.
		 * @param WP_REST_Response $args    Query arguments.
		 * @param WP_REST_Request  $request The request sent to the API.
		 *
		 * @since 2.0.30
		 */
		$related_posts = apply_filters( 'bbapp_related_get_posts', $combined_posts, $args, $request );

		$retval = array();

		if ( ! empty( $related_posts ) ) {
			foreach ( $related_posts as $post_single ) {
				$retval[] = $this->prepare_response_for_collection(
					$this->prepare_item_for_response( $post_single, $request )
				);
			}
		}

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after a list of Courses response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.0.30
		 */
		do_action( 'bbapp_related_post_items_response', $response, $request );

		return $response;
	}

	/**
	 * Checks if a given request has access to read comments.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since 2.0.30
	 * @return true|WP_Error True if the request has read access, error object otherwise.
	 */
	public function get_items_permissions_check( $request ) {
		/**
		 * Filters the related post items permissions check.
		 *
		 * @param bool            $permission True if the request has read access, false otherwise.
		 * @param WP_REST_Request $request    Full details about the request.
		 *
		 * @since 2.0.30
		 */
		return apply_filters( 'bbapp_related_post_items_permissions_check', true, $request );
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @return array
	 *
	 * @since 2.0.30
	 */
	public function get_related_posts_collection_params() {
		$params['post_id']             = array(
			'description' => __( 'Current post id.', 'buddyboss-app' ),
			'type'        => 'integer',
			'required'    => true,
		);
		$params['posts_per_page'] = array(
			'description'       => __( 'Number of items per page.', 'buddyboss-app' ),
			'type'              => 'integer',
			'default'           => 5,
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['paged']          = array(
			'description'       => __( 'Number of the page of the pagination.', 'buddyboss-app' ),
			'type'              => 'integer',
			'default'           => 1,
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}
}
