<?php
/**
 * Holds rest post extending functionality.
 *
 * @package BuddyBossApp\Api\WpCore\V2
 */

namespace BuddyBossApp\Api\WpCore\V2;

use BuddyBossApp\AccessControls\Core\Settings\Posts;
use WP_Post;

/**
 * Rest post controller.
 */
class BuddyBossApp_REST_Post_Controller {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return BuddyBossApp_REST_Post_Controller class object.
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();

			self::$instance->extend_post_object();
		}

		return self::$instance;
	}

	/**
	 * Extend post object.
	 */
	public function extend_post_object() {
		$this->hooks();

		register_rest_field(
			'post',
			'comments_count',
			array(
				'get_callback' => array( $this, 'bbapp_post_comment_counts' ),
				'schema'       => array(
					'description' => 'Post comments count.',
					'type'        => 'integer',
					'context'     => array( 'embed', 'view', 'edit' ),
				),
			)
		);
	}

	/**
	 * Define all Hooks.
	 */
	public function hooks() {
		add_filter( 'rest_post_collection_params', array( $this, 'bbapp_custom_parameters_post_collection' ), 10, 2 );
		add_filter( 'rest_post_query', array( $this, 'bbapp_modify_rest_post_query' ), 10, 2 );
		add_filter( 'option_sticky_posts', array( $this, 'option_sticky_posts' ), 10, 2 );
		add_action( 'rest_prepare_post', array( $this, 'rest_close_comments_for_old_post' ),10, 3 );
	}

	/**
	 * Function to add extra parameters to post rest endpoint.
	 *
	 * @since 1.4.4
	 *
	 * @param array  $query_params query parameters.
	 * @param string $post_type    Post type.
	 *
	 * @return mixed
	 */
	public function bbapp_custom_parameters_post_collection( $query_params, $post_type ) {
		$query_params['date'] = array(
			'description'       => __( 'Limit response to posts for specific date.', 'buddyboss-app' ),
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
		);

		return $query_params;
	}

	/**
	 * Function to modify the REST post query.
	 *
	 * @since 1.4.4
	 *
	 * @param array $args    post arguments.
	 * @param array $request REST request.
	 *
	 * @return array
	 */
	public function bbapp_modify_rest_post_query( $args, $request ) {
		global $bbapp_query_args;
		if ( ! empty( $request['date'] ) && 'all-dates' !== $request['date'] ) {
			$request_date = explode( '-', $request['date'] );

			if ( ! empty( $request_date[0] ) ) {
				$args['date_query'][0]['month'] = $request_date[0];
			}

			if ( ! empty( $request_date[1] ) ) {
				$args['date_query'][0]['year'] = $request_date[1];
			}
		}

		// Sticky post show on top.
		if ( null === $request->get_param( 'sticky' ) ) {
			$args['ignore_sticky_posts'] = false;
			$bbapp_query_args            = $args;
		}

		return $args;
	}

	/**
	 * Added new object into the post rest api.
	 *
	 * @param WP_Post $post      Post object.
	 * @param string  $attribute The REST Field key used into the REST response.
	 *
	 * @return int|string
	 */
	public function bbapp_post_comment_counts( $post, $attribute ) {
		$post_id = $post['id'];

		return get_comments_number( $post_id );
	}

	/**
	 * Exclude ignore sticky posts which is not valid.
	 *
	 * @param array  $db_sticky_posts Database sticky posts.
	 * @param string $option          Option name.
	 *
	 * @since 1.7.3
	 *
	 * @return array|mixed
	 */
	public function option_sticky_posts( $db_sticky_posts, $option ) {
		global $bbapp_query_args;

		if ( ! bbapp_is_rest() ) {
			return $db_sticky_posts;
		}

		if ( empty( $bbapp_query_args ) ) {
			return $db_sticky_posts;
		}

		if ( empty( $db_sticky_posts ) ) {
			return $db_sticky_posts;
		}

		if ( ! empty( $bbapp_query_args['post_type'] ) && 'post' !== $bbapp_query_args['post_type'] ) {
			return $db_sticky_posts;
		}

		$allow_sticky_posts     = array();
		$is_custom_sticky_posts = false;
		// Exclude sticky posts if access control filter enable.
		if ( isset( $bbapp_query_args['access_control_filter'] ) && true === (bool) $bbapp_query_args['access_control_filter'] ) {
			$is_custom_sticky_posts = true;
			$allow_sticky_posts     = bb_access_controls_get_access_items_by_item_ids( $db_sticky_posts, 'post' );
		}

		// Exclude sticky posts if bb_bookmark param pass.
		if ( isset( $bbapp_query_args['bb_bookmark'] ) ) {
			$is_custom_sticky_posts = true;
			if ( is_user_logged_in() ) {
				$item_ids         = array_intersect( $db_sticky_posts, $allow_sticky_posts );
				$bookmarked_items = bb_bookmarks_get_bookmark_by_item( 'post', $item_ids );
				if ( true === (bool) $bbapp_query_args['bb_bookmark'] ) {
					$allow_sticky_posts = wp_list_pluck( $bookmarked_items, 'item_id' );
				} else {
					$bookmarked_item_ids = wp_list_pluck( $bookmarked_items, 'item_id' );
					$allow_sticky_posts  = array_diff( $item_ids, $bookmarked_item_ids );
				}
			}
		}

		if ( true === $is_custom_sticky_posts ) {
			return $allow_sticky_posts;
		}

		return $db_sticky_posts;
	}

	/**
	 * Filters the post data for a REST API response.
	 *
	 *
	 * @param \WP_REST_Response $response The response object.
	 * @param \WP_Post          $post     Post object.
	 * @param \WP_REST_Request  $request  Request object.
	 *
	 * @since 1.7.90
	 *
	 * @return mixed
	 */
	public function rest_close_comments_for_old_post( $response, $post, $request ) {

		if ( ! get_option( 'close_comments_for_old_posts' ) ) {
			return $response;
		}

		$days_old = (int) get_option( 'close_comments_days_old' );
		if ( ! $days_old ) {
			return $response;
		}

		if ( time() - strtotime( $response->data['date_gmt'] ) > ( $days_old * DAY_IN_SECONDS ) ) {
			$response->data['comment_status'] = 'closed';
			$response->data['ping_status']    = 'closed';
		}

		return $response;
	}

}
