<?php
/**
 * Holds post archive rest functionality.
 *
 * @package BuddyBossApp\Api\WpCore\V2
 */

namespace BuddyBossApp\Api\WpCore\V2;

use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

/**
 * POst archive rest class.
 */
class BuddyBossApp_REST_Post_Archive_Controller extends \WP_REST_Controller {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->namespace = 'wp/v2';
		$this->rest_base = 'posts/archive';
	}

	/**
	 * Registers the routes for the objects of the controller.
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);
	}

	/**
	 * Retrieves a post archives items.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error Response object on success, or error object on failure.
	 */
	public function get_items( $request ) {
		$prepared_args = array();

		// Retrieve the list of registered collection query parameters.
		$registered = $this->get_collection_params();

		/*
		 * For each known parameter which is both registered and present in the request,
		 * set the parameter's value on the query $prepared_args.
		 */
		foreach ( $registered as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$prepared_args[ $api_param ] = $request[ $api_param ];
			}
		}

		$retval   = $this->bbapp_wp_get_archives( $prepared_args );
		$response = rest_ensure_response( $retval );

		return $response;
	}

	/**
	 * Checks if a given request has access to read comments.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return true|WP_Error True if the request has read access, error object otherwise.
	 */
	public function get_items_permissions_check( $request ) {
		return true;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array|mixed|void
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'post-archive',
			'type'       => 'object',
			'properties' => array(
				'day'      => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'A unique numeric ID for the activity.', 'buddyboss-app' ),
					'readonly'    => true,
					'type'        => 'integer',
				),
				'month'    => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The ID of some other object primarily associated with this one.', 'buddyboss-app' ),
					'type'        => 'integer',
				),
				'year'     => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The ID of some other object also associated with this one.', 'buddyboss-app' ),
					'type'        => 'integer',
				),
				'selected' => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The ID for the author of the activity.', 'buddyboss-app' ),
					'type'        => 'integer',
				),
				'count'    => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'User\'s display name for the activity.', 'buddyboss-app' ),
					'type'        => 'string',
				),
			),
		);

		/**
		 * Filters the posts archive schema.
		 *
		 * @param string $schema The endpoint schema.
		 */
		return apply_filters( 'bbapp_rest_posts_archive', $this->add_additional_fields_schema( $schema ) );
	}

	/**
	 * Retrieves the query params for collections.
	 *
	 * @return array Comments collection parameters.
	 */
	public function get_collection_params() {
		$query_params = array();

		$query_params['type'] = array(
			'description' => __( 'Type of archive to retrieve.', 'buddyboss-app' ),
			'type'        => 'string',
			'required'    => true,
			'default'     => 'monthly',
			'enum'        => array( 'daily', 'monthly', 'yearly' ),
		);

		$query_params['limit'] = array(
			'description' => __( 'Number of links to limit the query to. Default empty (no limit).', 'buddyboss-app' ),
			'type'        => 'integer',
		);

		$query_params['show_post_count'] = array(
			'description' => __( 'Whether to display the post count alongside the link.', 'buddyboss-app' ),
			'type'        => 'boolean',
			'default'     => true,
		);

		$query_params['order'] = array(
			'default'     => 'DESC',
			'description' => __( "Whether to use ascending or descending order. Accepts 'ASC', or 'DESC'.", 'buddyboss-app' ),
			'enum'        => array( 'ASC', 'DESC' ),
			'type'        => 'string',
		);

		$query_params['post_type'] = array(
			'description' => __( 'Post type.', 'buddyboss-app' ),
			'type'        => 'string',
			'default'     => 'post',
		);

		$query_params['year'] = array(
			'description' => __( 'Year. Default current year.', 'buddyboss-app' ),
			'type'        => 'string',
		);

		$query_params['monthnum'] = array(
			'description' => __( 'Month number. Default current month number.', 'buddyboss-app' ),
			'type'        => 'string',
		);

		$query_params['day'] = array(
			'description' => __( 'Day. Default current day.', 'buddyboss-app' ),
			'type'        => 'string',
		);

		/**
		 * Filter collection parameters for the archive controller.
		 *
		 * @param array $query_params JSON Schema-formatted collection parameters.
		 */
		return apply_filters( 'rest_post_archive_collection_params', $query_params );
	}

	/**
	 * Get posts Archive.
	 * - From wp_get_archives();
	 *
	 * @param array $args Arguments for the archive.
	 *
	 * @return array|void
	 */
	public function bbapp_wp_get_archives( $args = '' ) {
		global $wpdb, $wp_locale;

		$defaults = array(
			'type'            => 'monthly',
			'limit'           => '',
			'before'          => '',
			'after'           => '',
			'show_post_count' => true,
			'echo'            => 0,
			'order'           => 'DESC',
			'post_type'       => 'post',
			'year'            => gmdate( 'Y' ),
			'monthnum'        => gmdate( 'm' ),
			'day'             => gmdate( 'd' ),
			'w'               => '',
		);

		$parsed_args      = wp_parse_args( $args, $defaults );
		$post_type_object = get_post_type_object( $parsed_args['post_type'] );

		if ( ! is_post_type_viewable( $post_type_object ) ) {
			return;
		}

		$parsed_args['post_type'] = $post_type_object->name;

		if ( '' === $parsed_args['type'] ) {
			$parsed_args['type'] = 'monthly';
		}

		if ( ! empty( $parsed_args['limit'] ) ) {
			$parsed_args['limit'] = absint( $parsed_args['limit'] );
			$parsed_args['limit'] = ' LIMIT ' . $parsed_args['limit'];
		}

		$order = strtoupper( $parsed_args['order'] );

		if ( 'ASC' !== $order ) {
			$order = 'DESC';
		}

		$sql_where = $wpdb->prepare( "WHERE post_type = %s AND post_status = 'publish'", $parsed_args['post_type'] );

		/**
		 * Filters the SQL WHERE clause for retrieving archives.
		 *
		 * @since 2.2.0
		 *
		 * @param string $sql_where   Portion of SQL query containing the WHERE clause.
		 * @param array  $parsed_args An array of default arguments.
		 */
		$where = apply_filters( 'getarchives_where', $sql_where, $parsed_args );

		/**
		 * Filters the SQL JOIN clause for retrieving archives.
		 *
		 * @since 2.2.0
		 *
		 * @param string $sql_join    Portion of SQL query containing JOIN clause.
		 * @param array  $parsed_args An array of default arguments.
		 */
		$join = apply_filters( 'getarchives_join', '', $parsed_args );

		$last_changed = wp_cache_get_last_changed( 'posts' );
		$limit        = $parsed_args['limit'];
		$output       = array();

		if ( 'monthly' === $parsed_args['type'] ) {
			$query   = "SELECT YEAR(post_date) AS `year`, MONTH(post_date) AS `month`, count(ID) as posts FROM $wpdb->posts $join $where GROUP BY YEAR(post_date), MONTH(post_date) ORDER BY post_date $order $limit";
			$key     = md5( $query );
			$key     = "wp_get_archives:$key:$last_changed";
			$results = wp_cache_get( $key, 'posts' );

			if ( ! $results ) {
				$results = $wpdb->get_results( $query ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

				wp_cache_set( $key, $results, 'posts' );
			}

			if ( $results ) {
				foreach ( (array) $results as $result ) {
					$data = array(
						'day'      => '',
						'month'    => $result->month,
						'year'     => $result->year,
						'selected' => false,
					);

					if ( $parsed_args['show_post_count'] ) {
						$data['count'] = $result->posts;
					}

					$data['selected'] = (string) $parsed_args['year'] === $result->year && (string) $parsed_args['monthnum'] === $result->month;
					$output[]         = $data;
				}
			}
		} elseif ( 'yearly' === $parsed_args['type'] ) {
			$query   = "SELECT YEAR(post_date) AS `year`, count(ID) as posts FROM $wpdb->posts $join $where GROUP BY YEAR(post_date) ORDER BY post_date $order $limit";
			$key     = md5( $query );
			$key     = "wp_get_archives:$key:$last_changed";
			$results = wp_cache_get( $key, 'posts' );

			if ( ! $results ) {
				$results = $wpdb->get_results( $query ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

				wp_cache_set( $key, $results, 'posts' );
			}

			if ( $results ) {
				foreach ( (array) $results as $result ) {
					$data = array(
						'day'      => '',
						'month'    => '',
						'year'     => $result->year,
						'selected' => false,
					);

					if ( $parsed_args['show_post_count'] ) {
						$data['count'] = $result->posts;
					}

					$data['selected'] = (string) $parsed_args['year'] === $result->year;
					$output[]         = $data;
				}
			}
		} elseif ( 'daily' === $parsed_args['type'] ) {
			$query   = "SELECT YEAR(post_date) AS `year`, MONTH(post_date) AS `month`, DAYOFMONTH(post_date) AS `dayofmonth`, count(ID) as posts FROM $wpdb->posts $join $where GROUP BY YEAR(post_date), MONTH(post_date), DAYOFMONTH(post_date) ORDER BY post_date $order $limit";
			$key     = md5( $query );
			$key     = "wp_get_archives:$key:$last_changed";
			$results = wp_cache_get( $key, 'posts' );

			if ( ! $results ) {
				$results = $wpdb->get_results( $query ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared

				wp_cache_set( $key, $results, 'posts' );
			}

			if ( $results ) {
				foreach ( (array) $results as $result ) {
					$data = array(
						'day'      => $result->dayofmonth,
						'month'    => $result->month,
						'year'     => $result->year,
						'selected' => false,
					);

					if ( $parsed_args['show_post_count'] ) {
						$data['count'] = $result->posts;
					}

					$data['selected'] = (string) $parsed_args['year'] === $result->year && (string) $parsed_args['monthnum'] === $result->month && (string) $parsed_args['day'] === $result->dayofmonth;
					$output[]         = $data;
				}
			}
		}

		return $output;
	}
}
