<?php
/**
 * Holds Batch rest functionality.
 *
 * @package BuddyBossApp\Api\WpCore\V2
 */

namespace BuddyBossApp\Api\WpCore\V2;

use BuddyBoss\Performance\Cache;
use BuddyBoss\Performance\Helper;
use BuddyBoss\Performance\Integration\Integration_Abstract;
use BuddyBoss\Performance\Performance;
use WP_Error;
use WP_Http;
use WP_HTTP_Response;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Post archive rest class.
 */
class BuddyBossApp_REST_Batch_Controller extends WP_REST_Server {
	/**
	 * Constructor.
	 */
	public function __construct() {
		parent::__construct();
		add_filter( 'rest_endpoints', array( $this, 'register_bulk_endpoint' ) );
	}

	/**
	 * Register the batch endpoint.
	 *
	 * @param array $endpoints Existing endpoints.
	 *
	 * @since 2.0.80
	 *
	 * @return mixed
	 */
	public function register_bulk_endpoint( $endpoints ) {
		$endpoints['/buddyboss-app/v1/batch'] = array(
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'handle_batch_request' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'validation' => array(
						'type'    => 'string',
						'enum'    => array( 'require-all-validate', 'normal' ),
						'default' => 'normal',
					),
					'requests'   => array(
						'required' => true,
						'type'     => 'array',
						'maxItems' => $this->get_max_batch_size(),
						'items'    => array(
							'type'       => 'object',
							'properties' => array(
								'method' => array(
									'type'    => 'string',
									'enum'    => array( 'GET' ),
									'default' => 'GET',
								),
								'path'   => array(
									'type'     => 'string',
									'required' => true,
								),
							),
						),
					),
				),
			),
		);

		return $endpoints;
	}

	/**
	 * Handle a batch request.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since 2.0.80
	 *
	 * @return WP_Error|WP_REST_Response
	 */
	public function handle_batch_request( $request ) {
		$batch_data   = $request->get_json_params();
		$batch_header = $request->get_headers();

		if ( ! is_array( $batch_data ) ) {
			return new WP_Error( 'invalid_request', __( 'Invalid batch request data', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$requests = array();

		foreach ( $batch_data['requests'] as $args ) {
			$parsed_url = wp_parse_url( $args['path'] );

			if ( false === $parsed_url ) {
				$requests[] = new WP_Error( 'parse_path_failed', __( 'Could not parse the path.', 'buddyboss-app' ), array( 'status' => 400 ) );

				continue;
			}

			$single_request = new WP_REST_Request( isset( $args['method'] ) ? $args['method'] : 'GET', $parsed_url['path'] );

			if ( ! empty( $parsed_url['query'] ) ) {
				$query_args = null; // Satisfy linter.

				wp_parse_str( $parsed_url['query'], $query_args );
				$single_request->set_query_params( $query_args );
			}

			if ( ! empty( $args['_embed'] ) ) {
				// Parse and add the _embed query parameter to the request.
				$embed_args = array_map( 'trim', explode( ',', $args['_embed'] ) );

				$single_request->set_param( '_embed', $embed_args );
			}

			if ( ! empty( $args['body'] ) ) {
				$single_request->set_body_params( $args['body'] );
			}

			$args['headers'] = ! empty( $args['headers'] ) ? array_merge( $batch_header, $args['headers'] ) : $batch_header;

			if ( ! empty( $args['headers'] ) ) {
				$single_request->set_headers( $args['headers'] );
			}

			$requests[] = $single_request;
		}

		$matches    = array();
		$validation = array();
		$has_error  = false;

		foreach ( $requests as $single_request ) {
			$match     = $this->match_request_to_handler( $single_request );
			$matches[] = $match;
			$error     = null;

			if ( is_wp_error( $match ) ) {
				$error = $match;
			}

			if ( ! $error ) {
				list( $route, $handler ) = $match;

				if ( isset( $handler['allow_batch'] ) ) {
					$allow_batch = $handler['allow_batch'];
				} else {
					$route_options = $this->get_route_options( $route );
					$allow_batch   = isset( $route_options['allow_batch'] ) ? $route_options['allow_batch'] : false;
				}

				if ( ! is_array( $allow_batch ) || empty( $allow_batch['v1'] ) ) {
					$error = new WP_Error(
						'rest_batch_not_allowed',
						__( 'The requested route does not support batch requests.', 'buddyboss-app' ),
						array( 'status' => 400 )
					);
				}
			}

			if ( ! $error ) {
				$check_required = $single_request->has_valid_params();

				if ( is_wp_error( $check_required ) ) {
					$error = $check_required;
				}
			}

			if ( ! $error ) {
				$check_sanitized = $single_request->sanitize_params();

				if ( is_wp_error( $check_sanitized ) ) {
					$error = $check_sanitized;
				}
			}

			if ( $error ) {
				$has_error    = true;
				$validation[] = $error;
			} else {
				$validation[] = true;
			}
		}

		$responses = array();

		if ( $has_error && 'require-all-validate' === $batch_data['validation'] ) {
			foreach ( $validation as $valid ) {
				if ( is_wp_error( $valid ) ) {
					$responses[] = $this->envelope_response( $this->error_to_response( $valid ), false )->get_data();
				} else {
					$responses[] = null;
				}
			}

			return new WP_REST_Response(
				array(
					'failed'    => 'validation',
					'responses' => $responses,
				),
				WP_Http::MULTI_STATUS
			);
		}

		foreach ( $requests as $i => $single_request ) {
			$clean_request = clone $single_request;

			$clean_request->set_url_params( array() );
			$clean_request->set_attributes( array() );
			$clean_request->set_default_params( array() );

			/**
			 * Filters the pre-calculated result of a REST API dispatch request.
			 * This filter is documented in wp-includes/rest-api/class-wp-rest-server.php
			 *
			 * Allow hijacking the request before dispatching by returning a non-empty. The returned value
			 * will be used to serve the request instead.
			 *
			 * @param mixed           $result  Response to replace the requested version with. Can be anythinga normal endpoint can return, or null to not hijack the request.
			 * @param WP_REST_Server  $server  Server instance.
			 * @param WP_REST_Request $request Request used to generate the response.
			 *
			 * @since 2.0.80
			 */
			$result = apply_filters( 'rest_pre_dispatch', null, $this, $clean_request );

			if ( empty( $result ) ) {
				$match = $matches[ $i ];
				$error = null;

				if ( is_wp_error( $validation[ $i ] ) ) {
					$error = $validation[ $i ];
				}

				if ( is_wp_error( $match ) ) {
					$result = $this->error_to_response( $match );
				} else {
					list( $route, $handler ) = $match;

					if ( ! $error && ! is_callable( $handler['callback'] ) ) {
						$error = new WP_Error(
							'rest_invalid_handler',
							__( 'The handler for the route is invalid', 'buddyboss-app' ),
							array( 'status' => 500 )
						);
					}

					$cache_route = $batch_data['requests'][ $i ]['path'];

					// Initialize variables.
					$args             = array();
					$integration_name = '';

					if ( class_exists( 'BuddyBoss\Performance\Integration\Integration_Abstract' ) && ! empty( Integration_Abstract::$cache_endpoints ) ) {
						// Iterate through cache endpoints to find a matching integration.
						foreach ( Integration_Abstract::$cache_endpoints as $cache_integration_name => $endpoints ) {
							$current_endpoint = $this->get_current_endpoint( trim( $cache_route, '/' ) );

							if ( isset( $endpoints[ $current_endpoint ] ) ) {
								$integration_name = $cache_integration_name;
								$args             = $endpoints[ $current_endpoint ];
								break; // Found a matching endpoint, no need to continue the loop.
							}
						}
					}

					// Check if API performance component is enabled for the integration.
					if ( $this->is_api_performance_component_enabled( $integration_name ) ) {
						$cache_result = $this->get_cache_single_request( $cache_route, $integration_name, $args, $single_request );

						if ( ! empty( $cache_result ) ) {
							// Use cached result if available.
							$result = new WP_HTTP_Response( $cache_result['data'], WP_Http::OK, $this->prepare_cached_headers( $cache_result['header'] ) );
						} else {
							// If no cached result, process the request and cache the result.
							$result = $this->respond_to_request( $single_request, $route, $handler, $error );

							$this->do_cache_single_request( $result, $cache_route, $integration_name, $args, $this );
						}
					} else {
						// If API performance component is not enabled, process the request without caching.
						$result = $this->respond_to_request( $single_request, $route, $handler, $error );
					}
				}
			}

			/**
			 * Filters the REST API response.
			 *
			 * Allows modification of the response before returning.
			 * This filter is documented in wp-includes/rest-api/class-wp-rest-server.php
			 *
			 * @param WP_HTTP_Response $result  Result to send to the client. Usually a `WP_REST_Response`.
			 * @param WP_REST_Server   $server  Server instance.
			 * @param WP_REST_Request  $request Request used to generate the response.
			 *
			 * @since 2.0.80
			 */
			$result = apply_filters( 'rest_post_dispatch', rest_ensure_response( $result ), $this, $single_request );

			$embed = $single_request->get_param( '_embed' );

			// Check is null.
			if ( null === $embed ) {
				$embed = false;
			}

			$responses[] = $this->envelope_response( $result, $embed )->get_data();
		}

		return new WP_REST_Response( array( 'responses' => $responses ), WP_Http::MULTI_STATUS );
	}

	/**
	 * Cache the endpoint response.
	 *
	 * @param WP_REST_Response $results          Result to send to the client. Usually a `WP_REST_Response`.
	 * @param string           $route            Route.
	 * @param string           $integration_name Integration name.
	 * @param array            $args             Arguments.
	 * @param WP_REST_Server   $server           Server instance.
	 *
	 * @since 2.0.80
	 *
	 * @return void
	 */
	public function do_cache_single_request( $results, $route, $integration_name, $args, $server ) {
		$cache_name = '/wp-json' . $route;
		$user_id    = ( ! empty( $args ) && ! empty( $args['user_cache'] ) ) ? $this->get_loggedin_user_id() : 0;
		$unique_key = isset( $args['unique_id'] ) ? $args['unique_id'] : 'id';

		if ( is_array( $unique_key ) ) {
			$item_ids = array_map(
				function ( $el ) use ( $unique_key ) {
					return $this->prepare_key( $el, $unique_key );
				},
				$results->data
			);
		} else {
			$item_ids = array_column( $results->data, $unique_key );
		}

		if ( ! empty( $item_ids ) && 200 === $results->status ) {
			$cache_val = array(
				'data'   => $item_ids,
				'header' => $this->prepare_header( $results ),
			);

			Cache::instance()->set( $cache_name, $cache_val, $args['expire'], $integration_name, $user_id );

			// Prepare Embed links inside the request.
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.NonceVerification.Recommended
			$embed       = isset( $_GET['_embed'] ) ? rest_parse_embed_param( $_GET['_embed'] ) : false;
			$result_data = $server->response_to_data( $results, $embed );

			if ( ! empty( $result_data ) ) {
				foreach ( $result_data as $item ) {
					/**
					 * Filter to enable/disable cache for single item.
					 *
					 * @param bool   $is_cache_enabled Is cache enabled.
					 * @param array  $item             Item.
					 * @param array  $args             Arguments.
					 * @param string $integration_name Integration name.
					 *
					 * @since 2.0.80
					 * @return bool true/false.
					 */
					$is_cache_enabled = apply_filters( 'bbapp_performance_deep_cache_filter_item', true, $item, $args, $integration_name );

					if ( empty( $is_cache_enabled ) ) {
						continue;
					}

					if ( is_array( $unique_key ) ) {
						$item_id = $this->prepare_key( $item, $unique_key );
					} else {
						$item_id = $item[ $unique_key ];
					}

					if ( ! empty( $item ) && in_array( $item_id, $item_ids, true ) ) {
						Cache::instance()->set( $cache_name, $item, $args['expire'], $integration_name . '_' . $item_id, $user_id );
					}
				}
			}
		}
	}

	/**
	 * Added Pre Header.
	 *
	 * @param WP_HTTP_Response $results Result to send to the client. Usually a `WP_REST_Response`.
	 *
	 * @since 2.0.80
	 * @return array
	 */
	private function prepare_header( $results ) {
		$headers          = array();
		$disallow_headers = array(
			'bbapp-logged-in',
			'bbapp-unread-notifications',
			'bbp-unread-messages',
			'bbp-unread-notifications',
			'Expires',
			'Cache-Control',
		);

		/**
		 * Filter to disallow headers from cache.
		 * To add filter for this you need to execute code form mu level.
		 *
		 * @param array $disallow_headers Disallow headers.
		 *
		 * @since 2.0.80
		 * @return array
		 */
		$disallow_headers = apply_filters( 'rest_post_disprepare_header_cache', $disallow_headers );
		$header_list      = headers_list();

		if ( ! empty( $header_list ) ) {
			foreach ( $header_list as $header ) {
				$header = explode( ': ', $header, 2 );

				if ( ! in_array( $header[0], $disallow_headers, true ) && is_array( $header ) && count( $header ) === 2 ) {
					$headers[ $header[0] ] = $header[1];
				}
			}
		}

		$results_header = $results->get_headers();

		if ( ! empty( $results_header ) ) {
			foreach ( $results_header as $header_key => $header_val ) {
				if ( ! in_array( $header_key, $disallow_headers, true ) ) {
					$headers[ $header_key ] = $header_val;
				}
			}
		}

		return $headers;
	}

	/**
	 * Prepare items id form unique_key
	 *
	 * @param array $el         Element.
	 * @param array $unique_key Unique key.
	 *
	 * @since 2.0.80
	 * @return string
	 */
	private function prepare_key( $el, $unique_key ) {
		$group_name = array_intersect_key( $el, array_flip( $unique_key ) );
		$group_name = array_merge( array_flip( $unique_key ), $group_name );

		return implode( '_', $group_name );
	}

	/**
	 * Get cache single request.
	 *
	 * @param string          $route            Route.
	 * @param string          $integration_name Integration name.
	 * @param array           $args             Arguments.
	 * @param WP_REST_Request $single_request   Single request.
	 *
	 * @since 2.0.80
	 * @return array|false
	 */
	public function get_cache_single_request( $route, $integration_name, $args, $single_request ) {
		$user_id    = ( ! empty( $args ) && ! empty( $args['user_cache'] ) ) ? $this->get_loggedin_user_id() : 0;
		$results    = false;
		$cache_name = '/wp-json' . $route;
		$cache_val  = Cache::instance()->get( $cache_name, $user_id, get_current_blog_id(), $integration_name );

		if ( ! empty( $cache_val ) && ! empty( $cache_val['data'] ) ) {
			/**
			 * Filter to enable/disable cache for single item.
			 *
			 * @param array  $cache_val        Cache value.
			 * @param array  $args             Arguments.
			 * @param string $integration_name Integration name.
			 *
			 * @since 2.0.80
			 * @return array
			 */
			$cache_val['data'] = apply_filters( 'bbapp_performance_deep_filter_cached_data', $cache_val['data'], $args, $integration_name );

			$results           = array();
			$results['header'] = ( isset( $cache_val['header'] ) ) ? $cache_val['header'] : array();

			foreach ( $cache_val['data'] as $item_id ) {
				$get_cache = Cache::instance()->get( $cache_name, $user_id, get_current_blog_id(), $integration_name . '_' . $item_id );

				if ( false !== $get_cache ) {
					$results['data'][] = $get_cache;
				} elseif ( isset( $route ) ) {
					/**
					 * Fetch Single item data if any single item cache is cleared
					 */
					$server = rest_get_server();
					$retval = $server->dispatch( $single_request );

					if ( 200 !== $retval->status ) {
						/**
						 * Fetch Parent endpoint if single items data not found with fresh request
						 */
						$results = false;
						break;
					}

					if ( ! empty( $retval->data ) ) {
						/**
						 * Filter to key for single item key.
						 *
						 * @param string $column_key       Column key.
						 * @param array  $args             Arguments.
						 * @param string $integration_name Integration name.
						 *
						 * @since 2.2.20
						 * @return string
						 */
						$column_key = apply_filters( 'bbapp_performance_deep_cache_filter_key', 'id', $args, $integration_name );

						$filter_array    = array_column( $retval->data, null, $column_key );
						$filtered_object = $filter_array[ $item_id ] ?? current( $filter_array );

						/**
						 * Filter to enable/disable cache for single item.
						 *
						 * @param bool   $is_cache_enabled Is cache enabled.
						 * @param array  $item             Item.
						 * @param array  $args             Arguments.
						 * @param string $integration_name Integration name.
						 *
						 * @since 2.0.80
						 * @return bool true/false.
						 */
						$is_cache_enabled = apply_filters( 'bbapp_performance_deep_cache_filter_item', true, $filtered_object, $args, $integration_name );

						if ( ! empty( $is_cache_enabled ) ) {
							/**
							 * Set retrieve items response in cache for future use
							 */
							Cache::instance()->set( $cache_name, $filtered_object, $args['expire'], $integration_name . '_' . $item_id, $user_id );
						}

						$results['data'][] = $filtered_object;
					}
				} else {
					/**
					 * Fetch Parent endpoint if endpoint is not found correctly
					 */
					$results = false;
					break;
				}
			}
		}

		return $results;
	}

	/**
	 * Get logged in user id.
	 *
	 * @since 2.0.80
	 * @return int
	 */
	public function get_loggedin_user_id() {
		if ( class_exists( 'BuddyBoss\Performance\Performance' ) ) {
			if ( Performance::instance()->is_current_user_available() ) {
				return get_current_user_id();
			} else {
				$guessed_user_id = Performance::instance()->get_guessed_user_id();

				if ( ! $guessed_user_id ) {
					return 0;
				}

				return $guessed_user_id;
			}
		}

		return 0;
	}

	/**
	 * Prepare cached headers.
	 *
	 * @param array $results_header Results header.
	 *
	 * @since 2.0.80
	 * @return array
	 */
	public function prepare_cached_headers( $results_header ) {
		$results_header['bb-api-cache'] = 'hit';
		$headers                        = array();
		$disallow_headers               = array(
			'bbapp-logged-in',
			'bbapp-unread-notifications',
			'bbp-unread-messages',
			'bbp-unread-notifications',
			'Expires',
			'Cache-Control',
			'X-Powered-By',
			'Content-Type',
			'X-Robots-Tag',
			'X-Content-Type-Options',
			'Access-Control-Expose-Headers',
			'Access-Control-Allow-Headers',
			'Set-Cookie',
		);

		/**
		 * Filter to disallow headers from cache.
		 * To add filter for this you need to execute code form mu level.
		 *
		 * @param array $disallow_headers Disallow headers.
		 *
		 * @since 2.0.80
		 * @return array
		 */
		$disallow_headers = apply_filters( 'rest_post_disprepare_header_cache', $disallow_headers );

		if ( ! empty( $results_header ) ) {
			foreach ( $results_header as $header_key => $header_val ) {
				if ( ! in_array( $header_key, $disallow_headers, true ) ) {
					$headers[ $header_key ] = $header_val;
				}
			}
		}

		return $headers;
	}

	/**
	 * Get current endpoint.
	 *
	 * @param string $current_path Route.
	 *
	 * @since 2.0.80
	 */
	public function get_current_endpoint( $current_path ) {
		// remove query vars.
		if ( strpos( $current_path, '?' ) !== false ) {
			$current_path = explode( '?', $current_path );
			$current_path = $current_path[0];
		}

		return trim( $current_path );
	}

	/**
	 * Check if API performance component is enabled for the integration.
	 *
	 * @param string $integration_name Integration name.
	 *
	 * @since 2.0.80
	 * @return bool
	 */
	public function is_api_performance_component_enabled( $integration_name = '' ) {
		if ( ! class_exists( 'BuddyBoss\Performance\Helper' ) ) {
			return false;
		}

		$is_component_active = Helper::instance()->get_app_settings( 'cache_component', 'buddyboss-app' );

		switch ( $integration_name ) {
			case 'blog-post':
				$is_settings_enabled = Helper::instance()->get_app_settings( 'cache_blog_post', 'buddyboss-app' );
				break;
			case 'sfwd-courses':
				$is_settings_enabled = Helper::instance()->get_app_settings( 'cache_ld', 'buddyboss-app' );
				break;
			case 'bp-groups':
				$is_settings_enabled = Helper::instance()->get_app_settings( 'cache_bb_social_groups', 'buddyboss-app' );
				break;
			case 'bp-members':
				$is_settings_enabled = Helper::instance()->get_app_settings( 'cache_bb_members', 'buddyboss-app' );
				break;
			case 'bbp-forums':
				$is_settings_enabled = Helper::instance()->get_app_settings( 'cache_bb_forum_discussions', 'buddyboss-app' );
				break;
			case 'bp-activity':
				$is_settings_enabled = Helper::instance()->get_app_settings( 'cache_bb_activity_feeds', 'buddyboss-app' );
				break;
			case 'bp-notifications':
				$is_settings_enabled = Helper::instance()->get_app_settings( 'cache_bb_notifications', 'buddyboss-app' );
				break;
			default:
				/**
				 * Filter to enable/disable API performance component for custom integrations.
				 *
				 * @param bool   $is_settings_enabled Settings.
				 * @param string $integration_name    Integration name.
				 *
				 * @since 2.0.80
				 * @return bool true/false.
				 */
				$is_settings_enabled = apply_filters( 'is_api_performance_component_enabled', false, $integration_name );
		}

		return isset( $is_component_active ) && isset( $is_settings_enabled ) && $is_component_active && $is_settings_enabled;
	}
}
