<?php
/**
 * Holds WordPress API functionality.
 *
 * @package BuddyBossApp\Api\WpCore\V2
 */

namespace BuddyBossApp\Api\WpCore\V2;

use WP_Comment;
use WP_REST_Request;
use WP_REST_Response;

if ( ! function_exists( 'add_action' ) ) {
	die( "Sorry, you can't access this directly - Security established" );
}

/**
 * Wp rest api class.
 */
class RestAPI {

	/**
	 * Class variable.
	 *
	 * @var $c
	 */
	public $c; // Contain the classes.

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return RestAPI
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Load hooks.
	 */
	public function load() {
		$this->c = new \stdClass();

		// Register all hooks.
		$this->hooks();
	}

	/**
	 * Register all hooks
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'load_rest' ), 30 );
		add_action( 'rest_prepare_comment', array( $this, 'rest_prepare_comment' ), 10, 3 );
	}

	/**
	 * Prepare comment response.
	 *
	 * @param WP_REST_Response $response Rest response.
	 * @param WP_Comment       $comment  Wp comment.
	 * @param WP_REST_Request  $request  Rest request.
	 *
	 * @return mixed
	 */
	public function rest_prepare_comment( $response, $comment, $request ) {
		$response->data['can_delete'] = $this->check_comment_permission( $comment );

		return $response;
	}

	/**
	 * Checks if a comment can be edited or deleted.
	 *
	 * @since 4.7.0
	 *
	 * @param WP_Comment $comment Comment object.
	 *
	 * @return bool Whether the comment can be edited or deleted.
	 */
	protected function check_comment_permission( $comment ) {
		if ( 0 === (int) get_current_user_id() ) {
			return false;
		}

		if ( current_user_can( 'moderate_comments' ) ) {
			return true;
		}

		return current_user_can( 'edit_comment', $comment->comment_ID );
	}

	/**
	 * Load Rest APIs.
	 */
	public function load_rest() {
		// Bail early if no core rest support.
		if ( ! class_exists( 'WP_REST_Controller' ) ) {
			return;
		}

		$controller = new BuddyBossApp_REST_Comments_Controller();
		$controller->register_routes();

		$controller = new BuddyBossApp_REST_Post_Archive_Controller();
		$controller->register_routes();

		$controller = new BuddyBossApp_REST_Related_Post_Controller( 'post' );
		$controller->register_routes();

		BuddyBossApp_REST_Post_Controller::instance();

		$controller = new BuddyBossApp_REST_Users_Controller();
		$controller->register_routes();
	}
}
