<?php
/**
 * Holds TutorLMS API common function.
 *
 * @package BuddyBossApp\Api\TutorLMS
 */

use BuddyBossApp\Api\TutorLMS\Main;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

if ( ! function_exists( 'bbapp_tutor_response_add_total_headers' ) ) {
	/**
	 * Set headers to let the Client Script be aware of the pagination.
	 *
	 * @param WP_REST_Response $response The response data.
	 * @param integer          $total    The total number of found items.
	 * @param integer          $per_page The number of items per page of results.
	 *
	 * @since 2.2.80
	 * @return WP_REST_Response $response The response data.
	 */
	function bbapp_tutor_response_add_total_headers( WP_REST_Response $response, $total = 0, $per_page = 0 ) {
		if ( ! $total || ! $per_page ) {
			return $response;
		}

		$total_items = (int) $total;
		$max_pages   = ceil( $total_items / (int) $per_page );

		$response->header( 'X-WP-Total', $total_items );
		$response->header( 'X-WP-TotalPages', (int) $max_pages );

		return $response;
	}
}

if ( ! function_exists( 'bbapp_tutor_fix_relative_urls_protocol' ) ) {
	/**
	 * Fix the relative content urls protocol.
	 * E.g., Changes //example.com to https://example.com
	 *
	 * @param string $content Post content.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	function bbapp_tutor_fix_relative_urls_protocol( $content ) {
		$content = preg_replace( '/(src|href)=["|\']\/\/(.*?)["|\']/', '$1="https://$2"', $content );

		return trim( $content );
	}
}

if ( ! function_exists( 'bbapp_tutor_course_nav_items' ) ) {

	/**
	 * Get the course nav items.
	 *
	 * @param int $course_id The course ID.
	 * @param int $user_id   The user ID.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	function bbapp_tutor_course_nav_items( $course_id, $user_id = 0 ) {
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$app_tutor_instance = Main::instance();
		$course_nav_items   = $app_tutor_instance->course_nav_items( $course_id, $user_id );

		if ( function_exists( 'TUTOR_GB' ) ) {
			// Add the grade book nav item.
			$addon_config = tutor_utils()->get_addon_config( TUTOR_GB()->basename );
			$is_enable    = (bool) tutor_utils()->array_get( 'is_enable', $addon_config );

			if ( $is_enable ) {
				$course_nav_items = $app_tutor_instance->grade_book_add_course_nav_item( $course_nav_items, $course_id );
			}
		}

		if ( function_exists( 'TUTOR_CA' ) ) {
			// Add the course attachments nav item.
			$addon_config = tutor_utils()->get_addon_config( TUTOR_CA()->basename );
			$is_enable    = (bool) tutor_utils()->avalue_dot( 'is_enable', $addon_config );

			if ( $is_enable ) {
				$course_nav_items = $app_tutor_instance->course_attachments_course_nav_item( $course_nav_items, $course_id, $user_id );
			}
		}

		if ( function_exists( 'TUTOR_GC' ) ) {
			// Add the Google Classroom stream tab nav item.
			$addon_config = tutor_utils()->get_addon_config( TUTOR_GC()->basename );
			$is_enable    = (bool) tutor_utils()->avalue_dot( 'is_enable', $addon_config );

			if ( $is_enable ) {
				$course_nav_items = $app_tutor_instance->google_classroom_stream_tab( $course_nav_items, $course_id );
			}
		}

		// Enable/Disable course nav items.
		$course_nav_items = $app_tutor_instance->enable_disable_course_nav_items( $course_nav_items, $course_id, $user_id );

		/**
		 * Filter the course nav items.
		 *
		 * @param array $course_nav_items The course nav items.
		 * @param int   $course_id        The course ID.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_course_nav_items', $course_nav_items, $course_id );
	}
}
