<?php
/**
 * Holds the user notifications functionality.
 *
 * @package BuddyBoss\App\Api\TutorLMS\V1
 */

namespace BuddyBossApp\Api\TutorLMS\V1;

use BuddyBossApp\Api\TutorLMS\V1\Core\TutorRestController;
use BuddyBossApp\RestErrors;
use TUTOR_NOTIFICATIONS\Utils;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Holds the user notifications functionality.
 *
 * @since 2.2.80
 */
class UserNotifications extends TutorRestController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * UserNotifications instance.
	 *
	 * @since 2.2.80
	 * @return UserNotifications
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 2.2.80
	 */
	public function __construct() {
		$this->rest_base = 'notifications';

		parent::__construct();
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/mark-all-read',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'mark_all_read' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>\d+)/read',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'read' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/mark-all-unread',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'mark_all_unread' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Check if a given request has access to notification items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_items_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = RestErrors::instance()->user_not_logged_in();
		}

		if ( ! $this->load_notification_util_class() instanceof Utils ) {
			$retval = new WP_Error( 'notification_addon_disabled', __( 'Tutor Notifications addon is not enabled.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		/**
		 * Filter the notification `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_notifications_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve User Notifications.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/notifications Get User Notifications
	 * @apiName        GetUserNotification
	 * @apiGroup       Tutor LMS User Notification
	 * @apiDescription Retrieve User Notifications
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 */
	public function get_items( $request ) {
		$user_notifications = $this->load_notification_util_class()->get_all_notifications_by_current_user();

		/**
		 * Filters user notifications for Tutor LMS.
		 *
		 * @param array           $user_notifications User Notifications.
		 * @param WP_REST_Request $request            The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$retval = apply_filters( 'bbapp_tutor_get_notifications', $user_notifications, $request );

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after a list of notification response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_user_notification_items_response', $response, $request );

		return $response;
	}

	/**
	 * Mark all notifications as read.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response
	 * @api            {POST} /wp-json/buddyboss-app/tutor/v1/notifications/mark-all-read Mark All Read
	 * @apiName        MarkAllUserNotificationsRead
	 * @apiGroup       Tutor LMS User Notification
	 * @apiDescription Marl All User Notifications as Read
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 */
	public function mark_all_read( $request ) {
		$read_notifications = $this->load_notification_util_class()->mark_all_notifications_as_read();
		$data               = array(
			'success'            => true,
			'read_notifications' => $read_notifications,
		);

		$response = rest_ensure_response( $data );

		/**
		 * Fires after a read all notifications response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_user_notification_items_read_response', $response, $request );

		return $response;
	}

	/**
	 * Mark single notification as read.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response
	 * @api            {POST} /wp-json/buddyboss-app/tutor/v1/notifications/:id/read Mark Notification Read
	 * @apiName        MarkUserNotificationsRead
	 * @apiGroup       Tutor LMS User Notification
	 * @apiDescription Marl User Notifications as Read
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 */
	public function read( $request ) {
		$id = $request->get_param( 'id' );

		if ( ! $id ) {
			return rest_ensure_response( new WP_Error( 'notification_id_not_found', __( 'Notification ID not found.', 'buddyboss-app' ), array( 'status' => 404 ) ) );
		}

		$read_notification = $this->load_notification_util_class()->mark_single_notification_as_read( $id );
		$data              = array(
			'success'            => true,
			'read_notifications' => $read_notification,
		);

		$response = rest_ensure_response( $data );

		/**
		 * Fires after a read all notifications response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_user_notification_item_read_response', $response, $request );

		return $response;
	}

	/**
	 * Mark all notifications as unread.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response
	 * @api            {POST} /wp-json/buddyboss-app/tutor/v1/notifications/mark-all-unread Mark All Read
	 * @apiName        MarkAllUserNotificationsUnread
	 * @apiGroup       Tutor LMS User Notification
	 * @apiDescription Marl All User Notifications as Unread
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 */
	public function mark_all_unread( $request ) {
		$unread_notifications = $this->load_notification_util_class()->mark_all_notifications_as_unread();
		$data                 = array(
			'success'              => true,
			'unread_notifications' => $unread_notifications,
		);

		$response = rest_ensure_response( $data );

		/**
		 * Fires after an unread all notifications response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_user_notification_items_unread_response', $response, $request );

		return $response;
	}

	/**
	 * Load notification util class
	 *
	 * @since 2.2.80
	 * @return Utils | array
	 */
	public function load_notification_util_class() {
		$notification_util_cls_file = WP_PLUGIN_DIR . '/tutor-pro/addons/tutor-notifications/classes/Utils.php';
		$notification_util_cls      = array();

		if ( file_exists( $notification_util_cls_file ) && bbapp_is_tutor_lms_plugins_active() && tutils()->is_addon_enabled( TUTOR_NOTIFICATIONS()->basename ) ) {
			require_once $notification_util_cls_file;

			$notification_util_cls = new Utils();
		}

		return $notification_util_cls;
	}
}
