<?php
/**
 * Holds TutorLMS settings.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1
 */

namespace BuddyBossApp\Api\TutorLMS\V1;

use BuddyBossApp\Api\TutorLMS\V1\Core\TutorRestController;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * TutorLMS settings.
 */
class Settings extends TutorRestController {
	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Settings rest instance.
	 *
	 * @since 2.2.80
	 * @return Settings
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/settings',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_settings' ),
					'permission_callback' => array( $this, 'get_settings_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Check if a given request has access to settings.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_settings_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the settings permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_settings_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve Settings.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since          2.2.80
	 *
	 * @return WP_REST_Response
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/settings Get TutorLMS Settings
	 * @apiName        GetTutorSettings
	 * @apiGroup       Tutor Settings
	 * @apiDescription Retrieve Settings
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 */
	public function get_settings( $request ) {
		$settings = get_option( 'tutor_option', array() );

		if ( ! is_user_logged_in() ) {
			$setting_keys = array(
				'enable_course_about',
				'enable_course_level',
				'enable_course_share',
				'course_archive_filter',
				'course_retake_feature',
				'course_content_summary',
				'enable_course_material',
				'enable_course_duration',
				'enable_course_progress_bar',
				'enable_course_benefits',
				'enable_course_description',
				'enable_course_update_date',
				'enable_course_requirements',
				'display_course_instructors',
				'quiz_previous_button_enabled',
				'enable_course_total_enrolled',
				'enable_course_target_audience',
				'student_must_login_to_view_course',
				'quiz_permalink_base',
				'course_permalink_base',
				'lesson_permalink_base',
				'assignment_permalink_base',
				'supported_course_filters',
			);

			$setting_arr = array();

			foreach ( $setting_keys as $key ) {
				$setting_arr[ $key ] = $settings[ $key ] ?? null;
			}
		} else {
			$setting_arr = $settings;
		}

		/**
		 * Filter the settings.
		 *
		 * @param array           $settings Settings.
		 * @param WP_REST_Request $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$settings = apply_filters( 'bbapp_tutor_settings', $setting_arr, $request );

		return rest_ensure_response( $settings );
	}
}
