<?php
/**
 * Holds rest functionality for rest.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Quiz
 */

namespace BuddyBossApp\Api\TutorLMS\V1\Quiz;

use BuddyBossApp\Api\TutorLMS\V1\Core\TutorRestController;
use Tutor\Models\QuizModel;
use WP_Error;
use WP_Post;
use WP_Query;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Quiz rest class.
 */
class QuizRest extends TutorRestController {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Quiz post type.
	 *
	 * @var string $post_type
	 */
	protected $post_type = 'tutor_quiz';

	/**
	 * QuizRest instance.
	 *
	 * @since 2.2.80
	 * @return QuizRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 2.2.80
	 */
	public function __construct() {
		$this->rest_base = 'quiz';
		parent::__construct();
	}

	/**
	 * Check if a given request has access to Quiz items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_items_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the Quiz `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_quizzes_permissions_check', $retval, $request );
	}

	/**
	 * Check if a given request has access to quiz item.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_item_permissions_check( $request ) {
		$retval = true;

		/**
		 * Filter the quiz `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_quiz_permissions_check', $retval, $request );
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}


	/**
	 * Retrieve Quizzes.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since 2.2.80
	 *
	 * @return WP_REST_Response
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/quiz Get TutorLMS Quizzes
	 * @apiName        GetTutorQuizzes
	 * @apiGroup       Tutor Quizzes
	 * @apiDescription Retrieve Quizzes
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} [page] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {String} [search] Limit results to those matching a string.
	 * @apiParam {Array} [exclude] Ensure result set excludes specific IDs.
	 * @apiParam {Array} [include] Ensure a result set includes specific IDs.
	 * @apiParam {String} [after] Limit results to those published after a given ISO8601 compliant date.
	 * @apiParam {String} [before] Limit results to those published before a given ISO8601 compliant date.
	 * @apiParam {Array} [author] Limit results to those assigned to specific authors.
	 * @apiParam {Array} [author_exclude] Ensure results to exclude those assigned to specific authors.
	 * @apiParam {String=asc,desc} [order=desc] Sort result set by given order.
	 * @apiParam {String=date,id,title,menu_order} [orderby=date] Sort result set by given field.
	 * @apiParam {Array} [parent] Limit results to those assigned to specific parent.
	 */
	public function get_items( $request ) {
		$registered = $this->get_collection_params();

		/**
		 * Filter the request.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$request = apply_filters( 'bbapp_tutor_get_quizzes_request', $request );

		/**
		 * This array defines mappings between public API query parameters whose
		 * values are accepted as-passed, and their internal WP_Query parameter
		 * name equivalents (some are the same). Only values which are also
		 * present in $registered will be set.
		 */
		$parameter_mappings = array(
			'author'         => 'author__in',
			'author_exclude' => 'author__not_in',
			'exclude'        => 'post__not_in',
			'include'        => 'post__in',
			'offset'         => 'offset',
			'order'          => 'order',
			'orderby'        => 'orderby',
			'page'           => 'paged',
			'parent'         => 'post_parent__in',
			'parent_exclude' => 'post_parent__not_in',
			'search'         => 's',
			'slug'           => 'post_name__in',
			'status'         => 'post_status',
			'per_page'       => 'posts_per_page',
		);

		/**
		 * For each known parameter that is both registered and present in the request,
		 * set the parameter's value on the query $args.
		 */
		foreach ( $parameter_mappings as $api_param => $wp_param ) {
			if ( isset( $registered[ $api_param ], $request[ $api_param ] ) ) {
				$args[ $wp_param ] = $request[ $api_param ];
			} elseif ( isset( $registered[ $api_param ]['default'] ) ) {
				$args[ $wp_param ] = $registered[ $api_param ]['default'];
			}
		}

		// Check for & assign any parameters that require special handling or setting.
		$args['date_query'] = array();

		// Set before into a date query. Date query must be specified as an array of an array.
		if ( isset( $registered['before'], $request['before'] ) ) {
			$args['date_query'][0]['before'] = $request['before'];
		}

		// Set after into a date query. Date query must be specified as an array of an array.
		if ( isset( $registered['after'], $request['after'] ) ) {
			$args['date_query'][0]['after'] = $request['after'];
		}

		$args = $this->prepare_items_query( $args, $request );

		/**
		 * Filter the query arguments for the request.
		 *
		 * @param array           $args    Key value array of query var to query value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$args = apply_filters( 'bbapp_tutor_get_quizzes_args', $args, $request );

		$args['post_type'] = $this->post_type;

		$posts_query            = new WP_Query();
		$quizzes['posts']       = $posts_query->query( $args );
		$quizzes['total_posts'] = $posts_query->found_posts;

		// Todo::dips Check again this code needed or not.
		if ( $quizzes['total_posts'] < 1 ) {
			// Out-of-bounds, run the query again without LIMIT for total count.
			unset( $args['paged'] );
			$count_query = new WP_Query();
			$count_query->query( $args );
			$quizzes['total_posts'] = $count_query->found_posts;
		}

		/**
		 * Fires list of Quizzes is fetched via Query.
		 *
		 * @param array            $quizzes Fetched Quizzes.
		 * @param WP_REST_Response $args    Query arguments.
		 * @param WP_REST_Request  $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		$quizzes = apply_filters( 'bbapp_tutor_get_quizzes', $quizzes, $args, $request );

		$retval = array();

		foreach ( $quizzes['posts'] as $quize ) {
			if ( ! $this->check_read_permission( $quize ) ) {
				continue;
			}

			$retval[] = $this->prepare_response_for_collection( $this->prepare_item_for_response( $quize, $request ) );
		}

		$response       = rest_ensure_response( $retval );
		$total_posts    = ! empty( $quizzes['total_posts'] ) ? $quizzes['total_posts'] : 0;
		$posts_per_page = ! empty( $args['posts_per_page'] ) ? $args['posts_per_page'] : 0;
		$response       = bbapp_tutor_response_add_total_headers( $response, $total_posts, $posts_per_page );

		/**
		 * Fires after a list of quizzes response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_quiz_items_response', $response, $request );

		return $response;
	}

	/**
	 * Retrieve Quiz.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since 2.2.80
	 *
	 * @return WP_REST_Response|WP_Error
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/quiz/:id Get TutorLMS Topic
	 * @apiName        GetTutorQuiz
	 * @apiGroup       Tutor Quizzes
	 * @apiDescription Retrieve single Quiz
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Quiz.
	 */
	public function get_item( $request ) {
		$quiz_id = is_numeric( $request ) ? $request : (int) $request['id'];
		$quiz    = get_post( $quiz_id );

		if ( empty( $quiz ) || $this->post_type !== $quiz->post_type ) {
			return QuizError::instance()->invalid_quiz_id();
		}

		/**
		 * Fire after quiz is fetched via Query.
		 *
		 * @param array           $quiz    Fetched Quiz.
		 * @param WP_REST_Request $quiz_id Quiz id.
		 *
		 * @since 2.2.80
		 */
		$quiz = apply_filters( 'bbapp_tutor_get_quiz', $quiz, $quiz_id );

		$retval   = $this->prepare_response_for_collection( $this->prepare_item_for_response( $quiz, $request ) );
		$response = rest_ensure_response( $retval );

		/**
		 * Fires after an quiz response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		do_action( 'bbapp_tutor_quiz_item_response', $response, $request );

		return $response;
	}

	/**
	 * Prepare a single post-output for response.
	 *
	 * @param WP_Post         $post    Post object.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 2.2.80
	 * @return WP_REST_Response $data
	 */
	public function prepare_item_for_response( $post, $request ) {
		$GLOBALS['post'] = $post; //phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited

		setup_postdata( $post );

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$schema  = $this->get_public_item_schema();
		// Base fields for every post.
		$data = array(
			'id'           => $post->ID,
			'title'        => array(
				'raw'      => $post->post_title,
				'rendered' => get_the_title( $post->ID ),
			),
			'content'      => array(
				'raw'      => bbapp_tutor_fix_relative_urls_protocol( $post->post_content ),
				'rendered' => bbapp_tutor_fix_relative_urls_protocol( apply_filters( 'the_content', $post->post_content ) ),
			),
			'date'         => mysql_to_rfc3339( $post->post_date ),
			'date_gmt'     => mysql_to_rfc3339( $post->post_date_gmt ),
			'modified'     => mysql_to_rfc3339( $post->post_modified ),
			'modified_gmt' => mysql_to_rfc3339( $post->post_modified_gmt ),
			'link'         => get_permalink( $post->ID ),
			'slug'         => $post->post_name,
			'author'       => (int) $post->post_author,
			'excerpt'      => array(
				'raw'      => bbapp_tutor_fix_relative_urls_protocol( $post->post_excerpt ),
				'rendered' => bbapp_tutor_fix_relative_urls_protocol( apply_filters( 'the_excerpt', $post->post_excerpt ) ),
			),
			'menu_order'   => (int) $post->menu_order,
		);

		/**
		 * Feature Media.
		 */
		$post->featured_media            = $this->get_feature_media( $post );
		$data['featured_media']          = array();
		$data['featured_media']['small'] = ( is_array( $post->featured_media ) && isset( $post->featured_media['small'] ) ) ? $post->featured_media['small'] : null;
		$data['featured_media']['large'] = ( is_array( $post->featured_media ) && isset( $post->featured_media['large'] ) ) ? $post->featured_media['large'] : null;

		if ( ! empty( $schema['properties']['course'] ) && in_array( $context, $schema['properties']['course']['context'], true ) ) {
			$post->course   = $this->get_course_id( $post );
			$data['course'] = (int) $post->course;
		}

		if ( ! empty( $schema['properties']['topic'] ) && in_array( $context, $schema['properties']['topic']['context'], true ) ) {
			$post->topic   = $this->get_topic_id( $post );
			$data['topic'] = (int) $post->topic;
		}

		if ( ! empty( $schema['properties']['quiz_activity'] ) && in_array( $context, $schema['properties']['quiz_activity']['context'], true ) ) {
			$post->quiz_activity   = $this->get_quiz_activity( $post );
			$data['quiz_activity'] = $post->quiz_activity;
		}

		if ( ! empty( $schema['properties']['quiz_info'] ) && in_array( $context, $schema['properties']['quiz_info']['context'], true ) ) {
			$post->quiz_info   = $this->get_quiz_info( $post );
			$data['quiz_info'] = $post->quiz_info;
		}

		if ( ! empty( $schema['properties']['quiz_question_ans'] ) && in_array( $context, $schema['properties']['quiz_question_ans']['context'], true ) ) {
			$post->quiz_question_ans   = $this->quiz_question_ans( $post );
			$data['quiz_question_ans'] = $post->quiz_question_ans;
		}

		if ( ! empty( $schema['properties']['settings'] ) && in_array( $context, $schema['properties']['settings']['context'], true ) ) {
			$post->settings   = $this->get_settings( $post );
			$data['settings'] = $post->settings;
		}

		if ( ! empty( $schema['properties']['error_message'] ) && in_array( $context, $schema['properties']['error_message']['context'], true ) ) {
			$post->error_message = $this->get_error_message( $post );

			if ( ! empty( $post->error_message ) ) {
				$error_code            = $post->error_message->get_error_code();
				$data['error_message'] = array(
					'code'    => $error_code,
					'message' => $post->error_message->get_error_message(),
					'data'    => $post->error_message->get_error_data( $error_code ),

				);
			} else {
				$data['error_message'] = array();
			}
		}

		$data = $this->add_additional_fields_to_object( $data, $request );
		$data = $this->filter_response_by_context( $data, $context );

		// Wrap the data in a response object.
		$response = rest_ensure_response( $data );

		$response->add_links( $this->prepare_links( $data ) );

		/**
		 * Filters quiz response.
		 *
		 * @type WP_REST_Response $response Quiz response.
		 * @type WP_Post          $post     Post object.
		 * @type WP_REST_Request  $request  Request object.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_rest_prepare_quiz', $response, $post, $request );
	}

	/**
	 * Prepare items.
	 *
	 * @param array $prepared_args query parameters.
	 * @param null  $request       Request parameters.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function prepare_items_query( $prepared_args = array(), $request = null ) {
		$query_args = array();

		foreach ( $prepared_args as $key => $value ) {
			/**
			 * Filters the query_vars used in get_items() for the constructed query.
			 *
			 * The dynamic portion of the hook name, `$key`, refers to the query_var key.
			 *
			 * @param string $value The query_var value.
			 *
			 * @since 2.2.80
			 */
			$query_args[ $key ] = apply_filters( "rest_query_var-{$key}", $value ); // phpcs:ignore WordPress.NamingConventions.ValidHookName.UseUnderscores
		}

		$query_args['ignore_sticky_posts'] = true;

		// Map to proper WP_Query orderby param.
		if ( isset( $query_args['orderby'] ) && isset( $request['orderby'] ) ) {
			$orderby_mappings = array(
				'id'            => 'ID',
				'include'       => 'post__in',
				'slug'          => 'post_name',
				'include_slugs' => 'post_name__in',
			);

			if ( isset( $orderby_mappings[ $request['orderby'] ] ) ) {
				$query_args['orderby'] = $orderby_mappings[ $request['orderby'] ];
			}
		}

		return $query_args;
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param array $data item object.
	 *
	 * @since 2.2.80
	 * @return array Links for the given data.
	 */
	public function prepare_links( $data ) {
		$links = parent::prepare_links( $data );

		// Todo::dips add filter here for course/lesson/topic.

		return $links;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['after'] = array(
			'description'       => __( 'Limit response to resources published after a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['before'] = array(
			'description'       => __( 'Limit response to resources published before a given ISO8601 compliant date.', 'buddyboss-app' ),
			'type'              => 'string',
			'format'            => 'date-time',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author'] = array(
			'description'       => __( 'Limit result set to posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['author_exclude'] = array(
			'description'       => __( 'Ensure result set excludes posts assigned to specific authors.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);
		$params['include'] = array(
			'description'       => __( 'Limit result set to specific ids.', 'buddyboss-app' ),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		$params['offset']            = array(
			'description'       => __( 'Offset the result set by a specific number of items.', 'buddyboss-app' ),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['order']             = array(
			'description'       => __( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'desc',
			'enum'              => array( 'asc', 'desc' ),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']           = array(
			'description'       => __( 'Sort collection by object attribute.', 'buddyboss-app' ),
			'type'              => 'string',
			'default'           => 'date',
			'enum'              => array(
				'date',
				'id',
				'include',
				'title',
				'slug',
				'relevance',
			),
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['orderby']['enum'][] = 'menu_order';

		$params['parent']         = array(
			'description'       => __( 'Limit result set to those of particular parent IDs.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);
		$params['parent_exclude'] = array(
			'description'       => __( 'Limit result set to all items except those of a particular parent ID.', 'buddyboss-app' ),
			'type'              => 'array',
			'sanitize_callback' => 'wp_parse_id_list',
			'items'             => array( 'type' => 'integer' ),
		);

		$params['slug']   = array(
			'description'       => __( 'Limit result set to posts with a specific slug.', 'buddyboss-app' ),
			'type'              => 'string',
			'validate_callback' => 'rest_validate_request_arg',
		);
		$params['filter'] = array(
			'description' => __( 'Use WP Query arguments to modify the response; private query vars require appropriate authorization.', 'buddyboss-app' ),
		);

		return $params;
	}

	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'quiz',
			'type'       => 'object',
			'properties' => array(
				'id'             => array(
					'description' => __( 'Unique identifier for the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'title'          => array(
					'description' => __( 'The title for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Title for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML title for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'content'        => array(
					'description' => __( 'The content for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Content for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML content for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit' ),
						),
					),
				),
				'date'           => array(
					'description' => __( 'The date the object was published, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'date_gmt'       => array(
					'description' => __( 'The date the object was published, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'modified'       => array(
					'description' => __( 'The date the object was last modified, in the site\'s timezone.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'modified_gmt'   => array(
					'description' => __( 'The date the object was last modified, as GMT.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'date-time',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'link'           => array(
					'description' => __( 'URL to the object.', 'buddyboss-app' ),
					'type'        => 'string',
					'format'      => 'uri',
					'context'     => array( 'view', 'edit', 'embed' ),
					'readonly'    => true,
				),
				'slug'           => array(
					'description' => __( 'An alphanumeric identifier for the object unique to its type.', 'buddyboss-app' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit', 'embed' ),
					'arg_options' => array(
						'sanitize_callback' => 'sanitize_title',
					),
				),
				'author'         => array(
					'description' => __( 'The ID for the author of the object.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'excerpt'        => array(
					'description' => __( 'The excerpt for the object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit', 'embed' ),
					'properties'  => array(
						'raw'      => array(
							'description' => __( 'Excerpt for the object, as it exists in the database.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'edit' ),
						),
						'rendered' => array(
							'description' => __( 'HTML excerpt for the object, transformed for display.', 'buddyboss-app' ),
							'type'        => 'string',
							'context'     => array( 'view', 'edit', 'embed' ),
						),
					),
				),
				'featured_media' => array(
					'description' => __( 'Feature media object containing thumb and full URL of image.', 'buddyboss-app' ),
					'type'        => 'array',
					'context'     => array( 'view', 'edit', 'embed' ),
				),
				'menu_order'     => array(
					'description' => __( 'The order of the object in relation to other object of its type.', 'buddyboss-app' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
			),
		);

		$schema['properties']['course'] = array(
			'description' => __( 'The Course id for the object', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['topic'] = array(
			'description' => __( 'The Topic id for the object.', 'buddyboss-app' ),
			'type'        => 'integer',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['settings'] = array(
			'description' => __( 'Settings for object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view' ),
		);

		$schema['properties']['quiz_activity'] = array(
			'description' => __( 'Quiz resume activity for the object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view' ),
		);

		$schema['properties']['quiz_info'] = array(
			'description' => __( 'Quiz info for the object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view' ),
		);

		$schema['properties']['quiz_question_ans'] = array(
			'description' => __( 'Quiz question and answers for the object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view' ),
		);

		$schema['properties']['error_message'] = array(
			'description' => __( 'Error message for this object.', 'buddyboss-app' ),
			'type'        => 'array',
			'context'     => array( 'view', 'edit', 'embed' ),
		);

		$schema['properties']['materials'] = array(
			'description' => __( 'Materials for the object.', 'buddyboss-app' ),
			'type'        => 'string',
			'context'     => array( 'view', 'edit' ),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Get feature media.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function get_feature_media( $post ) {
		$return = array(
			'large' => null,
			'small' => null,
		);
		$large  = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'large' );
		$small  = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'medium' );

		if ( isset( $large[0] ) ) {
			$return['large'] = $large[0];
		}

		if ( isset( $small[0] ) ) {
			$return['small'] = $small[0];
		}

		return $return;
	}

	/**
	 * Get course id.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @since 2.2.80
	 * @return int
	 */
	protected function get_course_id( $post ) {
		$course_id = tutor_utils()->get_course_id_by_subcontent( $post->ID );

		if ( ! isset( $course_id ) ) {
			$course_id = 0;
		}

		return $course_id;
	}

	/**
	 * Get topic id.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @since 2.2.80
	 * @return int
	 */
	protected function get_topic_id( $post ) {
		$topic_id = $post->post_parent;

		if ( ! isset( $topic_id ) ) {
			$topic_id = 0;
		}

		return $topic_id;
	}

	/**
	 * Get quiz resume activity.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function get_quiz_activity( $post ) {
		$is_started_quiz        = tutor_utils()->is_started_quiz( $post->ID );
		$previous_attempts      = ( new QuizModel() )->quiz_attempts( $post->ID, get_current_user_id() );
		$attempted_count        = is_array( $previous_attempts ) ? count( $previous_attempts ) : 0;
		$feedback_mode          = tutor_utils()->get_quiz_option( $post->ID, 'feedback_mode', 0 );
		$attempts_allowed       = 'retry' !== $feedback_mode ? 1 : tutor_utils()->get_quiz_option( $post->ID, 'attempts_allowed', 0 );
		$attempt_remaining      = (int) $attempts_allowed - (int) $attempted_count;
		$previous_attempts_send = array();
		$attempt_ids            = ( ! empty( $previous_attempts ) && is_array( $previous_attempts ) ) ? array_column( $previous_attempts, 'attempt_id' ) : array();
		$answers_array          = QuizModel::get_quiz_answers_by_attempt_id( $attempt_ids, true );

		if ( ! empty( $previous_attempts ) ) {
			foreach ( $previous_attempts as $key => $attempt ) {
				$correct   = 0;
				$incorrect = 0;
				$answers   = $answers_array[ $attempt->attempt_id ] ?? array();

				if ( is_array( $answers ) && count( $answers ) > 0 ) {
					foreach ( $answers as $answer ) {
						if ( $answer->is_correct ) {
							++$correct;
						} elseif ( ! ( null === $answer->is_correct ) ) {
							++$incorrect;
						}
					}
				}

				$previous_attempts_send[ $key ]                   = $attempt;
				$previous_attempts_send[ $key ]->correct_answer   = $correct;
				$previous_attempts_send[ $key ]->incorrect_answer = $incorrect;
			}
		}

		return array(
			'is_started_quiz'   => null !== $is_started_quiz,
			'feedback_mode'     => $feedback_mode,
			'quiz_attempts'     => $previous_attempts_send,
			'attempted_count'   => $attempted_count,
			'attempts_allowed'  => $attempts_allowed,
			'attempt_remaining' => $attempt_remaining,
			'can_attempt'       => $attempt_remaining > 0 || 0 === $attempts_allowed,
		);
	}

	/**
	 * Get quiz resume activity.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function get_quiz_info( $post ) {
		$quiz_info                    = array();
		$time_limit                   = tutor_utils()->get_quiz_option( $post->ID, 'time_limit.time_value' );
		$passing_grade                = tutor_utils()->get_quiz_option( get_the_ID(), 'passing_grade', 0 );
		$quiz_info['total_questions'] = tutor_utils()->total_questions_for_student_by_quiz( $post->ID );
		$quiz_info['quiz_time']       = array();
		$quiz_info['passing_grade']   = $passing_grade;

		if ( $time_limit ) {
			$time_type           = tutor_utils()->get_quiz_option( $post->ID, 'time_limit.time_type' );
			$available_time_type = array(
				'seconds' => $time_limit > 1 ? __( 'Seconds', 'buddyboss-app' ) : __( 'Second', 'buddyboss-app' ),
				'minutes' => $time_limit > 1 ? __( 'Minutes', 'buddyboss-app' ) : __( 'Minute', 'buddyboss-app' ),
				'hours'   => $time_limit > 1 ? __( 'Hours', 'buddyboss-app' ) : __( 'Hour', 'buddyboss-app' ),
				'days'    => $time_limit > 1 ? __( 'Days', 'buddyboss-app' ) : __( 'Day', 'buddyboss-app' ),
				'weeks'   => $time_limit > 1 ? __( 'Weeks', 'buddyboss-app' ) : __( 'Week', 'buddyboss-app' ),
			);

			$quiz_info['quiz_time']['value']               = $time_limit;
			$quiz_info['quiz_time']['type']                = $time_type;
			$quiz_info['quiz_time']['available_time_type'] = $available_time_type;
		}

		return $quiz_info;
	}

	/**
	 * Get quiz question and answers.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	public function quiz_question_ans( $post ) {
		$questions_ans = array();

		if ( ! is_user_logged_in() ) {
			return $questions_ans;
		}

		$questions = tutor_utils()->get_random_questions_by_quiz( $post->ID );

		if ( $questions ) {
			foreach ( $questions as $question ) {
				$question_settings = maybe_unserialize( $question->question_settings );

				$question_data = array(
					'question_id'          => $question->question_id,
					'quiz_id'              => $question->quiz_id,
					'question_title'       => $question->question_title,
					'question_description' => $question->question_description,
					'question_type'        => $question->question_type,
					'question_mark'        => $question->question_mark,
					'question_order'       => $question->question_order,
					'question_settings'    => $question_settings,
				);

				$show_question_mark = (bool) tutor_utils()->avalue_dot( 'show_question_mark', $question_settings );
				$answer_required    = (bool) tutor_utils()->array_get( 'answer_required', $question_settings );

				$question_data['show_question_mark'] = $show_question_mark;
				$question_data['answer_required']    = $answer_required;

				$rand_choice = false;
				if ( 'single_choice' === $question->question_type || 'multiple_choice' === $question->question_type ) {
					$choice = maybe_unserialize( $question->question_settings );
					if ( isset( $choice['randomize_question'] ) ) {
						$rand_choice = 1 === $choice['randomize_question'];
					}
				}

				$answers_data = array();
				$answers      = QuizModel::get_answers_by_quiz_question( $question->question_id, $rand_choice );

				if ( $answers ) {
					foreach ( $answers as $answer ) {
						$answers_data[] = $this->get_answer_data( $answer );
					}
				}

				$questions_ans[] = array(
					'question'         => $question_data,
					'answers'          => $answers_data,
					'matching_answers' => 'matching' === $question->question_type ? QuizModel::get_answers_by_quiz_question(
						$question->question_id,
						$rand_choice
					) : array(),
				);
			}
		}

		return $questions_ans;
	}

	/**
	 * Get answer data.
	 *
	 * @param object $answer Answer object.
	 *
	 * @since 2.2.80
	 * @return array
	 */
	protected function get_answer_data( $answer ) {
		$choice_contexts = array(
			'true_false'      => 'radio',
			'single_choice'   => 'radio',
			'multiple_choice' => 'checkbox',
		);

		$answer_data = array(
			'answer_id'            => $answer->answer_id,
			'question_id'          => $answer->belongs_question_id,
			'question_type'        => $answer->belongs_question_type,
			'answer_title'         => $answer->answer_title,
			'is_correct'           => $answer->is_correct,
			'answer_two_gap_match' => $answer->answer_two_gap_match,
			'answer_view_format'   => $answer->answer_view_format,
			'answer_settings'      => $answer->answer_settings,
			'answer_order'         => $answer->answer_order,
		);

		if ( $answer->image_id ) {
			$answer_data['image_data'] = array(
				'image_id'  => $answer->image_id,
				'image_url' => esc_url( wp_get_attachment_image_url( $answer->image_id, 'full' ) ),
			);
		}

		if ( array_key_exists( $answer->belongs_question_type, $choice_contexts ) ) {
			$answer_data['choice_type'] = $choice_contexts[ $answer->belongs_question_type ];
		}

		return $answer_data;
	}

	/**
	 * Get settings.
	 *
	 * @param WP_Post $post quiz post object.
	 *
	 * @since 2.2.80
	 * @return mixed
	 */
	protected function get_settings( $post ) {
		$quiz_settings = tutor_utils()->get_quiz_option( $post->ID );
		if ( ! empty( $quiz_settings ) ) {
			return $quiz_settings;
		}

		return array();
	}

	/**
	 * Get an error message.
	 *
	 * @param WP_Post $post Post data.
	 *
	 * @since 2.2.80
	 * @return WP_Error|array
	 */
	public function get_error_message( $post ) {
		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_quiz_error', __( 'User not logged in.', 'buddyboss-app' ), array( 'status' => 401 ) );
		}

		return array();
	}
}
