<?php
/**
 * Holds rest functionality for quiz attempt.
 *
 * @package BuddyBossApp\Api\TutorLMS\V1\Quiz
 */

namespace BuddyBossApp\Api\TutorLMS\V1\Quiz;

use BuddyBossApp\Api\TutorLMS\V1\Core\TutorRestController;
use Tutor\Models\QuizModel;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Quiz rest class.
 */
class QuizAttemptRest extends TutorRestController {
	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * QuizRest instance.
	 *
	 * @since 2.2.80
	 * @return QuizAttemptRest
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 2.2.80
	 */
	public function __construct() {
		$this->rest_base = 'attempt';
		parent::__construct();
	}

	/**
	 * Register the component routes.
	 *
	 * @since 2.2.80
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/result',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_attempt_result' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Check if a given request has access to Quiz items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @since 2.2.80
	 * @return bool|WP_Error
	 */
	public function get_items_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = rest_ensure_response( QuizError::instance()->user_not_logged_in() );
		}

		/**
		 * Filter the Quiz `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_quiz_attempt_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve Quizzes.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since 2.2.80
	 *
	 * @return WP_REST_Response
	 * @api            {GET} /wp-json/buddyboss-app/tutor/v1/quiz/attempt/:id/result Get TutorLMS Quizzes
	 * @apiName        GetTutorQuizAttemptResult
	 * @apiGroup       Tutor Quizzes
	 * @apiDescription Retrieve Quizzes
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 */
	public function get_attempt_result( $request ) {
		$attempt_id = $request->get_param( 'id' );

		if ( empty( $attempt_id ) ) {
			return rest_ensure_response( QuizError::instance()->attempt_not_found() );
		}

		$attempt_data = tutor_utils()->get_attempt( $attempt_id );

		if ( empty( $attempt_data ) ) {
			return rest_ensure_response( QuizError::instance()->attempt_not_found() );
		}

		if ( get_current_user_id() !== (int) $attempt_data->user_id ) {
			return rest_ensure_response( QuizError::instance()->attempt_not_found() );
		}

		if ( ! function_exists( 'tutor_render_answer_list' ) ) {
			ob_start();
			tutor_load_template_from_custom_path(
				tutor()->path . '/views/quiz/attempt-details.php',
				array(
					'attempt_id'   => $attempt_id,
					'attempt_data' => $attempt_data,
					'user_id'      => get_current_user_id(),
					'context'      => 'frontend-dashboard-my-attempts',
				)
			);
			ob_get_clean();
		}

		$answers = QuizModel::get_quiz_answers_by_attempt_id( $attempt_id );
		$data    = array();

		if ( ! empty( $answers ) ) {
			foreach ( $answers as $key => $answer ) {
				$answer_status = 'wrong';

				if ( $answer->is_correct ) {
					$answer_status = 'correct';
				} elseif ( in_array(
					$answer->question_type,
					array(
						'open_ended',
						'short_answer',
						'image_answering',
					),
					true
				) ) {
					$answer_status = null === $answer->is_correct ? 'pending' : 'wrong';
				}

				$data[ $key ]['title']  = $answer->question_title;
				$data[ $key ]['status'] = $answer_status;
				$type                   = tutor_utils()->get_question_types( $answer->question_type );
				$data[ $key ]['type']   = array(
					'icon' => $type['icon'] ?? '',
					'name' => $type['name'] ?? '',
				);
				$data[ $key ]['result'] = $this->get_attempt_answer_result( $answer_status );

				$data[ $key ]['explanation']    = ( isset( $answer->answer_explanation ) ) ? wp_kses_post( wp_unslash( $answer->answer_explanation ) ) : '';
				$data[ $key ]['given_answer']   = $this->get_given_answer( $answer );
				$data[ $key ]['correct_answer'] = $this->get_correct_answer( $answer );
			}
		}

		$response = rest_ensure_response( $data );

		/**
		 * Fires after a list of quizzes response is prepared via the REST API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 *
		 * @since 2.2.80
		 */
		return apply_filters( 'bbapp_tutor_quiz_attempt_result_response', $response, $request );
	}

	/**
	 * Get answer given.
	 *
	 * @param object $answer Answer object.
	 *
	 * @since 2.2.80
	 * @return string
	 */
	public function get_given_answer( $answer ) {
		if ( ! isset( $answer ) || ! isset( $answer->question_type ) ) {
			return '';
		}

		$retval = '';

		ob_start();
		if ( 'single_choice' === $answer->question_type ) {
			$get_answers = tutor_utils()->get_answer_by_id( $answer->given_answer );
			tutor_render_answer_list( $get_answers );
		} elseif ( 'true_false' === $answer->question_type ) { // True false or single choice.
			$get_answers   = tutor_utils()->get_answer_by_id( $answer->given_answer );
			$answer_titles = wp_list_pluck( $get_answers, 'answer_title' );
			$answer_titles = array_map( 'stripslashes', $answer_titles );

			echo '<span class="tutor-fs-7 tutor-fw-medium tutor-color-black">' .
			     implode( '</p><p>', $answer_titles ) .  //phpcs:ignore
				'</span>';
		} elseif ( 'multiple_choice' === $answer->question_type ) { // Multiple choice.
			$get_answers = tutor_utils()->get_answer_by_id( maybe_unserialize( $answer->given_answer ) );

			tutor_render_answer_list( $get_answers );
		} elseif ( 'fill_in_the_blank' === $answer->question_type ) { // Fill in the blank.
			$answer_titles              = maybe_unserialize( $answer->given_answer );
			$get_db_answers_by_question = QuizModel::get_answers_by_quiz_question( $answer->question_id );

			tutor_render_fill_in_the_blank_answer( $get_db_answers_by_question, $answer_titles ); //phpcs:ignore --contain safe data
		} elseif ( 'open_ended' === $answer->question_type || 'short_answer' === $answer->question_type ) { // Open ended or short answer.
			if ( $answer->given_answer ) {
				echo wp_kses(
					wpautop( stripslashes( $answer->given_answer ) ),
					array(
						'p'    => true,
						'span' => true,
					)
				);
			}
		} elseif ( 'ordering' === $answer->question_type ) { // Ordering.
			$ordering_ids = maybe_unserialize( $answer->given_answer );
			foreach ( $ordering_ids as $ordering_id ) {
				$get_answers = tutor_utils()->get_answer_by_id( $ordering_id );
				tutor_render_answer_list( $get_answers );
			}
		} elseif ( 'matching' === $answer->question_type ) { // Matching.
			$question_settings      = ! empty( $answer->question_settings ) ? maybe_unserialize( $answer->question_settings ) : array();
			$is_image_matching      = isset( $question_settings['is_image_matching'] ) && '1' === $question_settings['is_image_matching'];
			$ordering_ids           = maybe_unserialize( $answer->given_answer );
			$original_saved_answers = QuizModel::get_answers_by_quiz_question( $answer->question_id );
			$answers                = array();

			foreach ( $original_saved_answers as $key => $original_saved_answer ) {
				$provided_answer_order_id    = isset( $ordering_ids[ $key ] ) ? $ordering_ids[ $key ] : 0;
				$provided_answer_order_by_id = tutor_utils()->get_answer_by_id( $provided_answer_order_id );

				if ( tutor_utils()->count( $provided_answer_order_by_id ) ) {
					foreach ( $provided_answer_order_by_id as $provided_answer_order ) {
						if ( $is_image_matching ) {
							$original_saved_answer->answer_view_format   = 'text_image';
							$original_saved_answer->answer_title         = $provided_answer_order->answer_title;
							$original_saved_answer->answer_two_gap_match = '';
							$answers[]                                   = $original_saved_answer;
						} else {
							$original_saved_answer->answer_two_gap_match = $provided_answer_order->answer_two_gap_match;
							$answers[]                                   = $original_saved_answer;
						}
					}
				}
			}

			tutor_render_answer_list( $answers );
		} elseif ( 'image_matching' === $answer->question_type ) {
			$ordering_ids           = maybe_unserialize( $answer->given_answer );
			$original_saved_answers = QuizModel::get_answers_by_quiz_question( $answer->question_id );
			$answers                = array();

			foreach ( $original_saved_answers as $key => $original_saved_answer ) {
				$provided_answer_order_id   = $ordering_ids[ $key ] ?? 0;
				$provided_answer_order_data = tutor_utils()->get_answer_by_id( $provided_answer_order_id );

				foreach ( $provided_answer_order_data as $provided_answer_order ) {
					if ( $provided_answer_order->answer_title ) {
						$original_saved_answer->answer_view_format = 'text_image';
						$original_saved_answer->answer_title       = $provided_answer_order->answer_title;
						$answers[]                                 = $original_saved_answer;
					}
				}
			}

			tutor_render_answer_list( $answers );
		} elseif ( 'image_answering' === $answer->question_type ) {
			$ordering_ids = maybe_unserialize( $answer->given_answer );
			$answers      = array();

			foreach ( $ordering_ids as $answer_id => $image_answer ) {
				$db_answers = tutor_utils()->get_answer_by_id( $answer_id );

				foreach ( $db_answers as $db_answer ) {
					$db_answer->answer_title       = $image_answer;
					$db_answer->answer_view_format = 'text_image';
					$answers[]                     = $db_answer;
				}
			}

			tutor_render_answer_list( $answers );
		}

		$retval .= ob_get_clean();

		return $retval;
	}

	/**
	 * Get attempt answer result.
	 *
	 * @param object $answer Answer data.
	 *
	 * @return string
	 * @since 2.2.80
	 */
	public function get_correct_answer( $answer ) {
		global $wpdb;

		$retval = '';

		ob_start();
		if ( ( 'open_ended' !== $answer->question_type && 'short_answer' !== $answer->question_type ) ) {
			if ( 'true_false' === $answer->question_type ) {
				$correct_answer = $wpdb->get_var( $wpdb->prepare( "SELECT answer_title FROM {$wpdb->prefix}tutor_quiz_question_answers WHERE belongs_question_id = %d AND belongs_question_type='true_false' AND is_correct = 1", $answer->question_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

				echo '<span class="tutor-fs-7 tutor-fw-medium tutor-color-black">' .
					esc_html( $correct_answer ) .
					'</span>';
			} elseif ( 'single_choice' === $answer->question_type ) { // Single choice.
				$correct_answer = $wpdb->get_results( $wpdb->prepare( "SELECT answer_title, image_id, answer_view_format FROM {$wpdb->prefix}tutor_quiz_question_answers WHERE belongs_question_id = %d AND belongs_question_type='single_choice' AND is_correct = 1", $answer->question_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

				tutor_render_answer_list( $correct_answer );
			} elseif ( 'multiple_choice' === $answer->question_type ) { // Multiple choice.
				$correct_answer = $wpdb->get_results( $wpdb->prepare( "SELECT answer_title, image_id, answer_view_format FROM {$wpdb->prefix}tutor_quiz_question_answers WHERE belongs_question_id = %d AND belongs_question_type='multiple_choice' AND is_correct = 1 ;", $answer->question_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

				tutor_render_answer_list( $correct_answer );
			} elseif ( 'fill_in_the_blank' === $answer->question_type ) { // Fill in the blanks.
				$correct_answer             = $wpdb->get_var( $wpdb->prepare( "SELECT answer_two_gap_match FROM {$wpdb->prefix}tutor_quiz_question_answers WHERE belongs_question_id = %d AND belongs_question_type='fill_in_the_blank'", $answer->question_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$answer_titles              = explode( '|', stripslashes( $correct_answer ) );
				$get_db_answers_by_question = QuizModel::get_answers_by_quiz_question( $answer->question_id );

				tutor_render_fill_in_the_blank_answer( $get_db_answers_by_question, $answer_titles );
			} elseif ( 'ordering' === $answer->question_type ) { // Ordering.
				$correct_answer = $wpdb->get_results( $wpdb->prepare( "SELECT answer_title, image_id, answer_view_format FROM {$wpdb->prefix}tutor_quiz_question_answers WHERE belongs_question_id = %d AND belongs_question_type='ordering' ORDER BY answer_order ASC;", $answer->question_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

				foreach ( $correct_answer as $ans ) {
					tutor_render_answer_list( array( $ans ) );
				}
			} elseif ( 'matching' === $answer->question_type ) { // Matching.
				$question_settings = ( ! empty( $answer->question_settings ) ) ? maybe_unserialize( $answer->question_settings ) : array();
				$is_image_matching = isset( $question_settings['is_image_matching'] ) && '1' === $question_settings['is_image_matching'];
				$correct_answer    = $wpdb->get_results( $wpdb->prepare( "SELECT answer_title, image_id, answer_two_gap_match, answer_view_format FROM {$wpdb->prefix}tutor_quiz_question_answers WHERE belongs_question_id = %d AND belongs_question_type='matching' ORDER BY answer_order ASC;", $answer->question_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

				if ( $is_image_matching ) {
					array_map(
						function( $ans ) {
							$ans->answer_view_format   = 'text_image';
							$ans->answer_two_gap_match = '';
						},
						$correct_answer
					);
				}

				tutor_render_answer_list( $correct_answer );
			} elseif ( 'image_matching' === $answer->question_type ) { // Image matching.
				$correct_answer = $wpdb->get_results( $wpdb->prepare( "SELECT answer_title, image_id, answer_two_gap_match FROM {$wpdb->prefix}tutor_quiz_question_answers WHERE belongs_question_id = %d AND belongs_question_type='image_matching' ORDER BY answer_order ASC;", $answer->question_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

				tutor_render_answer_list( $correct_answer, true );
			} elseif ( 'image_answering' === $answer->question_type ) { // Image Answering.
				$correct_answer = $wpdb->get_results( $wpdb->prepare( "SELECT answer_title, image_id, answer_two_gap_match FROM {$wpdb->prefix}tutor_quiz_question_answers WHERE belongs_question_id = %d AND belongs_question_type='image_answering' ORDER BY answer_order ASC;", $answer->question_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

				! is_array( $correct_answer ) ? $correct_answer = array() : 0;

				echo '<div class="answer-image-matched-wrap">';
				foreach ( $correct_answer as $image_answer ) {
					?>
					<div class="image-matching-item">
						<p class="dragged-img-rap"><img alt="" src="<?php echo esc_url( wp_get_attachment_image_url( $image_answer->image_id ) ); ?>" /> </p>
						<p class="dragged-caption"><?php echo esc_html( $image_answer->answer_title ); ?></p>
					</div>
					<?php
				}
				echo '</div>';
			}
		}

		$retval .= ob_get_clean();

		return $retval;
	}

	/**
	 * Get attempt answer result.
	 *
	 * @param string $answer_status Answer status.
	 *
	 * @since 2.2.80
	 * @return string
	 */
	public function get_attempt_answer_result( $answer_status ) {
		switch ( $answer_status ) {
			case 'correct':
				$result = esc_html__( 'Correct', 'buddyboss-app' );
				break;
			case 'pending':
				$result = esc_html__( 'Pending', 'buddyboss-app' );
				break;
			case 'wrong':
				$result = esc_html__( 'Incorrect', 'buddyboss-app' );
				break;
			default:
				$result = '';
		}

		return $result;
	}
}
